package jp.co.sra.jun.goodies.spirodesign;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunSpiroDesignController class
 * 
 *  @author    m-asada
 *  @created   2006/03/28 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun676 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSpiroDesignController.java,v 8.10 2008/02/20 06:32:03 nisinaka Exp $
 */
public class JunSpiroDesignController extends JunAbstractController {
	protected transient Point previus;
	protected transient int dragType;

	protected static final int DRAG_NONE = 0;
	protected static final int DRAG_SPIRO_PEN = 1;
	protected static final int DRAG_CENTER_OF_MOON_CIRCLE = 2;
	protected static final int DRAG_CENTER_OF_TERA_CIRCLE = 3;
	protected static final int DRAG_RADIUS_OF_MOON_CIRCLE = 4;
	protected static final int DRAG_RADIUS_OF_TERA_CIRCLE = 5;
	protected static final int DRAG_SCROLL = 6;

	/**
	 * Create a new instance of <code>JunSpiroDesignController</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunSpiroDesignController() {
		super();
	}
	
	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StController#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		super.initialize();
		previus = null;
		dragType = DRAG_NONE;
	}

	/**
	 * Drag center of moon circle on the view.
	 * 
	 * @category dragging
	 */
	protected void dragCenterOfMoonCircle_(MouseEvent evt) {
		Point point = this.getView().convertViewPointToModelPoint_(evt.getPoint());
		if (previus.equals(point) == false) {
			this.getModel().centerOfMoonCircle_(new Jun2dPoint(point));
			previus = point;
		}
	}

	/**
	 * Drag center of tera circle on the view.
	 * 
	 * @category dragging
	 */
	protected void dragCenterOfTeraCircle_(MouseEvent evt) {
		Point point = this.getView().convertViewPointToModelPoint_(evt.getPoint());
		if (previus.equals(point) == false) {
			this.getModel().centerOfTeraCircle_(new Jun2dPoint(point));
			previus = point;
		}
	}

	/**
	 * Drag radius of moon circle on the view.
	 * 
	 * @category dragging
	 */
	protected void dragRadiusOfMoonCircle_(MouseEvent evt) {
		Point point = this.getView().convertViewPointToModelPoint_(evt.getPoint());
		if (previus.equals(point) == false) {
			this.getModel().radiusOfMoonCircle_(this.getModel().moonCircle().center().distance_(new Jun2dPoint(point)));
			previus = point;
		}
	}

	/**
	 * Drag radius of tera circle on the view.
	 * 
	 * @category dragging
	 */
	protected void dragRadiusOfTeraCircle_(MouseEvent evt) {
		Point point = this.getView().convertViewPointToModelPoint_(evt.getPoint());
		if (previus.equals(point) == false) {
			this.getModel().radiusOfTeraCircle_(this.getModel().teraCircle().center().distance_(new Jun2dPoint(point)));
			previus = point;
		}
	}

	/**
	 * Drag spiro pen on the view.
	 * 
	 * @category dragging
	 */
	protected void dragScroll_(MouseEvent evt) {
		Point currentPoint = evt.getPoint();
		if (currentPoint.equals(previus) == false) {
			this.getView().scrollBy_(new Point(currentPoint.x - previus.x, currentPoint.y - previus.y));
			previus = currentPoint;
		}
	}

	/**
	 * Drag spiro pen on the view.
	 * 
	 * @category dragging
	 */
	protected void dragSpiroPen_(MouseEvent evt) {
		Point point = this.getView().convertViewPointToModelPoint_(evt.getPoint());
		if (previus.equals(point) == false) {
			this.getModel().pointOfSpiroPen_(new Jun2dPoint(point));
			previus = point;
		}
	}

	/**
	 * Answer the receiver's model as JunSpiroDesignAbstractModel.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesignAbstractModel
	 * @category model accessing
	 */
	public JunSpiroDesignAbstractModel getModel() {
		return (JunSpiroDesignAbstractModel) this.model();
	}

	/**
	 * Invoked when a mouse button has been pressed on the view.
	 * 
	 * @param evt java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent evt) {
		if (evt.isPopupTrigger() || evt.isMetaDown()) {
			super.mousePressed(evt);
			return;
		}
		
		if (this.spiroPenHasCursor()) {
			previus = this.getView().convertViewPointToModelPoint_(evt.getPoint());
			dragType = DRAG_SPIRO_PEN;
			return;
		}
		if (this.centerOfMoonCircleHasCursor()) {
			previus = this.getView().convertViewPointToModelPoint_(evt.getPoint());
			dragType = DRAG_CENTER_OF_MOON_CIRCLE;
			return;
		}
		if (this.centerOfTeraCircleHasCursor()) {
			previus = this.getView().convertViewPointToModelPoint_(evt.getPoint());
			dragType = DRAG_CENTER_OF_TERA_CIRCLE;
			return;
		}
		if (this.radiusOfMoonCircleHasCursor()) {
			previus = this.getView().convertViewPointToModelPoint_(evt.getPoint());
			dragType = DRAG_RADIUS_OF_MOON_CIRCLE;
			return;
		}
		if (this.radiusOfTeraCircleHasCursor()) {
			previus = this.getView().convertViewPointToModelPoint_(evt.getPoint());
			dragType = DRAG_RADIUS_OF_TERA_CIRCLE;
			return;
		}
		
		JunSpiroDesign spiroDesign = this.whichSpiroDesign();
		if (spiroDesign == null) {
			previus = evt.getPoint();
			dragType = DRAG_SCROLL;
			this.getView().toComponent().setCursor(JunCursors.Quarters2Cursor());
		} else {
			this.selectSpiroDesign_(spiroDesign);
		}
	}

	/**
	 * Invoked when a mouse button has been released on the view.
	 * 
	 * @param evt java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent evt) {
		if (evt.isPopupTrigger() || evt.isMetaDown()) {
			super.mousePressed(evt);
			return;
		}

		dragType = DRAG_NONE;
		previus = null;
		this.mouseMoved(evt);
	}

	/**
	 * Invoked when a mouse is dragged on the view.
	 *
	 * @param evt java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent evt) {
		switch(dragType) {
			case DRAG_SPIRO_PEN:
				this.dragSpiroPen_(evt);
				break;
			case DRAG_CENTER_OF_MOON_CIRCLE:
				this.dragCenterOfMoonCircle_(evt);
				break;
			case DRAG_CENTER_OF_TERA_CIRCLE:
				this.dragCenterOfTeraCircle_(evt);
				break;
			case DRAG_RADIUS_OF_MOON_CIRCLE:
				this.dragRadiusOfMoonCircle_(evt);
				break;
			case DRAG_RADIUS_OF_TERA_CIRCLE:
				this.dragRadiusOfTeraCircle_(evt);
				break;
			case DRAG_SCROLL:
				this.dragScroll_(evt);
				break;
			case DRAG_NONE:
				break;
		}
	}
	
	/**
	 * Invoked when a mouse is moved on the view.
	 *
	 * @param evt java.awt.event.MouseEvent 
	 * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseMoved(MouseEvent evt) {
		if (this.spiroPenHasCursor() || this.teraCircleHasCursor() || this.moonCircleHasCursor()) {
			this.getView().toComponent().setCursor(JunCursors.CrossCursor());
		} else {
			this.getView().toComponent().setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
		}
	}

	/**
	 * Select the spiro design and answer it.
	 * 
	 * @param spiroDesign jp.co.sra.jun.goodies.spirodesign.JunSpiroDesign
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesign
	 * @category selecting
	 */
	public JunSpiroDesign selectSpiroDesign_(JunSpiroDesign spiroDesign) {
		if (spiroDesign == null) {
			return null;
		}
		return this.getModel().selectSpiroDesign_(spiroDesign);
	}

	/**
	 * Which the spiro design and answer it.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesign
	 * @category selecting
	 */
	public JunSpiroDesign whichSpiroDesign() {
		if (this.getModel().isActiveSpiroDesignProcess()) {
			return null;
		}
		return this.getModel().whichSpiroDesign_(this.getView().convertViewPointToModelPoint_(this.cursorPoint()));
	}

	/**
	 * Answer true if the receiver's center of tera circle has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean centerOfTeraCircleHasCursor() {
		return this.indexOfTeraCircleMarks() == 0;
	}

	/**
	 * Answer true if the receiver's center of tera circle has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean centerOfMoonCircleHasCursor() {
		return this.indexOfMoonCircleMarks() == 0;
	}

	/**
	 * Answer true if the receiver's moon circle has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean moonCircleHasCursor() {
		return this.indexOfMoonCircleMarks() >= 0;
	}

	/**
	 * Answer true if the receiver's radius of moon circle has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean radiusOfMoonCircleHasCursor() {
		return this.indexOfMoonCircleMarks() > 0;
	}

	/**
	 * Answer true if the receiver's radius of tera circle has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean radiusOfTeraCircleHasCursor() {
		return this.indexOfTeraCircleMarks() > 0;
	}

	/**
	 * Answer true if the receiver's spiro pen has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean spiroPenHasCursor() {
		return this.indexOfSpiroPenMarks() >= 0;
	}

	/**
	 * Answer true if the receiver's tera circle has cursor, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean teraCircleHasCursor() {
		return this.indexOfTeraCircleMarks() >= 0;
	}

	/**
	 * Add the myself as a listener of the view.
	 * 
	 * @param newView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category view accessing
	 */
	protected void buildListener(StView newView) {
		Component aView = newView.toComponent();
		aView.addMouseListener(this);
		aView.addMouseMotionListener(this);
	}

	/**
	 * Answer the receiver's view as JunSpiroDesignView.
	 * 
	 * @return jp.co.sra.jun.goodies.spirodesign.JunSpiroDesignView
	 * @category view accessing
	 */
	public JunSpiroDesignView getView() {
		return (JunSpiroDesignView) this.view();
	}

	/**
	 * Answer the index of moon circle marks.
	 * 
	 * @return int
	 * @category private
	 */
	protected int indexOfMoonCircleMarks() {
		if (this.getModel().isActiveSpiroDesignProcess()) {
			return -1;
		}
		Point point = this.getView().convertViewPointToModelPoint_(this.cursorPoint());
		Rectangle[] marks = this.getModel().moonCircle().marks();
		if (marks == null || marks.length == 0) {
			return -1;
		}
		for (int i = 0; i < marks.length; i++) {
			if (marks[i].contains(point)) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Answer the index of spiro pen marks.
	 * 
	 * @return int
	 * @category private
	 */
	protected int indexOfSpiroPenMarks() {
		if (this.getModel().isActiveSpiroDesignProcess()) {
			return -1;
		}
		Point point = this.getView().convertViewPointToModelPoint_(this.cursorPoint());
		Rectangle[] marks = this.getModel().spiroPen().marks();
		if (marks == null || marks.length == 0) {
			return -1;
		}
		for (int i = 0; i < marks.length; i++) {
			if (marks[i].contains(point)) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Answer the index of tera circle marks.
	 * 
	 * @return int
	 * @category private
	 */
	protected int indexOfTeraCircleMarks() {
		if (this.getModel().isActiveSpiroDesignProcess()) {
			return -1;
		}
		Point point = this.getView().convertViewPointToModelPoint_(this.cursorPoint());
		Rectangle[] marks = this.getModel().teraCircle().marks();
		if (marks == null || marks.length == 0) {
			return -1;
		}
		for (int i = 0; i < marks.length; i++) {
			if (marks[i].contains(point)) {
				return i;
			}
		}
		return -1;
	}
}
