package jp.co.sra.jun.goodies.plotter;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.util.ListIterator;
import java.util.Vector;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.goodies.pen.JunPenAngle;
import jp.co.sra.jun.goodies.pen.JunPenLocation;
import jp.co.sra.jun.goodies.pen.JunPenTransformation;
import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunPlotterViewAwt class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2003/10/31 (by Nobuto Matsubara)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @updated   2006/03/16 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPlotterViewAwt.java,v 8.11 2008/02/20 06:32:01 nisinaka Exp $
 */
public class JunPlotterViewAwt extends JunAbstractViewCanvas implements JunPlotterView {
	private JunPenTransformation viewingTransformation;

	/**
	 * Create a new instance of <code>JunPlotterViewAwt</code> and initialize it.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.plotter.JunPlotter
	 * @category Instance creation
	 */
	public JunPlotterViewAwt(JunPlotter aModel) {
		super(aModel);
	}

	/**
	 * Answer a model as JunPlotter.
	 * 
	 * @return jp.co.sra.jun.goodies.plotter.JunPlotter
	 * @see jp.co.sra.jun.goodies.plotter.JunPlotterView#getModel()
	 * @category accessing
	 */
	public JunPlotter getModel() {
		return (JunPlotter) this.model();
	}

	/**
	 * Answer origin point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.goodies.plotter.JunPlotterView#originPoint()
	 * @category accessing
	 */
	public Jun2dPoint originPoint() {
		Jun2dPoint aPoint = new Jun2dPoint(this.getWidth() / 2, this.getHeight() / 2);
		return aPoint;
	}

	/**
	 * Answer transformation.
	 * 
	 * @return jp.co.sra.jun.goodies.pen.JunPenTransformation
	 * @category accessing
	 */
	public JunPenTransformation displayingTransformation() {
		JunPenTransformation aTransformation = this.viewingTransformation();
		aTransformation = aTransformation.product_(JunPenTransformation.Translate_(this.originPoint()));
		return aTransformation;
	}

	/**
	 * Answer a JunTransformation object.
	 * 
	 * @return jp.co.sra.jun.goodies.pen.JunPenTransformation
	 * @see jp.co.sra.jun.goodies.plotter.JunPlotterView#viewingTransformation()
	 * @category accessing
	 */
	public JunPenTransformation viewingTransformation() {
		if (viewingTransformation == null) {
			viewingTransformation = JunPenTransformation.Cabinet_(JunPenAngle.FromDeg_(45));
		}
		return viewingTransformation;
	}

	/**
	 * Set a JunTransformation object.
	 * 
	 * @param aTransformation jp.co.sra.jun.goodies.pen.JunPenTransformation
	 * @see jp.co.sra.jun.goodies.plotter.JunPlotterView#viewingTransformation_(jp.co.sra.jun.goodies.pen.JunPenTransformation)
	 * @category accessing
	 */
	public void viewingTransformation_(JunPenTransformation aTransformation) {
		viewingTransformation = aTransformation;
		this.repaint();
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewCanvas#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunPlotterController();
	}

	/**
	 * Answer a JunPlotterController object.
	 * 
	 * @return jp.co.sra.jun.goodies.plotter.JunPlotterController
	 * @category controller accessing
	 */
	public JunPlotterController plotterController() {
		return (JunPlotterController) this.controller();
	}

	/**
	 * Clear.
	 * 
	 * @category displaying
	 */
	public void clear() {
		Graphics aGraphics = this.getGraphics();
		try {
			aGraphics.clearRect(0, 0, this.getWidth(), this.getHeight());
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param graphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(final Graphics graphicsContext) {
		final JunPlotterViewAwt self = this;
		this.getModel().drawingsDo_(new StBlockClosure() {
			public Object value_(Object array) {
				self.draw_on_transform_((Vector) array, graphicsContext, self.displayingTransformation());
				return (Object) this;
			}
		});
	}

	/**
	 * Drawing an array on a graphics context.
	 * 
	 * @param anArray java.util.Vector
	 * @category displaying
	 */
	public void draw_(Vector anArray) {
		this.draw_on_(anArray, this.getGraphics());
	}

	/**
	 * Drawing an array on a graphics context.
	 * 
	 * @param anArray java.util.Vector
	 * @param graphicsContext java.awt.Graphics
	 * @category displaying
	 */
	public void draw_on_(Vector anArray, Graphics graphicsContext) {
		this.draw_on_transform_(anArray, graphicsContext, this.displayingTransformation());
	}

	/**
	 * Drawing an array on a graphics context.
	 * 
	 * @param anArray java.util.Vector
	 * @param graphicsContext java.awt.Graphics
	 * @param aTransformation jp.co.sra.jun.goodies.pen.JunPenTransformation
	 * @category displaying
	 */
	public void draw_on_transform_(Vector anArray, Graphics graphicsContext, JunPenTransformation aTransformation) {
		Rectangle clipBounds = graphicsContext.getClipBounds();
		if (clipBounds == null) {
			clipBounds = new Rectangle(0, 0, this.getWidth(), this.getHeight());
		}
		StSymbol tagSymbol = (StSymbol) anArray.get(0);
		if (tagSymbol == $("stroke")) {
			int nibWidth = ((Integer) anArray.get(1)).intValue();
			Color colorValue = (Color) anArray.get(2);
			JunPenLocation fromPoint = (JunPenLocation) anArray.get(3);
			JunPenLocation toPoint = (JunPenLocation) anArray.get(4);
			Jun2dPoint fromPoint2d = (fromPoint.transform_(aTransformation)).asPoint().rounded();
			Jun2dPoint toPoint2d = (toPoint.transform_(aTransformation)).asPoint().rounded();
			StRectangle aRectangle = StRectangle.Vertex_vertex_(fromPoint2d._toPoint(), toPoint2d._toPoint());
			aRectangle = aRectangle.expandedBy_(nibWidth);
			if (aRectangle.intersects_(new StRectangle(clipBounds))) {
				((Graphics2D) graphicsContext).setStroke(new BasicStroke(nibWidth));
				graphicsContext.setColor(colorValue);
				graphicsContext.drawLine((int) fromPoint2d.x(), (int) fromPoint2d.y(), (int) toPoint2d.x(), (int) toPoint2d.y());
			}
		}
		if (tagSymbol == $("fill")) {
			Color colorValue = (Color) anArray.get(1);
			Vector locationCollection = (Vector) anArray.get(2);
			Jun2dBoundingBox aRectangle = null;
			Polygon aPolygon = new Polygon();
			for (ListIterator aLocation = locationCollection.listIterator(); aLocation.hasNext();) {
				Jun2dPoint aPoint = ((JunPenLocation) aLocation.next()).transform_(aTransformation).asPoint().rounded();
				aPolygon.addPoint((int) aPoint.x(), (int) aPoint.y());
				if (aRectangle == null) {
					aRectangle = aPoint.extent_(new Jun2dPoint(1.0, 1.0));
				} else {
					aRectangle = aRectangle.merge_(aPoint.extent_(new Jun2dPoint(1.0, 1.0)));
				}
			}
			if (aRectangle.intersects_(new Jun2dBoundingBox(clipBounds))) {
				graphicsContext.setColor(colorValue);
				graphicsContext.fillPolygon(aPolygon);
			}
		}
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.setSize(300, 300);
	}

	/**
	 * Updating.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol aspect = evt.getAspect();
		Vector parameter = (Vector) evt.getParameter();
		if (aspect == $("clear")) {
			this.clear();
			return;
		}
		if (aspect == $("draw")) {
			this.draw_(parameter);
			return;
		}
		this.repaint();
	}
}
