package jp.co.sra.jun.goodies.plotter;

import java.awt.Color;
import java.awt.Frame;
import java.awt.List;
import java.lang.reflect.Method;
import java.util.Vector;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.pen.JunPen;
import jp.co.sra.jun.goodies.pen.JunPenDirection;
import jp.co.sra.jun.goodies.pen.JunPenLocation;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunPlotter class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2003/10/30 (by Nobuto Matsubara)
 *  @updated   2005/03/03 (by nisinaka)
 *  @updated   2006/03/16 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPlotter.java,v 8.13 2008/02/20 06:32:01 nisinaka Exp $
 */
public class JunPlotter extends JunApplicationModel {
	protected Vector drawings;
	protected StPopupMenu _popupMenu;

	/**
	 * Create a new instance of <code>JunPlotter</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunPlotter() {
		super();
	}

	/**
	 * Drawing a block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public void drawingsDo_(StBlockClosure aBlock) {
		Object[] arrays = drawings.toArray(new Object[drawings.size()]);
		for (int i = 0; i < arrays.length; i++) {
			aBlock.value_(arrays[i]);
		}
	}

	/**
	 * Answer methods of category programs.
	 * 
	 * @return java.lang.String[]
	 * @category accessing
	 */
	public String[] programSelectors() {
		String[] selectors = {
				"curveOfSinAndCos1",
				"curveOfSinAndCos2",
				"penExample",
				"sample1_triangles",
				"sample2_star",
				"sample3_dragon",
				"sample4_mandala",
				"sample5_spirals",
				"sample6_spirals",
				"sample7_polylines",
				"sample8_fill",
				"sample9_spirals",
				"sampleA_combination" };
		return selectors;
	}

	/**
	 * Answer a plotter controller object.
	 * 
	 * @return JunPlotterController
	 * @category controller accessing
	 */
	public JunPlotterController plotterController() {
		return (JunPlotterController) this.getController();
	}

	/**
	 * Flush menus.
	 * 
	 * @category flushing
	 */
	public void flushMenus() {
		_popupMenu = null;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		drawings = new Vector();
		_popupMenu = null;
	}

	/**
	 * Answer my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunPlotterViewAwt(this);
		} else {
			return new JunPlotterViewSwing(this);
		}
	}

	/**
	 * Answer the window title
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Pen Plotter");
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			// _popupMenu.add(new StMenuItem($String("Browse Programs", "Browse"), new MenuPerformer(this, "browsePrograms")));
			_popupMenu.add(new StMenuItem($String("Clear"), new MenuPerformer(this, "clearPaper")));
			_popupMenu.add(this._createExecuteMenu());
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem($String("Zoom in"), new MenuPerformer(this, "zoomIn")));
			_popupMenu.add(new StMenuItem($String("Zoom out"), new MenuPerformer(this, "zoomOut")));
			_popupMenu.add(new StMenuItem($String("Reset"), new MenuPerformer(this, "resetView")));
		}
		return _popupMenu;
	}

	/**
	 * Create an "Execute" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	public StMenu _createExecuteMenu() {
		String[] selectors = this.programSelectors();
		if (selectors == null) {
			return null;
		}

		StMenu executeMenu = new StMenu($String("Execute") + "...");
		for (int i = 0; i < selectors.length; ++i) {
			executeMenu.add(new StMenuItem(selectors[i], new MenuPerformer(this, selectors[i])));
		}
		return executeMenu;
	}

	/**
	 * Browse programs.
	 * 
	 * @category menu messages
	 */
	public void browsePrograms() {
		Method[] methods = this.getClass().getDeclaredMethods();
		Vector programs = new Vector();
		for (int i = 0; i < methods.length; ++i) {
			for (int j = 0; j < this.programSelectors().length; ++j) {
				if (methods[i].getName().equals(this.programSelectors()[j])) {
					programs.add(methods[i].getName());
				}
			}
		}
		List aList = new List(programs.size(), true);
		int width = 0, height = 0;
		StComposedText composedText = null;
		for (int i = 0; i < programs.size(); ++i) {
			aList.add((String) (programs.get(i)));
			composedText = new StComposedText((String) (programs.get(i)));
			if (width < composedText.width()) {
				width = composedText.width();
			}
			height += composedText.height();
		}
		Frame listWindow = new Frame();
		listWindow.add(aList);
		listWindow.setSize(width, height);
		listWindow.setVisible(true);
	}

	/**
	 * Clear drawing view.
	 * 
	 * @category menu messages
	 */
	public void clearPaper() {
		drawings = new Vector();
		this.changed_($("clear"));
	}

	/**
	 * Reset view.
	 * 
	 * @category menu messages
	 */
	public void resetView() {
		this.plotterController().resetView();
	}

	/**
	 * Zoom in.
	 * 
	 * @category menu messages
	 */
	public void zoomIn() {
		this.plotterController().zoomIn();
	}

	/**
	 * Zoom out.
	 * 
	 * @category menu messages
	 */
	public void zoomOut() {
		this.plotterController().zoomOut();
	}

	/**
	 * Program to draw a curve of sin and cos.
	 * 
	 * @category programs
	 */
	public void curveOfSinAndCos1() {
		this.clearPaper();

		final JunPen pen1 = this.newPen(Color.red);
		final JunPen pen2 = this.newPen(Color.blue);
		final Jun2dPoint scale = new Jun2dPoint(30, 30);

		final StBlockClosure curve = new StBlockClosure() {
			public Object value_value_(Object o1, Object o2) {
				final JunPen pen = (JunPen) o1;
				final StSymbol function = (StSymbol) o2;

				pen.up();
				for (int i = -300; i <= 300; i += 6) {
					final Double n = new Double(i / 100.0);
					JunPlotter.this.do_forMilliseconds_(new StBlockClosure() {
						public Object value() {
							double x = n.doubleValue() * Math.PI;
							double y = x;
							if (function == $("sin")) {
								y = Math.sin(y);
							} else if (function == $("cos")) {
								y = Math.cos(y);
							}

							Jun2dPoint p = new Jun2dPoint(x, y);
							p = p.scaledBy_(scale);
							pen.goto_(p);

							if (pen.isUp()) {
								pen.down();
							}
							return null;
						}
					}, 100);
				}

				return null;
			}
		};

		Thread aThread1 = new Thread() {
			public void run() {
				curve.value_value_(pen1, $("sin"));
			}
		};
		aThread1.setPriority(Thread.NORM_PRIORITY - 1);
		aThread1.start();

		Thread aThread2 = new Thread() {
			public void run() {
				curve.value_value_(pen2, $("cos"));
			}
		};
		aThread2.setPriority(Thread.NORM_PRIORITY - 1);
		aThread2.start();
	}

	/**
	 * Program to draw a curve of sin and cos with axes.
	 * 
	 * @category programs
	 */
	public void curveOfSinAndCos2() {
		this.curveOfSinAndCos1();

		final JunPen pen0 = this.newPen(Color.gray);
		final Jun2dPoint scale = new Jun2dPoint(30, 30);

		Thread aThread1 = new Thread() {
			public void run() {
				pen0.location_(new Jun2dPoint(-3 * Math.PI, 0).scaledBy_(scale));
				pen0.goto_(new Jun2dPoint(3 * Math.PI, 0).scaledBy_(scale));
			}
		};
		aThread1.setPriority(Thread.NORM_PRIORITY - 1);
		aThread1.start();

		Thread aThread2 = new Thread() {
			public void run() {
				pen0.location_(new Jun2dPoint(0, -1).scaledBy_(scale));
				pen0.goto_(new Jun2dPoint(0, 1).scaledBy_(scale));
			}
		};
		aThread2.setPriority(Thread.NORM_PRIORITY - 1);
		aThread2.start();
	}

	/**
	 * Program of a pen example.
	 * 
	 * @category programs
	 */
	public void penExample() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.nib_(4);
		aPen.color_(Color.gray);

		for (int i = 1; i <= 50; i++) {
			final Integer ii = new Integer(i);
			this.do_forMilliseconds_(new StBlockClosure() {
				public Object value() {
					aPen.go_(ii.intValue() * 4);
					aPen.turn_(89);
					return null;
				}
			}, 100);
		}
	}

	/**
	 * Program of triangles.
	 * 
	 * @category programs
	 */
	public void sample1_triangles() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.color_(Color.red);
		int increment = 45;

		for (int i = 0; i < 360; i += increment) {
			for (int j = 0; j < 3; j++) {
				this.do_forMilliseconds_(new StBlockClosure() {
					public Object value() {
						aPen.go_(100);
						aPen.turn_(120);
						return null;
					}
				}, 100);
			}
			aPen.turn_(increment);
		}
	}

	/**
	 * Program of a star.
	 * 
	 * @category programs
	 */
	public void sample2_star() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.nib_(3);
		aPen.color_(Color.red);
		aPen.spiral_angle_tick_(200, 144, 10);
	}

	/**
	 * Program of a dragon.
	 * 
	 * @category programs
	 */
	public void sample3_dragon() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.color_(Color.red);
		aPen.dragon_distance_tick_(9, 5, 3);
	}

	/**
	 * Program of a mandala.
	 * 
	 * @category programs
	 */
	public void sample4_mandala() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.color_(Color.red);
		aPen.mandala_diameter_tick_(24, 250, 25);
	}

	/**
	 * Program of a mandala.
	 * 
	 * @category programs
	 */
	public void sample5_spirals() {
		this.clearPaper();

		JunPen[] pen = new JunPen[5];
		pen[0] = this.newPen(Color.red);
		pen[1] = this.newPen(Color.red);
		pen[2] = this.newPen(Color.red);
		pen[3] = this.newPen(Color.red);
		pen[4] = this.newPen(Color.red);

		pen[0].location_(new Jun2dPoint(-75, -75));
		pen[1].location_(new Jun2dPoint(75, -75));
		pen[2].location_(new Jun2dPoint(75, 75));
		pen[3].location_(new Jun2dPoint(-75, 75));
		pen[4].location_(new Jun2dPoint(0, 0));

		for (int i = 0; i < pen.length; i++) {
			pen[i].spiral_angle_tick_(100, 88, 10);
		}
	}

	/**
	 * Program of sprials.
	 * 
	 * @category programs
	 */
	public void sample6_spirals() {
		this.clearPaper();

		JunPen[] pen = new JunPen[5];
		pen[0] = this.newPen(Color.red);
		pen[1] = this.newPen(Color.red);
		pen[2] = this.newPen(Color.red);
		pen[3] = this.newPen(Color.red);
		pen[4] = this.newPen(Color.red);

		pen[0].location_(new Jun2dPoint(-75, -75));
		pen[1].location_(new Jun2dPoint(75, -75));
		pen[2].location_(new Jun2dPoint(75, 75));
		pen[3].location_(new Jun2dPoint(-75, 75));
		pen[4].location_(new Jun2dPoint(0, 0));

		for (int i = 0; i < pen.length; i++) {
			final JunPen aPen = pen[i];
			Thread aThread = new Thread() {
				public void run() {
					aPen.spiral_angle_tick_(100, 88, 10);
				}
			};
			aThread.setPriority(Thread.NORM_PRIORITY - 1);
			aThread.start();
		}
	}

	/**
	 * Program of polylines.
	 * 
	 * @category programs
	 */
	public void sample7_polylines() {
		this.clearPaper();

		JunPen aPen = this.newPen();
		aPen.nib_(3);
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, -50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.blue);
		for (int i = 0; i < 4; i++) {
			aPen.go_(100);
			aPen.turn_(90);
		}
		aPen.location_(JunPenLocation.FromArray_(new double[] { 50, -50, -50 }));
		aPen.direction_(JunPenDirection.Longitude_(90));
		aPen.color_(Color.blue);
		for (int i = 0; i < 4; i++) {
			aPen.go_(100);
			aPen.tilt_(90);
		}
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, 50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.blue);
		for (int i = 0; i < 4; i++) {
			aPen.go_(100);
			aPen.tilt_(90);
		}
	}

	/**
	 * Program of fill.
	 * 
	 * @category programs
	 */
	public void sample8_fill() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.nib_(3);
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, -50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.red);
		aPen.fill_(new StBlockClosure() {
			public Object value() {
				for (int i = 0; i < 4; i++) {
					aPen.go_(100);
					aPen.turn_(90);
				}
				return null;
			}
		});
		aPen.location_(JunPenLocation.FromArray_(new double[] { 50, -50, -50 }));
		aPen.direction_(JunPenDirection.Longitude_(90));
		aPen.color_(Color.blue);
		aPen.fill_(new StBlockClosure() {
			public Object value() {
				for (int i = 0; i < 4; i++) {
					aPen.go_(100);
					aPen.tilt_(90);
				}
				return null;
			}
		});
		aPen.location_(JunPenLocation.FromArray_(new double[] { -50, 50, -50 }));
		aPen.direction_(JunPenDirection.Zero());
		aPen.color_(Color.green);
		aPen.fill_(new StBlockClosure() {
			public Object value() {
				for (int i = 0; i < 4; i++) {
					aPen.go_(100);
					aPen.tilt_(90);
				}
				return null;
			}
		});
	}

	/**
	 * Program of spirals
	 * 
	 * @category programs
	 */
	public void sample9_spirals() {
		this.clearPaper();

		final JunPen aPen = this.newPen();
		aPen.nib_(1);
		aPen.color_(Color.blue);

		StBlockClosure aBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				final double sign = ((Number) anObject).doubleValue();
				aPen.home();
				for (int i = 1; i <= 50; i++) {
					final Integer n = new Integer(i);
					JunPlotter.this.do_forMilliseconds_(new StBlockClosure() {
						public Object value() {
							JunPenLocation from = aPen.location();
							aPen.up();
							aPen.go_(n.doubleValue() * 4 * sign);
							aPen.goto_(aPen.location().plus_(JunPenLocation.X_y_z_(0, 2 * sign, 0)));
							JunPenLocation to = aPen.location();
							aPen.location_(from);
							aPen.down();
							aPen.goto_(to);
							aPen.tilt_(89);
							return null;
						}
					}, 100);

				}
				return null;
			}
		};

		aBlock.value_(new Double(1));
		aBlock.value_(new Double(-1));
	}

	/**
	 * Program of combination
	 * 
	 * @category programs
	 */
	public void sampleA_combination() {
		Thread aThread1 = new Thread() {
			public void run() {
				sample4_mandala();
			}
		};
		aThread1.setPriority(Thread.NORM_PRIORITY - 1);
		aThread1.start();

		Thread aThread2 = new Thread() {
			public void run() {
				sample9_spirals();
			}
		};
		aThread2.setPriority(Thread.NORM_PRIORITY - 1);
		aThread2.start();
	}

	/**
	 * Create a new JunPen.
	 * 
	 * @return jp.co.sra.jun.goodies.pen.JunPen
	 * @category menu messages
	 */
	protected JunPen newPen() {
		JunPen aPen = new JunPen();
		aPen.compute_(new StBlockClosure() {
			public Object value_(Object array) {
				drawings.add(array);
				JunPlotter.this.changed_with_($("draw"), array);
				return null;
			}
		});
		return aPen;
	}

	/**
	 * Create a new pen with the specified color.
	 * 
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.jun.goodies.pen.JunPen
	 * @category private
	 */
	protected JunPen newPen(Color aColor) {
		JunPen aPen = this.newPen();
		aPen.color_(aColor);
		return aPen;
	}

}
