package jp.co.sra.jun.goodies.movie.framework;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowEvent;
import java.io.File;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.DependentListener;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StCheckBoxMenuItem;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.qt4jun.JunQTMovie;
import jp.co.sra.qt4jun.JunQTMovieFactory;
import jp.co.sra.qt4jun.JunQTPort;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.gauge.JunSimpleGaugeModel;
import jp.co.sra.jun.goodies.sound.JunSoundMeterModel;
import jp.co.sra.jun.goodies.track.JunTrackSliderModel;
import jp.co.sra.jun.goodies.track.JunTrackSliderView;
import jp.co.sra.jun.goodies.track.JunTrackerModel;
import jp.co.sra.jun.goodies.utilities.JunImageUtility;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMoviePlayer class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2001/01/23 (by NISHIHARA Satoshi)
 *  @updated   2002/04/25 (by hoshi)
 *  @updated   2002/11/14 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun696 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMoviePlayer.java,v 8.15 2008/02/20 06:31:49 nisinaka Exp $
 */
public class JunMoviePlayer extends JunApplicationModel {

	protected static final int DefaultSoundMeterTimeStep = 20;
	protected static final int MinimumSoundLevelTimeStep = 50;
	protected static final int CalculateSoundWaveTimeStep = 600;

	protected JunMovieHandle movieHandle;
	protected JunQTPort moviePort;
	protected JunTrackerModel trackerModel;
	protected Thread trackingProcess;
	protected boolean trackingFlag;
	protected boolean playingFlag;
	protected double startTime;
	protected JunTrackSliderModel trackerModel2;
	protected int soundMeterTimeStep = DefaultSoundMeterTimeStep;
	protected StValueHolder soundValueHolder;
	protected JunSoundMeterModel soundSpectrum;
	protected JunSimpleGaugeModel volumeGauge;
	protected boolean keepAspect;

	protected StPopupMenu _popupMenu;

	/**
	 * Create a new instance of JunMoviePlayer.
	 * 
	 * @category Instance creation
	 */
	public JunMoviePlayer() {
		super();
	}

	/**
	 * Create a new instance of JunMoviePlayer with the specified File.
	 *
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunMoviePlayer(File aFile) {
		this(JunMovieHandle.Filename_(aFile));
	}

	/**
	 * Create a new instance of JunMoviePlayer and initialize it with a movie handle.
	 *
	 * @param aMovieHandle jp.co.sra.jun.goodies.movie.framework.JunMovieHandle
	 * @category Instance creation
	 */
	public JunMoviePlayer(JunMovieHandle aMovieHandle) {
		super();
		if (aMovieHandle != null) {
			this.movieHandle_(aMovieHandle);
		}
	}

	/**
	 * Create a new instance of JunMoviePlayer and open it with the specified movie file.
	 * 
	 * @param filename java.lang.String
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Instance creation
	 */
	public static JunMoviePlayer Filename_(String filename) {
		JunMovieHandle aMovieHandle = JunMovieHandle.Filename_(filename);
		if (aMovieHandle == null) {
			return null;
		}
		JunMoviePlayer aMoviePlayer = JunMoviePlayer.MovieHandle_(aMovieHandle);
		return aMoviePlayer;
	}

	/**
	 * Create a new instance of JunMoviePlayer and initialize it with a movie handle.
	 * 
	 * @param aMovieHandle jp.co.sra.jun.goodies.movie.framework.JunMovieHandle
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Instance creation
	 */
	public static JunMoviePlayer MovieHandle_(JunMovieHandle aMovieHandle) {
		return new JunMoviePlayer(aMovieHandle);
	}

	/**
	 * Create a new instance of JunMoviePlayer, choose a movie file, and open it.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Instance creation
	 */
	public static JunMoviePlayer OpenMovie() {
		JunMovieHandle aMovieHandle = JunMovieHandle.OpenMovie();
		if (aMovieHandle == null) {
			return null;
		}

		JunMoviePlayer aMoviePlayer = new JunMoviePlayer(aMovieHandle);
		aMoviePlayer.open();
		return aMoviePlayer;
	}

	/**
	 * Open a file dialog to choose a movie file and create a new instance of JunMoviePlayer for the movie.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Utilities
	 */
	public static JunMoviePlayer Request() {
		return Request_("Select a movie file.");
	}

	/**
	 * Open a file dialog to choose a movie file and create a new instance of JunMoviePlayer for the movie.
	 * 
	 * @param messageString java.lang.String
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Utilities
	 */
	public static JunMoviePlayer Request_(String messageString) {
		File aFile = RequestFile();
		if (aFile == null) {
			return null;
		}

		JunMoviePlayer aMoviePlayer = JunMoviePlayer.Filename_(aFile.getPath());
		return aMoviePlayer;
	}

	/**
	 * Open a file dialog to choose a movie file.
	 * 
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File RequestFile() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("movie files"), JunSystem.DefaultMovieExtensionPatterns()) };
		File aFile = JunFileRequesterDialog.Request($String("Select a movie file"), fileTypes, fileTypes[0]);
		return aFile;
	}

	/**
	 * Open a file dialog to choose a movie file, create a new instance of JunMoviePlayer for the movie, and open it.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Utilities
	 */
	public static JunMoviePlayer RequestOpen() {
		JunMoviePlayer aMoviePlayer = Request();
		if (aMoviePlayer == null) {
			return null;
		}

		aMoviePlayer.open();
		return aMoviePlayer;
	}

	/**
	 * Open a file dialog to choose a movie file, create a new instance of JunMoviePlayer for the movie, open it, and play it.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer
	 * @category Utilities
	 */
	public static JunMoviePlayer RequestOpenAndPlay() {
		JunMoviePlayer aMoviePlayer = Request();
		if (aMoviePlayer == null) {
			return null;
		}

		aMoviePlayer.openAndPlay();
		return aMoviePlayer;
	}

	/**
	 * Answer the aligned rectangle.
	 *
	 * @return java.awt.Rectangle
	 * @param aRectangle1 java.awt.Rectangle
	 * @param aRectangle2 java.awt.Rectangle
	 * @category Utilities
	 */
	public static Rectangle AlignRectangle_withRectangle_(Rectangle aRectangle1, Rectangle aRectangle2) {
		int width = aRectangle2.width;
		int height = aRectangle1.height * aRectangle2.width / aRectangle1.width;
		Point center2 = (new StRectangle(aRectangle2)).center();
		StRectangle box = new StRectangle(0, 0, width, height);
		box = box.align_with_(box.center(), center2);
		if (box.height() <= aRectangle2.height) {
			return box.toRectangle();
		}

		width = aRectangle1.width * aRectangle2.height / aRectangle1.height;
		height = aRectangle2.height;
		box = new StRectangle(0, 0, width, height);
		box = box.align_with_(box.center(), center2);
		return box.toRectangle();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		movieHandle = null;
		moviePort = null;
		trackerModel = null;
		trackingProcess = null;
		trackingFlag = false;
		playingFlag = false;
		startTime = -1;
		trackerModel2 = null;
		soundValueHolder = null;
		soundSpectrum = null;
		volumeGauge = null;
		keepAspect = true;

		_popupMenu = null;
	}

	/**
	 * Release the resources.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.releaseMovie();
		super.release();
	}

	/**
	 * Release the movie.
	 * 
	 * @category initialize-release
	 */
	protected void releaseMovie() {
		this.trackerModel().end();
		this.stop(); // The tracking process is terminated in the stop method.

		this.releaseMovieHandle();
		this.releaseMoviePort();
	}

	/**
	 * Release the movie handle.
	 * 
	 * @category initialize-release
	 */
	protected void releaseMovieHandle() {
		if (movieHandle == null) {
			return;
		}

		synchronized (movieHandle) {
			movieHandle.release();
			movieHandle = null;
		}
	}

	/**
	 * Release the graf pointer.
	 * 
	 * @deprecated since Jun500
	 * @category initialize-release
	 */
	protected void releaseGrafPtr() {
		if (moviePort != null) {
			moviePort.destroyPortAssociation();
			moviePort = null;
		}
	}

	/**
	 * Release the movie port.
	 *
	 * @category initialize-release
	 */
	protected void releaseMoviePort() {
		this.releaseGrafPtr();
	}

	/**
	 * Finalize the receiver.
	 * 
	 * @see java.lang.Object#finalize()
	 * @category initialize-release
	 */
	protected void finalize() {
		this.release();
	}

	/**
	 * Answer my current movie component.
	 *
	 * @return java.awt.Component
	 * @category accessing
	 */
	public Component movieComponent() {
		if (JunQTMovieFactory.portCreatesComponent()) {
			return this.moviePort(false).getComponent();
		} else {
			return this.getMovieView().toComponent();
		}
	}

	/**
	 * Answer my current movie handle.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMovieHandle
	 * @category accessing
	 */
	public JunMovieHandle movieHandle() {
		return movieHandle;
	}

	/**
	 * Set my new movie handle.
	 * 
	 * @param aMovieHandle jp.co.sra.jun.goodies.movie.framework.JunMovieHandle
	 * @category accessing
	 */
	public void movieHandle_(JunMovieHandle aMovieHandle) {
		if (movieHandle != null) {
			movieHandle.removeDependent_(this);
			this.releaseMovieHandle();
		}

		movieHandle = aMovieHandle;

		if (movieHandle != null) {
			movieHandle.addDependent_(this);
		}

		if (this.hasMovie()) {
			this.trackerModel().step_(1 / (double) movieHandle.duration());
			if (volumeGauge != null) {
				movieHandle.volume_(volumeGauge.doubleValue());
			}
		}
	}

	/**
	 * Answer my current tracker model.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackerModel
	 * @category accessing
	 */
	public JunTrackerModel trackerModel() {
		if (trackerModel == null) {
			trackerModel = this._newTrackerModel();
			trackerModel.addDependent_(this);
			trackerModel.playButton().action_(new StBlockClosure() {
				public Object value_(Object o) {
					playAction_((JunButtonModel) o);
					return null;
				}
			});
			trackerModel.loopButton().action_(new StBlockClosure() {
				public Object value_(Object o) {
					loopAction_((JunButtonModel) o);
					return null;
				}
			});
			trackerModel.nextButton().action_(new StBlockClosure() {
				public Object value_(Object o) {
					nextAction_((JunButtonModel) o);
					return null;
				}
			});
			trackerModel.previousButton().action_(new StBlockClosure() {
				public Object value_(Object o) {
					previousAction_((JunButtonModel) o);
					return null;
				}
			});
			trackerModel.speakerButton().selectedMenuItemHolder().compute_(new StBlockClosure() {
				public Object value_(Object o) {
					StMenuItem aMenuItem = (StMenuItem) o;
					if (aMenuItem == null) {
						System.out.println();
					}

					speakerAction_((Number) aMenuItem.value());
					return null;
				}
			});
			trackerModel.compute_(new StBlockClosure() {
				public Object value_(Object o) {
					setTime_(((Double) o).doubleValue());
					return null;
				}
			});
		}

		return trackerModel;
	}

	/**
	 * Answer my second tracker model.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @category accessing
	 */
	public JunTrackSliderModel trackerModel2() {
		if (trackerModel2 == null) {
			trackerModel2 = this._newTrackerModel2();
			trackerModel2.parentTracker_(this.trackerModel());
			trackerModel2.compute_(new StBlockClosure() {
				public Object value_(Object o) {
					setTime2_(((Double) o).doubleValue());
					return null;
				}
			});
		}

		return trackerModel2;
	}

	/**
	 * Answer my sound spectrum.
	 * 
	 * @return jp.co.sra.jun.goodies.sound.JunSoundMeterModel
	 * @category accessing
	 */
	public JunSoundMeterModel soundSpectrum() {
		if (soundSpectrum == null) {
			soundSpectrum = new JunSoundMeterModel();
			soundSpectrum.soundValueHolder_(this.soundValueHolder());
		}

		return soundSpectrum;
	}

	/**
	 * Answer my volume gauge model.
	 * 
	 * @return jp.co.sra.jun.goodies.gauge.JunSimpleGaugeModel
	 * @category accessing
	 */
	public JunSimpleGaugeModel volumeGauge() {
		if (volumeGauge == null) {
			volumeGauge = new JunSimpleGaugeModel(this.hasMovie() ? movieHandle().volume() : 1, 0, 1, 0.01d);
			volumeGauge.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					double volume = ((Number) anObject).doubleValue();
					if (hasMovie()) {
						movieHandle().volume_(volume);
					}

					String label = String.valueOf(Math.round(Math.round(volume / 0.1) * 0.1 * 100));
					StMenuItem aMenuItem = trackerModel().speakerButton().menu().atNameKey_($(label));
					trackerModel().speakerButton().selectedMenuItem_(aMenuItem);
					return null;
				}
			});
			volumeGauge.valueStringBlock_(new StBlockClosure() {
				public Object value_(Object volume) {
					return Long.toString(Math.round(((Double) volume).doubleValue() * 100));
				}
			});
		}

		return volumeGauge;
	}

	/**
	 * Answer whether to keep aspect or not.
	 *
	 * @return boolean
	 * @category accessing
	 */
	public boolean keepAspect() {
		return keepAspect;
	}

	/**
	 * Set whether to keep aspect or not.
	 *
	 * @param newKeepAspect boolean
	 * @category accessing
	 */
	public void keepAspect_(boolean newKeepAspect) {
		boolean oldBoolean = keepAspect;
		keepAspect = newKeepAspect;
		if (oldBoolean != keepAspect) {
			JunMovieView aView = this.getMovieView();
			if (aView != null) {
				aView.setMovieView();
				aView.toComponent().repaint();
			}
		}
	}

	/**
	 * Create a new instance of the tracker model.
	 *
	 * @return jp.co.sra.jun.goodies.track.JunTrackerModel
	 * @category accessing
	 */
	protected JunTrackerModel _newTrackerModel() {
		return new JunTrackerModel();
	}

	/**
	 * Create a new instance of the tracker model 2.
	 *
	 * @return jp.co.sra.jun.goodies.track.JunTrackerModel
	 * @category accessing
	 */
	protected JunTrackSliderModel _newTrackerModel2() {
		return new JunTrackSliderModel();
	}

	/**
	 * Answer my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category view accessing
	 */
	public StView defaultView() {
		return this.defaultViewWithTracker();
	}

	/**
	 * Answer my default view with two trackers.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category view accessing
	 */
	public StView defaultViewWithDoubleTracker() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunMovieViewAwt.WithDoubleTracker(this);
		} else {
			return JunMovieViewSwing.WithDoubleTracker(this);
		}
	}

	/**
	 * Answer my default view without a tracker.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category view accessing
	 */
	public StView defaultViewWithoutTracker() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunMovieViewAwt.WithoutTracker(this);
		} else {
			return JunMovieViewSwing.WithoutTracker(this);
		}
	}

	/**
	 * Answer my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category view accessing
	 */
	public StView defaultViewWithTracker() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunMovieViewAwt.WithTracker(this);
		} else {
			return JunMovieViewSwing.WithTracker(this);
		}
	}

	/**
	 * Create a clone of the JunMoviePlayer.
	 *
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#copy()
	 * @category copying
	 */
	public StObject copy() {
		JunMoviePlayer aMoviePlayer = this.hasMovie() ? new JunMoviePlayer(this.movieHandle().filename()) : new JunMoviePlayer();
		aMoviePlayer.goto_(this.now());
		aMoviePlayer.trackerModel().lastMarker_(this.trackerModel().lastMarker());
		aMoviePlayer.trackerModel().fixLastMarker_(this.trackerModel().fixLastMarker());
		aMoviePlayer.trackerModel().firstMarker_(this.trackerModel().firstMarker());
		aMoviePlayer.trackerModel().fixFirstMarker_(this.trackerModel().fixFirstMarker());
		aMoviePlayer.trackerModel().loopCondition_(this.trackerModel().loopCondition());
		if (this.trackerModel().areMarkersActive()) {
			aMoviePlayer.trackerModel().enableMarkers();
		} else {
			aMoviePlayer.trackerModel().disableMarkers();
		}
		aMoviePlayer.keepAspect_(keepAspect);
		return aMoviePlayer;
	}

	/**
	 * Answer true if the markers of the tracker are active, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean areMarkersActive() {
		return this.trackerModel().areMarkersActive();
	}

	/**
	 * Answer true if the player is at the start position, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean atStart() {
		return (this.now() <= 0);
	}

	/**
	 * Answer true if the player is at the end position, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean atEnd() {
		if (this.isIntervalNotEmpty() && (startTime <= this.trackerModel().lastMarker()) && (this.trackerModel().lastMarker() <= this.now())) {
			return true;
		}
		return (this.now() >= 1);
	}

	/**
	 * Answer true if the player has a movie, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasMovie() {
		return ((movieHandle() != null) && (movieHandle().hasMovie()));
	}

	/**
	 * Answer true if the imag model is embedded, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmbedded() {
		return false;
	}

	/**
	 * Answer true if the tracker has an interval which is not empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isIntervalNotEmpty() {
		return this.trackerModel().isIntervalNotEmpty();
	}

	/**
	 * Answer true if the player is now playing a movie, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean isPlay() {
		return this.trackerModel().playButton().value();
	}

	/**
	 * Answer true if the player is now playing a sound, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean isSound() {
		if (this.movieHandle() == null) {
			return false;
		}
		return this.movieHandle().isSound();
	}

	/**
	 * Answer true if the view has a tracker.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean viewHasTracker() {
		JunMovieView aView = this.getMovieView();
		if (aView == null) {
			return false;
		}

		return aView.hasTracker();
	}

	/**
	 * Answer true if the view has a tracker2.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean viewHasTracker2() {
		JunMovieView aView = this.getMovieView();
		if (aView == null) {
			return false;
		}

		return aView.hasTracker2();
	}

	/**
	 * Start playing.
	 * 
	 * @category playing
	 */
	public void start() {
		if (this.hasMovie()) {
			trackingProcess = new Thread() {
				public void run() {
					Thread.yield();
					JunTrackerModel tracker = trackerModel();
					tracker.playButtonVisual_(true);
					try {
						if (movieHandle().done()) {
							movieHandle().goToBeginning();
						}
						if (isIntervalNotEmpty() && (now() >= tracker.lastMarker())) {
							goto_(tracker.firstMarker());
						}
						startTime = now();
						movieHandle.timeValue_((int) Math.round(startTime * movieHandle.duration()));
						movieHandle().start();
						while (isPlay()) {
							next();
							Thread.yield();
						}
					} finally {
						tracker.playButtonVisual_(false);
						// trackingProcess = null;
					}
				}
			};

			trackingProcess.setPriority(Thread.NORM_PRIORITY);
			trackingProcess.start();
		}
	}

	/**
	 * Stop playing.
	 * 
	 * @category playing
	 */
	public void stop() {
		this.trackerModel().playButtonVisual_(false);
		if (this.hasMovie()) {
			movieHandle.stop();
		}
		startTime = -1;

		// Make sure to stop the tracking process.
		if (trackingProcess != null) {
			try {
				trackingProcess.join(1000);
			} catch (InterruptedException ie) {
			}
			trackingProcess = null;
		}
	}

	/**
	 * Play to the next.
	 * 
	 * @category playing
	 */
	public synchronized void next() {
		if (this.hasMovie()) {
			startTime = this.now();
			movieHandle.doMovieTask();
			this.setTrackerValue_(movieHandle.timeValue() / (double) movieHandle.duration());

			if (this.atEnd()) {
				if (movieHandle.done()) {
					this.setTrackerValue_(1);
				} else if (this.isIntervalNotEmpty()) {
					this.setTrackerValue_(this.trackerModel().lastMarker());
				}

				if (this.trackerModel().loopCondition() == $("loop")) {
					movieHandle.stop();
					if (this.isIntervalNotEmpty()) {
						this.goto_(this.trackerModel().firstMarker());
					} else {
						this.prologue();
					}
					movieHandle.start();
				} else {
					this.stop();
				}
			}
		} else {
			this.stop();
		}
	}

	/**
	 * Go forward by the specified amount.
	 *
	 * @param incrementNumber double
	 * @category playing
	 */
	public void go_(double incrementNumber) {
		this.goto_(this.now() + incrementNumber);
	}

	/**
	 * Go forward by the specified milliseconds.
	 *
	 * @param incrementMilliseconds double
	 * @category playing
	 */
	public void goInMilliseconds_(double incrementMilliseconds) {
		this.gotoInMilliseconds_(this.nowInMilliseconds() + incrementMilliseconds);
	}

	/**
	 * Go forward by the specified seconds.
	 *
	 * @param incrementSeconds double
	 * @category playing
	 */
	public void goInSeconds_(double incrementSeconds) {
		this.gotoInSeconds_(this.nowInSeconds() + incrementSeconds);
	}

	/**
	 * Go to the specified place.
	 *
	 * @param normalizedNumber double
	 * @category playing
	 */
	public void goto_(double normalizedNumber) {
		this.trackerModel().value_(Math.max(0, Math.min(normalizedNumber, 1)));
	}

	/**
	 * Go to the specified milliseconds.
	 *
	 * @param numberInMilliseconds double
	 * @category playing
	 */
	public void gotoInMilliseconds_(double numberInMilliseconds) {
		double totalMilliseconds = this.howLongInMilliseconds();
		double whereMilliseconds = Math.max(0, Math.min(numberInMilliseconds, totalMilliseconds));
		this.goto_(whereMilliseconds / totalMilliseconds);
	}

	/**
	 * Go to the specified seconds.
	 *
	 * @param numberInSeconds double
	 * @category playing
	 */
	public void gotoInSeconds_(double numberInSeconds) {
		double totalSeconds = this.howLongInSeconds();
		double whereSeconds = Math.max(0, Math.min(numberInSeconds, totalSeconds));
		this.goto_(whereSeconds / totalSeconds);
	}

	/**
	 * Go to the next interesting time.
	 * 
	 * @category playing
	 */
	public void gotoNextInterestingTime() {
		if (this.hasMovie()) {
			this.trackerModel().value_(movieHandle.nextInterestingTime() / (double) movieHandle.duration());
		}
	}

	/**
	 * Go to the previous interesting time.
	 * 
	 * @category playing
	 */
	public void gotoPreviousInterestingTime() {
		if (this.hasMovie()) {
			this.trackerModel().value_(movieHandle.previousInterestingTime() / (double) movieHandle.duration());
		}
	}

	/**
	 * Go to the prologue.
	 * 
	 * @category playing
	 */
	public void prologue() {
		this.goto_(0);
	}

	/**
	 * Go to the epilogue.
	 * 
	 * @category playing
	 */
	public void epilogue() {
		this.goto_(1);
	}

	/**
	 * Answer the current position.
	 * 
	 * @return double
	 * @category playing
	 */
	public double now() {
		return this.trackerModel().doubleValue();
	}

	/**
	 * Answer the current position in milliseconds.
	 *
	 * @return double
	 * @category playing
	 */
	public double nowInMilliseconds() {
		return this.now() * this.howLongInMilliseconds();
	}

	/**
	 * Answer the current position in seconds.
	 *
	 * @return double
	 * @category playing
	 */
	public double nowInSeconds() {
		return this.now() * this.howLongInSeconds();
	}

	/**
	 * Answer the total time of the movie in milliseconds.
	 *
	 * @return double
	 * @category playing
	 */
	public double howLongInMilliseconds() {
		return this.howLongInSeconds() * 1000;
	}

	/**
	 * Answer the total time of the movie in seconds.
	 *
	 * @return double
	 * @category playing
	 */
	public double howLongInSeconds() {
		return this.movieHandle().durationInSeconds();
	}

	/**
	 * Convert the receiver to an image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		Image anImage = this.toImage();
		return (anImage == null) ? null : new StImage(anImage);
	}

	/**
	 * Convert the receiver to an image.
	 * 
	 * @return java.awt.Image
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#toImage()
	 * @category converting
	 */
	public Image toImage() {
		if (this.hasMovie() == false) {
			return null;
		}
		return this.movieHandle().toImage();
	}

	/**
	 * Create a frame of the default view and open it.
	 * 
	 * @return java.awt.Frame
	 * @see jp.co.sra.smalltalk.StApplicationModel#open()
	 * @category interface opening
	 */
	public Frame open() {
		Frame aFrame = super.open();
		this.normalSize();
		return aFrame;
	}

	/**
	 * Open the player and play the movie.
	 * 
	 * @category interface opening
	 */
	public void openAndPlay() {
		this.open();
		this.start();
	}

	/**
	 * Open the player, play the movie, and close the player.
	 * 
	 * @category interface opening
	 */
	public void openAndPlayAndClose() {
		this.open();
		this.start();

		Thread aThread = new Thread() {
			public void run() {
				while (hasMovie() && atEnd() == false) {
					try {
						Thread.sleep(100);
					} catch (InterruptedException e) {
					}
				}
				closeRequest();
			}
		};
		aThread.start();
	}

	/**
	 * Open the player without a tracker, play the movie, and close the player.
	 * 
	 * @category interface opening
	 */
	public void openAndPlayAndCloseWithoutTracker() {
		this.openView_(this.defaultViewWithoutTracker());
		this.start();

		Thread aThread = new Thread() {
			public void run() {
				while (hasMovie() && atEnd() == false) {
					try {
						Thread.sleep(100);
					} catch (InterruptedException e) {
					}
				}
				closeRequest();
			}
		};
		aThread.start();
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Movie Player");
	}

	/**
	 * Called when the loop button is pressed.
	 *
	 * @param model jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	public void loopAction_(JunButtonModel model) {
		if (this.trackerModel().loopCondition() == $("oneWay")) {
			this.trackerModel().loop();
		} else {
			this.trackerModel().oneWay();
		}
	}

	/**
	 * Called when the next button is pressed.
	 *
	 * @param model jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	public void nextAction_(JunButtonModel model) {
		if (model._isPressedWithShiftDown()) {
			this.trackerModel().last();
		} else {
			if (this.isSound()) {
				this.go_(0.005);
			} else {
				this.gotoNextInterestingTime();
			}
		}
	}

	/**
	 * Called when the play button is pressed.
	 *
	 * @param model jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	public void playAction_(JunButtonModel model) {
		if (model.value()) {
			this.stop();
		} else {
			if (model._isPressedWithShiftDown()) {
				this.trackerModel().first();
			}
			this.start();
		}
	}

	/**
	 * Called when the previous button is pressed.
	 *
	 * @param model jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category button actions
	 */
	public void previousAction_(JunButtonModel model) {
		if (model._isPressedWithShiftDown()) {
			this.trackerModel().first();
		} else {
			if (this.isSound()) {
				this.go_(-0.005);
			} else {
				this.gotoPreviousInterestingTime();
			}
		}
	}

	/**
	 * Called when the speaker button is pressed.
	 *
	 * @param aNumber java.lang.Number
	 * @category button actions
	 */
	public void speakerAction_(Number aNumber) {
		float volume = aNumber.floatValue();
		if (volume == Math.round(this.volumeGauge().doubleValue() * 10) / 10.0f) {
			return;
		}

		if (this.movieHandle() != null) {
			this.volumeGauge().value_(volume);
		}
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Open") + "...", $("openMovie"), new MenuPerformer(this, "openMovie")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Open URL") + "...", $("openMovieFromURL"), new MenuPerformer(this, "openMovieFromURL")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Start"), $("start"), new MenuPerformer(this, "start")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Stop"), $("stop"), new MenuPerformer(this, "stop")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Half size"), $("halfSize"), new MenuPerformer(this, "halfSize")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Normal size"), $("normalSize"), new MenuPerformer(this, "normalSize")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Double size"), $("doubleSize"), new MenuPerformer(this, "doubleSize")));
			_popupMenu.add(new StCheckBoxMenuItem(JunSystem.$String("Keep aspect"), $("toggleKeepAspect"), this.keepAspect(), new MenuPerformer(this, "toggleKeepAspect")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Show tracker"), $("showTracker"), new MenuPerformer(this, "showTracker")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Hide tracker"), $("hideTracker"), new MenuPerformer(this, "hideTracker")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Sound volume"), $("soundVolume"), new MenuPerformer(this, "soundVolume")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Sound meter"), $("soundMeter"), new MenuPerformer(this, "soundMeter")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Calculate sound wave"), $("calculateSoundWave"), new MenuPerformer(this, "calculateSoundWave")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Spawn"), $("spawnMovie"), new MenuPerformer(this, "spawnMovie")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Spawn image"), $("spawnImage"), new MenuPerformer(this, "spawnImage")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Save selection") + "...", $("saveSelection"), new MenuPerformer(this, "saveSelection")));

			_popupMenu.atNameKey_($("openMovieFromURL")).disable();
		}
		return _popupMenu;
	}

	/**
	 * Update the popup menu.
	 * 
	 * @category menu accessing
	 */
	protected void _updatePopupMenu() {
		StMenuItem menuItem;

		menuItem = this._popupMenu().atNameKey_($("start"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.beEnabled(!this.isPlay());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("stop"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.beEnabled(this.isPlay());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("halfSize"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.beEnabled(!this.isEmbedded());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("normalSize"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.beEnabled(!this.isEmbedded());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("doubleSize"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.beEnabled(!this.isEmbedded());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("toggleKeepAspect"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.enable();
				((StCheckBoxMenuItem) menuItem).beSelected(this.keepAspect());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("showTracker"));
		if (menuItem != null) {
			if (this.viewHasTracker()) {
				menuItem.disable();
			} else {
				menuItem.beEnabled(!this.isEmbedded());
			}
		}
		menuItem = _popupMenu.atNameKey_($("hideTracker"));
		if (menuItem != null) {
			if (this.viewHasTracker()) {
				menuItem.beEnabled(!this.isEmbedded());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("soundVolume"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("soundMeter"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("calculateSoundWave"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("spawnImage"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.beEnabled(!this.isSound());
			} else {
				menuItem.disable();
			}
		}
		menuItem = _popupMenu.atNameKey_($("saveSelection"));
		if (menuItem != null) {
			if (this.hasMovie()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
			if (this.isIntervalNotEmpty()) {
				menuItem.label_(JunSystem.$String("Save selection") + "...");
			} else {
				menuItem.label_(JunSystem.$String("Save whole") + "...");
			}
		}
	}

	/**
	 * Make the view size double.
	 * 
	 * @category menu messages
	 */
	public void doubleSize() {
		if (this.hasMovie()) {
			Dimension oldSize = movieHandle.originalExtent();
			Dimension newSize = new Dimension(oldSize.width * 2, oldSize.height * 2);
			this.setSize_(newSize);
		}
	}

	/**
	 * Make the view size half.
	 * 
	 * @category menu messages
	 */
	public void halfSize() {
		if (this.hasMovie()) {
			Dimension oldSize = movieHandle.originalExtent();
			Dimension newSize = new Dimension(oldSize.width / 2, oldSize.height / 2);
			this.setSize_(newSize);
		}
	}

	/**
	 * Hide the tracker.
	 * 
	 * @category menu messages
	 */
	public void hideTracker() {
		JunMovieView aView = this.getMovieView();
		if (aView == null) {
			return;
		}
		Window aWindow = aView.topComponent();
		if (aWindow == null) {
			return;
		}
		if (aView.hasTracker() == false) {
			return;
		}

		int deltaHeight = aView.hasTracker2() ? -38 : -19;
		Rectangle displayRectangle = aWindow.getBounds();
		displayRectangle.height += deltaHeight;

		boolean isPlay = this.isPlay();
		if (isPlay) {
			this.stop();
		}
		JunMoviePlayer aMoviePlayer = (JunMoviePlayer) this.copy();
		aMoviePlayer.openViewIn_(aMoviePlayer.defaultViewWithoutTracker(), displayRectangle);
		this.closeRequest();
		if (isPlay) {
			aMoviePlayer.start();
		}
	}

	/**
	 * Make the view size normal.
	 * 
	 * @category menu messages
	 */
	public void normalSize() {
		if (this.hasMovie()) {
			this.setSize_(movieHandle.originalExtent());
		}
	}

	/**
	 * Open a movie.
	 * 
	 * @category menu messages
	 */
	public void openMovie() {
		JunMovieHandle aMovieHandle = JunMovieHandle.OpenMovie();

		if (aMovieHandle.hasMovie()) {
			if (movieHandle != null) {
				JunMoviePlayer aPlayer = JunMoviePlayer.MovieHandle_(aMovieHandle);
				aPlayer.open();
			} else {
				this.movieHandle_(aMovieHandle);
				this.normalSize();
			}
		}
	}

	/**
	 * Save the selected portion as a QuickTime movie file.
	 * 
	 * @category menu messages
	 */
	public void saveSelection() {
		if (this.hasMovie() == false) {
			return;
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType(JunSystem.$String("<1p> files", null, JunSystem.$String("Movie")), new String[] { "*.mov", "*.MOV" }) };
		File aFile = JunFileRequesterDialog.RequestNewFile(JunSystem.$String("Input a <1p> file.", null, "MOV"), new File(this.defaultBaseName() + ".mov"), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return;
		}

		this.saveSelectionTo_(aFile);
	}

	/**
	 * Save the selected portion as a QuickTime movie file.
	 *
	 * @param aFile java.io.File
	 * @category menu messages
	 */
	public void saveSelectionTo_(File aFile) {
		int start, duration;
		if (this.isIntervalNotEmpty()) {
			start = (int) Math.round(this.trackerModel().firstMarker() * this.movieHandle().duration());
			duration = (int) Math.round((this.trackerModel().lastMarker() - this.trackerModel().firstMarker()) * this.movieHandle().duration());
		} else {
			start = 0;
			duration = this.movieHandle().duration();
		}

		JunCursors executeCursor = new JunCursors(JunCursors.ExecuteCursor());
		JunQTMovie newMovie = null;
		try {
			executeCursor._show();
			this.movieHandle().selection_duration_(start, duration);
			newMovie = this.movieHandle().copyMovie();
		} finally {
			executeCursor._restore();
		}

		if (newMovie == null) {
			return;
		}

		JunCursors writeCursor = new JunCursors(JunCursors.WriteCursor());
		try {
			writeCursor._show();
			newMovie.flattenToFile(aFile);
		} finally {
			newMovie.release();
			writeCursor._restore();
		}
	}

	/**
	 * Show the tracker.
	 * 
	 * @category menu messages
	 */
	public void showTracker() {
		JunMovieView aView = this.getMovieView();
		if (aView == null) {
			return;
		}
		Window aWindow = aView.topComponent();
		if (aWindow == null) {
			return;
		}

		int deltaHeight;
		if (aView.hasTracker()) {
			if (aView.hasTracker2()) {
				if (this.areMarkersActive()) {
					return;
				}
				deltaHeight = -19;
			} else {
				if (this.areMarkersActive() == false) {
					return;
				}
				deltaHeight = 19;
			}
		} else {
			if (this.areMarkersActive()) {
				deltaHeight = 38;
			} else {
				deltaHeight = 19;
			}
		}
		Rectangle displayRectangle = aWindow.getBounds();
		displayRectangle.height += deltaHeight;

		boolean isPlay = this.isPlay();
		if (isPlay) {
			this.stop();
		}
		JunMoviePlayer aMoviePlayer = (JunMoviePlayer) this.copy();
		if (this.areMarkersActive()) {
			aMoviePlayer.openViewIn_(aMoviePlayer.defaultViewWithDoubleTracker(), displayRectangle);
		} else {
			aMoviePlayer.openViewIn_(aMoviePlayer.defaultViewWithTracker(), displayRectangle);
		}
		this.closeRequest();
		if (isPlay) {
			aMoviePlayer.start();
		}
	}

	/**
	 * Open the sound meter.
	 * 
	 * @category menu messages
	 */
	public void soundMeter() {
		JunSoundMeterModel model = this.soundSpectrum();
		if (model.builder().windows().length > 0) {
			model._windowExpandAndRaise();
		} else {
			StView view = this.getView();
			if (view == null) {
				model.open();
			} else {
				StRectangle box = new StRectangle(view.topComponent().getBounds());
				StRectangle area = new StRectangle(0, 0, 190, 45);
				area = area.expandedBy_(new StRectangle(0, 0, 8, 27)); // window border
				area = area.align_with_(area.bottomLeft(), new Point(box.left(), box.top() - 5));
				model.openIn_(area.toRectangle());
			}
			Frame[] frames = model.builder().frames();
			for (int i = 0; i < frames.length; ++i) {
				frames[i].setTitle(JunSystem.$String("Sound meter", "Meter"));
			}
		}
	}

	/**
	 * Open the volume gauge.
	 * 
	 * @category menu messages
	 */
	public void soundVolume() {
		JunSimpleGaugeModel model = this.volumeGauge();
		if (model.builder().windows().length > 0) {
			model._windowExpandAndRaise();
		} else {
			StView view = this.getView();
			if (view == null) {
				model.open();
			} else {
				StRectangle box = new StRectangle(view.topComponent().getBounds());
				StRectangle area = new StRectangle(0, 0, 124, 45);
				area = area.expandedBy_(new StRectangle(0, 0, 8, 27)); // window border
				area = area.align_with_(area.bottomRight(), new Point(box.right(), box.top() - 5));
				model.openIn_(area.toRectangle());
			}
			Frame[] frames = model.builder().frames();
			for (int i = 0; i < frames.length; ++i) {
				frames[i].setTitle(JunSystem.$String("Sound volume", "Volume"));
			}
		}
	}

	/**
	 * Spawn the current frame as an image.
	 * 
	 * @category menu messages
	 */
	public void spawnImage() {
		StImage anImage = this.asImage();
		if (anImage == null) {
			return;
		}
		JunImageUtility.Show_(anImage);
	}

	/**
	 * Spawn the movie player.
	 * 
	 * @category menu messages
	 */
	public void spawnMovie() {
		if (this.hasMovie() == false) {
			return;
		}

		JunMoviePlayer aMoviePlayer = new JunMoviePlayer(this.movieHandle().filename());
		aMoviePlayer.goto_(this.now());
		aMoviePlayer.open();
	}

	/**
	 * Toggle the keep aspect flag.
	 * 
	 * @category menu messages
	 */
	public void toggleKeepAspect() {
		this.keepAspect_(!this.keepAspect());
	}

	/**
	 * Return the sound level.
	 * 
	 * @param startNormalizedValue double
	 * @param endNormalizedValue double
	 * @param length int
	 * @return double[]
	 * @category sounds
	 */
	public double[] soundLevelFrom_to_length_(double startNormalizedValue, double endNormalizedValue, int length) {
		int duration = this.movieHandle().duration();
		int start = Math.max(0, Math.min((int) (startNormalizedValue * duration), duration));
		int end = Math.min(Math.max(start + Math.max(soundMeterTimeStep, MinimumSoundLevelTimeStep), (int) (endNormalizedValue * duration)), duration);
		byte[] data = this.movieHandle().asSoundDataFrom_duration_(start, end - start);
		if (data == null) {
			return null;
		}
		int cnt = data.length;
		if (end - start > soundMeterTimeStep) {
			cnt = cnt * soundMeterTimeStep / (end - start);
		}
		double[] array = new double[length * 2];
		for (int i = 0; i < length; ++i) {
			int low = 128, high = 128;
			int max = Math.min((i + 1) * cnt, data.length);
			for (int j = i * cnt; j < max; ++j) {
				int value = ((int) data[j]) & 0xff;
				if (value < low) {
					low = value;
				} else if (value > high) {
					high = value;
				}
			}
			array[i * 2] = low / 255.0;
			array[i * 2 + 1] = high / 255.0;
		}
		return array;
	}

	/**
	 * Answer my sound time step.
	 * 
	 * @return int
	 * @category sounds
	 */
	public int soundMeterTimeStep() {
		return soundMeterTimeStep;
	}

	/**
	 * Set my sound time step to aNumber.
	 * 
	 * @param aNumber int
	 * @category sounds
	 */
	public void soundMeterTimeStep_(int aNumber) {
		soundMeterTimeStep = aNumber;
	}

	/**
	 * Answer my sound value holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category sounds
	 */
	public StValueHolder soundValueHolder() {
		if (soundValueHolder == null) {
			soundValueHolder = new StValueHolder();
		}

		return soundValueHolder;
	}

	/**
	 * Calculate sound wave.
	 * 
	 * @category sounds
	 */
	public void calculateSoundWave() {
		this.soundMeter();

		Thread aThread = new Thread() {
			public void run() {
				int duration = movieHandle().duration();
				int start = 0, end = duration;
				if (isIntervalNotEmpty()) {
					start = (int) Math.round(trackerModel().firstMarker() * duration / soundMeterTimeStep) * soundMeterTimeStep;
					end = (int) (trackerModel().lastMarker() * duration);
				}
				int cnt = (CalculateSoundWaveTimeStep + soundMeterTimeStep - 1) / soundMeterTimeStep;
				JunSoundMeterModel soundMeter = soundSpectrum;
				for (int time = start; time < end; time += soundMeterTimeStep * cnt) {
					if (soundSpectrum != null) {
						double[] level = soundLevelFrom_to_length_((double) time / duration, (double) (time + soundMeterTimeStep * cnt) / duration, cnt);
						if (level != null) {
							for (int i = 0; i < cnt; ++i) {
								double t = (double) (time + soundMeterTimeStep * i) / duration;
								if (t < 1) {
									soundMeter.soundLevelData().put(new Double(t), new double[] { level[i * 2], level[i * 2 + 1] });
								}
							}
						}
						soundMeter.changed();
					}
				}
			}
		};
		aThread.start();
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose()
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		if (soundSpectrum != null) {
			soundSpectrum.closeRequest();
			soundSpectrum = null;
		}
		if (volumeGauge != null) {
			volumeGauge.closeRequest();
			volumeGauge = null;
		}

		this.release();

		super.noticeOfWindowClose(e);
	}

	/**
	 * Answer my KeyListener which handles my keyboard events.
	 *
	 * @return java.awt.event.KeyListener
	 * @category keyboard
	 */
	public KeyListener _keyListener() {
		return new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (_keyboardEvent(e)) {
					e.consume();
				}
			}
		};
	}

	/**
	 * Process the key event.
	 *
	 * @param ev java.awt.event.KeyEvent
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean _keyboardEvent(KeyEvent ev) {
		if (ev.isConsumed()) {
			return false;
		}
		if (this.hasMovie() == false) {
			return false;
		}

		int code = ev.getKeyCode();
		switch (code) {
			case KeyEvent.VK_SPACE:
			case KeyEvent.VK_ENTER:
				if (this.isPlay()) {
					this.stop();
				} else {
					this.start();
				}
				return true;
			case KeyEvent.VK_LEFT:
				this.previousAction_(this.trackerModel().previousButton());
				return true;
			case KeyEvent.VK_RIGHT:
				this.nextAction_(this.trackerModel().nextButton());
				return true;
			case (int) 'T':
				if (this.trackerModel().loopCondition() == $("oneWay")) {
					this.trackerModel().loopCondition_($("loop"));
				} else {
					this.trackerModel().loopCondition_($("oneWay"));
				}
				return true;
			case (int) '[':
				this.trackerModel().firstMarker_(this.now());
				return true;
			case (int) ']':
				this.trackerModel().lastMarker_(this.now());
				return true;
			case (int) 'P':
				this.prologue();
				return true;
			case (int) 'E':
				this.epilogue();
				return true;
			case (int) 'F':
				this.trackerModel().first();
				return true;
			case (int) 'L':
				this.trackerModel().last();
				return true;
		}

		return false;
	}

	/**
	 * Called to update the model and the dependents of the receiver.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol anAspect = evt.getAspect();
		if (anAspect == $("releasing")) {
			playingFlag = this.isPlay();
			if (playingFlag) {
				this.stop();
			}
		} else if (anAspect == $("recovered")) {
			if (this.hasMovie()) {
				this.trackerModel().step_(1 / (double) movieHandle.duration());
				movieHandle.timeValue_((int) Math.round(this.trackerModel().doubleValue() * movieHandle.duration()));
				JunMovieView aView = this.getMovieView();
				if (aView != null) {
					aView.setMovieView();
				}
				if (playingFlag == true) {
					this.start();
				}
			}
		} else if (anAspect == $("useMarkers")) {
			this.showTracker();
		} else if (anAspect == $("originalExtent")) {
			this.normalSize();
		}
	}

	/**
	 * Answer my current movie port.
	 * 
	 * @return jp.co.sra.qt4jun.JunQTPort
	 * @category private
	 */
	public JunQTPort moviePort(boolean inited) {
		if (moviePort == null) {
			Component mv = (Component) this.getMovieView();
			moviePort = JunQTMovieFactory.createPort(mv);
			moviePort.createPortAssociation1();
			if (JunQTMovieFactory.portCreatesComponent()) {
				Component comp = moviePort.getComponent();
				comp.addMouseListener(this.getMovieView().controller());
			}
		}
		if (inited) {
			moviePort.createPortAssociation2();
		}
		return moviePort;
	}

	/**
	 * Answer one of the movie views.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.JunMovieView
	 * @category private
	 */
	protected JunMovieView getMovieView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunMovieView) {
				return (JunMovieView) dependents[i];
			}
		}

		return null;
	}

	/**
	 * Answer the margins of the marker.
	 *
	 * @return double[]
	 * @category private
	 */
	protected double[] markerMargins() {
		return new double[] { 0.1, 0.1 };
	}

	/**
	 * Set the size for the view.
	 * 
	 * @param extent java.awt.Dimension
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#setSize_(java.awt.Dimension)
	 * @category private
	 */
	protected void setSize_(Dimension extent) {
		if (this.hasMovie() == false) {
			return;
		}

		// instead of super.setSize_(extent);
		JunMovieView aView = this.getMovieView();
		if (aView == null) {
			return;
		}

		Window aWindow = aView.topComponent();
		if (aWindow == null) {
			return;
		}

		Dimension oldExtent = aView.toComponent().getSize();
		Dimension newExtent = extent;
		int width = aWindow.getWidth() - oldExtent.width + newExtent.width;
		int height = aWindow.getHeight() - oldExtent.height + newExtent.height;
		aWindow.setSize(width, height);
		aWindow.validate();

		aView = this.getMovieView();
		if (aView == null) {
			return;
		}

		aView.setMovieView();
	}

	/**
	 * Set the sound value.
	 * 
	 * @param time double
	 * @category private
	 */
	protected void setSoundValue_(double time) {
		if (soundSpectrum != null) {
			int duration = this.movieHandle().duration();
			int timeValue = (int) Math.round(time * duration / soundMeterTimeStep) * soundMeterTimeStep;
			time = (double) timeValue / duration;
			Double key = new Double(time);
			double[] level = soundSpectrum.soundLevelDataAt_(key);
			if (level == null) {
				level = this.soundLevelFrom_to_length_(time, time, 1);
			}
			if (level != null) {
				this.soundValueHolder().value_(new Object[] { key, level });
			}
		}
	}

	/**
	 * Set the current time.
	 * 
	 * @param time double
	 * @category private
	 */
	protected synchronized void setTime_(double time) {
		if (trackingFlag != true && this.hasMovie()) {
			movieHandle.timeValue_((int) Math.round(time * movieHandle.duration()));
			movieHandle.doMovieTask();
		}
		this.setTime1_(time);
	}

	/**
	 * Set the current time for the first tracker.
	 * 
	 * @param time double
	 * @category private
	 */
	protected void setTime1_(double time) {
		boolean aBoolean = this.trackerModel2().areMarkersActive();
		if (this.trackerModel().isIntervalNotEmpty()) {
			this.trackerModel2().enableMarkers();
		} else {
			this.trackerModel2().disableMarkers();
		}
		if (aBoolean != this.trackerModel2().areMarkersActive()) {
			DependentListener[] dependents = this.trackerModel2().dependents();
			for (int i = 0; i < dependents.length; i++) {
				if (dependents[i] instanceof JunTrackSliderView) {
					dependents[i].update_(new DependentEvent(dependents[i], $("interval"), null));
				}
			}
		}

		double value;
		double[] markerMargins = this.markerMargins();
		if (this.trackerModel().isIntervalNotEmpty()) {
			value = (this.now() - this.trackerModel().firstMarker()) / (this.trackerModel().lastMarker() - this.trackerModel().firstMarker());
			value = value * (1 - markerMargins[0] - markerMargins[1]) + markerMargins[0];
			value = Math.max(0, Math.min(value, 1));
		} else {
			value = this.now();
		}
		this.trackerModel2().intervalHolder().value_(new double[] { markerMargins[0], 1 - markerMargins[1] });
		this.trackerModel2().valueHolder().value_(value);

		DependentListener[] dependents = this.trackerModel2().dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunTrackSliderView) {
				dependents[i].update_(new DependentEvent(dependents[i], $("value"), null));
			}
		}
	}

	/**
	 * Set the current time for the second tracker.
	 * 
	 * @param time double
	 * @category private
	 */
	protected synchronized void setTime2_(double time) {
		if (trackingFlag != true && this.hasMovie()) {
			if (this.trackerModel().isIntervalNotEmpty()) {
				double[] markerMargins = this.markerMargins();
				double value = (time - markerMargins[0]) / (1 - markerMargins[0] - markerMargins[1]);
				value = (this.trackerModel().lastMarker() - this.trackerModel().firstMarker()) * value + this.trackerModel().firstMarker();
				this.goto_(value);
			} else {
				this.goto_(time);
			}
		}
	}

	/**
	 * Set the tracker value.
	 * 
	 * @param value double
	 * @category private
	 */
	protected void setTrackerValue_(double value) {
		trackingFlag = true;
		try {
			this.trackerModel().value_(value);
		} finally {
			trackingFlag = false;
		}
		this.setSoundValue_(value);
	}

}
