package jp.co.sra.jun.goodies.itemlist;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.util.Vector;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.UIManager;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.StViewJPanel;
import jp.co.sra.smalltalk.WindowSpecLayout;
import jp.co.sra.jun.goodies.button.JunButtonViewSwing;
import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunOrderListViewSwing class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/12/09 (by Mitsuhiro Asada)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun465 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOrderListViewSwing.java,v 8.10 2008/02/20 06:31:36 nisinaka Exp $
 */
public class JunOrderListViewSwing extends JunAbstractViewJPanel implements JunOrderListView {

	protected Vector itemVisualRectangles;
	protected Vector itemRowRectangles;
	protected Rectangle boundingBox;

	protected JunButtonViewSwing upButtonView;
	protected JunButtonViewSwing downButtonView;

	/**
	 * Create a new instance of JunOrderListViewSwing.
	 * 
	 * @param newModel jp.co.sra.jun.goodies.itemlist.JunOrderListModel
	 * @category Instance creation
	 */
	public JunOrderListViewSwing(JunOrderListModel newModel) {
		super(newModel);
	}

	/**
	 * Create a new view for the JunOrderListViewSwing.
	 * The view consists of a JunOrderListViewSwing on a ScrollPane.
	 * The view having up/down buttons.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @param newModel jp.co.sra.jun.goodies.itemlist.JunOrderListModel
	 * @category Instance creation
	 */
	public static StView OnScrollPaneWithButtons(JunOrderListModel newModel) {
		return new StViewJPanel(newModel) {
			protected JunOrderListViewSwing orderListView;

			protected void buildComponent() {
				orderListView = new JunOrderListViewSwing((JunOrderListModel) this.model());
				JScrollPane scrollPane = new JScrollPane(orderListView, JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS) {
					public void setBounds(int x, int y, int width, int height) {
						super.setBounds(x, y, width, height);
						this.doLayout();
						orderListView.setBounds(0, 0, this.getViewport().getExtentSize().width, this.getViewport().getExtentSize().height);
					}
				};

				this.setLayout(new WindowSpecLayout());
				this.add(scrollPane, WindowSpecLayout.Position(0f, 1, 0f, 1, 1f, -20, 1f, -1));
				this.add(orderListView.upButtonView(), WindowSpecLayout.Position(1f, -19, 0.5f, -19, 1f, -1, 0.5f, -1));
				this.add(orderListView.downButtonView(), WindowSpecLayout.Position(1f, -19, 0.5f, 1, 1f, -1, 0.5f, 19));
				this.setPreferredSize(new Dimension(150, 150));
			}
		};
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.flushCachedInformation();
	}

	/**
	 * Answer my model as a JunOrderListModel.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListModel
	 * @see jp.co.sra.jun.goodies.itemlist.JunOrderListView#getModel()
	 * @category model accessing
	 */
	public JunOrderListModel getModel() {
		return (JunOrderListModel) this.model();
	}

	/**
	 * Answer my controller as a JunOrderListController.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListController
	 * @see jp.co.sra.jun.goodies.itemlist.JunOrderListView#getController()
	 * @category controller accessing
	 */
	public JunOrderListController getController() {
		return (JunOrderListController) this.controller();
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunOrderListController();
	}

	/**
	 * Answer my check mark rectangles.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector checkMarkRectangles() {
		return this.itemVisualRectangles();
	}

	/**
	 * Answer my check mark rectangles as Array.
	 * 
	 * @return java.awt.Rectangle[]
	 * @category accessing
	 */
	public Rectangle[] _checkMarkRectangleArray() {
		Rectangle[] rects = new Rectangle[this.checkMarkRectangles().size()];
		this.checkMarkRectangles().copyInto(rects);
		return rects;
	}

	/**
	 * Return the item object at the specified index in model's item object list.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListObject
	 * @param anIndex
	 * @category accessing
	 */
	public JunOrderListObject itemObjectAt_(int anIndex) {
		return this.getModel().itemObjectAt_(anIndex);
	}

	/**
	 * Answer the item object rectangles.
	 * 
	 * @return java.awt.Rectangle[]
	 * @category accessing
	 */
	public Rectangle[] itemObjectRectangles() {
		Rectangle[] checkMarkRects = this._checkMarkRectangleArray();
		Rectangle[] itemVisualRects = this._itemVisualRectangleArray();
		Rectangle[] aCollection = new Rectangle[checkMarkRects.length];
		for (int i = 0; i < checkMarkRects.length; i++) {
			Rectangle aBox = new Rectangle(checkMarkRects[i]);
			aBox.add(itemVisualRects[i]);
			aCollection[i] = aBox;
		}
		return aCollection;
	}

	/**
	 * Get the row rectangles.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector itemRowRectangles() {
		if (itemRowRectangles == null) {
			this.computeCachedInformation();
		}
		return itemRowRectangles;
	}

	/**
	 * Get the row rectangles as array.
	 * 
	 * @return java.awt.Rectangle[]
	 * @category accessing
	 */
	public Rectangle[] _itemRowRectangleArray() {
		Rectangle[] rects = new Rectangle[this.itemRowRectangles().size()];
		this.itemRowRectangles().copyInto(rects);
		return rects;
	}

	/**
	 * Get the item visual rectangles.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector itemVisualRectangles() {
		if (itemVisualRectangles == null) {
			this.computeCachedInformation();
		}
		return itemVisualRectangles;
	}

	/**
	 * Get the item visual rectangles as array.
	 * 
	 * @return java.awt.Rectangle[]
	 * @category accessing
	 */
	public Rectangle[] _itemVisualRectangleArray() {
		return this._itemVisualRectangleArray();
	}

	/**
	 * Answer my left margin size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int leftMargin() {
		return 0;
	}

	/**
	 * Answer my text margin size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int textMargin() {
		return 2;
	}

	/**
	 * Moves and resizes this component. The new location of the top-left
	 * corner is specified by <code>x</code> and <code>y</code>, and the
	 * new size is specified by <code>width</code> and <code>height</code>.
	 * @param x the new <i>x</i>-coordinate of this component
	 * @param y the new <i>y</i>-coordinate of this component
	 * @param width the new <code>width</code> of this component
	 * @param height the new <code>height</code> of this component
	 * 
	 * @see java.awt.Component#setBounds(int, int, int, int)
	 * @param x int
	 * @param y int
	 * @param width int
	 * @param height int
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		Rectangle oldBounds = new Rectangle(this.getBounds());
		super.setBounds(x, y, width, height);
		if (width != oldBounds.width || height != oldBounds.height) {
			this.flushCachedInformation();
		}
	}

	/**
	 * Gets the preferred bounds of this component.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle getPreferredBounds() {
		if (boundingBox == null) {
			this.computeCachedInformation();
		}
		return boundingBox;
	}

	/**
	 * Gets the preferred size of this component.
	 * 
	 * @return java.awt.Dimension
	 * @see java.awt.Component#getPreferredSize
	 * @category bounds accessing
	 */
	public Dimension getPreferredSize() {
		return this.getPreferredBounds().getSize();
	}

	/**
	 * Returns the rectangle at the specified position in this item row rectangles.
	 * 
	 * @return java.awt.Rectangle
	 * @param anIndex int
	 * @category bounds accessing
	 */
	public Rectangle rowBoundsAt_(int anIndex) {
		return (Rectangle) this.itemRowRectangles().get(anIndex);
	}

	/**
	 * Build this component.
	 * 
	 * @category interface opening
	 */
	protected void buildComponent() {
		this.setBackground(UIManager.getColor("Tree.background"));
		this.setForeground(UIManager.getColor("Tree.foreground"));
	}

	/**
	 * Answer my up button view.
	 *
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListView
	 * @category interface opening
	 */
	protected JunOrderListView itemListView() {
		return this;
	}

	/**
	 * Answer my up button view.
	 *
	 * @return jp.co.sra.jun.goodies.button.JunButtonViewSwing
	 * @category interface opening
	 */
	protected JunButtonViewSwing upButtonView() {
		if (upButtonView == null) {
			upButtonView = new JunButtonViewSwing(this.getModel().upButton());
		}
		return upButtonView;
	}

	/**
	 * Answer my down button view.
	 *
	 * @return jp.co.sra.jun.goodies.button.JunButtonViewSwing
	 * @category interface opening
	 */
	protected JunButtonViewSwing downButtonView() {
		if (downButtonView == null) {
			downButtonView = new JunButtonViewSwing(this.getModel().downButton());
		}
		return downButtonView;
	}

	/**
	 * Paints this component.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StViewCanvas#displayOn_(Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphicsContext) {
		if (this.isShowing() == false) {
			return;
		}

		Rectangle clippingBounds = aGraphicsContext.getClipBounds();
		if (clippingBounds == null) {
			clippingBounds = this.getBounds();
		}
		Graphics graphicsContext = aGraphicsContext.create();
		try {
			graphicsContext.setColor(this.getBackground());
			graphicsContext.fillRect(clippingBounds.x, clippingBounds.y, clippingBounds.width, clippingBounds.height);
			if (this.getModel().isEmpty()) {
				super.displayOn_(graphicsContext);
				return;
			}
			for (int anIndex = 0; anIndex < this.getModel().size(); anIndex++) {
				JunOrderListObject itemObject = this.itemObjectAt_(anIndex);
				Rectangle rowBox = this.rowBoundsAt_(anIndex);
				rowBox = new Rectangle(rowBox.x + this.leftMargin(), rowBox.y, rowBox.width - this.leftMargin(), rowBox.height);
				if (this.getModel().isSelected_(anIndex)) {
					graphicsContext.setColor(this.selectionBackgroundColor());
					graphicsContext.fillRect(rowBox.x, rowBox.y, rowBox.width, rowBox.height);
				}

				StComposedText composedText = new StComposedText(itemObject.itemString());
				if (this.getModel().isSelected_(anIndex)) {
					graphicsContext.setColor(this.selectionForegroundColor());
				} else {
					graphicsContext.setColor(this.getForeground());
				}
				composedText.displayOn_at_(graphicsContext, new Point(rowBox.x + this.textMargin(), (rowBox.y + rowBox.height / 2) - composedText.height() / 2));
			}
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
				graphicsContext = null;
			}
		}
	}

	/**
	 * Make visible bottom at an index position.
	 * 
	 * @param anIndex int
	 * @category displaying
	 */
	public void makeVisibleBottom_(int anIndex) {
		Window aWindow = this.topComponent();
		if (aWindow != null) {
			aWindow.repaint();
		}

		if (this.getParent() instanceof JViewport == false) {
			return;
		}

		JViewport viewport = (JViewport) this.getParent();
		Rectangle targetRowBox = this.rowBoundsAt_(anIndex);
		Rectangle viewBounds = new Rectangle(viewport.getViewPosition().x, viewport.getViewPosition().y, viewport.getWidth(), viewport.getHeight());
		if (viewBounds.y <= targetRowBox.y && targetRowBox.y + targetRowBox.height <= viewBounds.y + viewBounds.height) {
			return;
		}
		int scrollAmount = targetRowBox.y + this.getModel().lineGrid() - viewBounds.height;
		viewport.setViewPosition(new Point(viewport.getViewPosition().x, scrollAmount));
	}

	/**
	 * Make visible top at an index position.
	 * 
	 * @param anIndex int
	 * @category displaying
	 */
	public void makeVisibleTop_(int anIndex) {
		Window aWindow = this.topComponent();
		if (aWindow != null) {
			aWindow.repaint();
		}

		if (this.getParent() instanceof JViewport == false) {
			return;
		}

		JViewport viewport = (JViewport) this.getParent();
		Rectangle targetRowBox = this.rowBoundsAt_(anIndex);
		Rectangle viewBounds = new Rectangle(viewport.getViewPosition().x, viewport.getViewPosition().y, viewport.getWidth(), viewport.getHeight());
		if (viewBounds.y < targetRowBox.y && targetRowBox.y + targetRowBox.height < viewBounds.y + viewBounds.height) {
			return;
		}
		int scrollAmount = targetRowBox.y;
		viewport.setViewPosition(new Point(viewport.getViewPosition().x, scrollAmount));
	}

	/**
	 * Answer the selection background color.
	 * 
	 * @return java.awt.Color
	 * @category view properties
	 */
	public Color selectionBackgroundColor() {
		return this.isEnabled() ? UIManager.getColor("Tree.selectionBackground") : Color.darkGray;
	}

	/**
	 * Answer the selection foreground color.
	 * 
	 * @return java.awt.Color
	 * @category view properties
	 */
	public Color selectionForegroundColor() {
		return this.isEnabled() ? UIManager.getColor("Tree.selectionForeground") : new Color(212, 208, 200);

	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument evt.getAspect() is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param event jp.co.sra.smalltalk.DependentEvent
	 * @category updating
	 */
	public void update_(DependentEvent event) {
		if (this.isShowing() == false) {
			return;
		}
		StSymbol aspectSymbol = event.getAspect();
		if (aspectSymbol == $("list")) {
			this.flushCachedInformation();
		}
		if (aspectSymbol == $("grid")) {
			this.flushCachedInformation();
		}
		if (aspectSymbol == $("selections")) {
			Vector newIndexes = (Vector) ((Object[]) event.getParameter())[0];
			Vector oldIndexes = (Vector) ((Object[]) event.getParameter())[1];
			this.newSelectedIndexes_oldSelectedIndexes_(newIndexes, oldIndexes);
		}
		if (aspectSymbol == $("marks")) {
			Vector newIndexes = (Vector) ((Object[]) event.getParameter())[0];
			Vector oldIndexes = (Vector) ((Object[]) event.getParameter())[1];
			this.newSelectedIndexes_oldSelectedIndexes_(newIndexes, oldIndexes);
		}
		if (aspectSymbol == $("makeVisibleBottom")) {
			this.makeVisibleBottom_(((Integer) event.getParameter()).intValue());
		}
		if (aspectSymbol == $("makeVisibleTop")) {
			this.makeVisibleTop_(((Integer) event.getParameter()).intValue());
		}

		super.update_(event);
	}

	/**
	 * Selection list changed.
	 * 
	 * @param newIndexes java.util.Vector
	 * @param oldIndexes java.util.Vector
	 * @category selecting
	 */
	public void newSelectedIndexes_oldSelectedIndexes_(Vector newIndexes, Vector oldIndexes) {
		for (int i = 0; i < oldIndexes.size(); i++) {
			Integer anIndex = (Integer) oldIndexes.get(i);
			if (newIndexes.contains(anIndex) == false) {
				Rectangle aBox = this.rowBoundsAt_(anIndex.intValue());
				this.repaint(0, aBox.x, aBox.y, aBox.width, aBox.height);
			}
		}
		for (int i = 0; i < newIndexes.size(); i++) {
			Integer anIndex = (Integer) newIndexes.get(i);
			if (oldIndexes.contains(anIndex) == false) {
				Rectangle aBox = this.rowBoundsAt_(anIndex.intValue());
				this.repaint(0, aBox.x, aBox.y, aBox.width, aBox.height);
			}
		}
	}

	/**
	 * Answer the index position specified item rows contains the point.
	 * 
	 * @return int
	 * @param aPoint java.awt.Point
	 * @category selecting
	 */
	public int whichOfItemRowsAt_(Point aPoint) {
		Rectangle[] rects = this._itemRowRectangleArray();
		for (int anIndex = 0; anIndex < rects.length; anIndex++) {
			if (rects[anIndex].contains(aPoint)) {
				return anIndex;
			}
		}
		return -1;
	}

	/**
	 * Flush cached information.
	 * 
	 * @category private
	 */
	protected void flushCachedInformation() {
		itemVisualRectangles = null;
		itemRowRectangles = null;
		boundingBox = null;
	}

	/**
	 * Compute cached information.
	 * 
	 * @category private
	 */
	protected void computeCachedInformation() {
		itemVisualRectangles = new Vector();
		itemRowRectangles = new Vector();
		boundingBox = new Rectangle(0, 0, 0, 0);
		if (this.getModel().isEmpty()) {
			return;
		}
		final int lineGrid = this.getModel().lineGrid();
		final int leftMargin = this.leftMargin();
		final Point aPoint = new Point(0, 0);
		this.getModel().do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOrderListObject itemObject = (JunOrderListObject) anObject;

				Rectangle aBox = new Rectangle(aPoint.x + leftMargin, aPoint.y, JunOrderListViewSwing.this.getBounds().width - leftMargin, lineGrid);
				StComposedText composedText = new StComposedText(itemObject.itemString());
				aBox = new Rectangle(aBox.x + JunOrderListViewSwing.this.textMargin() * 2, (aBox.y + (aBox.height / 2)) - (composedText.height() / 2), composedText.width(), composedText.height());
				JunOrderListViewSwing.this.itemVisualRectangles.add(aBox);
				JunOrderListViewSwing.this.boundingBox.add(aBox);

				aBox = new Rectangle(aPoint.x, aPoint.y, JunOrderListViewSwing.this.getBounds().width, lineGrid);
				JunOrderListViewSwing.this.itemRowRectangles.add(aBox);
				JunOrderListViewSwing.this.boundingBox.add(aBox);

				aPoint.translate(0, lineGrid);
				return null;
			}
		});

		Rectangle[] rowRects = this._itemRowRectangleArray();
		for (int i = 0; i < rowRects.length; i++) {
			rowRects[i].width = boundingBox.width;
		}
	}
}
