package jp.co.sra.jun.goodies.itemlist;

import java.util.*;
import jp.co.sra.smalltalk.*;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOrderListModel class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/12/09 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun465 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOrderListModel.java,v 8.10 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunOrderListModel extends JunApplicationModel {
	protected Vector itemObjectList;
	protected Vector selectedIndexes;
	protected int lineGrid;

	/**
	 * Create a new instance of JunOrderListModel.
	 * 
	 * @category Instance creation
	 */
	public JunOrderListModel() {
		super();
		this.initialize();
	}

	/**
	 * Create a new instance of JunOrderListModel.
	 * 
	 * @param stringCollection java.lang.String[]
	 * @category Instance creation
	 */
	public JunOrderListModel(String[] stringCollection) {
		this();
		for (int i = 0; i < stringCollection.length; i++) {
			String aString;
			if (stringCollection[i] == null) {
				aString = "";
			} else {
				aString = stringCollection[i];
			}
			JunOrderListObject itemObject = this.defaultObject();
			itemObject.itemString_(aString);
			this.addItemObject_(itemObject);
		}
	}

	/**
	 * Initialize this object.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		itemObjectList = null;
		selectedIndexes = null;
		lineGrid = 0;
	}

	/**
	 * Return the index of the first occurrence of the item object in this item object list.
	 * 
	 * @return int
	 * @param itemObject jp.co.sra.jun.goodies.itemlist.JunOrderListObject
	 * @category accessing
	 */
	public int indexOfItemObject_(JunOrderListObject itemObject) {
		return this.itemObjectList().indexOf(itemObject);
	}

	/**
	 * Return the item object at the specified index in this item object list.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListObject
	 * @param anIndex
	 * @category accessing
	 */
	public JunOrderListObject itemObjectAt_(int anIndex) {
		return (JunOrderListObject) this.itemObjectList().elementAt(anIndex);
	}

	/**
	 * Get the item object lists.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector itemObjectList() {
		if (itemObjectList == null) {
			itemObjectList = new Vector();
		}
		return itemObjectList;
	}

	/**
	 * Get the item object lists.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListObject[]
	 * @category accessing
	 */
	public JunOrderListObject[] _itemObjectListArray() {
		JunOrderListObject[] arrayOfItem = new JunOrderListObject[this.itemObjectList().size()];
		this.itemObjectList().copyInto(arrayOfItem);
		return arrayOfItem;
	}

	/**
	 * Set the item object list.
	 * 
	 * @param aCollection java.util.Vector
	 * @category accessing
	 */
	public void itemObjectList_(Vector aCollection) {
		this.setItemObjectList_(aCollection);
		this.setSelectedIndexes_(this.selectedIndexes());
		this.changed_($("list"));
	}

	/**
	 * Get the line gird size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int lineGrid() {
		if (lineGrid == 0) {
			lineGrid = Math.max(SystemResourceSupport.getFontMetrics(SystemResourceSupport.getFont()).getHeight(), 18);
		}
		return lineGrid;
	}

	/**
	 * Set the line grid.
	 * 
	 * @param gridValue int
	 * @category accessing
	 */
	public void lineGrid_(int gridValue) {
		lineGrid = gridValue;
		this.changed_($("grid"));
	}

	/**
	 * Answer the size of this item object list.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return this.itemObjectList().size();
	}

	/**
	 * Answer the value in this item objects.
	 * 
	 * @return java.lang.Object[]
	 * @category accessing
	 */
	public Object[] value() {
		String[] value = new String[this.size()];
		for (int i = 0; i < value.length; i++) {
			value[i] = new String(this.itemObjectAt_(i).itemString());
		}
		return value;
	}

	/**
	 * Answer a view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category accessing
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunOrderListView && ((JunOrderListView) each).model() == this) {
				return (StView) each;
			}
		}
		return null;
	}

	/**
	 * Answer true if this item object list is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.itemObjectList().isEmpty();
	}

	/**
	 * Answer true if the item object specified index position is selected, otherwise false.
	 * 
	 * @return boolean
	 * @param anIndex java.lang.Integer
	 * @category testing
	 */
	public boolean isSelected_(Integer anIndex) {
		return this.selectedIndexes().contains(anIndex);
	}

	/**
	 * Answer true if the item object specified index position is selected, otherwise false.
	 * 
	 * @return boolean
	 * @param anIndex int
	 * @category testing
	 */
	public boolean isSelected_(int anIndex) {
		return this.isSelected_(new Integer(anIndex));
	}

	/**
	 * Add the item object.
	 * 
	 * @param itemObject jp.co.sra.jun.goodies.itemlist.JunOrderListObject
	 * @category adding
	 */
	public void addItemObject_(JunOrderListObject itemObject) {
		Vector copyCollection = (Vector) this.itemObjectList().clone();
		copyCollection.add(itemObject);
		this.itemObjectList_(copyCollection);
	}

	/**
	 * Answer the up button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel upButton() {
		JunButtonModel button = new JunButtonModel(false, JunCursors.UpArrowCursorImage(), new StBlockClosure() {
			public Object value_(Object obj) {
				JunOrderListModel.this.upButtonAction();
				return null;
			}
		});
		return button;
	}

	/**
	 * The action of up button.
	 * 
	 * @category buttons
	 */
	public void upButtonAction() {
		if (this.selectedIndexes().isEmpty()) {
			return;
		}
		JunOrderListObject[] selectedCollection = this.selectedItemObjects();
		Vector targetCollection = this.shiftDown_in_(this.selectedIndexes(), this.itemObjectList());
		if (this.itemObjectList() == targetCollection) {
			return;
		}
		this.setItemObjectList_(targetCollection);
		this.setSelectedItemObjects_(selectedCollection);
		Integer[] sortSelectedIndexes = this._selectedIndexArray();
		Arrays.sort(sortSelectedIndexes);
		this.changed_with_($("makeVisibleTop"), sortSelectedIndexes[0]);
	}

	/**
	 * Answer the down button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category buttons
	 */
	public JunButtonModel downButton() {
		JunButtonModel button = new JunButtonModel(false, JunCursors.DownArrowCursorImage(), new StBlockClosure() {
			public Object value_(Object obj) {
				JunOrderListModel.this.downButtonAction();
				return null;
			}
		});
		return button;
	}

	/**
	 * The action of down button.
	 * 
	 * @category buttons
	 */
	public void downButtonAction() {
		if (this.selectedIndexes().isEmpty()) {
			return;
		}
		JunOrderListObject[] selectedCollection = this.selectedItemObjects();
		Vector targetCollection = this.shiftUp_in_(this.selectedIndexes(), this.itemObjectList());
		if (this.itemObjectList() == targetCollection) {
			return;
		}
		this.setItemObjectList_(targetCollection);
		this.setSelectedItemObjects_(selectedCollection);
		Integer[] sortSelectedIndexes = this._selectedIndexArray();
		Arrays.sort(sortSelectedIndexes);
		this.changed_with_($("makeVisibleBottom"), sortSelectedIndexes[sortSelectedIndexes.length - 1]);
	}

	/**
	 * Get the selected indexes.
	 * 
	 * @return java.util.Vector
	 * @category selecting
	 */
	public Vector selectedIndexes() {
		if (selectedIndexes == null) {
			selectedIndexes = new Vector();
		}
		return selectedIndexes;
	}

	/**
	 * Get the selected indexes as Array.
	 * 
	 * @return java.util.Vector
	 * @category selecting
	 */
	public Integer[] _selectedIndexArray() {
		Integer[] array = new Integer[this.selectedIndexes().size()];
		this.selectedIndexes().copyInto(array);
		return array;
	}

	/**
	 * Set the selected indexes.
	 * 
	 * @param indexCollection java.util.Vector
	 * @category selecting
	 */
	public void selectedIndexes_(Vector indexCollection) {
		Vector oldSelectedIndexes = this.selectedIndexes();
		this.setSelectedIndexes_(indexCollection);
		this.changed_with_($("selection"), new Vector[] { selectedIndexes, oldSelectedIndexes });
	}

	/**
	 * Get the selected item objects.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListObject[]
	 */
	public JunOrderListObject[] selectedItemObjects() {
		Integer[] indexes = new Integer[this.selectedIndexes().size()];
		JunOrderListObject[] itemObjects = new JunOrderListObject[indexes.length];
		this.selectedIndexes().copyInto(indexes);
		for (int i = 0; i < indexes.length; i++) {
			itemObjects[i] = this.itemObjectAt_(indexes[i].intValue());
		}
		return itemObjects;
	}

	/**
	 * Set the selected item objects.
	 * 
	 * @param itemObjects jp.co.sra.jun.goodies.itemlist.JunOrderListObject[]
	 */
	public void selectedItemObjects_(JunOrderListObject[] itemObjects) {
		Vector indexes = new Vector();
		for (int i = 0; i < itemObjects.length; i++) {
			indexes.add(new Integer(this.indexOfItemObject_(itemObjects[i])));
		}
		this.selectedIndexes_(indexes);
	}

	/**
	 * Enumerating this item object list and execute block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void do_(StBlockClosure aBlock) {
		JunOrderListObject[] items = this._itemObjectListArray();
		for (int i = 0; i < items.length; i++) {
			aBlock.value_(items[i]);
		}
	}

	/**
	 * Answer the default item object.
	 * 
	 * @return jp.co.sra.jun.goodies.itemlist.JunOrderListObject
	 * @category defaults
	 */
	public JunOrderListObject defaultObject() {
		return new JunOrderListObject();
	}

	/**
	 * Answer the default view.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunOrderListViewAwt.OnScrollPaneWithButtons(this);
		} else {
			return JunOrderListViewSwing.OnScrollPaneWithButtons(this);
		}
	}

	/**
	 * Answer the default wait tick miliseconds.
	 * 
	 * @return int
	 */
	public int defaultWaitTick() {
		return 100;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @category viewing
	 */
	protected String windowTitle() {
		return JunSystem.$String("Order List");
	}

	/**
	 * Answer the number of max print.
	 * 
	 * @return int
	 * @category private
	 */
	private int maxPrint() {
		return 5000;
	}

	/**
	 * Set the item object list.
	 * 
	 * @param aCollection java.util.Vector
	 * @category private
	 */
	private void setItemObjectList_(Vector aCollection) {
		itemObjectList = aCollection;
	}

	/**
	 * Set the selected indexes.
	 * 
	 * @param indexCollection java.util.Vector
	 * @category private
	 */
	private void setSelectedIndexes_(Vector indexCollection) {
		selectedIndexes = new Vector();
		Integer[] ints = new Integer[indexCollection.size()];
		indexCollection.copyInto(ints);

		for (int i = 0; i < ints.length; i++) {
			int anIndex = ints[i].intValue();
			if (0 <= anIndex && anIndex <= this.itemObjectList().size() - 1 && selectedIndexes.contains(ints[i]) == false) {
				selectedIndexes.add(ints[i]);
			}
		}
	}

	/**
	 * Set the selected item objects.
	 * 
	 * @param itemObjects jp.co.sra.jun.goodies.itemlist.JunOrderListObject[]
	 */
	private void setSelectedItemObjects_(JunOrderListObject[] itemObjects) {
		Vector newSelectedIndexes = new Vector(itemObjects.length);
		for (int i = 0; i < itemObjects.length; i++) {
			newSelectedIndexes.add(new Integer(this.indexOfItemObject_(itemObjects[i])));
		}
		this.setSelectedIndexes_(newSelectedIndexes);
	}

	/**
	 * Answer the shift down collection at the index positions in the object collection.
	 * 
	 * @return java.util.Vector
	 * @param indexCollection java.util.Vector
	 * @param objectCollection java.util.Vector
	 * @category private
	 */
	private Vector shiftDown_in_(Vector indexCollection, Vector objectCollection) {
		Vector selectedCollection = new Vector();
		Vector remaindedCollection = new Vector();
		for (int i = 0; i < objectCollection.size(); i++) {
			if (indexCollection.contains(new Integer(i))) {
				selectedCollection.add(objectCollection.get(i));
			} else {
				remaindedCollection.add(objectCollection.get(i));
			}
		}
		Vector shiftIndexes = new Vector();
		Integer[] sortIndexCollection = new Integer[indexCollection.size()];
		indexCollection.copyInto(sortIndexCollection);
		Arrays.sort(sortIndexCollection);
		for (int i = 0; i < sortIndexCollection.length; i++) {
			int index = ((Integer) sortIndexCollection[i]).intValue() - 1;
			int anIndex = Math.max(0, Math.min(index, objectCollection.size() - 1));
			if (shiftIndexes.contains(new Integer(anIndex))) {
				shiftIndexes.add(new Integer(shiftIndexes.size()));
			} else {
				shiftIndexes.add(new Integer(anIndex));
			}
		}

		JunOrderListObject[] targetCollection = new JunOrderListObject[objectCollection.size()];
		for (int i = 0; i < shiftIndexes.size(); i++) {
			int index = ((Integer) shiftIndexes.get(i)).intValue();
			JunOrderListObject item = (JunOrderListObject) selectedCollection.get(i);
			targetCollection[index] = item;
		}
		for (int index = 0; index < targetCollection.length; index++) {
			if (targetCollection[index] == null) {
				targetCollection[index] = (JunOrderListObject) remaindedCollection.remove(0);
			}
		}

		Vector result = new Vector();
		for (int i = 0; i < targetCollection.length; i++) {
			result.add(i, targetCollection[i]);
		}
		return result;
	}

	/**
	 * Answer the shift up collection at the index positions in the object collection.
	 * 
	 * @return java.util.Vector
	 * @param indexCollection java.util.Vector
	 * @param objectCollection java.util.Vector
	 * @category private
	 */
	private Vector shiftUp_in_(Vector indexCollection, Vector objectCollection) {
		Vector selectedCollection = new Vector();
		Vector remaindedCollection = new Vector();
		for (int i = 0; i < objectCollection.size(); i++) {
			if (indexCollection.contains(new Integer(i))) {
				selectedCollection.add(objectCollection.get(i));
			} else {
				remaindedCollection.add(objectCollection.get(i));
			}
		}
		Vector shiftIndexes = new Vector();
		Integer[] sortIndexCollection = new Integer[indexCollection.size()];
		indexCollection.copyInto(sortIndexCollection);
		Arrays.sort(sortIndexCollection);
		for (int i = sortIndexCollection.length - 1; i >= 0; i--) {
			int index = ((Integer) sortIndexCollection[i]).intValue() + 1;
			int anIndex = Math.max(0, Math.min(index, objectCollection.size() - 1));
			if (shiftIndexes.contains(new Integer(anIndex))) {
				shiftIndexes.add(new Integer(objectCollection.size() - shiftIndexes.size() - 1));
			} else {
				shiftIndexes.add(new Integer(anIndex));
			}
		}

		JunOrderListObject[] targetCollection = new JunOrderListObject[objectCollection.size()];
		for (int i = 0; i < shiftIndexes.size(); i++) {
			int index = ((Integer) shiftIndexes.get(i)).intValue();
			JunOrderListObject item = (JunOrderListObject) selectedCollection.get(selectedCollection.size() - 1 - i);
			targetCollection[index] = item;
		}
		for (int index = 0; index < targetCollection.length; index++) {
			if (targetCollection[index] == null) {
				targetCollection[index] = (JunOrderListObject) remaindedCollection.remove(0);
			}
		}

		Vector result = new Vector();
		for (int i = 0; i < targetCollection.length; i++) {
			result.add(i, targetCollection[i]);
		}
		return result;
	}
}
