package jp.co.sra.jun.goodies.image.streams;

import java.awt.Color;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import jp.co.sra.smalltalk.StImage;

/**
 * JunSraBmpImageStream class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1999/01/20 (by Hirotsugu Kondo)
 *  @updated   2000/23/03 (by Mitsuhiro Asada)
 *  @updated   2005/08/15 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSraBmpImageStream.java,v 8.11 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunSraBmpImageStream extends JunBmpImageStream {
	protected int position = 0;
	private int bfType;
	private long bfSize;
	private int bfReserved1;
	private int bfReserved2;
	private long bfOffBits;
	private long biSize;
	private long biWidth;
	private long biHeight;
	private int biPlanes;
	private int biBitCount;
	private long biCompression;
	private long biSizeImage;
	private long biXPelsPerMeter;
	private long biYPelsPerMeter;
	private long biClrUsed;
	private long biClrImportant;
	private int[] imagePalette;

	/**
	 * Constructor with an input stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		return On_(new JunSraBmpImageStream(), stream);
	}

	/**
	 * Constructor with an output stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		throw new IOException("could not save image");
	}

	/**
	 * Read the image from the input stream.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category accessing
	 */
	public StImage nextImage() throws IOException {
		if (inStream != null) {
			boolean check = this.readBitmapFileHeader();
			if (check == true) {
				this.readBitmapInfoHeader();
				this.readImagePalette();
				boolean check2 = this.readImageData();
				if (check2 == false) {
					imageObject = null;
				}
			}
		}
		return imageObject;
	}

	/**
	 * Write the image on output stream.
	 * 
	 * @param newImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category accessing
	 */
	public void nextPutImage_(StImage newImage) throws IOException {
		throw new IOException("could not save image");
	}

	/**
	 * Answer next value.
	 * 
	 * @return int
	 * @exception java.io.IOException
	 * @category stream access
	 */
	private int next() throws IOException {
		int nextValue = -1;
		nextValue = this.inStream.read();
		this.position = this.position + 1;
		return nextValue;
	}

	/**
	 * Answer next long.
	 * 
	 * @return long
	 * @exception java.io.IOException
	 * @category stream access
	 */
	private long nextLong() throws IOException {
		int byte1 = this.next();
		int byte2 = this.next();
		int byte3 = this.next();
		int byte4 = this.next();
		return (byte1 << 24) + (byte2 << 16) + (byte3 << 8) + byte4;
	}

	/**
	 * Answer next LSB long.
	 * 
	 * @return long
	 * @exception java.io.IOException
	 * @category stream access
	 */
	private long nextLSBLong() throws IOException {
		int byte1 = this.next();
		int byte2 = this.next();
		int byte3 = this.next();
		int byte4 = this.next();
		return (byte4 << 24) + (byte3 << 16) + (byte2 << 8) + byte1;
	}

	/**
	 * Answer next LSB word.
	 * 
	 * @return int
	 * @exception java.io.IOException
	 * @category stream access
	 */
	private int nextLSBWord() throws IOException {
		int byte1 = this.next();
		int byte2 = this.next();
		return (byte2 << 8) + byte1;
	}

	/**
	 * Answer next word.
	 * 
	 * @return int
	 * @exception java.io.IOException
	 * @category stream access
	 */
	private int nextWord() throws IOException {
		int byte1 = this.next();
		int byte2 = this.next();
		return (byte1 << 8) + byte2;
	}

	/**
	 * Set the output stream.
	 * 
	 * @param stream java.io.OutputStream
	 * @exception IOException
	 * @category initialize-release
	 */
	protected void on_(OutputStream stream) throws IOException {
		throw new IOException("could not save image");
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return boolean
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private boolean readBitmapFileHeader() throws IOException {
		int checker = ((int) 'B' << 8) + (int) 'M';
		this.bfType = this.nextWord();
		if (checker != this.bfType) {
			return false;
		}
		this.bfSize = this.nextLSBLong();
		this.bfReserved1 = this.nextLSBWord();
		this.bfReserved2 = this.nextLSBWord();
		this.bfOffBits = this.nextLSBLong();
		return true;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 *
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private void readBitmapInfoHeader() throws IOException {
		this.biSize = this.nextLSBLong();
		this.biWidth = this.nextLSBLong();
		this.biHeight = this.nextLSBLong();
		this.biPlanes = this.nextLSBWord();
		this.biBitCount = this.nextLSBWord();
		this.biCompression = this.nextLSBLong();
		this.biSizeImage = this.nextLSBLong();
		this.biXPelsPerMeter = this.nextLSBLong();
		this.biYPelsPerMeter = this.nextLSBLong();
		this.biClrUsed = this.nextLSBLong();
		this.biClrImportant = this.nextLSBLong();
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private StImage readDepth1Data() throws IOException {
		System.out.print("1bit BMP detect");
		return null;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private StImage readDepth24Data() throws IOException {
		while (this.position < this.bfOffBits) {
			this.next();
		}
		int width = (int) this.biWidth;
		int height = (int) this.biHeight;
		StImage image = new StImage(width, height);

		for (int y = height - 1; y > -1; y--) {
			int count = 0;
			for (int x = 0; x < width; x++) {
				int rgb = (this.next() << 0) + (this.next() << 8) + (this.next() << 16);
				image.setPixel(x, y, new Color(rgb).getRGB());
				count = count + 3;
			}
			while ((count % 4) != 0) {
				this.next();
				count++;
			}
		}

		return image;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private StImage readDepth8Data() throws IOException {
		int[] palette = this.imagePalette;
		int width = (int) this.biWidth;
		int height = (int) this.biHeight;
		StImage image = new StImage(width, height);

		for (int y = height - 1; y > -1; y--) {
			int count = 0;
			for (int x = 0; x < width; x++) {
				int index = this.next();
				image.setPixel(x, y, palette[index]);
				count++;
			}
			while ((count % 4) != 0) {
				this.next();
				count++;
			}
		}

		return image;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private void readDepth8Palette() throws IOException {
		long size = 1 << this.biBitCount;
		if (this.biClrUsed != 0) {
			size = this.biClrUsed;
		}
		int[] colors = new int[(int) size];
		for (int index = 0; index < size; index++) {
			int rgb = this.next() + (this.next() << 8) + (this.next() << 16);
			this.next();
			colors[index] = 0xff000000 | rgb;
		}
		this.imagePalette = colors;
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return boolean
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private boolean readImageData() throws IOException {
		if (this.biCompression != 0) {
			return false;
		}

		int bit = this.biBitCount;
		this.imageObject = null;
		if (bit == 24) {
			this.imageObject = this.readDepth24Data();
		}
		if (bit == 8) {
			this.imageObject = this.readDepth8Data();
		}
		if (bit == 1) {
			this.imageObject = this.readDepth1Data();
		}
		return this.imageObject != null;
	}

	/**
	 * Palette ignored but depth 8 BMP.
	 *
	 * @exception java.io.IOException
	 * @category decoding
	 */
	private void readImagePalette() throws IOException {
		this.imagePalette = null;
		if (this.biBitCount == 8) {
			this.readDepth8Palette();
		}
	}
}
