package jp.co.sra.jun.goodies.image.streams;

import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.IndexColorModel;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.util.Arrays;
import java.util.Iterator;

import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.goodies.image.support.JunImageProcessor;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * JunGifAnimationStream class
 * 
 *  @author    nisinaka
 *  @created   2007/10/04 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGifAnimationStream.java,v 8.6 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunGifAnimationStream extends JunSraGifImageStream {

	protected boolean firstTime;
	protected int delayTime;
	protected int loopCount;

	private ImageReader _imageReader;
	private int _imageIndex;
	private ImageReadParam _imageReadParam;
	private ReadableByteChannel _byteChannel;
	private ByteBuffer _byteBuffer;
	private static int _ByteBufferSize = 256;

	/**
	 * Create a new instance of JunGifAnimationStream with the specified input stream.
	 * 
	 * @param stream java.io.InputStream
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		return On_(new JunGifAnimationStream(), stream);
	}

	/**
	 * Create a new instance of JunGifAnimationStream with the specified output stream.
	 * 
	 * @param stream java.io.InputStream
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		return On_(new JunGifAnimationStream(), stream);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		firstTime = true;
		delayTime = 0;
		loopCount = 1;
	}

	/**
	 * Set the input stream.
	 * 
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#on_(java.io.InputStream)
	 * @category initialize-release
	 */
	protected void on_(InputStream stream) throws IOException {
		this.initialize();
		super.on_(stream);

		if (stream instanceof FileInputStream) {
			_byteChannel = ((FileInputStream) stream).getChannel();

			try {
				this.readHeader();
				this.readApplicationExtensionBlock();
			} finally {
				((FileChannel) _byteChannel).position(0);
			}
		} else {
			_byteChannel = Channels.newChannel(stream);
		}

		ImageInputStream anImageInputStream = ImageIO.createImageInputStream(stream);
		if (anImageInputStream == null) {
			throw new IOException("Failed to create an image input stream.");
		}

		Iterator i = ImageIO.getImageReaders(anImageInputStream);
		if (i.hasNext() == false) {
			anImageInputStream.close();
			throw new IOException("No image readers are available.");
		}
		_imageReader = (ImageReader) i.next();
		_imageReader.setInput(anImageInputStream);
		_imageIndex = _imageReader.getMinIndex();
		_imageReadParam = _imageReader.getDefaultReadParam();
	}

	/**
	 * Set the output stream.
	 * 
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#on_(java.io.OutputStream)
	 * @category initialize-release
	 */
	protected void on_(OutputStream stream) throws IOException {
		this.initialize();
		super.on_(stream);
	}

	/**
	 * Answer my current loop count.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int loop() {
		return loopCount;
	}

	/**
	 * Set my new loop count.
	 * 
	 * @param howMany int
	 * @category accessing
	 */
	public void loop_(int howMany) {
		if (howMany < 0) {
			loopCount = 0;
		} else {
			loopCount = Math.max(0, Math.min(howMany, 0xFFFF));
		}
	}

	/**
	 * Answer my current ticking time.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int tick() {
		return delayTime * 10;
	}

	/**
	 * Set my new ticking time.
	 * 
	 * @param millisecondTime int
	 * @category accessing
	 */
	public void tick_(int millisecondTime) {
		delayTime = Math.max(0, Math.min(millisecondTime / 10, 0xFFFF));
	}

	/**
	 * Read the image from the input stream.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunGifImageStream#nextImage()
	 * @category accessing
	 */
	public StImage nextImage() throws IOException {
		try {
			IIOMetadata metadata = _imageReader.getImageMetadata(_imageIndex);
			String formatName = metadata.getNativeMetadataFormatName();
			Node node = metadata.getAsTree(formatName);
			node = node.getFirstChild();
			while (node != null) {
				if ("ImageDescriptor".equals(node.getNodeName())) {
					NamedNodeMap aMap = node.getAttributes();
					Node xNode = aMap.getNamedItem("imageLeftPosition");
					Node yNode = aMap.getNamedItem("imageTopPosition");
					if (xNode != null && yNode != null) {
						int x = Integer.parseInt(xNode.getNodeValue());
						int y = Integer.parseInt(yNode.getNodeValue());
						_imageReadParam.setDestinationOffset(new Point(x, y));
					}
				} else if ("GraphicControlExtension".equals(node.getNodeName())) {
					NamedNodeMap aMap = node.getAttributes();
					Node aNode = aMap.getNamedItem("delayTime");
					if (aNode != null) {
						delayTime = Integer.parseInt(aNode.getNodeValue());
					}
				}

				node = node.getNextSibling();
			}

			BufferedImage image = _imageReader.read(_imageIndex, _imageReadParam);
			_imageIndex++;
			_imageReadParam.setDestination(image);

			StImage anImage = new StImage(image);
			anImage = (StImage) anImage.clone();
			return anImage;

		} catch (IndexOutOfBoundsException e) {
			return null;
		} catch (Exception e) {
			e.printStackTrace();
			System.err.println("image index at " + _imageIndex);
			return null;
		}
	}

	/**
	 * Read a header of the image.
	 * 
	 * @throws java.io.IOException
	 * @category decoding
	 */
	protected void readHeader() throws IOException {
		if (this.hasMagicNumber_("GIF89a".getBytes()) == false) {
			throw SmalltalkException.Error("can't read the image");
		}

		this.readWord(); // width
		this.readWord(); // height

		byte b = this.next();
		boolean hasColorMap = (b & 128) != 0;
		bitsPerPixel = (b & 7) + 1;
		this.next(); // Background Color Index
		this.next(); // Pixel Aspect Ratio

		if (hasColorMap) {
			int size = 1 << bitsPerPixel;
			byte[] rPalette = new byte[size];
			byte[] gPalette = new byte[size];
			byte[] bPalette = new byte[size];
			for (int i = 0; i < size; i++) {
				rPalette[i] = this.next();
				gPalette[i] = this.next();
				bPalette[i] = this.next();
			}
			colorModel = new IndexColorModel(bitsPerPixel, size, rPalette, gPalette, bPalette);
		} else {
			System.err.println("GIF file does not have a color table.");
			colorModel = JunImageProcessor._BlackWhitePalette();
		}
	}

	/**
	 * Read an application extension block of the image.
	 * 
	 * @throws java.io.IOException
	 * @category decoding
	 */
	protected void readApplicationExtensionBlock() throws IOException {
		if (this.next() != Extension) {
			return;
		}
		if (this.next() != ApplicationExtensionLabel) {
			return;
		}
		if (this.next() != 0x0b) {
			return;
		}

		byte[] bytes = "NETSCAPE".getBytes();
		if (Arrays.equals(bytes, this.next_(bytes.length)) == false) {
			return;
		}

		bytes = "2.0".getBytes();
		if (Arrays.equals(bytes, this.next_(bytes.length)) == false) {
			return;
		}

		if (this.next() != 0x03) {
			// return;
		}
		if (this.next() != 0x01) {
			// return;
		}

		loopCount = this.readWord();

		if (this.next() != 0) {
			// return;
		}
	}

	/**
	 * Read the word from the current position.
	 * 
	 * @return int
	 * @throws java.io.IOException
	 * @category decoding
	 */
	protected int readWord() throws IOException {
		return (this.next() & 0xFF) + (this.next() << 8);
	}

	/**
	 * Answer true if the receiver has bytes as a magic number from the current position, otherwise false.
	 * 
	 * @param bytes byte[]
	 * @return boolean
	 * @throws java.io.IOException
	 * @category private
	 */
	protected boolean hasMagicNumber_(byte[] bytes) throws IOException {
		int size = this._fillBuffer(bytes.length);
		if (size < bytes.length) {
			return false;
		}

		_byteBuffer.mark();
		byte[] array = new byte[bytes.length];
		_byteBuffer.get(array);
		if (Arrays.equals(bytes, array)) {
			return true;
		}

		_byteBuffer.reset();
		return false;
	}

	/**
	 * Fill the byte buffer enough for the size.
	 * 
	 * @param size int
	 * @return int
	 * @throws java.io.IOException
	 * @category buffer accessing
	 */
	protected int _fillBuffer(int size) throws IOException {
		if (_byteBuffer == null) {
			_byteBuffer = ByteBuffer.allocate(Math.max(size, _ByteBufferSize));
			size = _byteChannel.read(_byteBuffer);
			_byteBuffer.clear();
			return size;
		}

		int number = _byteBuffer.limit() - _byteBuffer.position();
		if (number >= size) {
			return number;
		}

		ByteBuffer buffer;
		if (_byteBuffer.capacity() < size) {
			buffer = ByteBuffer.allocate(Math.max(size, _ByteBufferSize));
			buffer.put(_byteBuffer);
			_byteBuffer = buffer;
		} else {
			_byteBuffer.compact();
		}
		buffer = ByteBuffer.allocate(_byteBuffer.limit() - _byteBuffer.position());
		_byteChannel.read(buffer);
		buffer.clear();
		_byteBuffer.put(buffer);
		_byteBuffer.flip();

		return _byteBuffer.limit() - _byteBuffer.position();
	}

	/**
	 * Write the image on the output stream.
	 *
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunGifImageStream#nextPutImage_(jp.co.sra.smalltalk.StImage)
	 * @category accessing
	 */
	public void nextPutImage_(StImage anImage) throws IOException {
		byte[] bits = this.bitsFor_(anImage);

		if (firstTime) {
			this.writeHeader();
			this.writeApplicationExtensionBlock();
			firstTime = false;
		}

		this.writeGraphicControlExtensionBlock();
		this.writeImageBlock_(bits);
	}

	/**
	 * Write the application extension block of the image.
	 * 
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeApplicationExtensionBlock() throws IOException {
		this.nextPut_(Extension);
		this.nextPut_(ApplicationExtensionLabel);
		this.nextPut_(0x0b); // Block Size #1
		this.nextPutAll_("NETSCAPE".getBytes()); // Application Identifier
		this.nextPutAll_("2.0".getBytes()); // Application Authentication Code
		this.nextPut_(0x03); // Block Size #2
		this.nextPut_(0x01);
		this.writeWord_(loopCount);
		this.nextPut_(0);
	}

	/**
	 * Write the graphic control extension block of the image.
	 * 
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunSraGifImageStream#writeGraphicControlExtensionBlock()
	 * @category encoding
	 */
	protected void writeGraphicControlExtensionBlock() throws IOException {
		this.nextPut_(Extension);
		this.nextPut_(GraphicControlLabel);
		this.nextPut_(4);
		if (transparentColorIndex >= 0) {
			this.nextPut_(1);
		} else {
			this.nextPut_(0);
		}
		this.writeWord_(delayTime);
		if (transparentColorIndex >= 0) {
			this.nextPut_(transparentColorIndex);
		} else {
			this.nextPut_(0);
		}
		this.nextPut_(0); // Block Terminator
	}

	/**
	 * Write the image block of the image.
	 * 
	 * @param bits byte[]
	 * @throws java.io.IOException
	 * @category encoding
	 */
	protected void writeImageBlock_(byte[] bits) throws IOException {
		this.nextPut_(ImageSeparator);
		this.writeWord_(0);
		this.writeWord_(0);
		this.writeWord_(width);
		this.writeWord_(height);
		this.nextPut_(interlace ? 64 : 0);

		this.writeBitData_(bits);
	}

	/**
	 * Answer true if the receiver reaches at the end, otherwise false.
	 * 
	 * @return boolean
	 * @throws java.io.IOException
	 * @category stream access
	 */
	public boolean atEnd() throws IOException {
		return this._fillBuffer(1) == 0;
	}

	/**
	 * Answer the byte in the current position.
	 * 
	 * @return byte
	 * @throws java.io.IOException
	 * @category stream access
	 */
	public byte next() throws IOException {
		this._fillBuffer(1);
		return _byteBuffer.get();
	}

	/**
	 * Answer the bytes from the current position.
	 * 
	 * @param size int
	 * @return byte[]
	 * @throws java.io.IOException
	 * @category stream access
	 */
	public byte[] next_(int size) throws IOException {
		if (this._fillBuffer(size) < size) {
			return null;
		}

		byte[] bytes = new byte[size];
		_byteBuffer.get(bytes);
		return bytes;
	}

	/**
	 * Close the stream.
	 * 
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#close()
	 * @category stream access
	 */
	public void close() throws IOException {
		if (outStream != null) {
			this.nextPut_(Terminator);
		}

		if (_imageReader != null) {
			ImageInputStream anImageInputStream = (ImageInputStream) _imageReader.getInput();
			anImageInputStream.close();
			_imageReader.dispose();
			_imageReader = null;
		}

		if (_byteChannel != null) {
			_byteChannel.close();
			_byteChannel = null;
			inStream = null;
		}

		super.close();
	}

}
