package jp.co.sra.jun.goodies.font;

import java.awt.Font;
import java.awt.Frame;
import java.awt.GraphicsEnvironment;
import java.io.PrintWriter;
import java.io.StringWriter;

import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.goodies.display.JunEmbeddedDisplayModel;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunFontModel class
 * 
 *  @author    nisinaka
 *  @created   2003/02/28 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun642 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFontModel.java,v 8.13 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunFontModel extends JunApplicationModel {

	public static final int DefaultTextStyleHeight = 72;
	protected static final String TextStyleString = "abcdefghijklmnopqrstuvwxyz";

	/**
	 * Answer the text style with the default height.
	 * 
	 * @return java.awt.Font
	 * @category Accessing
	 */
	public static Font TextStyle() {
		return TextStyle(DefaultTextStyleHeight);
	}

	/**
	 * Answer the text style with the specified height.
	 * 
	 * @param height float
	 * @return java.awt.Font
	 * @category Accessing
	 */
	public static Font TextStyle(float height) {
		Font aFont = null;
		Font defaultFont = StComposedText.DefaultFont();
		int maxScaleFactor = 256;
		for (int scaleFactor = 1; scaleFactor <= maxScaleFactor; scaleFactor++) {
			aFont = defaultFont.deriveFont((float) scaleFactor);
			StComposedText text = new StComposedText(TextStyleString, aFont);
			if (text.height() > height) {
				break;
			}
		}
		return aFont;
	}

	////////////////////////////////////////////////////////////////////////////////

	protected JunEmbeddedDisplayModel displayModel;
	protected String[] fontNames;
	protected String fontName;
	protected int[] fontSizes;
	protected int fontSize;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		fontName = null;
		fontSize = 0;
	}

	/**
	 * Answer my current display model.
	 *
	 * @return jp.co.sra.jun.goodies.display.JunEmbeddedDisplayModel
	 * @category accessing
	 */
	public JunEmbeddedDisplayModel displayModel() {
		if (displayModel == null) {
			displayModel = new JunEmbeddedDisplayModel();
		}
		return displayModel;
	}

	/**
	 * Answer the current font names.
	 *
	 * @return java.lang.String[]
	 * @category accessing
	 */
	public String[] fontNames() {
		if (fontNames == null) {
			Font[] fonts = GraphicsEnvironment.getLocalGraphicsEnvironment().getAllFonts();
			fontNames = new String[fonts.length];
			for (int i = 0; i < fonts.length; i++) {
				fontNames[i] = fonts[i].getFontName();
			}
		}
		return fontNames;
	}

	/**
	 * Answer the current font name.
	 *
	 * @return java.lang.String
	 * @category accessing
	 */
	public String fontName() {
		if (fontName == null) {
			fontName = this.defaultFontName();
		}
		return fontName;
	}

	/**
	 * Set the new font name.
	 *
	 * @param newFontName java.lang.String
	 * @category accessing
	 */
	public void fontName_(String newFontName) {
		fontName = newFontName;
	}

	/**
	 * Answer the current font sizes.
	 *
	 * @return int[]
	 * @category accessing
	 */
	public int[] fontSizes() {
		if (fontSizes == null) {
			fontSizes = this.defaultFontSizes();
		}
		return fontSizes;
	}

	/**
	 * Answer the current font size.
	 *
	 * @return int
	 * @category accessing
	 */
	public int fontSize() {
		if (fontSize <= 0) {
			fontSize = this.defaultFontSize();
		}
		return fontSize;
	}

	/**
	 * Set the new font size.
	 *
	 * @param newFontSize int
	 * @category accessing
	 */
	public void fontSize_(int newFontSize) {
		fontSize = newFontSize;
	}

	/**
	 * Answer the current font.
	 *
	 * @return java.awt.Font
	 * @category accessing
	 */
	public Font getFont() {
		return new Font(this.fontName(), Font.PLAIN, this.fontSize());
	}

	/**
	 * Answer the default font name.
	 *
	 * @return java.lang.String
	 * @category defaults
	 */
	protected String defaultFontName() {
		return "Dialog";
	}

	/**
	 * Answer the default font size.
	 *
	 * @return int
	 * @category defaults
	 */
	protected int defaultFontSize() {
		return 12;
	}

	/**
	 * Answer the default font sizes.
	 *
	 * @return int[]
	 * @category defaults
	 */
	protected int[] defaultFontSizes() {
		return new int[] { 9, 10, 12, 14, 18, 24, 36, 48, 72 };
	}

	/**
	 * Answer the default text.
	 *
	 * @return java.lang.String
	 * @category defaults
	 */
	protected String defaultText() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		try {
			pw.print("abcdefghijklmnopqrstuvwxyz 0123456789_~!\"'|\\,");
			pw.println();
			pw.print("ABCDEFGHIJKLMNOPQRSTUVWXYZ +-*/=<>{}()#$%&@^.");
			pw.println();
			pw.print($String("AIUEO", ""));
		} finally {
			pw.flush();
			pw.close();
		}
		return sw.toString();
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFontViewAwt(this);
		} else {
			return new JunFontViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Font Model");
	}

	/**
	 * Update the visual object when opened.
	 *
	 * @return java.awt.Frame
	 * @param aView jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	protected Frame allButOpenView_(StView aView) {
		Frame aFrame = super.allButOpenView_(aView);
		this.updateVisualObject();
		return aFrame;
	}

	/**
	 * Update the visual object.
	 * 
	 * @category actions
	 */
	protected void updateVisualObject() {
		this.displayModel().visual_(new StComposedText(this.defaultText(), this.getFont()));
	}

	/**
	 * Answer the index of the current font name.
	 *
	 * @return int
	 * @category aspects
	 */
	protected int getFontNameIndex() {
		int index = -1;
		String fontName = this.fontName();
		String[] fontNames = this.fontNames();
		for (int i = 0; i < fontNames.length; i++) {
			if (fontName.equals(fontNames[i])) {
				index = i;
				break;
			}
		}
		return index;
	}

	/**
	 * Set the new index of the font name.
	 *
	 * @param newFontNameIndex int
	 * @category aspects
	 */
	protected void setFontNameIndex(int newFontNameIndex) {
		fontName = this.fontNames()[newFontNameIndex];
		this.updateVisualObject();
	}

	/**
	 * Answer the index of the current font size.
	 *
	 * @return int
	 * @category aspects
	 */
	protected int getFontSizeIndex() {
		int index = -1;
		int fontSize = this.fontSize();
		int[] fontSizes = this.fontSizes();
		for (int i = 0; i < fontSizes.length; i++) {
			if (fontSize == fontSizes[i]) {
				index = i;
				break;
			}
		}
		return index;
	}

	/**
	 * Set the new index of the font size.
	 *
	 * @param newFontSizeIndex int
	 * @category aspects
	 */
	protected void setFontSizeIndex(int newFontSizeIndex) {
		fontSize = this.fontSizes()[newFontSizeIndex];
		this.updateVisualObject();
	}

}
