package jp.co.sra.jun.goodies.files;

import java.awt.Dialog;
import java.io.File;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSimpleDialog;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunFileDialog class
 * 
 *  @author    nisinaka
 *  @created   2000/12/18 (by nisinaka)
 *  @updated   2003/03/09 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileDialog.java,v 8.13 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunFileDialog extends StSimpleDialog {

	protected static File PreviousPath = null;

	protected JunFileModel fileModel;
	protected StBlockClosure acceptBlock;
	protected StBlockClosure cancelBlock;

	/**
	 * Flush the previous path.
	 * 
	 * @category Class initialization
	 */
	public static void FlushPrevious() {
		PreviousPath = null;
	}

	/**
	 * Answer the current previous path.
	 *
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File PreviousPath() {
		return PreviousPath;
	}

	/**
	 * Set the new previous path.
	 *
	 * @param aFile java.io.File
	 * @category Utilities
	 */
	public static void PreviousPath_(File aFile) {
		PreviousPath = aFile;
	}

	/**
	 * Open a file dialog and let a user to choose.
	 * 
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request() {
		return Request(null);
	}

	/**
	 * Open a file dialog showing the message string and let a user to choose.
	 *
	 * @return java.io.File
	 * @param messageString java.lang.String
	 * @category Utilities
	 */
	public static File Request(String messageString) {
		return Request(messageString, null, null, null, null, null);
	}

	/**
	 * Open a file dialog and let a user to choose.
	 *
	 * @param messageString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return Request(messageString, null, fileTypes, initialFileType, null, null);
	}

	/**
	 * Open a file dialog and let a user to choose.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @param acceptBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param cancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString, File initialFile, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType, StBlockClosure acceptBlock, StBlockClosure cancelBlock) {
		JunFileDialog fileDialog = new JunFileDialog();
		JunFileModel fileModel = fileDialog.fileModel();

		if (fileTypes != null) {
			fileModel.fileTypes_(fileTypes);
		}

		if (initialFileType != null) {
			fileModel.fileType().value_(initialFileType);
		} else {
			fileTypes = fileModel.fileTypes();
			if (fileTypes != null && fileTypes.length > 0) {
				fileModel.fileType().value_(fileTypes[0]);
			}
		}

		if (initialFile == null) {
			if (PreviousPath != null && PreviousPath.exists()) {
				fileModel.value_(PreviousPath);
			} else {
				FlushPrevious();
			}
		} else {
			if (initialFile.isAbsolute()) {
				fileModel.value_(initialFile);
			} else {
				if (PreviousPath != null && PreviousPath.exists()) {
					fileModel.value_(new File(PreviousPath, initialFile.getPath()));
				} else {
					FlushPrevious();
					fileModel.value_(initialFile);
				}
			}
		}

		Dialog dialog = fileDialog.allButOpenDialog();
		if (messageString != null) {
			dialog.setTitle(messageString);
		}
		if (acceptBlock != null) {
			fileDialog.acceptBlock_(acceptBlock);
		}
		if (cancelBlock != null) {
			fileDialog.cancelBlock_(cancelBlock);
		}
		_ShowAtMousePoint(dialog);
		if (fileDialog.accept()._booleanValue()) {
			File aFile = (File) fileModel.value();
			if (aFile.exists() && aFile.isDirectory()) {
				PreviousPath = aFile;
			} else {
				PreviousPath = aFile.getParentFile();
			}
			return aFile;
		} else {
			return null;
		}
	}

	/**
	 * Open a file dialog and let a user to choose.
	 *
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @param initialFilenameString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @param acceptBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param cancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File Request(String messageString, File initialDirectory, String initialFilenameString, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType, StBlockClosure acceptBlock, StBlockClosure cancelBlock) {
		File initialFile = null;
		if (initialDirectory != null) {
			initialFile = initialDirectory;
		}
		if (initialFilenameString != null && initialFilenameString.length() > 0) {
			initialFile = new File(initialFile, initialFilenameString);
		}

		return Request(messageString, initialFile, fileTypes, initialFileType, acceptBlock, cancelBlock);
	}

	/**
	 * Request for a directory.
	 *
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestDirectory() {
		return RequestDirectory($String("Select a directory."));
	}

	/**
	 * Request for a directory.
	 *
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestDirectory(String messageString) {
		return RequestDirectory(messageString, null);
	}

	/**
	 * Request for a directory.
	 *
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestDirectory(String messageString, File initialDirectory) {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { JunFileModel.FileType.All(JunSystem.$String("All files")) };
		return RequestDirectory(messageString, initialDirectory, fileTypes, fileTypes[0]);
	}

	/**
	 * Request for a directory.
	 *
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestDirectory(String messageString, File initialDirectory, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return Request(messageString, initialDirectory, null, fileTypes, initialFileType, new StBlockClosure() {
			public Object value_(Object anObject) {
				JunFileModel fileModel = (JunFileModel) anObject;
				File aFile = (File) fileModel.value();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						return Boolean.TRUE;
					} else {
						message += aFile.getPath() + JunSystem.$String(" is not a directory.") + "\n";
						message += "(" + JunSystem.$String("It may be a file.") + ")\n";
						message += JunSystem.$String("Input a directory.");
						JunDialog.Warn_(message);
						fileModel.fileName().value_(new String());
						return Boolean.FALSE;
					}
				} else {
					String message = aFile.getPath() + JunSystem.$String(" does not exist.") + "\n";
					message += JunSystem.$String("Input a directory.");
					JunDialog.Warn_(message);
					fileModel.fileName().value_(new String());
					return Boolean.FALSE;
				}
			}
		}, new StBlockClosure() {
			public Object value_(Object fileModel) {
				return Boolean.TRUE;
			}
		});
	}

	/**
	 * Request for a file.
	 *
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile() {
		return RequestFile($String("Select a file."));
	}

	/**
	 * Request for a file.
	 *
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString) {
		return RequestFile(messageString, null);
	}

	/**
	 * Request for a file.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString, File initialFile) {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { JunFileModel.FileType.All(JunSystem.$String("All files")) };
		return RequestFile(messageString, initialFile, fileTypes, fileTypes[0]);
	}

	/**
	 * Request for a file.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestFile(String messageString, File initialFile, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return Request(messageString, initialFile, null, fileTypes, initialFileType, new StBlockClosure() {
			public Object value_(Object anObject) {
				JunFileModel fileModel = (JunFileModel) anObject;
				File aFile = (File) fileModel.value();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isFile()) {
						return Boolean.TRUE;
					} else {
						message += aFile.getPath() + JunSystem.$String(" is not a file.") + "\n";
						message += "(" + JunSystem.$String("It may be a directory.") + ")\n";
						message += JunSystem.$String("Input a file.");
						JunDialog.Warn_(message);
						fileModel.fileName().value_(new String());
						return Boolean.FALSE;
					}
				} else {
					String message = aFile.getPath() + JunSystem.$String(" does not exist.") + "\n";
					message += JunSystem.$String("Input a file.");
					JunDialog.Warn_(message);
					fileModel.fileName().value_(new String());
					return Boolean.FALSE;
				}
			}
		}, new StBlockClosure() {
			public Object value_(Object fileModel) {
				return Boolean.TRUE;
			}
		});
	}

	/**
	 * Request for a new directory.
	 *
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewDirectory(String messageString) {
		return RequestNewDirectory(messageString, null);
	}

	/**
	 * Request for a new directory.
	 *
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewDirectory(String messageString, File initialDirectory) {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { JunFileModel.FileType.All(JunSystem.$String("All files")) };
		return RequestNewDirectory(messageString, initialDirectory, fileTypes, fileTypes[0]);
	}

	/**
	 * Request for a new directory.
	 *
	 * @param messageString java.lang.String
	 * @param initialDirectory java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewDirectory(String messageString, File initialDirectory, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return Request(messageString, initialDirectory, null, fileTypes, initialFileType, new StBlockClosure() {
			public Object value_(Object fileModel) {
				File aFile = (File) ((JunFileModel) fileModel).value();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						message += aFile.getPath() + JunSystem.$String(" already exists.") + "\n";
						message += JunSystem.$String("Do you want to go ahead?");
						return new Boolean(JunDialog.Confirm_(message, false));
					} else {
						message += aFile.getPath() + JunSystem.$String(" already exists as file.") + "\n";
						message += JunSystem.$String("Input a directory.");
						JunDialog.Warn_(message);
						return Boolean.FALSE;
					}
				} else {
					return Boolean.TRUE;
				}
			}
		}, new StBlockClosure() {
			public Object value_(Object fileModel) {
				return Boolean.TRUE;
			}
		});
	}

	/**
	 * Request for a new file.
	 *
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString) {
		return RequestNewFile(messageString, null, null, null);
	}

	/**
	 * Request for a new file.
	 *
	 * @param messageString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return RequestNewFile(messageString, null, fileTypes, initialFileType);
	}

	/**
	 * Request for a new file.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString, File initialFile) {
		return RequestNewFile(messageString, initialFile, null, null);
	}

	/**
	 * Request for a new file.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 */
	public static File RequestNewFile(String messageString, File initialFile, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return Request(messageString, initialFile, fileTypes, initialFileType, new StBlockClosure() {
			public Object value_(Object fileModel) {
				File aFile = (File) ((JunFileModel) fileModel).value();
				if (aFile.exists()) {
					String message = "";
					if (aFile.isDirectory()) {
						message += aFile.getPath() + JunSystem.$String(" is not a file.") + "\n";
						message += "(" + JunSystem.$String("It may be a directory.") + ")\n";
						message += JunSystem.$String("Input a file.");
						JunDialog.Warn_(message);
						return Boolean.FALSE;
					} else {
						message += aFile.getPath() + JunSystem.$String(" already exists.") + "\n";
						message += JunSystem.$String("Do you want to go ahead?");
						return new Boolean(JunDialog.Confirm_(message, false));
					}
				} else {
					return Boolean.TRUE;
				}
			}
		}, new StBlockClosure() {
			public Object value_(Object fileModel) {
				return Boolean.TRUE;
			}
		});
	}

	/**
	 * Request for a new file name.
	 *
	 * @param messageString java.lang.String
	 * @return java.io.File
	 * @category Friendly utilities
	 * @deprecated since Jun637, use RequestNewFile.
	 */
	public static File RequestNewFilename(String messageString) {
		return RequestNewFile(messageString);
	}

	/**
	 * Request for a new file name.
	 *
	 * @param messageString java.lang.String
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 * @deprecated since Jun637, use RequestNewFile.
	 */
	public static File RequestNewFilename(String messageString, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return RequestNewFile(messageString, fileTypes, initialFileType);
	}

	/**
	 * Request for a new file name.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @return java.io.File
	 * @category Friendly utilities
	 * @deprecated since Jun637, use RequestNewFile.
	 */
	public static File RequestNewFilename(String messageString, File initialFile) {
		return RequestNewFile(messageString, initialFile);
	}

	/**
	 * Request for a new file name.
	 *
	 * @param messageString java.lang.String
	 * @param initialFile java.io.File
	 * @param fileTypes jp.co.sra.jun.goodies.files.JunFileModel.FileType[]
	 * @param initialFileType jp.co.sra.jun.goodies.files.JunFileModel.FileType
	 * @return java.io.File
	 * @category Friendly utilities
	 * @deprecated since Jun637, use RequestNewFile.
	 */
	public static File RequestNewFilename(String messageString, File initialFile, JunFileModel.FileType[] fileTypes, JunFileModel.FileType initialFileType) {
		return RequestNewFile(messageString, initialFile, fileTypes, initialFileType);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		this.fileModel().currentDirectory();
		acceptBlock = null;
		cancelBlock = null;
	}

	/**
	 * Answer my file model.
	 *
	 * @return jp.co.sra.jun.goodies.files.JunFileModel
	 * @category accessing
	 */
	public JunFileModel fileModel() {
		if (fileModel == null) {
			fileModel = new JunFileModel();
		}
		return fileModel;
	}

	/**
	 * Set the new accept block.
	 *
	 * @param newAcceptBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	protected void acceptBlock_(StBlockClosure newAcceptBlock) {
		acceptBlock = newAcceptBlock;
	}

	/**
	 * Set the new cancel block.
	 *
	 * @param newCancelBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	protected void cancelBlock_(StBlockClosure newCancelBlock) {
		cancelBlock = newCancelBlock;
	}

	/**
	 * Called when an accept button is pressed.
	 *
	 * @return boolean
	 * @category actions
	 */
	public boolean acceptButtonPressed() {
		if (acceptBlock != null) {
			return ((Boolean) acceptBlock.value_(this.fileModel())).booleanValue();
		}
		return true;
	}

	/**
	 * Called when a cancel button is pressed.
	 *
	 * @return boolean
	 * @category actions
	 */
	public boolean cancelButtonPressed() {
		if (cancelBlock != null) {
			return ((Boolean) cancelBlock.value_(this.fileModel())).booleanValue();
		}
		return true;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunFileDialogViewAwt(this);
		} else {
			return new JunFileDialogViewSwing(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("File Dialog");
	}

}
