package jp.co.sra.jun.goodies.drawing.map;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StReadStream;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.calendar.JunCalendarModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.drawing.element.JunDrawingElement;
import jp.co.sra.jun.goodies.drawing.element.JunEllipseElement;
import jp.co.sra.jun.goodies.drawing.element.JunFreehandElement;
import jp.co.sra.jun.goodies.drawing.element.JunImageElement;
import jp.co.sra.jun.goodies.drawing.element.JunLabelElement;
import jp.co.sra.jun.goodies.drawing.element.JunLinkElement;
import jp.co.sra.jun.goodies.drawing.element.JunNodeElement;
import jp.co.sra.jun.goodies.drawing.element.JunPathElement;
import jp.co.sra.jun.goodies.drawing.element.JunRectangleElement;
import jp.co.sra.jun.goodies.drawing.element.JunRoundRectangleElement;
import jp.co.sra.jun.goodies.drawing.element.JunTextElement;
import jp.co.sra.jun.goodies.drawing.element.JunTextboxElement;
import jp.co.sra.jun.goodies.drawing.element.JunVertexesElement;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispParser;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunDrawingMapModel class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawingMapModel.java,v 8.12 2008/02/20 06:31:24 nisinaka Exp $
 */
public class JunDrawingMapModel extends JunApplicationModel implements ClipboardOwner {
	protected JunDrawingMap mapObject;
	protected Jun2dPoint scalePoint;

	protected HashMap popupMenuTable;
	protected StMenuBar _menuBar;

	protected transient File fileName;

	public static final String DefaultHeader = "% Jun Drawing Map";

	/**
	 * Create a new instance of JunDrawingMapModel and read from user.
	 * 
	 * @param initialFile java.io.File
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel
	 * @category Instance creation
	 */
	public static JunDrawingMapModel FromUser_(File initialFile) {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "DRW"), new String[] { "*.drw", "*.DRW" }) };
		File file = (initialFile == null) ? null : initialFile;
		file = JunFileRequesterDialog.Request($String("Select a <1p> file.", null, "DRW"), file, fileTypes, fileTypes[0], null, null);
		if (file == null) {
			return null;
		}

		JunDrawingMapModel aModel = null;
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new InputStreamReader(new FileInputStream(file)));
			String line = reader.readLine();
			if (line.length() < DefaultHeader.length()) {
				JunDialog.Warn_($String("<1p> files", null, file.getName()) + $String(" can not read."));
				return null;
			}
			String header = line.substring(0, DefaultHeader.length());
			if (header.equals(DefaultHeader) == false) {
				JunDialog.Warn_($String("<1p> files", null, file.getName()) + $String(" can not read."));
				return null;
			}
			reader.close();
			reader = null;

			reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), System.getProperty("file.encoding")));
			StringWriter sw = new StringWriter();
			String lineSeparator = (String) System.getProperties().get("line.separator");

			while ((line = reader.readLine()) != null) {
				sw.write(line);
				sw.write(lineSeparator);
			}
			reader.close();
			reader = null;

			JunLispCons list = (JunLispCons) JunLispParser.Parse_(new StReadStream(sw.toString()));
			aModel = new JunDrawingMapModel(new JunDrawingMap(list));
			aModel.fileName_(file);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		} finally {
			if (reader != null) {
				try {
					reader.close();
					reader = null;
				} catch (IOException e) {
					throw new SmalltalkException(e);
				}
			}
		}
		return aModel;
	}

	/**
	 * Create a new instance of JunDrawingMapModel and initialize it.
	 * 
	 * @param aMap jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category Instance creation
	 */
	public JunDrawingMapModel(JunDrawingMap aMap) {
		super();
		this.mapObject_(aMap);
	}

	/**
	 * Initialize the JunDrawingMapModel.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		mapObject = null;
		scalePoint = null;

		popupMenuTable = null;
		_menuBar = null;

		fileName = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();
		this.mapObject_(null);
	}

	/**
	 * Answer the display object.
	 * 
	 * @return display object
	 * @category accessing
	 */
	public StDisplayable displayObject() {
		return this.mapObject();
	}

	/**
	 * Answer the receiver's map object.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category accessing
	 */
	public JunDrawingMap mapObject() {
		return mapObject;
	}

	/**
	 * Set the receiver's map object.
	 * 
	 * @param aMapObject jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category accessing
	 */
	public void mapObject_(JunDrawingMap aMapObject) {
		if (mapObject != null) {
			mapObject.removeDependentListener(this);
		}

		mapObject = aMapObject;
		if (mapObject != null) {
			mapObject.addDependentListener(this);
		}
	}

	/**
	 * Answer the receiver's scale point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint scalePoint() {
		if (scalePoint == null) {
			scalePoint = Jun2dPoint.Unity();
		}
		return scalePoint;
	}

	/**
	 * Set the receiver's scale point.
	 * 
	 * @param newScalePoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void scalePoint_(Jun2dPoint newScalePoint) {
		if (newScalePoint != null) {
			double scaleX = Math.max(0.01, Math.min(Math.round(newScalePoint.x() * 100) / 100.0, 100.0));
			double scaleY = Math.max(0.01, Math.min(Math.round(newScalePoint.y() * 100) / 100.0, 100.0));
			scalePoint = new Jun2dPoint(scaleX, scaleY);
		} else {
			scalePoint = null;
		}
		this.mapObject().flushBounds();
		this.changed_($("zoom"));
	}

	/**
	 * Set the receiver's scale point.
	 * 
	 * @param newScalePoint double
	 * @category accessing
	 */
	public void scalePoint_(double newScalePoint) {
		this.scalePoint_(newScalePoint, newScalePoint);
	}

	/**
	 * Set the receiver's scale point.
	 * 
	 * @param newScalePointX double
	 * @param newScalePointY double
	 * @category accessing
	 */
	public void scalePoint_(double newScalePointX, double newScalePointY) {
		this.scalePoint_(new Jun2dPoint(newScalePointX, newScalePointY));
	}

	/**
	 * Answer the receiver's file name.
	 *
	 * @return java.io.File
	 * @category accessing
	 */
	public File fileName() {
		return fileName;
	}

	/**
	 * Answer the receiver's scaled size. 
	 * 
	 * @return java.awt.Dimension
	 * @category bounds accessing
	 */
	public Dimension scaledSize() {
		Dimension extent = this.mapObject().extent();
		return new Dimension((int) Math.round(extent.width * this.scalePoint().x()), (int) Math.round(extent.height * this.scalePoint().y()));
	}

	/**
	 * Answer a controller as JunDrawingMapController.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapController
	 * @category controller accessing
	 */
	public JunDrawingMapController getDrawingMapController() {
		JunDrawingMapView aView = this.getDrawingMapView();
		if (aView == null) {
			return null;
		}
		return aView.getDrawingMapController();
	}

	/**
	 * Answer a view as JunDrawingMapView.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapView
	 * @category view accessing
	 */
	public JunDrawingMapView getDrawingMapView() {
		return (JunDrawingMapView) this.getView();
	}

	/**
	 * Notifies this object that it is no longer the owner of
	 * the contents of the clipboard.
	 *
	 * @param clipboard the clipboard that is no longer owned
	 * @param contents the contents which this owner had placed on the clipboard
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
	 * @category clipboards
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}

	/**
	 * Answer the default file.
	 * 
	 * @return java.io.File
	 * @category defaults
	 */
	protected File defaultFile() {
		return new File(this.defaultBaseName() + ".drw");
	}

	/**
	 * Answer the string of default header.
	 *
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultHeader() {
		return JunDrawingMapModel.DefaultHeader;
	}

	/**
	 * Answer the default view of the model.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunDrawingMapViewAwt(this);

		} else {
			return new JunDrawingMapViewSwing(this);
		}
	}

	/**
	 * Answer the receiver's window title.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Drawing");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);
		this.mapObject_(null);
	}

	/**
	 * Update the menu indication.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		this.updateFileMenuIndication();
		this.updateElementMenuIndication();
		this.updateMiscMenuIndication();
	}

	/**
	 * Update the file menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateFileMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("fileMenu"));

		StMenuItem menuElement = aMenu.atNameKey_($("saveMap"));
		if (menuElement != null) {
			if (this.mapObject().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}

		menuElement = aMenu.atNameKey_($("saveAsMap"));
		if (menuElement != null) {
			if (this.mapObject().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}
	}

	/**
	 * Update the element menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateElementMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("elementMenu"));

		StMenuItem menuElement = aMenu.atNameKey_($("addMenu"));
		if (menuElement != null) {
			StMenuItem subMenuElement = ((StMenu) menuElement).atNameKey_($("addLinkElement"));
			if (subMenuElement != null) {
				if (this.mapObject().selectedElements().isEmpty()) {
					subMenuElement.beEnabled(false);
				} else {
					subMenuElement.beEnabled(true);
				}
			}

			subMenuElement = ((StMenu) menuElement).atNameKey_($("addLabelElement"));
			if (subMenuElement != null) {
				JunDrawingElement anElement = this.mapObject().currentElement();
				if (anElement != null && anElement.isLink()) {
					subMenuElement.beEnabled(true);
				} else {
					subMenuElement.beEnabled(false);
				}
			}
		}

		menuElement = aMenu.atNameKey_($("copyElement"));
		if (menuElement != null) {
			if (this.mapObject().selectedElements().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}

		menuElement = aMenu.atNameKey_($("cutElement"));
		if (menuElement != null) {
			if (this.mapObject().selectedElements().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}

		menuElement = aMenu.atNameKey_($("clearElement"));
		if (menuElement != null) {
			if (this.mapObject().selectedElements().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}

		menuElement = aMenu.atNameKey_($("selectAll"));
		if (menuElement != null) {
			if (this.mapObject().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}

		menuElement = aMenu.atNameKey_($("orderMenu"));
		if (menuElement != null) {
			if (this.mapObject().selectedElements().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}

		menuElement = aMenu.atNameKey_($("openProperties"));
		if (menuElement != null) {
			if ((this.mapObject().selectedElements().isEmpty() == false) && (this.mapObject().currentElement().propertiesModel() != null)) {
				menuElement.beEnabled(true);
			} else {
				menuElement.beEnabled(false);
			}
		}
	}

	/**
	 * Update the misc menu indication.
	 * 
	 * @category menu accessing
	 */
	public void updateMiscMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("miscMenu"));

		StMenuItem menuElement = aMenu.atNameKey_($("fitInSpace"));
		if (menuElement != null) {
			if (this.mapObject().isEmpty()) {
				menuElement.beEnabled(false);
			} else {
				menuElement.beEnabled(true);
			}
		}
	}

	/**
	 * Create new model and open it.
	 *
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMapModel
	 * @category menu messages
	 */
	public JunDrawingMapModel newMap() {
		JunDrawingMapModel mapModel = new JunDrawingMapModel(new JunDrawingMap());
		mapModel.open();
		return mapModel;
	}

	/**
	 * Open DrawingMap file.
	 * 
	 * @category menu messages
	 */
	public void openMap() {
		JunDrawingMapModel aModel = FromUser_(this.fileName());
		if (aModel == null) {
			return;
		}

		this.fileName_(aModel.fileName());
		this.mapObject_(aModel.mapObject());
		this.changed_($("redisplay"));
	}

	/**
	 * Save DrawingMap file.
	 * 
	 * @category menu messages
	 */
	public void saveMap() {
		if (this.mapObject().isEmpty()) {
			return;
		}
		if (this.fileName() == null) {
			this.saveAsMap();
			return;
		}

		JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
		BufferedWriter writer = null;
		try {
			cursor._show();
			writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(this.fileName()), System.getProperty("file.encoding")));
			writer.write(this.defaultHeader());
			writer.write(' ');
			writer.write('(');
			writer.write(JunCalendarModel.StringFromDateAndTime_(new Date()));
			writer.write(')');
			writer.newLine();
			writer.newLine();
			this.mapObject().toLispList().saveOn_(writer);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		} finally {
			if (writer != null) {
				try {
					writer.flush();
					writer.close();
					writer = null;
				} catch (IOException e) {
					throw new SmalltalkException(e);
				}
			}
			cursor._restore();
		}
	}

	/**
	 * Save as DrawingMap file.
	 * 
	 * @category menu messages
	 */
	public void saveAsMap() {
		if (this.mapObject().isEmpty()) {
			return;
		}

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "DRW"), new String[] { "*.drw", "*.DRW" }) };
		File file = (this.fileName() == null) ? this.defaultFile() : this.fileName();
		file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, "DRW"), file, fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}
		this.fileName_(file);
		this.saveMap();
	}

	/**
	 * Quit this window.
	 * 
	 * @category menu messages
	 */
	public void quitDoing() {
		this.closeRequest();
	}

	/**
	 * Zoom in the receiver.
	 * 
	 * @category menu messages
	 */
	public void zoomIn() {
		this.scalePoint_(this.scalePoint().x() + 0.1);
	}

	/**
	 * Zoom out the receiver.
	 * 
	 * @category menu messages
	 */
	public void zoomOut() {
		this.scalePoint_(this.scalePoint().x() - 0.1);
	}

	/**
	 * The receiver's zoom lens is 1.0
	 *
	 * @category menu messages
	 */
	public void zoomX1() {
		this.scalePoint_(1.0);
	}

	/**
	 * Copy the receiver's selected elements to clipboard.
	 * 
	 * @category menu messages
	 */
	public void copyElement() {
		if (this.mapObject().isEmpty() || this.mapObject().selectedElements().isEmpty()) {
			return;
		}

		JunDrawingMap theMap = (JunDrawingMap) _New(this.mapObject().getClass());
		JunDrawingElement[] selectedElements = this.mapObject()._selectedElements();
		try {
			for (int i = 0; i < selectedElements.length; i++) {
				theMap.addElement_(selectedElements[i]);
				if (selectedElements[i].isLink()) {
					JunLinkElement linkElement = (JunLinkElement) selectedElements[i];
					if (linkElement.hasLabel() && this.mapObject().selectedElements().contains(linkElement.labelElement()) == false) {
						theMap.addElement_((JunDrawingElement) linkElement.labelElement());
					}
				}
			}

			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			try {
				pw.print(this.defaultHeader());
				pw.print(' ');
				pw.print('(');
				pw.print(JunCalendarModel.StringFromDateAndTime_(new Date()));
				pw.print(')');
				pw.println();
				pw.println();
				theMap.toLispList().saveOn_(pw);
			} catch (IOException e) {
				System.out.println(e.getMessage());
				e.printStackTrace();
			} finally {
				if (pw != null) {
					pw.flush();
					pw.close();
					pw = null;
				}
				if (sw != null) {
					try {
						sw.close();
						sw.flush();
					} catch (IOException e) {
						System.out.println(e.getMessage());
						e.printStackTrace();
					}
				}
			}

			Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
			StringSelection contents = new StringSelection(sw.toString());
			clipboard.setContents(contents, this);
		} finally {
			JunDrawingElement[] elements = theMap._componentElements();
			for (int i = 0; i < elements.length; i++) {
				elements[i].parent_(this.mapObject());
			}
		}
	}

	/**
	 * Cut the receiver's selected elements and copy its to clipboard.
	 * 
	 * @category menu messages
	 */
	public void cutElement() {
		this.copyElement();
		this.mapObject().removeSelectedElements();
		this.mapObject().changed_($("redisplay"));
	}

	/**
	 * Paset the receiver's elements from clipboard and selected its.
	 * 
	 * @category menu messages
	 */
	public void pasteElement() {
		Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		Transferable contents = clipboard.getContents(this);
		BufferedReader reader = null;
		try {
			// check header
			reader = new BufferedReader(new StringReader((String) contents.getTransferData(DataFlavor.stringFlavor)));
			String line = reader.readLine();
			if (line.length() < this.defaultHeader().length()) {
				JunDialog.Warn_($String("The contents of copy buffer is not <1p>.", null, $String("Element")));
				return;
			}
			String header = line.substring(0, this.defaultHeader().length());
			if (header.equals(this.defaultHeader()) == false) {
				JunDialog.Warn_($String("The contents of copy buffer is not <1p>.", null, $String("Element")));
				return;
			}
			reader.close();
			reader = null;

			// create element from clipboard
			JunLispCons list = (JunLispCons) JunLispParser.Parse_(new StReadStream((String) contents.getTransferData(DataFlavor.stringFlavor)));
			JunDrawingMap theMap = (JunDrawingMap) _New(this.mapObject().getClass(), list);
			JunDrawingElement[] elements = theMap._componentElements();
			if (elements == null || elements.length == 0) {
				return;
			}

			// check repetition of id
			JunLinkElement[] linkElements = theMap._linkElements();
			JunLabelElement[] labelElements = theMap._labelElements();
			for (int i = 0; i < elements.length; i++) {
				long oldId = elements[i].id();
				if (this.mapObject().findElement_(oldId) != null) {
					long newId = elements[i].assignId();
					if (elements[i].isNode()) {
						for (int j = 0; j < linkElements.length; j++) {
							if (linkElements[j].fromElementId() == oldId) {
								linkElements[j].fromElement_(null);
								linkElements[j].fromElementId_(newId);
							}
							if (linkElements[j].toElementId() == oldId) {
								linkElements[j].toElement_(null);
								linkElements[j].toElementId_(newId);
							}
							if (linkElements[j].labelElementId() == oldId) {
								linkElements[j].labelElement_(null);
								linkElements[j].labelElementId_(newId);
							}
						}
					} else if (elements[i].isLink()) {
						for (int j = 0; j < labelElements.length; j++) {
							if (labelElements[j].baseElementId() == oldId) {
								labelElements[j].baseElement_(null);
								labelElements[j].baseElementId_(newId);
							}
						}
					}
				}
			}

			// add elements
			for (int i = 0; i < elements.length; i++) {
				this.mapObject().addElement_(elements[i]);
			}
			for (int i = 0; i < linkElements.length; i++) {
				if (linkElements[i].checkElement() == false) {
					this.mapObject().removeElement_(linkElements[i]);
				} else {
					linkElements[i].updateFromToPoint();
				}
			}
			for (int i = 0; i < labelElements.length; i++) {
				if (labelElements[i].checkElement() == false || (labelElements[i].baseElement().hasLabel() && labelElements[i].baseElement().labelElement() != labelElements[i])) {
					this.mapObject().removeElement_(labelElements[i]);
				} else {
					labelElements[i].baseElement().labelElement_(labelElements[i]);
				}
			}

			// add selected elements
			this.mapObject().clearSelectedElements();
			for (int i = 0; i < elements.length; i++) {
				this.mapObject().addSelectedElement_(elements[i]);
			}
			this.mapObject().changed_($("bounds"));
		} catch (IOException e) {
			throw new SmalltalkException(e);
		} catch (UnsupportedFlavorException e) {
		} finally {
			if (reader != null) {
				try {
					reader.close();
					reader = null;
				} catch (IOException e) {
					throw new SmalltalkException(e);
				}
			}
		}
	}

	/**
	 * Clear the receiver's selected elements.
	 * 
	 * @category menu messages
	 */
	public void clearElement() {
		if (this.mapObject().selectedElements().isEmpty()) {
			return;
		}

		if (JunDialog.Confirm_($String("Really clear?"), true)) {
			this.mapObject().removeSelectedElements();
			this.mapObject().changed_($("redisplay"));
		}
	}

	/**
	 * Select the receiver's all elements.
	 * 
	 * @category menu messages
	 */
	public void selectAll() {
		JunDrawingMap aMap = this.mapObject();
		aMap.flushSelectedElements();
		JunDrawingElement[] elements = this.mapObject()._componentElements();
		for (int i = 0; i < elements.length; i++) {
			aMap.addSelectedElement_(elements[i]);
		}
		aMap.changed_($("selection"));
	}

	/**
	 * Move the receiver's selected elements to the top space.
	 * 
	 * @category menu messages
	 */
	public void topElement() {
		JunDrawingMap aMap = this.mapObject();
		if (aMap.selectedElements().isEmpty()) {
			return;
		}

		JunDrawingElement[] selectedElements = aMap._selectedElements();
		for (int i = 0; i < selectedElements.length; i++) {
			aMap.removeElement_(selectedElements[i]);
			aMap.addElement_(selectedElements[i]);
			aMap.addSelectedElement_(selectedElements[i]);
		}
		aMap.changed_($("selection"));
	}

	/**
	 * Move the receiver's selected elements to the front space.
	 * 
	 * @category menu messages
	 */
	public void forwardElement() {
		JunDrawingMap aMap = this.mapObject();
		if (aMap.selectedElements().isEmpty()) {
			return;
		}

		ArrayList elements = aMap.componentElements();
		JunDrawingElement[] selectedElements = aMap._selectedElements();
		for (int i = 0; i < selectedElements.length; i++) {
			JunDrawingElement element = selectedElements[i];
			int index = aMap.componentElementIndexOf_(element);
			if (index + 1 < aMap.componentElementSize() && aMap.selectedElements().contains(elements.get(index + 1)) == false) {
				aMap.removeElement_(element);
				aMap.addElement_beforeIndex_(element, index + 1);
				aMap.addSelectedElement_(element);
			}
		}
		aMap.changed_($("selection"));
	}

	/**
	 * Move the receiver's selected elements to the backward space.
	 * 
	 * @category menu messages
	 */
	public void backwardElement() {
		JunDrawingMap aMap = this.mapObject();
		if (aMap.selectedElements().isEmpty()) {
			return;
		}

		ArrayList elements = aMap.componentElements();
		JunDrawingElement[] selectedElements = aMap._selectedElements();
		for (int i = selectedElements.length - 1; i >= 0; i--) {
			JunDrawingElement element = selectedElements[i];
			int index = aMap.componentElementIndexOf_(element);
			if (index > 0 && aMap.selectedElements().contains(elements.get(index - 1)) == false) {
				aMap.removeElement_(element);
				aMap.addElement_beforeIndex_(element, index - 1);
				aMap.addSelectedElement_(element);
			}
		}
		aMap.changed_($("selection"));
	}

	/**
	 * Move the receiver's selected elements to the bottom space.
	 * 
	 * @category menu messages
	 */
	public void bottomElement() {
		JunDrawingMap aMap = this.mapObject();
		if (aMap.selectedElements().isEmpty()) {
			return;
		}

		JunDrawingElement[] selectedElements = aMap._selectedElements();
		for (int i = 0; i < selectedElements.length; i++) {
			aMap.removeElement_(selectedElements[i]);
			aMap.addElement_beforeIndex_(selectedElements[i], 0);
			aMap.addSelectedElement_(selectedElements[i]);
		}
		aMap.changed_($("selection"));
	}

	/**
	 * Open the receiver's selected element properties.
	 * 
	 * @category menu messages
	 */
	public void openProperties() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null) {
			return;
		}
		anElement.openProperties();
	}

	/**
	 * Add new ellipse element.
	 * 
	 * @category menu messages
	 */
	public void addEllipseElement() {
		JunEllipseElement anElement = new JunEllipseElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new image element.
	 * 
	 * @category menu messages
	 */
	public void addImageElement() {
		JunImageElement anElement = new JunImageElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new label element.
	 * 
	 * @category menu messages
	 */
	public void addLabelElement() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null) {
			return;
		}
		if (anElement.isLink() == false) {
			JunDialog.Warn_($String("The selected element cannot create a <1p>.", null, $String("Label")));
			return;
		}

		JunLinkElement linkElement = (JunLinkElement) anElement;
		if (linkElement.hasLabel()) {
			JunDialog.Warn_($String("The selected element cannot create a <1p>.", null, $String("Label")));
			return;
		}

		JunLabelElement labelElement = new JunLabelElement();
		linkElement.labelElement_(labelElement);
		this.addElement_(labelElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(labelElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new link element.
	 * 
	 * @category menu messages
	 */
	public void addLinkElement() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null) {
			return;
		}
		if (anElement.isNode() == false) {
			JunDialog.Warn_($String("The selected element cannot create a <1p>.", null, $String("Link")));
			return;
		}

		JunNodeElement nodeElement = (JunNodeElement) anElement;
		JunDrawingMapView aView = this.getDrawingMapView();
		if (aView == null || aView.toComponent().isShowing() == false) {
			return;
		}
		JunDrawingMapController mapController = aView.getDrawingMapController();
		mapController.displayLinkPointFrom_to_(nodeElement.center(), mapController.cursorPointInModel());
	}

	/**
	 * Add new path element.
	 * 
	 * @category menu messages
	 */
	public void addPathElement() {
		JunPathElement anElement = new JunPathElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new rectangle element.
	 * 
	 * @category menu messages
	 */
	public void addRectangleElement() {
		JunRectangleElement anElement = new JunRectangleElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new round rectangle element.
	 * 
	 * @category menu messages
	 */
	public void addRoundRectangleElement() {
		JunRoundRectangleElement anElement = new JunRoundRectangleElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new text element.
	 * 
	 * @category menu messages
	 */
	public void addTextElement() {
		JunTextElement anElement = new JunTextElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new textbox element.
	 * 
	 * @category menu messages
	 */
	public void addTextboxElement() {
		JunTextboxElement anElement = new JunTextboxElement();
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new freehand element with specified the points.
	 * 
	 * @param points java.util.ArrayList
	 * @category menu messages
	 */
	public void addFreehandElement(ArrayList points) {
		JunFreehandElement anElement = new JunFreehandElement();
		anElement.points_(points);
		this.addElement_(anElement);
		this.mapObject().clearSelectedElements();
		this.mapObject().addSelectedElement_(anElement);
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Add new specified element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category menu messages
	 */
	public void addElement_(JunDrawingElement anElement) {
		if (anElement == null) {
			return;
		}

		Point centerPoint = null;
		JunDrawingMapView aView = this.getDrawingMapView();
		if (aView != null && anElement.isLink() == false) {
			centerPoint = aView.getDrawingMapController().cursorPoint();
			centerPoint = new Point((int) (centerPoint.x / this.scalePoint().x()), (int) (centerPoint.y / this.scalePoint().y()));
		} else {
			centerPoint = new Point(100, 100);
		}

		if (anElement.isLink()) {
			// no operation
		} else if (anElement.isVertexes()) {
			JunVertexesElement pathElement = (JunVertexesElement) anElement;
			if (pathElement.points().isEmpty()) {
				ArrayList points = new ArrayList();
				points.add(new Point(centerPoint.x - 50, centerPoint.y));
				points.add(new Point(centerPoint.x + 50, centerPoint.y));
				pathElement.points_(points);
			}
		} else if (anElement.isLabel()) {
			JunLabelElement labelElement = (JunLabelElement) anElement;
			int width = 0;
			if (labelElement.text() != null && labelElement.text().length() > 0) {
				width = this.getDrawingMapView().toComponent().getFontMetrics(labelElement.font()).stringWidth(labelElement.text());
			} else {
				width = labelElement.defaultExtent().width;
			}
			labelElement.extent_(width, labelElement.defaultExtent().height);
			Point baseCenter = labelElement.baseElement().center();
			labelElement.location_(baseCenter.x - (width / 2), baseCenter.y - (labelElement.height() / 2));
		} else if (anElement.isRectangular()) {
			Dimension extent = anElement.defaultExtent();
			anElement.location_(centerPoint.x - (extent.width / 2), centerPoint.y - (extent.height / 2));
			anElement.extent_(extent);
		}

		this.mapObject().addElement_(anElement);
	}

	/**
	 * Add the receiver's new vertex
	 * 
	 * @category menu messages
	 */
	public void addVertex() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null || (anElement.isLink() || anElement.isPath()) == false) {
			return;
		}
		JunDrawingMapController controller = this.getDrawingMapController();
		if (controller == null) {
			return;
		}
		Point aPoint = controller.popupMenuPoint();
		if (anElement.containsPoint_(aPoint) == false) {
			return;
		}
		if (anElement.isPath()) {
			((JunPathElement) anElement).addOnLine_(aPoint);
		} else if (anElement.isLink()) {
			((JunLinkElement) anElement).addOnLine_(aPoint);
		}
		this.mapObject().changed_with_($("redisplay"), anElement.bounds());
	}

	/**
	 * Remove the receiver's selected vertex.
	 * 
	 * @category menu messages
	 */
	public void removeVertex() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null || (anElement.isLink() || anElement.isPath()) == false) {
			return;
		}
		JunDrawingMapController controller = this.getDrawingMapController();
		if (controller == null) {
			return;
		}
		Rectangle bounds = new Rectangle(anElement.bounds());

		if (anElement.isPath()) {
			JunPathElement pathElement = ((JunPathElement) anElement);
			Point removePoint = pathElement.pointInControllPoint_(controller.popupMenuPoint());
			if (removePoint == null) {
				return;
			}
			pathElement.remove_(removePoint);
		}
		if (anElement.isLink()) {
			JunLinkElement linkElement = ((JunLinkElement) anElement);
			Point removePoint = linkElement.pathElement().pointInControllPoint_(controller.popupMenuPoint());
			if (removePoint == null) {
				return;
			}
			linkElement.remove_(removePoint);
		}

		bounds.add(anElement.bounds());
		bounds.grow(JunDrawingElement.CONTROLL_AREA_SIZE / 2, JunDrawingElement.CONTROLL_AREA_SIZE / 2);
		this.mapObject().changed_with_($("redisplay"), bounds);
	}

	/**
	 * Change the selected image element from file.
	 * 
	 * @category menu messages
	 */
	public void imageFromFile() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null || anElement.isImage() == false) {
			return;
		}

		JunImageElement imageElement = (JunImageElement) anElement;
		File aFile = imageElement.file() != null ? imageElement.file() : new File(JunSystem.DefaultBaseName() + "." + JunSystem.DefaultImageExtension());
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Image files"), JunSystem.DefaultImageExtensionPatterns()), };
		aFile = JunFileRequesterDialog.Request($String("Select a <1p> file.", null, $String("Image files")), aFile, fileTypes, fileTypes[0], null, null);
		if (aFile == null || aFile.equals(imageElement.file())) {
			return;
		}
		imageElement.file_(aFile);
		this.mapObject().changed_with_($("redisplay"), imageElement.bounds());
	}

	/**
	 * Chnage the selected image element from user.
	 * 
	 * @category menu messages
	 */
	public void imageFromUser() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null || anElement.isImage() == false) {
			return;
		}

		JunImageElement imageElement = (JunImageElement) anElement;
		imageElement.image_(StImage._FromUser());
		this.mapObject().changed_with_($("redisplay"), imageElement.bounds());
	}

	/**
	 * Chnage the selected image element from user.
	 * 
	 * @category menu messages
	 */
	public void imageTakeAway() {
		JunDrawingElement anElement = this.mapObject().currentElement();
		if (anElement == null || anElement.isImage() == false) {
			return;
		}

		JunImageElement imageElement = (JunImageElement) anElement;
		imageElement.image_(null);
		this.mapObject().changed_with_($("redisplay"), imageElement.bounds());
	}

	/**
	 * Fit in space.
	 * 
	 * @category menu messages
	 */
	public void fitInSpace() {
		this.mapObject().flushBounds();
		this.mapObject().changed_($("bounds"));
	}

	/**
	 * Answer the receiver's menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createElementMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	public StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));
		fileMenu.add(new StMenuItem($String("New"), $("newMap"), new MenuPerformer(this, "newMap"), KeyEvent.VK_N, KeyEvent.CTRL_MASK));
		fileMenu.add(new StMenuItem($String("Open") + "...", $("openMap"), new MenuPerformer(this, "openMap"), KeyEvent.VK_O, KeyEvent.CTRL_MASK));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Save") + "...", $("saveMap"), new MenuPerformer(this, "saveMap"), KeyEvent.VK_S, KeyEvent.ALT_MASK));
		fileMenu.add(new StMenuItem($String("Save as..."), $("saveAsMap"), new MenuPerformer(this, "saveAsMap")));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Close"), $("quitDoing"), new MenuPerformer(this, "quitDoing")));
		return fileMenu;
	}

	/**
	 * Create a "View" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	public StMenu _createViewMenu() {
		StMenu viewMenu = new StMenu($String("View Menu", "View"), $("viewMenu"));

		StMenu zoomMenu = new StMenu($String("Zooming"), $("zoomMenu"));
		zoomMenu.add(new StMenuItem($String("Zoom in"), $("zoomIn"), new MenuPerformer(this, "zoomIn")));
		zoomMenu.add(new StMenuItem($String("x 1"), $("zoomX1"), new MenuPerformer(this, "zoomX1")));
		zoomMenu.add(new StMenuItem($String("Zoom out"), $("zoomOut"), new MenuPerformer(this, "zoomOut")));
		viewMenu.add(zoomMenu);
		return viewMenu;
	}

	/**
	 * Create a "Element" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category resources
	 */
	public StMenu _createElementMenu() {
		StMenu elementMenu = new StMenu($String("Element"), $("elementMenu"));

		StMenu addMenu = new StMenu($String("Add"), $("addMenu"));
		addMenu.add(new StMenuItem($String("Text"), $("addTextElement"), new MenuPerformer(this, "addTextElement")));
		addMenu.add(new StMenuItem($String("Ellipse"), $("addEllipseElement"), new MenuPerformer(this, "addEllipseElement")));
		addMenu.add(new StMenuItem($String("Rectangle"), $("addRectangleElement"), new MenuPerformer(this, "addRectangleElement")));
		addMenu.add(new StMenuItem($String("Round Rectangle"), $("addRoundRectangleElement"), new MenuPerformer(this, "addRoundRectangleElement")));
		addMenu.add(new StMenuItem($String("Path"), $("addPathElement"), new MenuPerformer(this, "addPathElement")));
		addMenu.add(new StMenuItem($String("Textbox"), $("addTextboxElement"), new MenuPerformer(this, "addTextboxElement")));
		addMenu.addSeparator();
		addMenu.add(new StMenuItem($String("Link"), $("addLinkElement"), new MenuPerformer(this, "addLinkElement")));
		addMenu.add(new StMenuItem($String("Label"), $("addLabelElement"), new MenuPerformer(this, "addLabelElement")));
		elementMenu.add(addMenu);
		elementMenu.addSeparator();

		elementMenu.add(new StMenuItem($String("Cut"), $("cutElement"), new MenuPerformer(this, "cutElement"), KeyEvent.VK_X, KeyEvent.CTRL_MASK));
		elementMenu.add(new StMenuItem($String("Copy"), $("copyElement"), new MenuPerformer(this, "copyElement"), KeyEvent.VK_C, KeyEvent.CTRL_MASK));
		elementMenu.add(new StMenuItem($String("Paste"), $("pasteElement"), new MenuPerformer(this, "pasteElement"), KeyEvent.VK_V, KeyEvent.CTRL_MASK));
		elementMenu.add(new StMenuItem($String("Clear"), $("clearElement"), new MenuPerformer(this, "clearElement")));
		elementMenu.addSeparator();
		elementMenu.add(new StMenuItem($String("Select all"), $("selectAll"), new MenuPerformer(this, "selectAll"), KeyEvent.VK_A, KeyEvent.CTRL_MASK));
		elementMenu.addSeparator();

		StMenu orderMenu = new StMenu($String("Order"), $("orderMenu"));
		orderMenu.add(new StMenuItem($String("Bring to Front"), $("topElement"), new MenuPerformer(this, "topElement")));
		orderMenu.add(new StMenuItem($String("Bring Forward"), $("frontElement"), new MenuPerformer(this, "forwardElement")));
		orderMenu.add(new StMenuItem($String("Send Backward"), $("backElement"), new MenuPerformer(this, "backwardElement")));
		orderMenu.add(new StMenuItem($String("Send to Back"), $("bottomElement"), new MenuPerformer(this, "bottomElement")));
		elementMenu.add(orderMenu);
		elementMenu.addSeparator();

		elementMenu.add(new StMenuItem($String("Properties"), $("openProperties"), new MenuPerformer(this, "openProperties")));

		return elementMenu;
	}

	/**
	 * Create a "Image Element" menu for JunPathElement.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu[]
	 * @category resources
	 */
	public StMenu[] _createElementMenuForImageElement() {
		StMenu imageMenu = new StMenu($String("Image"), $("imageMenu"));
		imageMenu.add(new StMenuItem($String("From file..."), $("imageFromFile"), new MenuPerformer(this, "imageFromFile")));
		imageMenu.add(new StMenuItem($String("From user..."), $("imageFromUser"), new MenuPerformer(this, "imageFromUser")));
		imageMenu.addSeparator();
		imageMenu.add(new StMenuItem($String("Take away"), $("imageTakeAway"), new MenuPerformer(this, "imageTakeAway")));

		return new StMenu[] { imageMenu };
	}

	/**
	 * Create a "Path Element" menu for JunPathElement.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu[]
	 * @category resources
	 */
	public StMenu[] _createElementMenuForPathElement() {
		StMenu vertexMenu = new StMenu($String("Vertex"), $("vertexMenu"));
		vertexMenu.add(new StMenuItem($String("Add"), $("addVertex"), new MenuPerformer(this, "addVertex")));
		vertexMenu.add(new StMenuItem($String("Remove"), $("removeVertex"), new MenuPerformer(this, "removeVertex")));

		return new StMenu[] { vertexMenu };
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.StMenu
	 * @category resources
	 */
	public StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu($String("Misc"), $("miscMenu"));
		miscMenu.add(new StMenuItem($String("Fit in space"), $("fitInSpace"), new MenuPerformer(this, "fitInSpace")));
		return miscMenu;
	}

	/**
	 * Answer the receiver's popup menu table.
	 * 
	 * @return java.util.HashMap
	 * @category resources
	 */
	public HashMap popupMenuTable() {
		if (popupMenuTable == null) {
			popupMenuTable = new HashMap();
		}
		return popupMenuTable;
	}

	/**
	 * Answer the receiver's popup menu for an element.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	public StPopupMenu popupMenuFor_(JunDrawingElement anElement) {
		StSymbol key = (anElement != null) ? anElement._className() : null;
		if (this.popupMenuTable().containsKey(key) == false) {
			StPopupMenu aPopupMenu = new StPopupMenu();
			StMenuItem[] defaultMenus = ((StMenu) this._menuBar().atNameKey_($($String("elementMenu")))).menuItems();
			for (int i = 0; i < defaultMenus.length; i++) {
				aPopupMenu.add(defaultMenus[i]);
			}
			if (anElement != null && (anElement.isPath() || anElement.isLink())) {
				StMenu[] elementMenus = this._createElementMenuForPathElement();
				if (elementMenus != null && elementMenus.length > 0) {
					for (int i = 0; i < elementMenus.length; i++) {
						aPopupMenu.add(elementMenus[i]);
					}
				}
			}
			if (anElement != null && anElement.isImage()) {
				StMenu[] elementMenus = this._createElementMenuForImageElement();
				if (elementMenus != null && elementMenus.length > 0) {
					for (int i = 0; i < elementMenus.length; i++) {
						aPopupMenu.add(elementMenus[i]);
					}
				}
			}
			this.popupMenuTable().put(key, aPopupMenu);
		}
		return (StPopupMenu) this.popupMenuTable().get(key);
	}

	/**
	 * Answer true if the receiver is zooming, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isZooming() {
		return (Jun2dPoint.Unity().equal_(this.scalePoint()) == false);
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument evt.getAspect() is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		super.update_(evt);
		this.changed_(evt);
	}

	/**
	 * Set the receiver's file name.
	 *
	 * @param aFilename java.io.File
	 * @category private
	 */
	protected void fileName_(File aFilename) {
		fileName = aFilename;
	}
}
