package jp.co.sra.jun.goodies.drawing.element;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.Ellipse2D;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.drawing.properties.JunVertexesPropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunVertexesElement class
 * 
 *  @author    m-asada
 *  @created   2005/10/14 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunVertexesElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public abstract class JunVertexesElement extends JunNodeElement {
	protected List points;
	protected Color foregroundColor;
	protected int lineWidth;

	protected transient Rectangle preferredBounds;
	protected transient Rectangle pointBounds;
	protected transient Shape[] shapes;
	protected transient BasicStroke stroke;
	protected transient JunVertexesPropertiesModel propertiesModel;

	public static final StSymbol CONTROLL_POINT_BEGIN = $("begin");
	public static final StSymbol CONTROLL_POINT_END = $("end");

	public static final int MIN_LINE_WIDTH = 1;
	public static final int MAX_LINE_WIDTH = 10;

	/**
	 * Create a new instance of <code>JunVertexesElement</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunVertexesElement() {
		super();
	}

	/**
	 * Create a new instance of <code>JunVertexesElement</code> and initialize it.
	 * 
	 * @param aList
	 * @category Instance creation
	 */
	public JunVertexesElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		points = null;
		foregroundColor = Color.black;
		lineWidth = 1;

		preferredBounds = null;
		pointBounds = null;
		shapes = null;
		stroke = null;
		propertiesModel = null;
	}

	/**
	 * Release the resources of the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		super.release();

		if (propertiesModel != null) {
			propertiesModel.closeRequest();
			propertiesModel = null;
		}
	}

	/**
	 * Answer the receiver's points.
	 * 
	 * @return java.util.List
	 * @category accessing
	 */
	public List points() {
		if (points == null) {
			points = new ArrayList();
		}
		return points;
	}

	/**
	 * Set the receiver's points.
	 * 
	 * @param newPoints java.util.List
	 * @category accessing
	 */
	public void points_(List newPoints) {
		points = newPoints;
		this.flushBounds();
	}

	/**
	 * Answer the array of receiver's points.
	 * 
	 * @return java.awt.Point[]
	 * @category accessing
	 */
	public Point[] _points() {
		Point[] points = new Point[this.points().size()];
		this.points().toArray(points);
		return points;
	}

	/**
	 * Replaces the element at the specified position in this points with the specified point.
	 * 
	 * @param anInteger int
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IndexOutOfBoundsException
	 * @category accessing
	 */
	public void at_put_(int anInteger, Point aPoint) {
		if (anInteger < 0 || this.points().size() < anInteger) {
			throw new IndexOutOfBoundsException();
		}
		Point newPoint = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		if (this.points().get(anInteger).equals(newPoint) == false) {
			this.points().set(anInteger, newPoint);
			this.flushBounds();
		}
	}

	/**
	 * Answer the point at the specified position in this points.
	 * 
	 * @param anIndex int
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point pointAt_(int anIndex) {
		if (this.points().isEmpty()) {
			return null;
		}
		if (anIndex < 0 || this.points().size() <= anIndex) {
			return null;
		}
		return (Point) this.points().get(anIndex);
	}

	/**
	 * Answer the index of the first occurrence of point within the receiver.
	 * If the receiver does not contain anElement, answer -1.
	 * 
	 * @param aPoint java.awt.Point
	 * @return int
	 * @see java.util.ArrayList#indexOf(java.lang.Object);
	 * @category accessing
	 */
	public int pointIndexOf_(Point aPoint) {
		return this.points().indexOf(aPoint);
	}

	/**
	 * Answer the receiver's begin points.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point begin() {
		return this.pointAt_(0);
	}

	/**
	 * Set the receiver's begin points.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public void begin_(Point aPoint) {
		this.at_put_(0, aPoint);
	}

	/**
	 * Answer the receiver's end points.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point end() {
		return this.pointAt_(this.points().size() - 1);
	}

	/**
	 * Set the receiver's end points.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public void end_(Point aPoint) {
		this.at_put_(this.points().size() - 1, aPoint);
	}

	/**
	 * Add the new point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @category adding
	 */
	public Point add_(Point aPoint) {
		Point newPoint = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		this.points().add(newPoint);
		this.flushBounds();
		return newPoint;
	}

	/**
	 * Add the new point at the specified position in this points.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @category adding
	 */
	public Point add_at_(Point aPoint, int index) {
		Point newPoint = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		this.points().add(index, newPoint);
		this.flushBounds();
		return newPoint;
	}

	/**
	 * Add the argument, newPoint, as an element of the receiver's point.
	 * Put it in the position just succeeding oldPoint.  Answer newPoint.
	 *
	 * @param newPoint java.awt.Point
	 * @param oldPoint java.awt.Point
	 * @return java.awt.Point
	 * @category adding
	 */
	public Point add_after_(Point newPoint, Point oldPoint) {
		int index = this.points().indexOf(oldPoint);
		if (index < 0) {
			return null;
		}
		return this.add_at_(newPoint, index + 1);
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		if (preferredBounds == null) {
			Shape[] shapes = this.shapes();
			for (int i = 0; i < shapes.length; i++) {
				Rectangle shapeBounds = (shapes[i] instanceof Polygon || shapes[i] instanceof Ellipse2D) ? (shapes[i].getBounds()) : (this.stroke().createStrokedShape(shapes[i]).getBounds());
				if (preferredBounds != null) {
					preferredBounds.add(shapeBounds);
				} else {
					preferredBounds = shapeBounds;
				}
			}
			preferredBounds.setSize(preferredBounds.width + 1, preferredBounds.height + 1);
		}
		return preferredBounds;
	}

	/**
	 * Answer the receiver's bounds contains this points.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	protected Rectangle pointBounds() {
		if (pointBounds == null) {
			Point[] pathPoints = this._points();
			for (int i = 0; i < pathPoints.length; i++) {
				if (pointBounds == null) {
					pointBounds = new Rectangle(pathPoints[i].x, pathPoints[i].y, 1, 1);
				} else {
					pointBounds.add(pathPoints[i]);
				}
			}
			pointBounds.setSize(pointBounds.width + 1, pointBounds.height + 1);
		}
		return pointBounds;
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		List newPoints = new ArrayList();
		Point[] points = this._points();
		for (int i = 0; i < points.length; i++) {
			newPoints.add(new Point(points[i]));
		}

		super.postCopy(context);

		this.points_(newPoints);
		this.foregroundColor_(new Color(foregroundColor.getRGB(), true));
		preferredBounds = null;
		pointBounds = null;
		shapes = null;
		stroke = null;
		propertiesModel = null;

		return this;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		Rectangle clipBounds = aGraphics.getClipBounds();
		if (clipBounds != null && clipBounds.intersects(this.bounds()) == false) {
			return;
		}

		Graphics2D graphicsContext = (Graphics2D) aGraphics.create();
		try {
			graphicsContext.setColor(this.foregroundColor());
			graphicsContext.setStroke(this.stroke());
			Shape[] shapes = this.shapes();
			for (int i = 0; i < shapes.length; i++) {
				if (shapes[i] instanceof Polygon || shapes[i] instanceof Ellipse2D) {
					graphicsContext.fill(shapes[i]);
				} else {
					graphicsContext.draw(shapes[i]);
				}
			}
		} finally {
			if (graphicsContext != null) {
				graphicsContext.dispose();
			}
		}
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @category flushing
	 */
	protected void flushBounds() {
		preferredBounds = null;
		pointBounds = null;
		shapes = null;
		stroke = null;
	}

	/**
	 * Answer the point in the specified controll point. 
	 * 
	 * @param point java.awt.Point
	 * @return java.awt.Point
	 * @throws java.lang.NumberFormatException
	 * @category functions
	 */
	public Point pointInControllPoint_(Point point) {
		StSymbol aSymbol = this.controllPointAreaName_(point);
		if (aSymbol == null) {
			return null;
		}
		if (aSymbol == CONTROLL_POINT_BEGIN) {
			return this.begin();
		} else if (aSymbol == CONTROLL_POINT_END) {
			return this.end();
		} else {
			String index = aSymbol.toString().substring(aSymbol.toString().indexOf('_') + 1);
			return this.pointAt_(Integer.parseInt(index));
		}
	}

	/**
	 * Remove the point at the specified position in this points.
	 * 
	 * @param index int
	 * @return java.awt.Point
	 * @category removing
	 */
	public Point removeAt_(int index) {
		if (index < 0 || this.points().size() < index || this.points.size() <= 2) {
			return null;
		}
		Point aPoint = (Point) this.points().remove(index);
		this.flushBounds();
		return aPoint;
	}

	/**
	 * Remove the point in this points.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @category removing
	 */
	public Point remove_(Point aPoint) {
		if (this.points().contains(aPoint) == false || this.points.size() <= 2) {
			return null;
		}
		this.points().remove(aPoint);
		this.flushBounds();
		return aPoint;
	}

	/**
	 * Answer true if receiver is vertexes element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isVertexes()
	 * @category testing
	 */
	public boolean isVertexes() {
		return true;
	}

	/**
	 * Answer true if receiver's bounding box contains the specified a point, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#containsPoint_(java.awt.Point)
	 * @category testing
	 */
	public boolean containsPoint_(Point aPoint) {
		if (super.containsPoint_(aPoint) == false) {
			return false;
		}

		Shape[] shapes = this.shapes();
		for (int i = 0; i < shapes.length; i++) {
			if (shapes[i] instanceof Polygon) {
				if (shapes[i].contains(aPoint)) {
					return true;
				}
			} else if (this.stroke().createStrokedShape(shapes[i]).contains(aPoint)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Answer true if receiver's element intersects the specified a rectangle, otherwise false.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#intersects_(java.awt.Rectangle)
	 * @category testing
	 */
	public boolean intersects_(Rectangle aRectangle) {
		if (super.intersects_(aRectangle) == false) {
			return false;
		}

		Shape[] shapes = this.shapes();
		for (int i = 0; i < shapes.length; i++) {
			if (shapes[i] instanceof Polygon) {
				if (shapes[i].intersects(aRectangle)) {
					return true;
				}
			} else if (this.stroke().createStrokedShape(shapes[i]).intersects(aRectangle)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Answer the receiver's line width.
	 * 
	 * @return int
	 * @category visual properties
	 */
	public int lineWidth() {
		return lineWidth;
	}

	/**
	 * Set the receiver's line width.
	 * 
	 * @param aNumber int
	 * @category visual properties
	 */
	public void lineWidth_(int aNumber) {
		lineWidth = Math.min(Math.max(aNumber, MIN_LINE_WIDTH), MAX_LINE_WIDTH);
		this.flushBounds();
	}

	/**
	 * Answer the foreground color.
	 * 
	 * @return java.awt.Color
	 * @category visual properties
	 */
	public Color foregroundColor() {
		return foregroundColor;
	}

	/**
	 * Set the foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void foregroundColor_(Color aColor) {
		if (aColor != null) {
			foregroundColor = aColor;
		}
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#propertiesModel()
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		if (propertiesModel == null) {
			propertiesModel = new JunVertexesPropertiesModel(this);
		}
		return propertiesModel;
	}

	/**
	 * Answer the receiver's shapes.
	 * 
	 * @return java.awt.geom.Shape[]
	 * @category private
	 */
	protected abstract Shape[] shapes();

	/**
	 * Answer the receiver's stroke
	 * 
	 * @return java.awt.BasicStroke
	 * @category private
	 */
	protected abstract BasicStroke stroke();

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.pointsToLispList());
		list.add_(this.foregroundColorToLispList());
		list.add_(this.lineWidthToLispList());
		return list;
	}

	/**
	 * Convert the receiver's points to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons pointsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("points"));
		for (Iterator iterator = this.points().iterator(); iterator.hasNext();) {
			list.add_(new Jun2dPoint((Point) iterator.next()));
		}
		return list;
	}

	/**
	 * Convert the foreground color to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons foregroundColorToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("foregroundColor"));
		list.tail_(this.foregroundColor());
		return list;
	}

	/**
	 * Convert the line width to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons lineWidthToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("lineWidth"));
		list.tail_(new Integer(this.lineWidth()));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.pointsFromLispList_(aList);
		this.foregroundColorFromLispList_(aList);
		this.lineWidthFromLispList_(aList);
	}

	/**
	 * Get the receiver's points from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void pointsFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("points"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] objects = ((JunLispList) list.tail()).asArray();
		for (int i = 0; i < objects.length; i++) {
			this.add_(((Jun2dPoint) objects[i])._toPoint());
		}
	}

	/**
	 * Get the foreground color from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void foregroundColorFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("foregroundColor"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.foregroundColor_((Color) list.tail());
	}

	/**
	 * Get the line width from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void lineWidthFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("lineWidth"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.lineWidth_(((Number) list.tail()).intValue());
	}

}
