package jp.co.sra.jun.goodies.drawing.element;

import java.awt.BasicStroke;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.datatransfer.DataFlavor;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;
import java.awt.geom.Line2D;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.drawing.properties.JunPathPropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunPathElement class
 * 
 *  @author    m-asada
 *  @created   2005/03/03 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPathElement.java,v 8.11 2008/02/20 06:31:23 nisinaka Exp $
 */
public class JunPathElement extends JunVertexesElement {
	protected int beginStyle;
	protected int endStyle;

	protected transient Dimension minimumExtent;

	public static final int ARROW_STYLE_SQUARE = 0;
	public static final int ARROW_STYLE_ARROW = 1;
	public static final int ARROW_STYLE_DIAMOND = 2;
	public static final int ARROW_STYLE_CIRCLE = 3;

	public static DataFlavor DataFlavor = new DataFlavor(JunPathElement.class, "JunPathElement");

	/**
	 * Create a new instance of JunPathElement and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunPathElement() {
		super();
	}

	/**
	 * Create a new instance of JunPathElement and initialize it.
	 *
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunPathElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		beginStyle = ARROW_STYLE_SQUARE;
		endStyle = ARROW_STYLE_SQUARE;
		minimumExtent = null;
	}

	/**
	 * Add the new point on receiver's line segment.
	 * 
	 * @param aPoint java.awt.Point
	 * @return java.awt.Point
	 * @category adding
	 */
	public Point addOnLine_(Point aPoint) {
		Shape[] shapes = this.shapes();
		Point newPoint;
		for (int i = 0; i < this.points().size() - 1; i++) {
			if (this.stroke().createStrokedShape(shapes[i]).contains(aPoint)) {
				newPoint = this.add_after_(aPoint, this.pointAt_(i));
				return newPoint;
			}
		}
		if (this.begin().distance(aPoint) <= this.end().distance(aPoint)) {
			newPoint = this.add_at_(aPoint, 0);
		} else {
			newPoint = this.add_(aPoint);
		}
		return newPoint;
	}

	/**
	 * Set the receiver's rectangle.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#bounds_(java.awt.Rectangle)
	 * @category bounds accessing
	 */
	public void bounds_(Rectangle aRectangle) {
		Rectangle pointBounds = this.pointBounds();

		if (pointBounds.height == 1) {
			this.location_(aRectangle.x, this.y());
			this.extent_(aRectangle.width, this.height());
		} else if (pointBounds.width == 1) {
			this.location_(this.x(), aRectangle.y);
			this.extent_(this.width(), aRectangle.height);
		} else {
			super.bounds_(aRectangle);
		}
	}

	/**
	 * Set the receiver's location point.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#location_(java.awt.Point)
	 * @category bounds accessing
	 */
	public void location_(Point aPoint) {
		if (this.location().equals(aPoint)) {
			return;
		}

		Point newLocation = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		int deltaX = newLocation.x - this.x();
		int deltaY = newLocation.y - this.y();
		ArrayList newPoints = new ArrayList(this.points().size());
		for (Iterator iterator = this.points().iterator(); iterator.hasNext();) {
			Point pathPoint = (Point) iterator.next();
			pathPoint.translate(deltaX, deltaY);
			newPoints.add(new Point(Math.max(pathPoint.x, 0), Math.max(pathPoint.y, 0)));
		}
		this.points_(newPoints);
	}

	/**
	 * Set the receiver's extent size.
	 * 
	 * @param aDimension java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#extent_(java.awt.Dimension)
	 * @category bounds accessing
	 */
	public void extent_(Dimension aDimension) {
		if (this.extent().equals(aDimension)) {
			return;
		}

		Rectangle pointBounds = this.pointBounds();
		Point baseLocation = pointBounds.getLocation();
		Dimension baseSize = pointBounds.getSize();
		int newWidth = Math.max(aDimension.width, this.defaultMinimumExtent().width) - (this.preferredBounds().width - baseSize.width);
		int newHeight = Math.max(aDimension.height, this.defaultMinimumExtent().height) - (this.preferredBounds().height - baseSize.height);
		Dimension newSize = new Dimension(newWidth, newHeight);
		ArrayList newPoints = new ArrayList(this.points().size());
		for (Iterator iterator = this.points().iterator(); iterator.hasNext();) {
			Point pathPoint = (Point) iterator.next();
			int pointX = (pointBounds.width == 1) ? pathPoint.x : (baseLocation.x + Math.max(Math.round((pathPoint.x - baseLocation.x) * newSize.width / (float) baseSize.width), 0));
			int pointY = (pointBounds.height == 1) ? pathPoint.y : (baseLocation.y + Math.max(Math.round((pathPoint.y - baseLocation.y) * newSize.height / (float) baseSize.height), 0));
			newPoints.add(new Point(pointX, pointY));
		}
		this.points_(newPoints);
	}

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.awt.HashMap
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#controllPoints()
	 * @category bounds accessing
	 */
	public HashMap controllPoints() {
		Point[] arrayOfPoint = (Point[]) this.points().toArray(new Point[this.points().size()]);

		HashMap pointMap = new HashMap();
		pointMap.put(JunPathElement.CONTROLL_POINT_BEGIN, arrayOfPoint[0]);
		for (int i = 1; i < arrayOfPoint.length - 1; i++) {
			pointMap.put($("point_" + i), arrayOfPoint[i]);
		}
		pointMap.put(JunPathElement.CONTROLL_POINT_END, arrayOfPoint[arrayOfPoint.length - 1]);
		return pointMap;
	}

	/**
	 * Replaces the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#controllPointAt_put_(jp.co.sra.smalltalk.StSymbol, java.awt.Point)
	 * @category bounds accessing
	 */
	public void controllPointAt_put_(StSymbol aSymbol, Point aPoint) {
		Point sourcePoint = this.controllPointAt_(aSymbol);
		if (sourcePoint == null) {
			throw new IllegalArgumentException("'" + aSymbol.toString() + "' does not exist.");
		}
		Point newPoint = new Point(Math.max(aPoint.x, 0), Math.max(aPoint.y, 0));
		if (sourcePoint.equals(newPoint) == false) {
			sourcePoint.setLocation(newPoint);
			this.flushBounds();
		}
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);
		minimumExtent = null;

		return this;
	}

	/**
	 * Get the default minimum extent.
	 *
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#defaultMinimumExtent()
	 * @category defaults
	 */
	public Dimension defaultMinimumExtent() {
		if (minimumExtent == null) {
			Rectangle preferredBounds = this.preferredBounds();
			Rectangle pointBounds = this.pointBounds();
			minimumExtent = new Dimension(20, 20);
			minimumExtent.setSize(minimumExtent.width + preferredBounds.width - pointBounds.width, minimumExtent.height + preferredBounds.height - pointBounds.height);
		}
		return minimumExtent;
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunVertexesElement#flushBounds()
	 * @category flushing
	 */
	protected void flushBounds() {
		super.flushBounds();
		minimumExtent = null;
	}

	/**
	 * Answer true if receiver is path element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isPath()
	 * @category testing
	 */
	public boolean isPath() {
		return true;
	}

	/**
	 * Returns an array of DataFlavor objects indicating the flavors the data can be provided in. The array should be
	 * ordered according to preference for providing the data (from most richly descriptive to least descriptive).
	 * 
	 * @return java.awt.datatransfer.DataFlavor[]
	 * @see java.awt.datatransfer.Transferable#getTransferDataFlavors()
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#getTransferDataFlavors()
	 * @category transfering
	 */
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { JunPathElement.DataFlavor };
	}

	/**
	 * Answer the receiver's begin style.
	 * 
	 * @return int
	 * @category visual properties
	 */
	public int beginStyle() {
		return beginStyle;
	}

	/**
	 * Set the receiver's begin style.
	 * 
	 * @param aNumber int
	 * @throws java.lang.IllegalArgumentException
	 * @category visual properties
	 */
	public void beginStyle_(int aNumber) {
		if (this.beginStyle() != aNumber) {
			switch (aNumber) {
				case ARROW_STYLE_SQUARE:
				case ARROW_STYLE_ARROW:
				case ARROW_STYLE_DIAMOND:
				case ARROW_STYLE_CIRCLE:
					beginStyle = aNumber;
					this.flushBounds();
					break;
				default:
					throw new IllegalArgumentException("The number of arrow type as'" + aNumber + "' is faild");
			}
		}
	}

	/**
	 * Answer the receiver's the end arrow style.
	 * 
	 * @return fromArrowType
	 * @category visual properties
	 */
	public int endStyle() {
		return endStyle;
	}

	/**
	 * Set the receiver's the end arrow style.
	 * 
	 * @param aNumber int
	 * @throws java.lang.IllegalArgumentException
	 * @category visual properties
	 */
	public void endStyle_(int aNumber) {
		if (this.endStyle() != aNumber) {
			switch (aNumber) {
				case ARROW_STYLE_SQUARE:
				case ARROW_STYLE_ARROW:
				case ARROW_STYLE_DIAMOND:
				case ARROW_STYLE_CIRCLE:
					endStyle = aNumber;
					this.flushBounds();
					break;
				default:
					throw new IllegalArgumentException("The number of arrow type as'" + aNumber + "' is faild");
			}
		}
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#propertiesModel()
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		if (propertiesModel == null) {
			propertiesModel = new JunPathPropertiesModel(this);
		}
		return propertiesModel;
	}

	/**
	 * Answer the receiver's shapes.
	 * 
	 * @return java.awt.geom.Shape[]
	 * @see jp.co.sra.jun.goodies.drawing.element.JunVertexesElement#shapes()
	 * @category private
	 */
	protected Shape[] shapes() {
		if (shapes == null) {
			ArrayList anArray = new ArrayList();
			Point[] pointArray = this._points();

			for (int i = 1; i < pointArray.length; i++) {
				anArray.add(new Line2D.Double(pointArray[i - 1].x, pointArray[i - 1].y, pointArray[i].x, pointArray[i].y));
			}
			Shape[] fromArrowShapes = this.arrowShapes_point_with_(this.beginStyle(), pointArray[0], pointArray[1]);
			for (int i = 0; i < fromArrowShapes.length; i++) {
				anArray.add(fromArrowShapes[i]);
			}
			Shape[] toArrowShapes = this.arrowShapes_point_with_(this.endStyle(), pointArray[pointArray.length - 1], pointArray[pointArray.length - 2]);
			for (int i = 0; i < toArrowShapes.length; i++) {
				anArray.add(toArrowShapes[i]);
			}

			shapes = (Shape[]) anArray.toArray(new Shape[anArray.size()]);
		}
		return shapes;
	}

	/**
	 * Answer the receiver's arrow shapes from two points.
	 * 
	 * @param aPoint java.awt.Point
	 * @param anotherPoint java.awt.Point
	 * @return java.awt.geom.Shape[]
	 * @category private
	 */
	protected Shape[] arrowShapes_point_with_(int arrowType, Point aPoint, Point anotherPoint) {
		if (arrowType == ARROW_STYLE_SQUARE) {
			return new Line2D.Double[0];
		}

		if (arrowType == ARROW_STYLE_ARROW) {
			double arrowRotateAngle = new Jun2dPoint(aPoint).minus_(new Jun2dPoint(anotherPoint)).theta();
			Jun2dTransformation transformation = Jun2dTransformation.RotateZ_(arrowRotateAngle);
			transformation = transformation.product_(Jun2dTransformation.Translate_(new Jun2dPoint(anotherPoint)));

			Jun2dLine arrow1 = new Jun2dLine(new Jun2dPoint(aPoint.x - 13 - this.lineWidth() * 2, aPoint.y - 3 - this.lineWidth()), new Jun2dPoint(aPoint));
			arrow1 = arrow1.transform_(transformation);
			arrow1 = arrow1.translatedBy_((arrow1.to().minus_(new Jun2dPoint(aPoint))).negated());
			Point arrow1From = arrow1.from()._toPoint();

			Jun2dLine arrow2 = new Jun2dLine(new Jun2dPoint(aPoint.x - 13 - this.lineWidth() * 2, aPoint.y + 3 + this.lineWidth()), new Jun2dPoint(aPoint));
			arrow2 = arrow2.transform_(transformation);
			arrow2 = arrow2.translatedBy_((arrow2.to().minus_(new Jun2dPoint(aPoint))).negated());
			Point arrow2From = arrow2.from()._toPoint();

			GeneralPath path = new GeneralPath();
			path.moveTo(arrow1From.x, arrow1From.y);
			path.lineTo(aPoint.x, aPoint.y);
			path.lineTo(arrow2From.x, arrow2From.y);
			return new GeneralPath[] { path };
		}

		if (arrowType == ARROW_STYLE_DIAMOND) {
			double arrowRotateAngle = new Jun2dPoint(aPoint).minus_(new Jun2dPoint(anotherPoint)).theta();
			Jun2dTransformation transformation = Jun2dTransformation.RotateZ_(arrowRotateAngle);
			transformation = transformation.product_(Jun2dTransformation.Translate_(new Jun2dPoint(anotherPoint)));

			int size = 10 + this.lineWidth() * 2;
			int[] xPoints = new int[] { aPoint.x, aPoint.x - size, aPoint.x - (size * 2), aPoint.x - size };
			int[] yPoints = new int[] { aPoint.y, aPoint.y - (size / 2), aPoint.y, aPoint.y + (size / 2) };
			Jun2dPoint translatedPoint = null;
			for (int i = 0; i < xPoints.length; i++) {
				Jun2dPoint point = new Jun2dPoint(xPoints[i], yPoints[i]);
				point = point.transform_(transformation);
				if (translatedPoint == null) {
					translatedPoint = (new Jun2dPoint(point).minus_(new Jun2dPoint(aPoint))).negated();
				}
				point = point.translatedBy_(translatedPoint);
				xPoints[i] = point._toPoint().x;
				yPoints[i] = point._toPoint().y;
			}
			return new Polygon[] { new Polygon(xPoints, yPoints, xPoints.length) };
		}

		if (arrowType == ARROW_STYLE_CIRCLE) {
			double arrowRotateAngle = new Jun2dPoint(aPoint).minus_(new Jun2dPoint(anotherPoint)).theta();
			Jun2dTransformation transformation = Jun2dTransformation.RotateZ_(arrowRotateAngle);
			transformation = transformation.product_(Jun2dTransformation.Translate_(new Jun2dPoint(anotherPoint)));

			int size = 10 + this.lineWidth() * 2;
			Jun2dPoint point = new Jun2dPoint(aPoint.x - (size / 2), aPoint.y);
			point = point.transform_(transformation);
			Jun2dPoint translatedPoint = (new Jun2dPoint(aPoint).transform_(transformation).minus_(new Jun2dPoint(aPoint))).negated();
			point = point.translatedBy_(translatedPoint);

			return new Ellipse2D.Double[] { new Ellipse2D.Double(point._toPoint().x - size / 2, point._toPoint().y - size / 2, size, size) };
		}

		throw new IllegalArgumentException("The number of arrow type as'" + arrowType + "' is faild");
	}

	/**
	 * Answer the receiver's stroke
	 * 
	 * @return java.awt.BasicStroke
	 * @see jp.co.sra.jun.goodies.drawing.element.JunVertexesElement#stroke()
	 * @category private
	 */
	protected BasicStroke stroke() {
		if (stroke == null) {
			stroke = new BasicStroke(this.lineWidth(), BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL);
		}
		return stroke;
	}

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.beginStyleToLispList());
		list.add_(this.endStyleToLispList());
		return list;
	}

	/**
	 * Convert the begin style to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons beginStyleToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("beginStyle"));
		list.tail_(new Integer(this.beginStyle()));
		return list;
	}

	/**
	 * Convert the end style to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons endStyleToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("endStyle"));
		list.tail_(new Integer(this.endStyle()));
		return list;
	}

	/**
	 * Get the receiver from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.beginStyleFromLispList_(aList);
		this.endStyleFromLispList_(aList);
	}

	/**
	 * Get the begin style from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void beginStyleFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("beginStyle"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.beginStyle_(((Number) list.tail()).intValue());
	}

	/**
	 * Get the end style from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void endStyleFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("endStyle"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		this.endStyle_(((Number) list.tail()).intValue());
	}
}
