package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Point;
import java.awt.Rectangle;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.goodies.drawing.map.JunDrawingMap;
import jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunDrawingElement class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDrawingElement.java,v 8.10 2008/02/20 06:31:23 nisinaka Exp $
 */
public abstract class JunDrawingElement extends JunDrawingVisual {
	protected JunDrawingVisual parent;

	public static final int CONTROLL_AREA_SIZE = 6;

	public static final StSymbol CONTROLL_POINT_TOP_LEFT = $("topLeft");
	public static final StSymbol CONTROLL_POINT_TOP_RIGHT = $("topRight");
	public static final StSymbol CONTROLL_POINT_BOTTOM_LEFT = $("bottomLeft");
	public static final StSymbol CONTROLL_POINT_BOTTOM_RIGHT = $("bottomRight");

	/**
	 * Create a JunDrawingElement from the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Lisp support
	 */
	public static JunDrawingElement FromLispList_(JunLispList aList) {
		JunDrawingElement element = null;
		StSymbol className = (StSymbol) aList.head();
		if (className == $("JunEllipseElement")) {
			element = new JunEllipseElement(aList);
		} else if (className == $("JunFreehandElement")) {
			element = new JunFreehandElement(aList);
		} else if (className == $("JunImageElement")) {
			element = new JunImageElement(aList);
		} else if (className == $("JunLabelElement")) {
			element = new JunLabelElement(aList);
		} else if (className == $("JunLinkElement")) {
			element = new JunLinkElement(aList);
		} else if (className == $("JunPathElement")) {
			element = new JunPathElement(aList);
		} else if (className == $("JunRectangleElement")) {
			element = new JunRectangleElement(aList);
		} else if (className == $("JunRoundRectangleElement")) {
			element = new JunRoundRectangleElement(aList);
		} else if (className == $("JunTextElement")) {
			element = new JunTextElement(aList);
		} else if (className == $("JunTextboxElement")) {
			element = new JunTextboxElement(aList);
		}
		return element;
	}

	/**
	 * Create a new instance of JunDrawingElement and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunDrawingElement() {
		super();
	}

	/**
	 * Create a new instance of JunDrawingElement and initialize it.
	 *
	 * @param aList
	 * @category Instance creation
	 */
	public JunDrawingElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		parent = null;
	}

	/**
	 * Answer the receiver's parent.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @category accessing
	 */
	public JunDrawingVisual parent() {
		return parent;
	}

	/**
	 * Set the receiver's parent.
	 * 
	 * @param newParent jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @category accessing
	 */
	public void parent_(JunDrawingVisual newParent) {
		parent = newParent;
	}

	/**
	 * Answer the receiver's map object.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.map.JunDrawingMap
	 * @category accessing
	 */
	public JunDrawingMap map() {
		if (this.parent() instanceof JunDrawingMap) {
			return (JunDrawingMap) this.parent();
		} else if (this.parent() instanceof JunDrawingElement) {
			return ((JunDrawingElement) this.parent()).map();
		}
		return null;
	}

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.util.HashMap
	 * @category bounds accessing
	 */
	public abstract HashMap controllPoints();

	/**
	 * Answer the collection of receiver's controll point.
	 * 
	 * @return java.util.HashMap
	 * @category bounds accessing
	 */
	public HashMap controllPointAreas() {
		HashMap pointMap = this.controllPoints();
		if (pointMap == null || pointMap.size() == 0) {
			return new HashMap(0);
		}

		HashMap areaMap = new HashMap();
		for (Iterator iterator = pointMap.keySet().iterator(); iterator.hasNext();) {
			StSymbol key = (StSymbol) iterator.next();
			Point value = (Point) pointMap.get(key);
			Rectangle area = new Rectangle(value.x, value.y, CONTROLL_AREA_SIZE, CONTROLL_AREA_SIZE);
			area.translate(-CONTROLL_AREA_SIZE / 2, -CONTROLL_AREA_SIZE / 2);
			areaMap.put(key, area);
		}
		return areaMap;
	}

	/**
	 * Answer the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @return aPoint java.awt.Point
	 * @category bounds accessing
	 */
	public Point controllPointAt_(StSymbol aSymbol) {
		return (Point) this.controllPoints().get(aSymbol);
	}

	/**
	 * Replaces the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IllegalArgumentException
	 * @category bounds accessing
	 */
	public abstract void controllPointAt_put_(StSymbol aSymbol, Point aPoint);

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		JunDrawingVisual parent = this.parent();
		if (parent != null) {
			parent = (JunDrawingVisual) context.get(parent);
		}
		super.postCopy(context);
		this.parent_(parent);

		return this;
	}

	/**
	 * Find a drawing element with the specified id number.
	 * 
	 * @param findId long
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category finding
	 */
	public JunDrawingElement findElement_(long findId) {
		if (findId < 0) {
			return null;
		}
		if (this.id() == findId) {
			return this;
		}
		return null;
	}

	/**
	 * Answer the controll point area symbol contains specified the point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category functions
	 */
	public StSymbol controllPointAreaName_(Point aPoint) {
		HashMap areaMap = this.controllPointAreas();
		for (Iterator iterator = areaMap.keySet().iterator(); iterator.hasNext();) {
			StSymbol key = (StSymbol) iterator.next();
			Rectangle value = (Rectangle) areaMap.get(key);
			if (value.contains(aPoint)) {
				return key;
			}
		}
		return null;
	}

	/**
	 * Answer true if receiver's controll areas contains the specified a point, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPointInControllArea_(Point aPoint) {
		for (Iterator iterator = this.controllPointAreas().values().iterator(); iterator.hasNext();) {
			if (((Rectangle) iterator.next()).contains(aPoint)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Answer true if receiver is element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isElement()
	 * @category testing
	 */
	public boolean isElement() {
		return true;
	}

	/**
	 * Open the receiver's properties
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#openProperties()
	 * @category visual properties
	 */
	public void openProperties() {
		if (this.propertiesModel() == null) {
			return;
		}
		this.propertiesModel()._show();
	}

	/**
	 * Answer the receiver's properties model.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.properties.JunDrawingElementPropertiesModel
	 * @category visual properties
	 */
	public JunDrawingElementPropertiesModel propertiesModel() {
		return null;
	}
}
