package jp.co.sra.jun.goodies.drawing.element;

import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunCompositeElement class
 * 
 *  @author    m-asada
 *  @created   2005/03/01 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCompositeElement.java,v 8.10 2008/02/20 06:31:23 nisinaka Exp $
 */
public abstract class JunCompositeElement extends JunNodeElement {
	protected ArrayList componentElements;
	protected transient Rectangle preferredBounds;

	/**
	 * Create a new instance of JunCompositeElement and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunCompositeElement() {
		super();
	}

	/**
	 * Create a new instance of JunCompositeElement and initialize it.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunCompositeElement(JunLispList aList) {
		super(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		componentElements = null;
		preferredBounds = null;
	}

	/**
	 * Initialize the receiver's component elements.
	 * 
	 * @category initialize-release
	 */
	protected void initializeComponentElements() {
		this.flushComponentElements();
	}

	/**
	 * Get the receiver's component elements.
	 * 
	 * @return java.uti.ArrayList
	 * @category accessing
	 */
	public ArrayList componentElements() {
		if (componentElements == null) {
			componentElements = new ArrayList();
		}
		return componentElements;
	}

	/**
	 * Set the receiver's component elements.
	 * 
	 * @param anArray java.uti.ArrayList
	 * @category accessing
	 */
	public void componentElements_(ArrayList anArray) {
		componentElements = anArray;
		if (componentElements != null && componentElements.size() > 0) {
			JunDrawingElement[] elements = this._componentElements();
			for (int i = 0; i < elements.length; i++) {
				elements[i].parent_(this);
			}
		}
		this.flushBounds();
	}

	/**
	 * Get an array of the receiver's component elements.
	 * 
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement[]
	 * @category accessing
	 */
	public JunDrawingElement[] _componentElements() {
		JunDrawingElement[] elements = new JunDrawingElement[this.componentElementSize()];
		this.componentElements().toArray(elements);
		return elements;
	}

	/**
	 * Answer the element at the specified position in this components.
	 * 
	 * @param anIndex int
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category accessing
	 */
	public JunDrawingElement componentElementAt_(int anIndex) {
		if (this.componentElements().isEmpty()) {
			return null;
		}
		if (anIndex < 0 || this.componentElementSize() <= anIndex) {
			return null;
		}
		return (JunDrawingElement) this.componentElements().get(anIndex);
	}

	/**
	 * Answer the index of the first occurrence of anElement within the receiver.
	 * If the receiver does not contain anElement, answer -1.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return int
	 * @see java.util.ArrayList#indexOf(java.lang.Object);
	 * @category accessing
	 */
	public int componentElementIndexOf_(JunDrawingElement anElement) {
		return this.componentElements().indexOf(anElement);
	}

	/**
	 * Answer how many elements the receiver contains.
	 * 
	 * @return int
	 * @see java.util.ArrayList#size();
	 * @category accessing
	 */
	public int componentElementSize() {
		return this.componentElements().size();
	}

	/**
	 * Add the specified element to the end of this component elements.
	 * 
	 * @param newElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return boolean
	 * @category adding
	 */
	public boolean addElement_(JunDrawingElement newElement) {
		if (newElement == null || this.componentElements().contains(newElement)) {
			return false;
		}
		this.flushBounds();
		newElement.parent_(this);
		return this.componentElements().add(newElement);
	}

	/**
	 * Add the specified element to the position of this component element.
	 * 
	 * @param newElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @param anIndex int
	 * @return boolean
	 * @category adding
	 */
	public void addElement_beforeIndex_(JunDrawingElement newElement, int anIndex) {
		if (newElement == null || this.componentElements().contains(newElement)) {
			return;
		}
		newElement.parent_(this);
		if (0 <= anIndex && anIndex < this.componentElementSize()) {
			this.componentElements().add(anIndex, newElement);
		} else {
			this.componentElements().add(newElement);
		}
		this.flushBounds();
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		if (preferredBounds == null) {
			JunDrawingElement[] elements = this._componentElements();
			for (int i = 0; i < elements.length; i++) {
				if (preferredBounds == null) {
					preferredBounds = new Rectangle(elements[i].bounds());
				} else {
					preferredBounds.add(elements[i].bounds());
				}
			}
		}
		return preferredBounds;
	}

	/**
	 * Replaces the receiver's controll point with the specified position.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @param aPoint java.awt.Point
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#controllPointAt_put_(jp.co.sra.smalltalk.StSymbol, java.awt.Point)
	 * @category bounds accessing
	 */
	public void controllPointAt_put_(StSymbol aSymbol, Point aPoint) {
		Rectangle bounds = this.bounds();
		Point controllPoint = this.controllPointAt_(aSymbol);
		boolean changeVertical = (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_RIGHT);
		boolean changeHorizontal = (aSymbol == JunDrawingElement.CONTROLL_POINT_TOP_LEFT || aSymbol == JunDrawingElement.CONTROLL_POINT_BOTTOM_LEFT);
		int deltaWidth = Math.max(aPoint.x, 0) - controllPoint.x;
		int deltaHeight = Math.max(aPoint.y, 0) - controllPoint.y;
		int newX = (changeHorizontal) ? (Math.max(bounds.x + deltaWidth, 0)) : (bounds.x);
		int newY = (changeVertical) ? (Math.max(bounds.y + deltaHeight, 0)) : (bounds.y);
		int newWidth = (changeHorizontal) ? (bounds.width + (bounds.x - newX)) : (Math.max(bounds.width + deltaWidth, this.defaultMinimumExtent().width));
		int newHeight = (changeVertical) ? (bounds.height + (bounds.y - newY)) : (Math.max(bounds.height + deltaHeight, this.defaultMinimumExtent().height));
		if (newWidth < this.defaultMinimumExtent().width) {
			newX = newX + (newWidth - this.defaultMinimumExtent().width);
			newWidth = this.defaultMinimumExtent().width;
		}
		if (newHeight < this.defaultMinimumExtent().height) {
			newY = newY + (newHeight - this.defaultMinimumExtent().height);
			newHeight = this.defaultMinimumExtent().height;
		}
		this.bounds_(new Rectangle(newX, newY, newWidth, newHeight));
	}

	/**
	 * Do the receiver specific copy process after the shallow copy.
	 * 
	 * @param context java.util.Map
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#postCopy(java.util.Map)
	 * @category copying
	 */
	public JunDrawingVisual postCopy(Map context) {
		super.postCopy(context);
		ArrayList newComponents = new ArrayList();
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			newComponents.add(context.containsKey(elements[i]) ? context.get(elements[i]) : elements[i].copy(context));
		}
		this.componentElements_(newComponents);
		return this;
	}

	/**
	 * Display this element icon on a canvas.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		Rectangle clipBounds = aGraphics.getClipBounds();
		if (clipBounds == null || clipBounds.intersects(this.bounds())) {
			JunDrawingElement[] elements = this._componentElements();
			for (int i = 0; i < elements.length; i++) {
				elements[i].displayOn_(aGraphics);
			}
		}
	}

	/**
	 * Evaluate aBlock with each of the receiver's elements as the argument.
	 * 
	 * @param aBlock sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public Object elementsDo_(StBlockClosure aBlock) {
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			aBlock.value_(elements[i]);
			if (elements[i].isComposite()) {
				((JunCompositeElement) elements[i]).elementsDo_(aBlock);
			}
		}
		return this;
	}

	/**
	 * Find a drawing element with the specified id number.
	 * 
	 * @param findId long
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingElement#findElement_(long)
	 * @category finding
	 */
	public JunDrawingElement findElement_(long findId) {
		JunDrawingElement findElement = super.findElement_(findId);
		if (findElement != null) {
			return findElement;
		}
		for (Iterator iterator = this.componentElements().iterator(); iterator.hasNext();) {
			findElement = ((JunDrawingElement) iterator.next()).findElement_(findId);
			if (findElement != null) {
				return findElement;
			}
		}
		return null;
	}

	/**
	 * Flush the receiver's preferred bounds.
	 * 
	 * @category flushing
	 */
	protected void flushBounds() {
		preferredBounds = null;
	}

	/**
	 * Flush the receiver's component elements.
	 * 
	 * @category flushing
	 */
	protected void flushComponentElements() {
		componentElements = null;
	}

	/**
	 * Remove the element from receiver's component elements.
	 * 
	 * @param anElement jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @return boolean
	 * @category removing
	 */
	public boolean removeElement_(JunDrawingElement anElement) {
		if (this.componentElements().contains(anElement) == false) {
			return false;
		}
		boolean result = this.componentElements().remove(anElement);
		anElement.parent_(null);
		this.flushBounds();
		return result;
	}

	/**
	 * Answer the graph element including a point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.goodies.drawing.element.JunDrawingElement
	 * @category selecting
	 */
	public JunDrawingElement which_(Point aPoint) {
		for (ListIterator iterator = this.componentElements().listIterator(this.componentElementSize()); iterator.hasPrevious();) {
			JunDrawingElement element = (JunDrawingElement) iterator.previous();
			if (element.containsPoint_(aPoint)) {
				return element;
			}
		}
		return null;
	}

	/**
	 * Answer true if receiver is composite element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isComposite()
	 * @category testing
	 */
	public boolean isComposite() {
		return true;
	}

	/**
	 * Answer true if the receiver is empty, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.componentElements().isEmpty();
	}

	/**
	 * Answer true if receiver is rectangular element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#isRectangular()
	 * @category testing
	 */
	public boolean isRectangular() {
		return true;
	}

	/**
	 * Convert the receiver to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		JunLispCons componentElementsList = this.componentElementsToLispList();
		if (componentElementsList != null) {
			list.add_(componentElementsList);
		}
		return list;
	}

	/**
	 * Convert the receiver's component elements to the lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons componentElementsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("componentElements"));
		JunDrawingElement[] elements = this._componentElements();
		for (int i = 0; i < elements.length; i++) {
			list.add_(elements[i].toLispList());
		}
		return list;
	}

	/**
	 * Get the component element from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.drawing.element.JunDrawingVisual#fromLispList_(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.componentElementsFromLispList_(aList);
	}

	/**
	 * Get the receiver's component elements from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void componentElementsFromLispList_(JunLispList aList) {
		JunLispList list = (JunLispList) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("componentElements"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.flushComponentElements();
		Object[] objects = ((JunLispList) list.tail()).asArray();
		for (int i = 0; i < objects.length; i++) {
			JunDrawingElement anElement = JunDrawingElement.FromLispList_((JunLispList) objects[i]);
			this.addElement_(anElement);
		}
	}
}
