package jp.co.sra.jun.goodies.colors;

import java.awt.Color;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSmallCompiler;

/**
 * JunColorChoiceModel class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/11/05 (by Hirotsugu Kondo)
 *  @updated   2000/08/21 (by Mitsuhiro Asada)
 *  @updated   2003/04/02 (by Mitsuhiro Asada)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun490 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunColorChoiceModel.java,v 8.11 2008/02/20 06:31:22 nisinaka Exp $
 */
public abstract class JunColorChoiceModel extends JunApplicationModel implements StBlockValued, ClipboardOwner {

	protected Jun2dPoint xy = new Jun2dPoint(0, 0);
	protected double z = 0;
	protected StPopupMenu _popupMenu;

	/**
	 * Constructor.
	 * 
	 * @category Instance creation
	 */
	public JunColorChoiceModel() {
		super();
		this.color_(Color.white);
	}

	/**
	 * Answer the area of bar.
	 * 
	 * @return Rectangle
	 * @category accessing
	 */
	public static Rectangle _barArea() {
		int width = 16;
		int height = _Intervals();

		return new Rectangle(0, 0, width, height);
	}

	/**
	 * Answer the area of plate.
	 * 
	 * @return Rectangle
	 * @category accessing
	 */
	public static Rectangle _plateArea() {
		int width = _Intervals();
		int height = _Intervals();

		return new Rectangle(0, 0, width, height);
	}

	/**
	 * gap is a color grid size.
	 * 
	 * @return int
	 * @category Accessing
	 */
	protected static int _Gap() {
		return 2;
	}

	/**
	 * intervals is a color plate width and height.
	 * 
	 * @return int
	 * @category Accessing
	 */
	protected static int _Intervals() {
		return 100;
	}

	/**
	 * Copy selected color to clipboard.
	 * 
	 * @category menu messages
	 */
	public void _copy() {
		JunColorChoiceView colorChoiceView = (JunColorChoiceView) this.getView();
		if (colorChoiceView == null) {
			return;
		}

		String colorString = colorChoiceView.colorString();
		if (colorString != null && colorString.length() > 0) {
			Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
			StringSelection contents = new StringSelection(colorString);
			clipboard.setContents(contents, this);
		}
	}

	/**
	 * Paste a color from clipboard.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void _paste() {
		String string = null;
		try {
			Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
			Transferable contents = clipboard.getContents(this);
			string = (String) contents.getTransferData(DataFlavor.stringFlavor);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		} catch (UnsupportedFlavorException e) {
		}

		if (string == null || string.indexOf("ColorValue", 0) == -1) {
			return;
		}

		Object color = JunSmallCompiler.Evaluate_logged_(string, false);
		if (color instanceof Color == false) {
			return;
		}
		this.color_((Color) color);
	}

	/**
	 * Return current selected bar.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category image accessing
	 */
	public StImage bar() {
		StImage[] thisBars = this.bars();
		int index = (int) (this.xy.x() * (thisBars.length - 1));
		StImage value = thisBars[index];

		return value;
	}

	/**
	 * Answer the images of bar.
	 * 
	 * @return jp.co.sra.smalltalk.StImage[]
	 * @category image accessing
	 */
	public abstract StImage[] bars();

	/**
	 * Answer the brighness value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public abstract double brightness();

	/**
	 * Set the brighness value.
	 * 
	 * @param value double
	 * @category accessing
	 */
	public abstract void brightness_(double value);

	/**
	 * Answer a color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color color() {
		Color color = Color.getHSBColor((float) this.hue(), (float) this.saturation(), (float) this.brightness());
		return color;
	}

	/**
	 * Set the color and mapping this.
	 * 
	 * @param color java.awt.Color
	 * @category accessing
	 */
	public abstract void color_(Color color);

	/**
	 * Answer a BlockValue that computes aBlock with the receiver's value  as
	 * the argument. aBlock will become a dependent of the receiver, and will
	 * be sent the message value: when the receiver is sent the message
	 * value:.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunColorChoiceViewAwt(this);
		} else {
			return new JunColorChoiceViewSwing(this);
		}
	}

	/**
	 * Answer the hue value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public abstract double hue();

	/**
	 * Set the hue value.
	 * 
	 * @param value double
	 * @category accessing
	 */
	public abstract void hue_(double value);

	/**
	 * Notifies this object that it is no longer the owner of
	 * the contents of the clipboard.
	 *
	 * @param clipboard the clipboard that is no longer owned
	 * @param contents the contents which this owner had placed on the clipboard
	 * @see java.awt.datatransfer.ClipboardOwner#lostOwnership(java.awt.datatransfer.Clipboard, java.awt.datatransfer.Transferable)
	 * @category clipboards
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}

	/**
	 * Return current selected plate.
	 * 
	 * @return jp.co.sra.smalltalk.StImage[]
	 * @category image accessing
	 */
	public StImage plate() {
		StImage[] thisPlates = this.plates();
		int index = (int) (this.z * (thisPlates.length - 1));
		StImage value = thisPlates[index];

		return value;
	}

	/**
	 * Answer the images of color plate.
	 * 
	 * @return jp.co.sra.smalltalk.StImage[]
	 * @category image accessing
	 */
	public abstract StImage[] plates();

	/**
	 * Answer the saturation value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public abstract double saturation();

	/**
	 * Set the saturation value.
	 * 
	 * @param value double
	 * @category accessing
	 */
	public abstract void saturation_(double value);

	/**
	 * Answer my value.
	 *
	 * @return java.lang.Object
	 * @category accessing
	 */
	public Object value() {
		return this.color();
	}

	/**
	 * Set my value.
	 *
	 * @param newValue java.lang.Object
	 * @category accessing
	 */
	public void value_(Object newValue) {
		if (newValue instanceof Color) {
			this.color_((Color) newValue);
		}
	}

	/**
	 * Answer the xy value.
	 * 
	 * @return Jun2dPoint
	 * @category basic accessing
	 */
	public Jun2dPoint xy() {
		return xy;
	}

	/**
	 * Set the xy value.
	 * 
	 * @param aPoint Jun2dPoint
	 * @category basic accessing
	 */
	public void xy_(Jun2dPoint aPoint) {
		xy = aPoint;
		this.changed_($("xy"));
	}

	/**
	 * Answer the z value.
	 * 
	 * @return double
	 * @category basic accessing
	 */
	public double z() {
		return z;
	}

	/**
	 * Set the z value.
	 * 
	 * @param aDouble double
	 * @category basic accessing
	 */
	public void z_(double aDouble) {
		z = aDouble;
		this.changed_($("z"));
	}

	/**
	 * Answer a view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunColorChoiceView && ((JunColorChoiceView) each).model() == this) {
				return (JunColorChoiceView) each;
			}
		}
		return null;
	}

	/**
	 * Initialize the ApplicationModel when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.color_(Color.white);
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem($String("Copy"), new MenuPerformer(this, "_copy")));
			_popupMenu.add(new StMenuItem($String("Paste"), new MenuPerformer(this, "_paste")));
		}
		return _popupMenu;
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Color Choice");
	}

}