package jp.co.sra.jun.goodies.button;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunButtonController class
 * 
 *  @author    He Weijie
 *  @created   1998/08/27 (by He Weijie)
 *  @updated   1999/11/12 (by MATSUDA Ryouichi)
 *  @updated   2002/11/03 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun540 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunButtonController.java,v 8.10 2008/02/20 06:31:11 nisinaka Exp $
 */
public class JunButtonController extends JunAbstractController implements MouseListener {

	protected boolean _mousePressed = false;
	protected Thread _repeatThread;

	/**
	 * Answer my model as a JunButtonModel.
	 * 
	 * @return JunButtonModel
	 * @category model accessing
	 */
	protected JunButtonModel getButtonModel() {
		return (JunButtonModel) this.model();
	}

	/**
	 * Answer my view as a JunButtonView.
	 * 
	 * @return JunButtonView interface
	 * @category view accessing
	 */
	protected JunButtonView getButtonView() {
		return (JunButtonView) this.view();
	}

	/**
	 * Add myself as a listener of the view.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#release()
	 * @category view accessing
	 */
	protected void buildListener(StView aView) {
		aView.toComponent().addMouseListener(this);
	}

	/**
	 * Invoked when a mouse button has been pressed on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent e) {
		if (this.getButtonModel().isActive() == false) {
			return;
		}

		this.getButtonModel()._isPressedWithShiftDown(e.isShiftDown());

		Graphics aGraphics = null;
		try {
			aGraphics = this.getButtonView().toComponent().getGraphics();
			if (this.getButtonModel().value() == false) {
				this.getButtonView().displayTrueOn_(aGraphics);
			}
			aGraphics.setColor(Color.black);
			Dimension size = this.getButtonView().toComponent().getSize();
			aGraphics.drawRect(0, 0, size.width - 1, size.height - 1);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}

		// start the repeat action.
		if (this.getButtonModel().repeatAction()) {
			this._startRepeatThread();
		}
	}

	/**
	 * Invoked when a mouse button has been released on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent e) {
		if (this.getButtonModel().isActive() == false) {
			return;
		}

		// stop the repeat action.		
		if (_repeatThread != null) {
			this._stopRepeatThread();
		}

		// update the display view.
		this.view().toComponent().repaint();

		// and execute the action.
		this.executeAction();
	}

	/**
	 * Do an action.
	 * 
	 * @category private
	 */
	protected synchronized void executeAction() {
		StBlockClosure action = this.getButtonModel().action();
		switch (action.numArgs()) {
			case 0:
				action.value();
				break;
			case 1:
				action.value_(this.getButtonModel());
				break;
			case 2:
				action.value_value_(this.getButtonModel(), null);
				break;
		}
	}

	/**
	 * Start a thread for the repeat action.
	 * 
	 * @category private
	 */
	protected synchronized void _startRepeatThread() {
		_mousePressed = true;
		final long time = System.currentTimeMillis();
		_repeatThread = new Thread() {
			public void run() {
				while (_mousePressed) {
					if (getButtonModel().repeatAction() && (System.currentTimeMillis() - time > getButtonModel().repeatTick())) {
						executeAction();
					}
					Thread.yield();
				}
			}
		};
		_repeatThread.setPriority(Thread.NORM_PRIORITY + 1);
		_repeatThread.start();
	}

	/**
	 * Stop the thread for the repeat action.
	 * 
	 * @category private
	 */
	protected synchronized void _stopRepeatThread() {
		_mousePressed = false;

		if (_repeatThread != null) {
			try {
				_repeatThread.join();
			} catch (Exception ex) {
				ex.printStackTrace();
			} finally {
				_repeatThread = null;
			}
		}
	}

}
