package jp.co.sra.jun.goodies.animation;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Timer;
import java.util.TimerTask;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.DependentListener;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunGifAnimationStream;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.movie.support.JunImagesToMovie;
import jp.co.sra.jun.goodies.progress.JunProgress;
import jp.co.sra.jun.goodies.track.JunTrackSliderModel;
import jp.co.sra.jun.goodies.track.JunTrackSliderView;
import jp.co.sra.jun.goodies.track.JunTrackerModel;
import jp.co.sra.jun.goodies.utilities.JunControlUtility;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunCartoonMovie class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2004/02/05 (by Nobuto Matsubara)
 *  @updated   2005/03/03 (by nisinaka)
 *  @updated   2007/11/05 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCartoonMovie.java,v 8.20 2008/02/20 06:31:11 nisinaka Exp $
 */
public class JunCartoonMovie extends JunApplicationModel {

	protected JunTrackerModel trackerModel;
	protected JunTrackSliderModel trackerModel2;
	protected ArrayList imageCollection;
	protected ArrayList timeCollection;
	protected StValueHolder progressText;
	protected Timer playProcess;
	protected String displayName;
	protected StMenuBar _menuBar;

	/**
	 * Create a new instance of JunCartoonMovie and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunCartoonMovie() {
		super();
	}

	/**
	 * Create a new instance of JunCartoonMovie and initialize it.
	 *
	 * @param imageSequenceDirectory java.io.File
	 * @param jpegImageFileNamePattern java.lang.String
	 * @param framesPerSecond int
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public JunCartoonMovie(File imageSequenceDirectory, final String jpegImageFileNamePattern, int framesPerSecond) throws IOException {
		if (imageSequenceDirectory.exists() == false) {
			throw new IOException(imageSequenceDirectory.getPath() + " does not exist.");
		}
		if (imageSequenceDirectory.isDirectory() == false) {
			throw new IOException(imageSequenceDirectory.getPath() + " is not a directory.");
		}

		File[] files = imageSequenceDirectory.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return JunStringUtility.StringMatch_and_(name, jpegImageFileNamePattern);
			}
		});
		if (files == null || files.length == 0) {
			throw new IOException("No image in " + imageSequenceDirectory.getPath());
		}

		JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
		try {
			cursor._show();

			for (int i = 0; i < files.length; i++) {
				JunImageStream aStream = null;
				try {
					aStream = JunJpegImageStream.On_(new FileInputStream(files[i]));
					this.addImage_framesPerSecond_(aStream.nextImage(), framesPerSecond);
				} finally {
					if (aStream != null) {
						aStream.close();
					}
				}
			}
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Create a new instance of JunCartoonMovie and initialize it with an animation GIF file.
	 *
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category Instance creation
	 */
	public JunCartoonMovie(File aFile) throws IOException {
		this.openGifAnimationFrom_(aFile);
	}

	/**
	 * Initializing.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		trackerModel = null;
		imageCollection = null;
		timeCollection = null;
		progressText = null;
		playProcess = null;
		_menuBar = null;
	}

	/**
	 * Answer cumulative time until index.
	 * 
	 * @param anIndex int
	 * @return double
	 * @category accessing
	 */
	public double cumulativeTimeUntilIndex_(int anIndex) {
		if (anIndex < 0) {
			return 0;
		}
		if (anIndex >= this.timeCollection().size()) {
			return this.totalMilliseconds();
		}
		double millisecondClockValue = 0;
		for (int i = 0; i < anIndex + 1; ++i) {
			millisecondClockValue = millisecondClockValue + ((Double) this.timeCollection().get(i)).doubleValue();
		}
		return millisecondClockValue;
	}

	/**
	 * Answer current milliseconds.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double currentMilliseconds() {
		return this.now() * this.totalMilliseconds();
	}

	/**
	 * Answer display name.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String displayName() {
		if (displayName == null) {
			displayName = "";
		}
		return displayName;
	}

	/**
	 * Set display name.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void displayName_(String aString) {
		displayName = aString;
	}

	/**
	 * Answer an image at cumulative time.
	 * 
	 * @param cumulativeTime double
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage imageAtCumulativeTime_(double cumulativeTime) {
		int imageIndex = this.indexAtCumulativeTime_(cumulativeTime);
		if (imageIndex < 0) {
			return this.defaultBlueImage();
		}
		return (StImage) this.imageCollection().get(imageIndex);
	}

	/**
	 * Answer image collection.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	public ArrayList imageCollection() {
		if (imageCollection == null) {
			imageCollection = new ArrayList();
		}
		return imageCollection;
	}

	/**
	 * Answer an image to display.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage imageToDisplay() {
		return this.imageAtCumulativeTime_(this.currentMilliseconds());
	}

	/**
	 * Answer cumulative time at a collection.
	 * 
	 * @param cumulativeTime double
	 * @return int
	 * @category accessing
	 */
	public int indexAtCumulativeTime_(double cumulativeTime) {
		int maxIndex = this.timeCollection().size();
		if (cumulativeTime <= 0 || cumulativeTime > this.totalMilliseconds()) {
			return 0;
		}
		for (int i = 0; i < maxIndex; ++i) {
			if (this.cumulativeTimeUntilIndex_(i - 1) <= cumulativeTime && cumulativeTime < this.cumulativeTimeUntilIndex_(i)) {
				return i;
			}
		}
		return maxIndex - 1;
	}

	/**
	 * Answer progress text.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder progressText() {
		if (progressText == null) {
			progressText = new StValueHolder("");
			updateProgressText();
		}
		return progressText;
	}

	/**
	 * Answer time collection.
	 * 
	 * @return java.util.ArrayList
	 * @category accessing
	 */
	public ArrayList timeCollection() {
		if (timeCollection == null) {
			timeCollection = new ArrayList();
		}
		return timeCollection;
	}

	/**
	 * Answer total millisecond of current animation.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double totalMilliseconds() {
		double millisecondClockValue = 0;
		for (int i = 0; i < this.timeCollection().size(); ++i) {
			double each = ((Double) this.timeCollection().get(i)).doubleValue();
			millisecondClockValue = millisecondClockValue + each;
		}
		return millisecondClockValue;
	}

	/**
	 * Answer a JunTrackerModel.
	 * If trackerModel is null, set trackerModel a new JunTrackerModel.
	 * 
	 * @return JunTrackerModel
	 * @category accessing
	 */
	public JunTrackerModel trackerModel() {
		if (trackerModel != null) {
			return trackerModel;
		}
		trackerModel = new JunTrackerModel();
		trackerModel.addDependent_(this);
		final JunCartoonMovie this_ = this;
		trackerModel.playButton().action_(new StBlockClosure() {
			public Object value_(Object anObject) {
				this_.playAction_((JunButtonModel) anObject);
				return null;
			}
		});
		trackerModel.loopButton().action_(new StBlockClosure() {
			public Object value_(Object anObject) {
				this_.loopAction_((JunButtonModel) anObject);
				return null;
			}
		});
		trackerModel.nextButton().action_(new StBlockClosure() {
			public Object value_(Object anObject) {
				this_.nextAction_((JunButtonModel) anObject);
				return null;
			}
		});
		trackerModel.previousButton().action_(new StBlockClosure() {
			public Object value_(Object anObject) {
				this_.previousAction_((JunButtonModel) anObject);
				return null;
			}
		});
		trackerModel.compute_(new StBlockClosure() {
			public Object value_(Object anObject) {
				this_.setTime_(((Double) anObject).doubleValue());
				return null;
			}
		});
		trackerModel.enableMarkers();
		return trackerModel;
	}

	/**
	 * Answer a JunTrackSliderModel.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackSliderModel
	 * @category accessing
	 */
	public JunTrackSliderModel trackerModel2() {
		if (trackerModel2 == null) {
			trackerModel2 = new JunTrackSliderModel();
			trackerModel2.parentTracker_(this.trackerModel());
			final JunCartoonMovie this_ = this;
			trackerModel2.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					this_.setTime2_(((Double) anObject).doubleValue());
					return null;
				}
			});
		}
		return trackerModel2;
	}

	/**
	 * Add an image and set the number of frames per second.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @param aNumber int
	 * @category adding
	 */
	public void addImage_framesPerSecond_(StImage anImage, int aNumber) {
		this.addImage_keepTime_(anImage, 1000.0d / Math.min(Math.max(1, aNumber), 1000));
	}

	/**
	 * Add an image and set keep time.
	 * 
	 * @param anImage StImage
	 * @param millisecondClockValue double
	 * @category adding
	 */
	public void addImage_keepTime_(StImage anImage, double millisecondClockValue) {
		this.imageCollection().add(anImage);
		this.timeCollection().add(new Double(millisecondClockValue));
		double totalValue = this.totalMilliseconds();
		if (totalValue <= 0) {
			this.trackerModel().step_(1);
		} else {
			this.trackerModel().step_(1.0d / totalValue);
		}
		this.updateMenuIndication();
	}

	/**
	 * Set loop button action.
	 * 
	 * @param model JunButtonModel
	 * @category button actions
	 */
	public void loopAction_(JunButtonModel model) {
		if (this.trackerModel().loopCondition() == $("oneWay")) {
			this.trackerModel().loop();
		} else {
			this.trackerModel().oneWay();
		}
		this.updatePlayMenuIndication();
	}

	/**
	 * Set next button action.
	 * 
	 * @param model JunButtonModel
	 * @category button actions
	 */
	public void nextAction_(JunButtonModel model) {
		if (model._isPressedWithShiftDown()) {
			this.last();
		} else {
			this.next();
		}
	}

	/**
	 * Set play button action.
	 * 
	 * @param model JunButtonModel
	 * @category button actions
	 */
	public void playAction_(JunButtonModel model) {
		if (model.value()) {
			this.stop();
		} else {
			if (model._isPressedWithShiftDown()) {
				this.trackerModel().first();
			}
			this.start();
		}
		this.updatePlayMenuIndication();
	}

	/**
	 * Set previous button action.
	 * 
	 * @param model JunButtonModel
	 * @category button actions
	 */
	public void previousAction_(JunButtonModel model) {
		if (model._isPressedWithShiftDown()) {
			this.first();
		} else {
			this.previous();
		}
	}

	/**
	 * Convert the receiver to an image as StImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		return this.imageToDisplay();
	}

	/**
	 * Answer default "blue image".
	 * 
	 * @return StImage
	 * @category defaults
	 */
	public StImage defaultBlueImage() {
		StImage aImage = new StImage(this.defaultBlueImageExtent());
		Graphics aGraphics = aImage.image().getGraphics();
		aGraphics.setColor(Color.blue);
		aGraphics.fillRect(0, 0, aImage.width(), aImage.height());
		return aImage;
	}

	/**
	 * Answer default blue image extent.
	 * 
	 * @return Point
	 * @category defaults
	 */
	public Point defaultBlueImageExtent() {
		return new Point(320, 240);
	}

	/**
	 * Answer default display name color.
	 * 
	 * @return Color
	 * @category defaults
	 */
	public Color defaultDisplayNameColor() {
		Color black = Color.black;
		Color green = Color.green;
		Color blendColor = new Color(black.getRed() + green.getRed(), black.getGreen() + green.getGreen(), black.getBlue() + green.getBlue());
		return blendColor;
	}

	/**
	 * Do images and times.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object imagesAndTimesDo_(StBlockClosure aBlock) {
		for (int i = 0; i < this.imageCollection().size(); ++i) {
			StImage anImage = (StImage) this.imageCollection().get(i);
			Double millisecondClockValue = (Double) this.timeCollection().get(i);
			Object result = aBlock.value_value_(anImage, millisecondClockValue);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		if (playProcess != null) {
			playProcess.cancel();
			playProcess = null;
		}

		super.noticeOfWindowClose(e);
	}

	/**
	 * Open and play.
	 * 
	 * @category interface opening
	 */
	public void openAndPlay() {
		this.openLightWeightWindowSpecFitFlag_(true);
		this.start();
	}

	/**
	 * Open with the light weight window spec view.
	 * 
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame openLightWeightWindowSpec() {
		return this.openLightWeightWindowSpecAt_fitFlag_markerFlag_(null, true, false);
	}

	/**
	 * Open with the light weight window spec view.
	 * 
	 * @param displayPoint java.awt.Point
	 * @param fitBoolean boolean
	 * @param markerBoolean
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame openLightWeightWindowSpecAt_fitFlag_markerFlag_(Point displayPoint, boolean fitBoolean, boolean markerBoolean) {
		Frame aFrame = null;

		StView aView = this.defaultViewWithSimpleTracker();
		if (displayPoint == null) {
			aFrame = this.openView_(aView);
		} else {
			aFrame = this.openViewAt_(aView, displayPoint);
		}

		if (fitBoolean) {
			this.normalSize();
		}

		if (markerBoolean) {
			this.trackerModel().enableMarkers();
		} else {
			this.trackerModel().disableMarkers();
		}

		return aFrame;
	}

	/**
	 * Open with the light weight window spec view.
	 * 
	 * @param fitBoolean boolean
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame openLightWeightWindowSpecFitFlag_(boolean fitBoolean) {
		return this.openLightWeightWindowSpecAt_fitFlag_markerFlag_(null, fitBoolean, false);
	}

	/**
	 * Open with the light weight window spec view.
	 * 
	 * @param fitBoolean boolean
	 * @param markerBoolean boolean
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame openLightWeightWindowSpecFitFlag_markerFlag_(boolean fitBoolean, boolean markerBoolean) {
		return this.openLightWeightWindowSpecAt_fitFlag_markerFlag_(null, fitBoolean, markerBoolean);
	}

	/**
	 * Open with the light weight window spec view.
	 * 
	 * @param markerBoolean boolean
	 * @return java.awt.Frame
	 * @category interface opening
	 */
	public Frame openLightWeightWindowSpecMarkerFlag_(boolean markerBoolean) {
		return this.openLightWeightWindowSpecAt_fitFlag_markerFlag_(null, false, markerBoolean);
	}

	/**
	 * Answer my default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		return this.defaultViewWithTracker();
	}

	/**
	 * Answer my default view with tracker.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultViewWithTracker() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunCartoonMovieViewAwt.WithTracker(this);
		} else {
			return JunCartoonMovieViewSwing.WithTracker(this);
		}
	}

	/**
	 * Answer my default view with a simple tracker.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultViewWithSimpleTracker() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunCartoonMovieViewAwt.WithSimpleTracker(this);
		} else {
			return JunCartoonMovieViewSwing.WithSimpleTracker(this);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Cartoon Movie Player");
	}

	/**
	 * Answer my KeyListener which handles my keyboard events.
	 *
	 * @return java.awt.event.KeyListener
	 * @category keyboard
	 */
	public KeyListener _keyListener() {
		return new KeyAdapter() {
			public void keyPressed(KeyEvent e) {
				if (_keyboardEvent(e)) {
					e.consume();
				}
			}
		};
	}

	/**
	 * Process the key event.
	 *
	 * @param ev java.awt.event.KeyEvent
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean _keyboardEvent(KeyEvent ev) {
		if (ev.isConsumed()) {
			return false;
		}

		int code = ev.getKeyCode();
		switch (code) {
			case KeyEvent.VK_SPACE:
			case KeyEvent.VK_ENTER:
				if (this.isPlay()) {
					this.stop();
				} else {
					this.start();
				}
				return true;
			case KeyEvent.VK_LEFT:
				this.previousAction_(this.trackerModel().previousButton());
				return true;
			case KeyEvent.VK_RIGHT:
				this.nextAction_(this.trackerModel().nextButton());
				return true;
			case KeyEvent.VK_TAB:
				if (this.trackerModel().loopCondition() == $("oneWay")) {
					this.trackerModel().loopCondition_($("loop"));
				} else {
					this.trackerModel().loopCondition_($("oneWay"));
				}
				return true;
			case (int) '[':
				this.trackerModel().firstMarker_(this.now());
				return true;
			case (int) ']':
				this.trackerModel().lastMarker_(this.now());
				return true;
			case (int) 'P':
				this.prologue();
				return true;
			case (int) 'E':
				this.epilogue();
				return true;
			case (int) 'F':
				this.trackerModel().first();
				return true;
			case (int) 'L':
				this.trackerModel().last();
				return true;
		}
		return false;
	}

	/**
	 * Update the menu indication.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#updateMenuIndication()
	 * @category menu accessing
	 */
	public void updateMenuIndication() {
		super.updateMenuIndication();
		this.updateFileMenuIndication();
		this.updatePlayMenuIndication();
		this.updatePositionMenuIndication();
		this.updateMiscMenuIndication();
	}

	/**
	 * Update the file menu indication.
	 * 
	 * @category menu accessing
	 */
	protected void updateFileMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("fileMenu"));
		if (aMenu == null) {
			return;
		}

		StMenuItem menuItem;
		menuItem = aMenu.atNameKey_($("saveAsGifAnimation"));
		if (menuItem != null) {
			if (this.isEmpty()) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}

		menuItem = aMenu.atNameKey_($("saveAsMovie"));
		if (menuItem != null) {
			if (this.isEmpty()) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}
	}

	/**
	 * Update the play menu indication.
	 * 
	 * @category menu accessing
	 */
	protected void updatePlayMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("playMenu"));
		if (aMenu == null) {
			return;
		}

		StMenuItem menuItem;
		menuItem = aMenu.atNameKey_($("start"));
		if (menuItem != null) {
			if (this.isEmpty()) {
				menuItem.disable();
			} else if (this.isPlay()) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}

		menuItem = aMenu.atNameKey_($("stop"));
		if (menuItem != null) {
			if (this.isEmpty()) {
				menuItem.disable();
			} else if (this.isPlay()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}

		menuItem = aMenu.atNameKey_($("loopPlay"));
		if (menuItem != null) {
			if (this.trackerModel().loopCondition() == $("loop")) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}

		menuItem = aMenu.atNameKey_($("oneWayPlay"));
		if (menuItem != null) {
			if (this.trackerModel().loopCondition() == $("oneWay")) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}
	}

	/**
	 * Update the position menu indication.
	 * 
	 * @category menu accessing
	 */
	protected void updatePositionMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("positionMenu"));
		if (aMenu == null) {
			return;
		}

		StMenuItem menuItem;
		menuItem = aMenu.atNameKey_($("gotoPrevious"));
		if (menuItem != null) {
			if (this.isEmpty()) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}

		menuItem = aMenu.atNameKey_($("gotoNext"));
		if (menuItem != null) {
			if (this.isEmpty()) {
				menuItem.disable();
			} else {
				menuItem.enable();
			}
		}

		menuItem = aMenu.atNameKey_($("gotoFirst"));
		if (menuItem != null) {
			if (this.trackerModel().areMarkersActive()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}

		menuItem = aMenu.atNameKey_($("gotoLast"));
		if (menuItem != null) {
			if (this.trackerModel().areMarkersActive()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}

		menuItem = aMenu.atNameKey_($("setFirstMarker"));
		if (menuItem != null) {
			if (this.trackerModel().areMarkersActive()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}

		menuItem = aMenu.atNameKey_($("setLastMarker"));
		if (menuItem != null) {
			if (this.trackerModel().areMarkersActive()) {
				menuItem.enable();
			} else {
				menuItem.disable();
			}
		}
	}

	/**
	 * Update the misc menu indication.
	 * 
	 * @category menu accessing
	 */
	protected void updateMiscMenuIndication() {
		StMenu aMenu = (StMenu) this._menuBar().atNameKey_($("positionMenu"));
		if (aMenu == null) {
			return;
		}

		// nothing to do
	}

	/**
	 * Set the size to double.
	 * 
	 * @category menu messages
	 */
	public void doubleSize() {
		int width = -1;
		int height = -1;
		StImage[] images = (StImage[]) this.imageCollection().toArray(new StImage[this.imageCollection().size()]);
		for (int i = 0; i < images.length; i++) {
			width = Math.max(width, images[i].width());
			height = Math.max(height, images[i].height());
		}
		if (width <= 0 || height <= 0) {
			return;
		}

		this.setSize_(new Dimension(width * 2, height * 2));
	}

	/**
	 * Go to epilogue menu message.
	 * 
	 * @category menu messages
	 */
	public void gotoEpilogue() {
		this.epilogue();
	}

	/**
	 * Go to first menu message.
	 * 
	 * @category menu messages
	 */
	public void gotoFirst() {
		this.first();
	}

	/**
	 * Go to last menu message.
	 * 
	 * @category menu messages
	 */
	public void gotoLast() {
		this.last();
	}

	/**
	 * Go to next menu message.
	 * 
	 * @category menu messages
	 */
	public void gotoNext() {
		this.next();
	}

	/**
	 * Go to previous menu message.
	 * 
	 * @category menu messages
	 */
	public void gotoPrevious() {
		this.previous();
	}

	/**
	 * Go to prologue menu message.
	 * 
	 * @category menu messages
	 */
	public void gotoPrologue() {
		this.prologue();
	}

	/**
	 * Set the size to half.
	 * 
	 * @category menu messages
	 */
	public void halfSize() {
		int width = -1;
		int height = -1;
		StImage[] images = (StImage[]) this.imageCollection().toArray(new StImage[this.imageCollection().size()]);
		for (int i = 0; i < images.length; i++) {
			width = Math.max(width, images[i].width());
			height = Math.max(height, images[i].height());
		}
		if (width <= 0 || height <= 0) {
			return;
		}

		this.setSize_(new Dimension(width / 2, height / 2));
	}

	/**
	 * Loop play menu message.
	 * 
	 * @category menu messages
	 */
	public void loopPlay() {
		this.trackerModel().loop();
		this.updatePlayMenuIndication();
	}

	/**
	 * Set the size to normal.
	 * 
	 * @category menu messages
	 */
	public void normalSize() {
		int width = -1;
		int height = -1;
		StImage[] images = (StImage[]) this.imageCollection().toArray(new StImage[this.imageCollection().size()]);
		for (int i = 0; i < images.length; i++) {
			width = Math.max(width, images[i].width());
			height = Math.max(height, images[i].height());
		}
		if (width <= 0 || height <= 0) {
			return;
		}

		this.setSize_(new Dimension(width, height));
	}

	/**
	 * OneWay play menu message.
	 * 
	 * @category menu messages
	 */
	public void oneWayPlay() {
		this.trackerModel().oneWay();
		this.updatePlayMenuIndication();
	}

	/**
	 * Open a GIF animation.
	 * 
	 * @throws java.io.IOException
	 * @category menu messages
	 */
	public void openGifAnimation() throws IOException {
		File aFile = RequestFile();
		if (aFile == null) {
			return;
		}

		this.openGifAnimationFrom_(aFile);
	}

	/**
	 * Open a GIF animation from the file.
	 * 
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category menu messages
	 */
	public void openGifAnimationFrom_(File aFile) throws IOException {
		if (aFile == null || aFile.exists() == false) {
			return;
		}

		ArrayList aList = new ArrayList();

		JunGifAnimationStream stream = null;
		try {
			stream = (JunGifAnimationStream) JunGifAnimationStream.On_(new FileInputStream(aFile));

			JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
			try {
				cursor._show();

				StImage image = null;
				while ((image = stream.nextImage()) != null) {
					int tick = stream.tick();
					aList.add(new Object[] { image, new Integer(tick) });
				}

			} finally {
				cursor._restore();
			}

		} finally {
			if (stream != null) {
				stream.close();
			}
		}

		imageCollection = null;
		timeCollection = null;

		if (stream.loop() == 0) {
			this.loopPlay();
		}
		for (int i = 0; i < aList.size(); i++) {
			Object[] entity = (Object[]) aList.get(i);
			StImage image = (StImage) entity[0];
			int tick = ((Number) entity[1]).intValue();
			this.addImage_keepTime_(image, tick);
		}

		this.normalSize();
		this.goto_(0);
	}

	/**
	 * Quit menu message.
	 * 
	 * @category menu messages
	 */
	public void quitDoing() {
		this.closeRequest();
	}

	/**
	 * Save as a GIF animation file.
	 * 
	 * @throws java.io.IOException
	 * @category menu messages
	 */
	public void saveAsGifAnimation() throws IOException {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("GifAnimation files"), new String[] { "*.gif", "*.GIF" }) };
		File aFile = JunFileRequesterDialog.RequestNewFile($String("Input a GifAnimation file."), new File(this.defaultBaseName() + ".gif"), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return;
		}

		this.saveAsGifAnimationTo_(aFile);
	}

	/**
	 * SAve as a GIF animation to the file.
	 * 
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category menu messages
	 */
	public void saveAsGifAnimationTo_(File aFile) throws IOException {
		JunGifAnimationStream stream = null;
		try {
			stream = (JunGifAnimationStream) JunGifAnimationStream.On_(new FileOutputStream(aFile));

			if (this.trackerModel().loopCondition() == $("loop")) {
				stream.loop_(0);
			} else {
				stream.loop_(1);
			}

			JunCursors cursor = new JunCursors(JunCursors.WriteCursor());
			try {
				cursor._show();

				final JunGifAnimationStream _stream = stream;
				Object result = this.imagesAndTimesDo_(new StBlockClosure() {
					public Object value_value_(Object image, Object tick) {
						_stream.tick_(((Number) tick).intValue());
						try {
							_stream.nextPutImage_((StImage) image);
						} catch (IOException e) {
							return e;
						}
						return null;
					}
				});
				if (result != null && result instanceof IOException) {
					throw (IOException) result;
				}

			} finally {
				cursor._restore();
			}
		} finally {
			if (stream != null) {
				stream.close();
			}
		}
	}

	/**
	 * Save as movie menu message.
	 * 
	 * @category menu messages
	 */
	public void saveAsMovie() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Movie files"), new String[] { "*.mov", "*.MOV" }) };
		File aFile = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, $String("Movie")), new File(this.defaultBaseName() + ".mov"), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return;
		}

		this.saveAsMovieTo_(aFile);
		// (new JunMoviePlayer(aFile)).open();
	}

	/**
	 * Save as movie to the file.
	 * 
	 * @param aFile java.lang.String
	 * @category menu messages
	 */
	public void saveAsMovieTo_(File aFile) {
		final StView aView = this.getView();
		StRectangle aBox;
		if (aView == null) {
			aBox = new StRectangle(0, 0, 320, 240);
		} else {
			aBox = new StRectangle(aView.toComponent().getBounds());
		}
		JunImagesToMovie.File_extent_do_(aFile, new Dimension(aBox.width(), aBox.height()), new StBlockClosure() {
			public Object value_(Object o) {
				final JunImagesToMovie imagesToMovie = (JunImagesToMovie) o;
				if (aView != null) {
					imagesToMovie.background_(aView.toComponent().getBackground());
				}
				(new JunProgress()).do_(new StBlockClosure() {
					public Object value_(Object o) {
						final JunProgress aProgress = (JunProgress) o;
						final Integer[] timeAndCount = { new Integer(0), new Integer(0) };
						final int size = JunCartoonMovie.this.imageCollection().size();
						aProgress.value_(timeAndCount[1].intValue() / size);
						aProgress.message_($String("writing..."));
						JunCartoonMovie.this.imagesAndTimesDo_(new StBlockClosure() {
							public Object value_value_(Object A, Object B) {
								StImage anImage = (StImage) A;
								int aTick = ((Double) B).intValue();
								if (aTick > 0) {
									imagesToMovie.add_milliseconds_(anImage, aTick);
								}
								timeAndCount[1] = new Integer(timeAndCount[1].intValue() + 1);
								timeAndCount[0] = new Integer(timeAndCount[0].intValue() + aTick);
								aProgress.value_(timeAndCount[1].intValue() / size);
								aProgress.message_($String("writing..."));
								return null;
							}
						});
						return Boolean.TRUE;
					}
				});
				return null;
			}
		});
	}

	/**
	 * Set first marker menu message.
	 * 
	 * @category menu messages
	 */
	public void setFirstMarker() {
		this.trackerModel().firstMarkerButton().action().value_(this.trackerModel().firstMarkerButton());
	}

	/**
	 * Set last marker menu message.
	 * 
	 * @category menu messages
	 */
	public void setLastMarker() {
		this.trackerModel().lastMarkerButton().action().value_(this.trackerModel().lastMarkerButton());
	}

	/**
	 * Go to frame 1 (End of movie).
	 * 
	 * @category playing
	 */
	public void epilogue() {
		this.goto_(1);
	}

	/**
	 * Go to first frame of a movie file.
	 * 
	 * @category playing
	 */
	public void first() {
		this.trackerModel().first();
	}

	/**
	 * Go to normalizedValue time.
	 * 
	 * @param normalizedValue double
	 * @category playing
	 */
	public void goto_(double normalizedValue) {
		this.trackerModel().value_(normalizedValue);
	}

	/**
	 * Go to last of movie.
	 * 
	 * @category playing
	 */
	public void last() {
		this.trackerModel().last();
	}

	/**
	 * Go to next frame of a movie file.
	 * 
	 * @category playing
	 */
	public void next() {
		if (this.isEmpty()) {
			return;
		}
		int currentIndex = this.indexAtCumulativeTime_(this.currentMilliseconds());
		double cumulativeTime = this.cumulativeTimeUntilIndex_(currentIndex);
		double totalValue = this.totalMilliseconds();
		double normalizedValue;
		if (totalValue <= 0) {
			normalizedValue = 1;
		} else {
			normalizedValue = cumulativeTime / totalValue;
		}
		this.goto_(normalizedValue);
	}

	/**
	 * Answer now.
	 * 
	 * @return double
	 * @category playing
	 */
	public double now() {
		return this.trackerModel().doubleValue();
	}

	/**
	 * Go to previous frame of a movie file.
	 * 
	 * @category playing
	 */
	public void previous() {
		if (this.isEmpty()) {
			return;
		}
		int currentIndex = this.indexAtCumulativeTime_(this.currentMilliseconds());
		double cumulativeTime = this.cumulativeTimeUntilIndex_(currentIndex);
		double keepTime = ((Double) this.timeCollection().get(currentIndex)).doubleValue();
		cumulativeTime = cumulativeTime - keepTime;
		if (this.currentMilliseconds() == cumulativeTime) {
			currentIndex = Math.max(currentIndex - 1, 0);
			keepTime = ((Double) this.timeCollection().get(currentIndex)).doubleValue();
			while (currentIndex > 1 && keepTime == 0) {
				currentIndex = Math.max(currentIndex - 1, 0);
				keepTime = ((Double) this.timeCollection().get(currentIndex)).doubleValue();
			}
			cumulativeTime = this.cumulativeTimeUntilIndex_(currentIndex);
			keepTime = ((Double) this.timeCollection().get(currentIndex)).doubleValue();
			cumulativeTime = cumulativeTime - keepTime;
		}
		double totalValue = this.totalMilliseconds();
		double normalizedValue;
		if (totalValue <= 0) {
			normalizedValue = 0;
		} else {
			normalizedValue = cumulativeTime / totalValue;
		}
		this.goto_(normalizedValue);
	}

	/**
	 * Go to frame 0 (First of movie).
	 *
	 * @category playing
	 */
	public void prologue() {
		this.goto_(0);
	}

	/**
	 * Start to play a movie.
	 * 
	 * @category playing
	 */
	public void start() {
		if (this.isEmpty()) {
			return;
		}

		if (playProcess != null) {
			playProcess.cancel();
		}

		this.trackerModel().playButtonVisual_(true);
		if (this.now() >= 1) {
			this.prologue();
		}
		if (this.trackerModel().isIntervalNotEmpty() && (this.now() <= this.trackerModel().firstMarker() || this.trackerModel().lastMarker() <= this.now())) {
			this.first();
		}

		double previousTime = this.currentMilliseconds();
		double endTime = this.totalMilliseconds();
		final double[] time = new double[] { previousTime, endTime };

		playProcess = new Timer();
		playProcess.scheduleAtFixedRate(new TimerTask() {
			public void run() {
				double previousTime = time[0];
				double endTime = time[1];
				double currentTime = currentMilliseconds() + tickTime();
				if (previousTime == currentTime) {
					currentTime += tickTime();
				}
				if (endTime <= 0) {
					epilogue();
				} else {
					goto_(currentTime / endTime);
				}
				if (currentTime >= endTime) {
					if (trackerModel().loopCondition() == $("loop")) {
						prologue();
					} else {
						stop();
						last();
					}
				}
				if (trackerModel().isIntervalNotEmpty()) {
					if (currentTime >= trackerModel().lastMarker() * endTime) {
						if (trackerModel().loopCondition() == $("loop")) {
							first();
						} else {
							stop();
							last();
						}
					}
				}

				time[0] = currentTime;
			}
		}, 0, this.tickTime());

		this.updatePlayMenuIndication();
	}

	/**
	 * Stop to playing movie.
	 * 
	 * @category playing
	 */
	public void stop() {
		if (playProcess != null) {
			playProcess.cancel();
			playProcess = null;
		}

		if (this.isEmpty()) {
			return;
		}

		this.trackerModel().playButtonVisual_(false);
		this.updatePlayMenuIndication();
	}

	/**
	 * Execute playing movie thread.
	 *
	 * @category playing
	 */
	protected void run() {
		if (this.isEmpty()) {
			return;
		}

		if (this.now() >= 1) {
			this.prologue();
		}
		if (this.trackerModel().isIntervalNotEmpty() && (this.now() <= this.trackerModel().firstMarker() || this.now() >= this.trackerModel().lastMarker())) {
			this.first();
		}

		double previousTime = this.currentMilliseconds();
		double endTime = this.totalMilliseconds();
		while (this.isPlay()) {
			JunControlUtility.NextMillisecondClockValue_(this.tickTime()); // (Delay untilMilliseconds: nextTime) wait

			double currentTime = this.currentMilliseconds() + this.tickTime();
			if (previousTime == currentTime) {
				currentTime += this.tickTime();
			}
			if (endTime <= 0) {
				this.epilogue();
			} else {
				this.goto_(currentTime / endTime);
			}
			if (currentTime >= endTime) {
				if (this.trackerModel().loopCondition() == $("loop")) {
					this.prologue();
				} else {
					this.stop();
					this.last();
				}
			}
			if (this.trackerModel().isIntervalNotEmpty()) {
				if (currentTime >= this.trackerModel().lastMarker() * endTime) {
					if (this.trackerModel().loopCondition() == $("loop")) {
						this.first();
					} else {
						this.stop();
						this.last();
					}
				}
			}
			previousTime = currentTime;
		}
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));
			fileMenu.add(new StMenuItem($String("Open GifAnimation..."), new MenuPerformer(this, "openGifAnimation")));
			fileMenu.addSeparator();
			fileMenu.add(new StMenuItem($String("Save as GifAnimation..."), $("saveAsGifAnimation"), new MenuPerformer(this, "saveAsGifAnimation")));
			fileMenu.add(new StMenuItem($String("Save as movie..."), $("saveAsMovie"), new MenuPerformer(this, "saveAsMovie")));
			fileMenu.addSeparator();
			fileMenu.add(new StMenuItem($String("Quit"), new MenuPerformer(this, "quitDoing")));
			_menuBar.add(fileMenu);

			StMenu playMenu = new StMenu($String("Play"), $("playMenu"));
			playMenu.add(new StMenuItem($String("Start"), $("start"), new MenuPerformer(this, "start")));
			playMenu.add(new StMenuItem($String("Stop"), $("stop"), new MenuPerformer(this, "stop")));
			playMenu.addSeparator();
			playMenu.add(new StMenuItem($String("Loop"), $("loopPlay"), new MenuPerformer(this, "loopPlay")));
			playMenu.add(new StMenuItem($String("One way"), $("oneWayPlay"), new MenuPerformer(this, "oneWayPlay")));
			_menuBar.add(playMenu);

			StMenu positionMenu = new StMenu($String("Position"), $("positionMenu"));
			positionMenu.add(new StMenuItem($String("Previous"), $("gotoPrevious"), new MenuPerformer(this, "gotoPrevious")));
			positionMenu.add(new StMenuItem($String("Next"), $("gotoNext"), new MenuPerformer(this, "gotoNext")));
			positionMenu.addSeparator();
			positionMenu.add(new StMenuItem($String("-> Prologue"), new MenuPerformer(this, "gotoPrologue")));
			positionMenu.add(new StMenuItem($String("-> Epilogue"), new MenuPerformer(this, "gotoEpilogue")));
			positionMenu.addSeparator();
			positionMenu.add(new StMenuItem($String("-> First"), $("gotoFirst"), new MenuPerformer(this, "gotoFirst")));
			positionMenu.add(new StMenuItem($String("-> Last"), $("gotoLast"), new MenuPerformer(this, "gotoLast")));
			positionMenu.addSeparator();
			positionMenu.add(new StMenuItem($String("Set first marker"), $("setFirstMarker"), new MenuPerformer(this, "setFirstMarker")));
			positionMenu.add(new StMenuItem($String("Set last marker"), $("setLastMarker"), new MenuPerformer(this, "setLastMarker")));
			_menuBar.add(positionMenu);

			StMenu miscMenu = new StMenu($String("Misc"), $("miscMenu"));
			miscMenu.add(new StMenuItem($String("Half size"), new MenuPerformer(this, "halfSize")));
			miscMenu.add(new StMenuItem($String("Normal size"), new MenuPerformer(this, "normalSize")));
			miscMenu.add(new StMenuItem($String("Double size"), new MenuPerformer(this, "doubleSize")));
			_menuBar.add(miscMenu);
		}
		return _menuBar;
	}

	/**
	 * Answer current animation is empty or not.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmpty() {
		return this.imageCollection().size() == 0;
	}

	/**
	 * Answer current animation is playing or not.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPlay() {
		return this.trackerModel().playButton().value() == true;
	}

	/**
	 * Update progress text.
	 * 
	 * @category updating
	 */
	public void updateProgressText() {
		this.progressText().value_(this.stringFromMilliseconds_((long) this.currentMilliseconds()) + " / " + this.stringFromMilliseconds_((long) this.totalMilliseconds()));
	}

	/**
	 * Answer a view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#getView()
	 * @category private
	 */
	public StView getView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			Object each = dependents[i];
			if (each instanceof JunCartoonMovieView && ((StView) each).model() == this) {
				return (StView) each;
			}
		}
		return null;
	}

	/**
	 * Answer marker margins.
	 * 
	 * @return double[]
	 * @category private
	 */
	public double[] markerMargins() {
		return new double[] { 0.1, 0.1 };
	}

	/**
	 * Set time.
	 * 
	 * @param time Time
	 * @category private
	 */
	public void setTime_(double time) {
		this.updateProgressText();
		this.changed_($("image"));
		this.setTime1_(time);
	}

	/**
	 * Set time.
	 * 
	 * @param  time Time
	 * @category private
	 */
	public void setTime1_(double time) {
		boolean aBoolean = this.trackerModel2().areMarkersActive();
		if (this.trackerModel().isIntervalNotEmpty()) {
			this.trackerModel2().enableMarkers();
		} else {
			this.trackerModel2().disableMarkers();
		}
		if (!(aBoolean == this.trackerModel2().areMarkersActive())) {
			DependentListener[] dependents = this.trackerModel2().dependents();
			for (int i = 0; i < dependents.length; ++i) {
				if (dependents[i] instanceof JunTrackSliderView) {
					dependents[i].update_(new DependentEvent(dependents[i], $("interval"), null));
				}
			}
		}
		double value;
		if (this.trackerModel().isIntervalNotEmpty()) {
			value = (this.trackerModel().doubleValue() - this.trackerModel().firstMarker()) / (this.trackerModel().lastMarker() - this.trackerModel().firstMarker());
			value = value * (1 - this.markerMargins()[0] - this.markerMargins()[1]) + this.markerMargins()[0];
			value = Math.max(0, Math.min(value, 1));
		} else {
			value = this.trackerModel().doubleValue();
		}
		double[] margins = { this.markerMargins()[0], 1 - this.markerMargins()[1] };
		this.trackerModel2().intervalHolder().value_(margins);
		this.trackerModel2().valueHolder().value_(value);
		DependentListener[] dependents = this.trackerModel2().dependents();
		for (int i = 0; i < dependents.length; ++i) {
			if (dependents[i] instanceof JunTrackSliderView) {
				dependents[i].update_(new DependentEvent(dependents[i], $("value"), null));
			}
		}
	}

	/**
	 * Set time.
	 * 
	 * @param time double
	 * @category private
	 */
	public void setTime2_(double time) {
		double value;
		if (this.trackerModel().isIntervalNotEmpty()) {
			value = (time - this.markerMargins()[0]) / (1 - this.markerMargins()[0] - this.markerMargins()[1]);
			value = (this.trackerModel().lastMarker() - this.trackerModel().firstMarker()) * value + this.trackerModel().firstMarker();
			this.trackerModel().value_(value);
		} else {
			this.trackerModel().value_(time);
		}
	}

	/**
	 * Answer tick time.
	 * 
	 * @return int
	 * @category private
	 */
	public int tickTime() {
		return 50;
	}

	/**
	 * Convert the number in milliseconds to a string representation.
	 *
	 * @param milliseconds long
	 * @return java.lang.String
	 * @category private
	 */
	protected String stringFromMilliseconds_(long milliseconds) {
		long decimal = milliseconds % 1000;
		long value = milliseconds / 1000;
		int seconds = (int) (value % 60);
		int minutes = (int) (value / 60 % 60);
		int hours = (int) (value / 3600);

		StringBuffer buf = new StringBuffer();
		if (hours < 10) {
			buf.append('0');
		}
		buf.append(hours);
		buf.append(':');
		if (minutes < 10) {
			buf.append('0');
		}
		buf.append(minutes);
		buf.append(':');
		if (seconds < 10) {
			buf.append('0');
		}
		buf.append(seconds);
		buf.append('.');
		if (decimal < 100) {
			buf.append('0');
		}
		if (decimal < 10) {
			buf.append('0');
		}
		buf.append(decimal);
		return buf.toString();
	}

	/**
	 * Create a movie file from the images in the specified directory.
	 * 
	 * @param aDirectory java.io.File
	 * @param aString java.lang.String
	 * @param aNumber int
	 * @param aFile java.io.File
	 * @throws java.io.IOException
	 * @category Utilities
	 */
	public static void ImageSequenceDirectory_jpegImageFileNamePattern_framesPerSecond_movieFile_(File aDirectory, String aString, int aNumber, File aFile) throws IOException {
		JunImagesToMovie.ImageSequenceDirectory_jpegImageFileNamePattern_framesPerSecond_movieFile_(aDirectory, aString, aNumber, aFile);
	}

	/**
	 * Open a file requester dialog to choose a GIF animation file.
	 * 
	 * @return java.io.File
	 * @category Utilities
	 */
	public static File RequestFile() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("GifAnimation files"), new String[] { "*.gif", "*.GIF" }) };
		File aFile = JunFileRequesterDialog.Request($String("Select a file"), fileTypes, fileTypes[0]);
		return aFile;
	}

	/**
	 * Create a JunCartoonMovie with a GIF file and show it.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.goodies.animation.JunCartoonMovie
	 * @category Utilities
	 */
	public static JunCartoonMovie ShowFile_(File aFile) {
		if (aFile == null || aFile.exists() == false) {
			return null;
		}

		try {
			JunCartoonMovie aCartoonMovie = new JunCartoonMovie(aFile);
			aCartoonMovie.openLightWeightWindowSpecFitFlag_markerFlag_(true, false);
			aCartoonMovie.start();
			return aCartoonMovie;
		} catch (IOException e) {
			e.printStackTrace();
			return null;
		}
	}

}
