package jp.co.sra.jun.geometry.transformations;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.jun.collections.sequences.JunDoubleMatrix;
import jp.co.sra.jun.collections.sequences.JunLinearEquations;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.curves.JunLine;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun3dTransformation class
 * 
 *  @author    nisinaka
 *  @created   1998/10/09 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2003/07/01 (by nisinaka)
 *  @updated   2004/09/29 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun673 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dTransformation.java,v 8.13 2008/02/20 06:31:11 nisinaka Exp $
 */
public class Jun3dTransformation extends JunTransformation {

	protected double a;
	protected double b;
	protected double c;
	protected double p;
	protected double d;
	protected double e;
	protected double f;
	protected double q;
	protected double g;
	protected double h;
	protected double i;
	protected double r;
	protected double l;
	protected double m;
	protected double n;
	protected double s;

	/**
	 * Create a new transformation which translate the alignmentPoint to the relativePoint.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param alignmentPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param relativePoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Align_with_(Jun3dPoint alignmentPoint, Jun3dPoint relativePoint) {
		return Jun3dTransformation.Translate_(relativePoint.minus_(alignmentPoint));
	}

	/**
	 * Create a new Jun3dTransformation which aligns the specified points.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anArrayOfPointArray jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category Typical transformation
	 */
	public static Jun3dTransformation AlignPoints_(Jun3dPoint[][] anArrayOfPointArray) {
		int size = anArrayOfPointArray.length;
		double[] xEquations = new double[size * 5];
		double[] yEquations = new double[size * 5];
		double[] zEquations = new double[size * 5];
		int index = 0;
		for (int i = 0; i < size; i++) {
			double[] points = anArrayOfPointArray[i][0].asArray();
			xEquations[index] = yEquations[index] = zEquations[index] = points[0];
			index++;
			xEquations[index] = yEquations[index] = zEquations[index] = points[1];
			index++;
			xEquations[index] = yEquations[index] = zEquations[index] = points[2];
			index++;
			xEquations[index] = yEquations[index] = zEquations[index] = 1.0d;
			index++;
			Jun3dPoint aPoint = anArrayOfPointArray[i][1];
			xEquations[index] = -aPoint.x();
			yEquations[index] = -aPoint.y();
			zEquations[index] = -aPoint.z();
			index++;
		}

		double[] xSolution = (new JunLinearEquations(size, 5, xEquations)).solution();
		double[] ySolution = (new JunLinearEquations(size, 5, yEquations)).solution();
		double[] zSolution = (new JunLinearEquations(size, 5, zEquations)).solution();
		Jun3dTransformation aTransformation = new Jun3dTransformation();
		aTransformation.a = xSolution[0];
		aTransformation.b = ySolution[0];
		aTransformation.c = zSolution[0];
		aTransformation.d = xSolution[1];
		aTransformation.e = ySolution[1];
		aTransformation.f = zSolution[1];
		aTransformation.g = xSolution[2];
		aTransformation.h = ySolution[2];
		aTransformation.i = zSolution[2];
		aTransformation.l = xSolution[3];
		aTransformation.m = ySolution[3];
		aTransformation.n = zSolution[3];
		aTransformation.p = 0.0d;
		aTransformation.q = 0.0d;
		aTransformation.r = 0.0d;
		aTransformation.s = 1.0d;
		return aTransformation;
	}

	/**
	 * Create a new Jun3dTransformation which aligns two vectors.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Typical transformation
	 */
	public static Jun3dTransformation AlignVector_to_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2) {
		if (aJun3dPoint1.minus_(aJun3dPoint2).length() < aJun3dPoint2.ACCURACY) {
			return Jun3dTransformation.Unity();
		}

		Jun3dPoint zero = Jun3dPoint.Zero();
		Jun3dPoint axisVector = aJun3dPoint1.product_(aJun3dPoint2).unitVector();
		Jun3dPoint normalVector1 = aJun3dPoint1.product_(axisVector);
		Jun3dPoint normalVector2 = aJun3dPoint2.product_(axisVector);
		return Jun3dTransformation.AlignPoints_(new Jun3dPoint[][] { { aJun3dPoint1, aJun3dPoint2 }, { axisVector, axisVector }, { normalVector1, normalVector2 }, { zero, zero } });
	}

	/**
	 * Create a new Jun3dTransformation which aligns two vectors.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Typical transformation
	 */
	public static Jun3dTransformation AlignVector_withVector_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2) {
		Jun3dTransformation aTransformation = Jun3dTransformation.AlignVector_to_(aJun3dPoint1, aJun3dPoint2);
		return aTransformation;
	}

	/**
	 * Create a new instance of Jun3dTransformation and initialize it with an array.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anArray double[]
	 * @category Instance creation
	 */
	public static Jun3dTransformation FromArray_(double[] anArray) {
		Jun3dTransformation aTransformation = new Jun3dTransformation();
		aTransformation.a = anArray[0];
		aTransformation.b = anArray[1];
		aTransformation.c = anArray[2];
		aTransformation.p = anArray[3];
		aTransformation.d = anArray[4];
		aTransformation.e = anArray[5];
		aTransformation.f = anArray[6];
		aTransformation.q = anArray[7];
		aTransformation.g = anArray[8];
		aTransformation.h = anArray[9];
		aTransformation.i = anArray[10];
		aTransformation.r = anArray[11];
		aTransformation.l = anArray[12];
		aTransformation.m = anArray[13];
		aTransformation.n = anArray[14];
		aTransformation.s = anArray[15];
		return aTransformation;
	}

	/**
	 * Typical transformation - mirror x.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category Typical transformation
	 */
	public static Jun3dTransformation MirrorX() {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.a = -1.0d;
		return aTransformation;
	}

	/**
	 * Typical transformation - mirror y.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category Typical transformation
	 */
	public static Jun3dTransformation MirrorY() {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.e = -1.0d;
		return aTransformation;
	}

	/**
	 * Typical transformation - mirror x.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category Typical transformation
	 */
	public static Jun3dTransformation MirrorZ() {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.i = -1.0d;
		return aTransformation;
	}

	/**
	 * Typical transformation - Perspective.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Perspective_(Jun3dPoint aPoint) {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.a = -aPoint.z();
		aTransformation.e = -aPoint.z();
		aTransformation.g = aPoint.x();
		aTransformation.h = aPoint.y();
		aTransformation.i = -1;
		aTransformation.r = 1;
		aTransformation.s = -aPoint.z();
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the z axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Rotate_(JunAngle anAngle) {
		return RotateZ_(anAngle);
	}

	/**
	 * Typical transformation - Rotate around a line.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.curves.JunLine
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Rotate_around_(JunAngle anAngle, JunLine aLine) {
		JunAngle theAngle = (JunAngle) anAngle.copy();
		Jun3dPoint firstPoint = null;
		Jun3dPoint lastPoint = null;
		if (aLine instanceof Jun3dLine) {
			Jun3dLine theLine = (Jun3dLine) aLine;
			firstPoint = (Jun3dPoint) theLine.first().copy();
			lastPoint = (Jun3dPoint) theLine.last().copy();
		} else {
			Jun2dLine theLine = (Jun2dLine) aLine;
			firstPoint = Jun3dPoint.Coerce_(theLine.first());
			lastPoint = Jun3dPoint.Coerce_(theLine.last());
		}
		Jun3dPoint aVector = (Jun3dPoint) lastPoint.minus_(firstPoint);
		double vx = aVector.x();
		double vy = aVector.y();
		double vz = aVector.z();
		double aV = Math.sqrt((vy * vy) + (vz * vz));
		double aL = Math.sqrt((vx * vx) + (vy * vy) + (vz * vz));
		Jun3dTransformation aT = Translate_(firstPoint.negated());
		Jun3dTransformation aRx = Unity();
		if (aV != 0.0d) {
			aRx.e = vz / aV;
			aRx.f = vy / aV;
			aRx.h = -vy / aV;
			aRx.i = vz / aV;
		}
		Jun3dTransformation aRy = Unity();
		if (aL != 0.0d) {
			aRy.a = aV / aL;
			aRy.c = vx / aL;
			aRy.g = -vx / aL;
			aRy.i = aV / aL;
		}
		Jun3dTransformation aRz = Rotate_(theAngle);
		Jun3dTransformation rRy = Unity();
		if (aL != 0.0d) {
			rRy.a = aV / aL;
			rRy.c = -vx / aL;
			rRy.g = vx / aL;
			rRy.i = aV / aL;
		}
		Jun3dTransformation rRx = Unity();
		if (aV != 0.0d) {
			rRx.e = vz / aV;
			rRx.f = -vy / aV;
			rRx.h = vy / aV;
			rRx.i = vz / aV;
		}
		Jun3dTransformation rT = Translate_(firstPoint);
		Jun3dTransformation aTransformation = aT.product_(aRx);
		aTransformation = aTransformation.product_(aRy);
		aTransformation = aTransformation.product_(aRz);
		aTransformation = aTransformation.product_(rRy);
		aTransformation = aTransformation.product_(rRx);
		aTransformation = aTransformation.product_(rT);
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the x axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Typical transformation
	 */
	public static Jun3dTransformation RotateX_(JunAngle anAngle) {
		double cos = anAngle.cos();
		double sin = anAngle.sin();
		Jun3dTransformation aTransformation = Unity();
		aTransformation.e = cos;
		aTransformation.f = sin;
		aTransformation.h = -sin;
		aTransformation.i = cos;
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the x axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle double
	 * @category Typical transformation
	 */
	public static Jun3dTransformation RotateX_(double anAngle) {
		return Jun3dTransformation.RotateX_(JunAngle.FromRad_(anAngle));
	}

	/**
	 * Typical transformation - Rotate around the y axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Typical transformation
	 */
	public static Jun3dTransformation RotateY_(JunAngle anAngle) {
		double cos = anAngle.cos();
		double sin = anAngle.sin();
		Jun3dTransformation aTransformation = Unity();
		aTransformation.a = cos;
		aTransformation.c = -sin;
		aTransformation.g = sin;
		aTransformation.i = cos;
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the y axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle double
	 * @category Typical transformation
	 */
	public static Jun3dTransformation RotateY_(double anAngle) {
		return Jun3dTransformation.RotateY_(JunAngle.FromRad_(anAngle));
	}

	/**
	 * Typical transformation - Rotate around the z axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Typical transformation
	 */
	public static Jun3dTransformation RotateZ_(JunAngle anAngle) {
		double cos = anAngle.cos();
		double sin = anAngle.sin();
		Jun3dTransformation aTransformation = Unity();
		aTransformation.a = cos;
		aTransformation.b = sin;
		aTransformation.d = -sin;
		aTransformation.e = cos;
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the z axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param anAngle double
	 * @category Typical transformation
	 */
	public static Jun3dTransformation RotateZ_(double anAngle) {
		return Jun3dTransformation.RotateZ_(JunAngle.FromRad_(anAngle));
	}

	/**
	 * Create a new instance of Jun3dTransformation with the specified scale.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aNumber double
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Scale_(double aNumber) {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.a = aNumber;
		aTransformation.e = aNumber;
		aTransformation.i = aNumber;
		return aTransformation;
	}

	/**
	 * Create a new instance of JunTransformation with the specified scale.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Scale_(Jun3dPoint aPoint) {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.a = aPoint.x();
		aTransformation.e = aPoint.y();
		aTransformation.i = aPoint.z();
		return aTransformation;
	}

	/**
	 * Create a new instance of Jun3dTransformation with the specified
	 * translation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aNumber double
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Translate_(double aNumber) {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.l = aNumber;
		aTransformation.m = aNumber;
		aTransformation.n = aNumber;
		return aTransformation;
	}

	/**
	 * Create a new instance of Jun3dTransformation with the specified
	 * translation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Typical transformation
	 */
	public static Jun3dTransformation Translate_(Jun3dPoint aPoint) {
		Jun3dTransformation aTransformation = Unity();
		aTransformation.l = aPoint.x();
		aTransformation.m = aPoint.y();
		aTransformation.n = aPoint.z();
		return aTransformation;
	}

	/**
	 * Answer a new Jun3dTransformation with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category Constants access
	 */
	public static Jun3dTransformation Unity() {
		Jun3dTransformation aTransformation = new Jun3dTransformation();
		aTransformation.a = 1;
		aTransformation.b = 0;
		aTransformation.c = 0;
		aTransformation.p = 0;
		aTransformation.d = 0;
		aTransformation.e = 1;
		aTransformation.f = 0;
		aTransformation.q = 0;
		aTransformation.g = 0;
		aTransformation.h = 0;
		aTransformation.i = 1;
		aTransformation.r = 0;
		aTransformation.l = 0;
		aTransformation.m = 0;
		aTransformation.n = 0;
		aTransformation.s = 1;
		return aTransformation;
	}

	/**
	 * Answer a new Jun3dTransformation with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category Constants access
	 */
	public static Jun3dTransformation Zero() {
		Jun3dTransformation aTransformation = new Jun3dTransformation();
		aTransformation.a = 0;
		aTransformation.b = 0;
		aTransformation.c = 0;
		aTransformation.p = 0;
		aTransformation.d = 0;
		aTransformation.e = 0;
		aTransformation.f = 0;
		aTransformation.q = 0;
		aTransformation.g = 0;
		aTransformation.h = 0;
		aTransformation.i = 0;
		aTransformation.r = 0;
		aTransformation.l = 0;
		aTransformation.m = 0;
		aTransformation.n = 0;
		aTransformation.s = 0;
		return aTransformation;
	}

	/**
	 * Answer the receiver's a value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double a() {
		return a;
	}

	/**
	 * Answer the receiver's b value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double b() {
		return b;
	}

	/**
	 * Answer the receiver's c value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double c() {
		return c;
	}

	/**
	 * Answer the receiver's d value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double d() {
		return d;
	}

	/**
	 * Answer the receiver's e value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double e() {
		return e;
	}

	/**
	 * Answer the receiver's f value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double f() {
		return f;
	}

	/**
	 * Answer the receiver's g value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double g() {
		return g;
	}

	/**
	 * Answer the receiver's h value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double h() {
		return h;
	}

	/**
	 * Answer the receiver's i value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double i() {
		return i;
	}

	/**
	 * Answer the receiver's l value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double l() {
		return l;
	}

	/**
	 * Answer the receiver's m value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double m() {
		return m;
	}

	/**
	 * Answer the receiver's n value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double n() {
		return n;
	}

	/**
	 * Answer the receiver's p value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double p() {
		return p;
	}

	/**
	 * Answer the receiver's q value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double q() {
		return q;
	}

	/**
	 * Answer the receiver's r value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double r() {
		return r;
	}

	/**
	 * Answer the receiver's s value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double s() {
		return s;
	}

	/**
	 * Apply this transformation to aPoint and return the result.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category applying transformation
	 */
	public Jun3dPoint applyTo_(Jun3dPoint aPoint) {
		return aPoint.transform_(this);
	}

	/**
	 * Apply this transformation to anObject and return the result.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param anObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category applying transformation
	 */
	public JunOpenGL3dObject applyTo_(JunOpenGL3dObject anObject) {
		return anObject.transform_(this);
	}

	/**
	 * Apply this transformation to aPoint and return the result.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category applying transformation
	 */
	public Jun3dPoint applyToPoint_(JunPoint aPoint) {
		Jun3dPoint point = aPoint instanceof Jun3dPoint ? (Jun3dPoint) aPoint : Jun3dPoint.Coerce_(aPoint);
		double nx = (point.x() * a) + (point.y() * d) + (point.z() * g) + l;
		double ny = (point.x() * b) + (point.y() * e) + (point.z() * h) + m;
		double nz = (point.x() * c) + (point.y() * f) + (point.z() * i) + n;
		double scale = (point.x() * p) + (point.y() * q) + (point.z() * r) + s;
		if (scale == 0.0d) {
			scale = ACCURACY;
		}
		return new Jun3dPoint(nx / scale, ny / scale, nz / scale);
	}

	/**
	 * Convert this transformation as an array.
	 * 
	 * @return double[]
	 * @see jp.co.sra.jun.geometry.transformations.JunTransformation#asArray()
	 * @category converting
	 */
	public double[] asArray() {
		double[] array = new double[16];
		array[0] = a;
		array[1] = b;
		array[2] = c;
		array[3] = p;
		array[4] = d;
		array[5] = e;
		array[6] = f;
		array[7] = q;
		array[8] = g;
		array[9] = h;
		array[10] = i;
		array[11] = r;
		array[12] = l;
		array[13] = m;
		array[14] = n;
		array[15] = s;
		return array;
	}

	/**
	 * Convert this transformation as a matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @see jp.co.sra.jun.geometry.transformations.JunTransformation#asMatrix()
	 * @category converting
	 */
	public JunDoubleMatrix asMatrix() {
		return new JunDoubleMatrix(4, this.asArray());
	}

	/**
	 * Convert the receiver as a rotation matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category converting
	 */
	public JunDoubleMatrix asRotationMatrix() {
		double[] anArray = new double[3 * 3];
		anArray[0] = a;
		anArray[1] = b;
		anArray[2] = c;
		anArray[3] = d;
		anArray[4] = e;
		anArray[5] = f;
		anArray[6] = g;
		anArray[7] = h;
		anArray[8] = i;
		return new JunDoubleMatrix(3, anArray);
	}

	/**
	 * Convert the receiver as a translation point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category converting
	 */
	public Jun3dPoint asTranslation() {
		return new Jun3dPoint(l, m, n);
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dTransformation aTransformation = (Jun3dTransformation) anObject;
		return this.isEqualNumber_to_(a, aTransformation.a) && this.isEqualNumber_to_(b, aTransformation.b) && this.isEqualNumber_to_(c, aTransformation.c) && this.isEqualNumber_to_(p, aTransformation.p) && this.isEqualNumber_to_(d, aTransformation.d)
				&& this.isEqualNumber_to_(e, aTransformation.e) && this.isEqualNumber_to_(f, aTransformation.f) && this.isEqualNumber_to_(q, aTransformation.q) && this.isEqualNumber_to_(g, aTransformation.g) && this.isEqualNumber_to_(h, aTransformation.h)
				&& this.isEqualNumber_to_(i, aTransformation.i) && this.isEqualNumber_to_(r, aTransformation.r) && this.isEqualNumber_to_(l, aTransformation.l) && this.isEqualNumber_to_(m, aTransformation.m) && this.isEqualNumber_to_(n, aTransformation.n)
				&& this.isEqualNumber_to_(s, aTransformation.s);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun3dTransformation aTransformation = (Jun3dTransformation) anObject;
		return a == aTransformation.a && b == aTransformation.b && c == aTransformation.c && p == aTransformation.p && d == aTransformation.d && e == aTransformation.e && f == aTransformation.f && q == aTransformation.q && g == aTransformation.g
				&& h == aTransformation.h && i == aTransformation.i && r == aTransformation.r && l == aTransformation.l && m == aTransformation.m && n == aTransformation.n && s == aTransformation.s;
	}

	/**
	 * Answer the inverse of this transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming functions
	 */
	public Jun3dTransformation inverse() {
		return Jun3dTransformation.FromArray_(this.asMatrix().inverse().asArrayOfDouble());
	}

	/**
	 * Answer the inverse of this transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @see jp.co.sra.jun.geometry.transformations.JunTransformation#_inverse()
	 * @category transforming functions
	 */
	public JunTransformation _inverse() {
		return this.inverse();
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("3dTransformation (");
		aWriter.write(a + " " + b + " " + c + " " + p);
		aWriter.write(" ,  ");
		aWriter.write(d + " " + e + " " + f + " " + q);
		aWriter.write(" ,  ");
		aWriter.write(g + " " + h + " " + i + " " + r);
		aWriter.write(" ,  ");
		aWriter.write(l + " " + m + " " + n + " " + s);
		aWriter.write(')');
	}

	/**
	 * Answer a new transformation producted by another instance of
	 * Jun3dTransformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming functions
	 */
	public Jun3dTransformation product_(Jun3dTransformation aTransformation) {
		Jun3dTransformation newTransformation = new Jun3dTransformation();
		newTransformation.a = (a * aTransformation.a) + (b * aTransformation.d) + (c * aTransformation.g) + (p * aTransformation.l);
		newTransformation.b = (a * aTransformation.b) + (b * aTransformation.e) + (c * aTransformation.h) + (p * aTransformation.m);
		newTransformation.c = (a * aTransformation.c) + (b * aTransformation.f) + (c * aTransformation.i) + (p * aTransformation.n);
		newTransformation.p = (a * aTransformation.p) + (b * aTransformation.q) + (c * aTransformation.r) + (p * aTransformation.s);
		newTransformation.d = (d * aTransformation.a) + (e * aTransformation.d) + (f * aTransformation.g) + (q * aTransformation.l);
		newTransformation.e = (d * aTransformation.b) + (e * aTransformation.e) + (f * aTransformation.h) + (q * aTransformation.m);
		newTransformation.f = (d * aTransformation.c) + (e * aTransformation.f) + (f * aTransformation.i) + (q * aTransformation.n);
		newTransformation.q = (d * aTransformation.p) + (e * aTransformation.q) + (f * aTransformation.r) + (q * aTransformation.s);
		newTransformation.g = (g * aTransformation.a) + (h * aTransformation.d) + (i * aTransformation.g) + (r * aTransformation.l);
		newTransformation.h = (g * aTransformation.b) + (h * aTransformation.e) + (i * aTransformation.h) + (r * aTransformation.m);
		newTransformation.i = (g * aTransformation.c) + (h * aTransformation.f) + (i * aTransformation.i) + (r * aTransformation.n);
		newTransformation.r = (g * aTransformation.p) + (h * aTransformation.q) + (i * aTransformation.r) + (r * aTransformation.s);
		newTransformation.l = (l * aTransformation.a) + (m * aTransformation.d) + (n * aTransformation.g) + (s * aTransformation.l);
		newTransformation.m = (l * aTransformation.b) + (m * aTransformation.e) + (n * aTransformation.h) + (s * aTransformation.m);
		newTransformation.n = (l * aTransformation.c) + (m * aTransformation.f) + (n * aTransformation.i) + (s * aTransformation.n);
		newTransformation.s = (l * aTransformation.p) + (m * aTransformation.q) + (n * aTransformation.r) + (s * aTransformation.s);
		return newTransformation;
	}

	/**
	 * Answer the rotation angle of the transformation.
	 *
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category function
	 */
	public JunAngle rotationAngle() {
		return JunAngle.FromRad_(Math.acos((a + e + i - 1) / 2));
	}

	/**
	 * Answer the rotation vector of the transformation.
	 *
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category function
	 */
	public Jun3dPoint rotationVector() {
		double denominator = Math.sqrt((f - h) * (f - h) + (c - g) * (c - g) + (b - d) * (b - d));
		return new Jun3dPoint((h - f) / denominator, (c - g) / denominator, (d - b) / denominator);
	}

	/**
	 * Answer a new transformation transformd by another instance of
	 * Jun3dTransformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @param aJunTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public Jun3dTransformation transform_(Jun3dTransformation aJunTransformation) {
		return this.product_(aJunTransformation);
	}
}
