package jp.co.sra.jun.geometry.transformations;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.jun.collections.sequences.JunDoubleMatrix;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;

/**
 * Jun2dTransformation class
 * 
 *  @author    nisinaka
 *  @created   1998/12/01 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/09/29 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun610 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dTransformation.java,v 8.13 2008/02/20 06:31:11 nisinaka Exp $
 */
public class Jun2dTransformation extends JunTransformation {

	protected double a;
	protected double b;
	protected double p;
	protected double c;
	protected double d;
	protected double q;
	protected double m;
	protected double n;
	protected double s;

	/**
	 * Create a new instance of Jun2dTransformation and initialize it with an array.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2Transformation
	 * @param anArray double[]
	 * @category Instance creation
	 */
	public static Jun2dTransformation FromArray_(double[] anArray) {
		Jun2dTransformation aTransformation = new Jun2dTransformation();
		aTransformation.a = anArray[0];
		aTransformation.b = anArray[1];
		aTransformation.p = anArray[2];
		aTransformation.c = anArray[3];
		aTransformation.d = anArray[4];
		aTransformation.q = anArray[5];
		aTransformation.m = anArray[6];
		aTransformation.n = anArray[7];
		aTransformation.s = anArray[8];
		return aTransformation;
	}

	/**
	 * Typical transofrmation - Mirror the x axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category Typical transformation
	 */
	public static Jun2dTransformation MirrorX() {
		Jun2dTransformation aTransformation = Unity();
		aTransformation.a = -1.0d;
		return aTransformation;
	}

	/**
	 * Typical transofrmation - Mirror the y axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category Typical transformation
	 */
	public static Jun2dTransformation MirrorY() {
		Jun2dTransformation aTransformation = Unity();
		aTransformation.d = -1.0d;
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the z axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Typical transformation
	 */
	public static Jun2dTransformation Rotate_(JunAngle anAngle) {
		return RotateZ_(anAngle);
	}

	/**
	 * Typical transformation - Rotate around.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category Typical transformation
	 */
	public static Jun2dTransformation Rotate_around_(JunAngle anAngle, Jun2dPoint aPoint) {
		JunAngle theAngle = (JunAngle) anAngle.copy();
		Jun2dPoint thePoint = new Jun2dPoint(aPoint);
		Jun2dTransformation aT = Jun2dTransformation.Translate_(thePoint.negated());
		Jun2dTransformation aRz = Jun2dTransformation.Rotate_(theAngle);
		Jun2dTransformation rT = Jun2dTransformation.Translate_(thePoint);
		Jun2dTransformation aTransformation = aT.product_(aRz);
		aTransformation = aTransformation.product_(rT);
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the z axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category Typical transformation
	 */
	public static Jun2dTransformation RotateZ_(JunAngle anAngle) {
		double cos = anAngle.cos();
		double sin = anAngle.sin();
		Jun2dTransformation aTransformation = Unity();
		aTransformation.a = cos;
		aTransformation.b = sin;
		aTransformation.c = -sin;
		aTransformation.d = cos;
		return aTransformation;
	}

	/**
	 * Typical transformation - Rotate around the z axis.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param anAngle double
	 * @category Typical transformation
	 */
	public static Jun2dTransformation RotateZ_(double anAngle) {
		return Jun2dTransformation.RotateZ_(JunAngle.FromRad_(anAngle));
	}

	/**
	 * Create a new instance of JunTransformation with the specified scale.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param aNumber double
	 * @category Typical transformation
	 */
	public static Jun2dTransformation Scale_(double aNumber) {
		Jun2dTransformation aTransformation = Unity();
		aTransformation.a = aNumber;
		aTransformation.d = aNumber;
		return aTransformation;
	}

	/**
	 * Create a new instance of Jun2dTransformation with the specified scale.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Typical transformation
	 */
	public static Jun2dTransformation Scale_(Jun2dPoint aPoint) {
		Jun2dTransformation aTransformation = Unity();
		aTransformation.a = aPoint.x();
		aTransformation.d = aPoint.y();
		return aTransformation;
	}

	/**
	 * Create a new instance of Jun2dTransformation with the specified translation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param aNumber double
	 * @category Typical transformation
	 */
	public static Jun2dTransformation Translate_(double aNumber) {
		Jun2dTransformation aTransformation = Unity();
		aTransformation.m = aNumber;
		aTransformation.n = aNumber;
		return aTransformation;
	}

	/**
	 * Create a new instance of Jun2dTransformation with the specified
	 * translation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Typical transformation
	 */
	public static Jun2dTransformation Translate_(Jun2dPoint aPoint) {
		Jun2dTransformation aTransformation = Unity();
		aTransformation.m = aPoint.x();
		aTransformation.n = aPoint.y();
		return aTransformation;
	}

	/**
	 * Answer a new Jun2dTransformation with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category Constants access
	 */
	public static Jun2dTransformation Unity() {
		Jun2dTransformation aTransformation = new Jun2dTransformation();
		aTransformation.a = 1;
		aTransformation.b = 0;
		aTransformation.p = 0;
		aTransformation.c = 0;
		aTransformation.d = 1;
		aTransformation.q = 0;
		aTransformation.m = 0;
		aTransformation.n = 0;
		aTransformation.s = 1;
		return aTransformation;
	}

	/**
	 * Answer a new Jun2dTransformation with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category Constants access
	 */
	public static Jun2dTransformation Zero() {
		Jun2dTransformation aTransformation = new Jun2dTransformation();
		aTransformation.a = 0;
		aTransformation.b = 0;
		aTransformation.p = 0;
		aTransformation.c = 0;
		aTransformation.d = 0;
		aTransformation.q = 0;
		aTransformation.m = 0;
		aTransformation.n = 0;
		aTransformation.s = 0;
		return aTransformation;
	}

	/**
	 * Answer the receiver's a value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double a() {
		return a;
	}

	/**
	 * Answer the receiver's b value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double b() {
		return b;
	}

	/**
	 * Answer the receiver's c value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double c() {
		return c;
	}

	/**
	 * Answer the receiver's d value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double d() {
		return d;
	}

	/**
	 * Answer the receiver's m value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double m() {
		return m;
	}

	/**
	 * Answer the receiver's n value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double n() {
		return n;
	}

	/**
	 * Answer the receiver's p value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double p() {
		return p;
	}

	/**
	 * Answer the receiver's q value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double q() {
		return q;
	}

	/**
	 * Answer the receiver's s value.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double s() {
		return s;
	}

	/**
	 * Apply this transformation to the Jun2dPoint and return the result.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category applying transformation
	 */
	public Jun2dPoint applyToPoint_(JunPoint aPoint) {
		double x = aPoint.x();
		double y = aPoint.y();
		double nx = (x * a) + (y * c) + m;
		double ny = (x * b) + (y * d) + n;
		double scale = (x * p) + (y * q) + s;
		if (scale == 0) {
			scale = ACCURACY;
		}
		return new Jun2dPoint(nx / scale, ny / scale);
	}

	/**
	 * Convert this transformation as an array.
	 * 
	 * @return double[]
	 * @see jp.co.sra.jun.geometry.transformations.JunTransformation#asArray()
	 * @category converting
	 */
	public double[] asArray() {
		double[] array = new double[9];
		array[0] = a;
		array[1] = b;
		array[2] = p;
		array[3] = c;
		array[4] = d;
		array[5] = q;
		array[6] = m;
		array[7] = n;
		array[8] = s;
		return array;
	}

	/**
	 * Convert this transformation as a matrix.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @see jp.co.sra.jun.geometry.transformations.JunTransformation#asMatrix()
	 * @category converting
	 */
	public JunDoubleMatrix asMatrix() {
		return new JunDoubleMatrix(3, this.asArray());
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dTransformation aTransformation = (Jun2dTransformation) anObject;
		return this.isEqualNumber_to_(a, aTransformation.a) && this.isEqualNumber_to_(b, aTransformation.b) && this.isEqualNumber_to_(p, aTransformation.p) && this.isEqualNumber_to_(c, aTransformation.c) && this.isEqualNumber_to_(d, aTransformation.d)
				&& this.isEqualNumber_to_(q, aTransformation.q) && this.isEqualNumber_to_(m, aTransformation.m) && this.isEqualNumber_to_(n, aTransformation.n) && this.isEqualNumber_to_(s, aTransformation.s);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dTransformation aTransformation = (Jun2dTransformation) anObject;
		return a == aTransformation.a && b == aTransformation.b && p == aTransformation.p && c == aTransformation.c && d == aTransformation.d && q == aTransformation.q && m == aTransformation.m && n == aTransformation.n && s == aTransformation.s;
	}

	/**
	 * Answer the inverse of this transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming functions
	 */
	public Jun2dTransformation inverse() {
		return Jun2dTransformation.FromArray_(this.asMatrix().inverse().asArrayOfDouble());
	}

	/**
	 * Answer the inverse of this transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @see jp.co.sra.jun.geometry.transformations.JunTransformation#_inverse()
	 * @category transforming functions
	 */
	public JunTransformation _inverse() {
		return this.inverse();
	}

	/**
	 * Answer true if the receiver is a 2d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("2dTransformation (");
		aWriter.write(a + " " + b + " " + p);
		aWriter.write(" ,  ");
		aWriter.write(c + " " + d + " " + q);
		aWriter.write(" ,  ");
		aWriter.write(m + " " + n + " " + s);
		aWriter.write(')');
	}

	/**
	 * Answer a new transformation producted by another instance of
	 * Jun2dTransformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming functions
	 */
	public Jun2dTransformation product_(Jun2dTransformation aTransformation) {
		Jun2dTransformation newTransformation = new Jun2dTransformation();
		newTransformation.a = (a * aTransformation.a) + (b * aTransformation.c) + (p * aTransformation.m);
		newTransformation.b = (a * aTransformation.b) + (b * aTransformation.d) + (p * aTransformation.n);
		newTransformation.p = (a * aTransformation.p) + (b * aTransformation.q) + (p * aTransformation.s);
		newTransformation.c = (c * aTransformation.a) + (d * aTransformation.c) + (q * aTransformation.m);
		newTransformation.d = (c * aTransformation.b) + (d * aTransformation.d) + (q * aTransformation.n);
		newTransformation.q = (c * aTransformation.p) + (d * aTransformation.q) + (q * aTransformation.s);
		newTransformation.m = (m * aTransformation.a) + (n * aTransformation.c) + (s * aTransformation.m);
		newTransformation.n = (m * aTransformation.b) + (n * aTransformation.d) + (s * aTransformation.n);
		newTransformation.s = (m * aTransformation.p) + (n * aTransformation.q) + (s * aTransformation.s);
		return newTransformation;
	}

	/**
	 * Answer a new transformation transformd by another instance of
	 * Jun2dTransformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @param aJunTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dTransformation transform_(Jun2dTransformation aJunTransformation) {
		return this.product_(aJunTransformation);
	}

}
