package jp.co.sra.jun.geometry.surfaces;

import java.io.IOException;
import java.io.Writer;
import java.util.Arrays;

import jp.co.sra.jun.geometry.abstracts.JunSurface;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.curves.JunLine;

/**
 * JunTriangle class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/14 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun664 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTriangle.java,v 8.15 2008/02/20 06:30:59 nisinaka Exp $
 */
public abstract class JunTriangle extends JunSurface {
	/**
	 * Answer my p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	protected abstract JunPoint _p1();

	/**
	 * Answer my p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	protected abstract JunPoint _p2();

	/**
	 * Answer my p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	protected abstract JunPoint _p3();

	/**
	 * Convert the receiver as an array of JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		return new JunPlane[] { this.asPlane() };
	}

	/**
	 * Convert to a JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category converting
	 */
	public abstract JunPlane asPlane();

	/**
	 * Answer the angle with the specified line.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.curves.JunLine
	 * @category functions
	 */
	public JunAngle angleWithLine_(JunLine aLine) {
		return this.asPlane().angleWithLine_(aLine);
	}

	/**
	 * Answer the angle with the specified plane.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public JunAngle angleWithLine_(JunPlane aPlane) {
		return this.asPlane().angleWithPlane_(aPlane);
	}

	/**
	 * Answer the angle with the specified triangle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.surfaces.JunTriangle
	 * @category functions
	 */
	public JunAngle angleWithTriangle_(JunTriangle aTriangle) {
		return this.asPlane().angleWithTriangle_(aTriangle);
	}

	/**
	 * Answer the receiver's area with sign.
	 * 
	 * @return double
	 * @category functions
	 */
	public abstract double areaWithSign();

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException if failed.
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		this._p1().printOn_(aWriter);
		aWriter.write(" triangle: ");
		this._p2().printOn_(aWriter);
		aWriter.write(" and: ");
		this._p3().printOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" on: ");
		this._p1().storeOn_(aWriter);
		aWriter.write(" on: ");
		this._p2().storeOn_(aWriter);
		aWriter.write(" on: ");
		this._p3().storeOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Answer true if the receiver has an area, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasArea() {
		return (this._p1().equal_(this._p2()) || this._p2().equal_(this._p3()) || this._p3().equal_(this._p1())) == false;
	}

	/**
	 * Answer true if the receiver is a triangle, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isTriangle()
	 * @category testing
	 */
	public boolean isTriangle() {
		return true;
	}

	/**
	 * Answer the value which side of on a plane.
	 * 
	 * @param bisector jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return int
	 * @category testing
	 */
	public int whichSideOf_(JunPlane bisector) {
		JunPoint[] points = new JunPoint[] { this._p1(), this._p2(), this._p3() };
		int[] anArray = new int[points.length];
		for (int i = 0; i < points.length; i++) {
			anArray[i] = points[i].whichSideOf_(bisector);
		}
		boolean hasPlus = Arrays.binarySearch(anArray, 1) != -1;
		boolean hasMinus = Arrays.binarySearch(anArray, -1) != -1;
		if (hasPlus && hasMinus) {
			return 0;
		}
		boolean hasZero = Arrays.binarySearch(anArray, 0) != -1;
		if (hasZero) {
			return 0;
		}
		if (hasMinus && hasPlus == false) {
			return -1;
		}
		return 1;
	}
}
