package jp.co.sra.jun.geometry.surfaces;

import java.awt.Color;
import java.awt.Point;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StColorValue;

import jp.co.sra.jun.geometry.abstracts.JunSurface;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.curves.JunLine;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;

/**
 * JunPlane class
 * 
 *  @author    nisinaka
 *  @created   1998/11/09 (by nisinaka)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2004/09/29 (by m-asada)
 *  @updated   2004/10/20 (by Mitsuhiro Asada)
 *  @updated   2004/12/02 (by Mitsuhiro Asada)
 *  @updated   2007/04/26 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun678 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPlane.java,v 8.20 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunPlane extends JunSurface {

	/** The parameters. */
	protected double a;

	/** The parameters. */
	protected double b;

	/** The parameters. */
	protected double c;

	/** The parameters. */
	protected double d;

	/** Three points to determine the plane. */
	protected Jun3dPoint p1;

	/** Three points to determine the plane. */
	protected Jun3dPoint p2;

	/** Three points to determine the plane. */
	protected Jun3dPoint p3;

	/**
	 * Answer a new <code>JunPlane</code> with xy plane.
	 * 
	 * @return jp.co.sra.jun.geometry.surface.JunPlane
	 * @category Constants
	 */
	public static final JunPlane Xy() {
		return new JunPlane(new Jun3dPoint(0, 0, 0), new Jun3dPoint(1, 0, 0), new Jun3dPoint(0, 1, 0));
	}

	/**
	 * Answer a new <code>JunPlane</code> with yz plane.
	 * 
	 * @return jp.co.sra.jun.geometry.surface.JunPlane
	 * @category Constants
	 */
	public static final JunPlane Yz() {
		return new JunPlane(new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 1, 0), new Jun3dPoint(0, 0, 1));
	}

	/**
	 * Answer a new <code>JunPlane</code> with zx plane.
	 * 
	 * @return jp.co.sra.jun.geometry.surface.JunPlane
	 * @category Constants
	 */
	public static final JunPlane Zx() {
		return new JunPlane(new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 1), new Jun3dPoint(1, 0, 0));
	}

	/**
	 * Create a new instance of JunPlane with three JunPoints. This
	 * corresponds to the instance creation method "on:on:on:".
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public JunPlane(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3) {
		this._initialize(aPoint1, aPoint2, aPoint3);
	}

	/**
	 * Create a new instance of JunPlane with four double values. This
	 * corresponds to the instance creation method "a:b:c:d:".
	 * 
	 * @param aNumber1 double  the parameter a
	 * @param aNumber2 double  the parameter b
	 * @param aNumber3 double  the parameter c
	 * @param aNumber4 double  the parameter d
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category Instance creation
	 */
	public JunPlane(double aNumber1, double aNumber2, double aNumber3, double aNumber4) {
		super();
		this.setA_(aNumber1);
		this.setB_(aNumber2);
		this.setC_(aNumber3);
		this.setD_(aNumber4);
		Jun3dPoint normalVector = new Jun3dPoint(aNumber1, aNumber2, aNumber3);
		if (normalVector.length() < this.Accuracy()) {
			throw new SmalltalkException("Invalid parameters");
		}
		Jun3dPoint pa = null;
		Jun3dPoint pb = null;
		Jun3dPoint pc = null;
		if (Math.abs(this.a()) > this.Accuracy()) {
			pa = new Jun3dPoint(0 - this.d() / this.a(), 0, 0);
		}
		if (Math.abs(this.b()) > this.Accuracy()) {
			pb = new Jun3dPoint(0, 0 - this.d() / this.b(), 0);
		}
		if (Math.abs(this.c()) > this.Accuracy()) {
			pc = new Jun3dPoint(0, 0, 0 - this.d() / this.c());
		}
		Jun3dPoint p1 = null;
		Jun3dPoint p2 = null;
		Jun3dPoint p3 = null;
		if (pa != null) {
			p1 = pa;
			if (pb != null) {
				p2 = pb;
			} else {
				if (pc != null) {
					p2 = pc;
				} else {
					p2 = new Jun3dPoint(pa.x(), 0.0d, 1.0d);
				}
			}
		} else {
			if (pb != null) {
				p1 = pb;
				if (pc != null) {
					p2 = pc;
				} else {
					p2 = new Jun3dPoint(1.0d, pb.y(), 0.0d);
				}
			} else {
				if (pc != null) {
					p1 = pc;
					p2 = new Jun3dPoint(1.0d, 0.0d, pc.z());
				} else {
					throw new SmalltalkException("Invalid parameters");
				}
			}
		}
		p2 = p1.plus_(p2.minus_(p1).unitVector());
		p3 = (p1.minus_(p2)).vectorProduct_(normalVector.unitVector());
		this.setP1_(p1);
		this.setP2_(p2);
		this.setP3_(p3);
	}

	/**
	 * The default constructor.
	 * 
	 * @category Instance creation
	 */
	protected JunPlane() {
	}

	/**
	 * Create a new instance of the JunPlane with two points.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category Instance creation
	 */
	public static final JunPlane Between_and_(JunPoint aPoint1, JunPoint aPoint2) {
		Jun3dPoint a3dPoint1 = aPoint1.as3dPoint();
		Jun3dPoint a3dPoint2 = aPoint2.as3dPoint();
		Jun3dPoint centerPoint = a3dPoint1.center_(a3dPoint2);
		return JunPlane.On_vertical_(centerPoint, a3dPoint1.to_(a3dPoint2));
	}

	/**
	 * Create a new instance of the JunPlane with a point and a line.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aLine jp.co.sra.jun.geometry.curves.JunLine
	 * @category Instance creation
	 */
	public static final JunPlane On_vertical_(JunPoint aPoint, JunLine aLine) {
		Jun3dPoint thePoint = Jun3dPoint.Coerce_(aPoint);
		Jun3dLine theLine = null;
		if (aLine instanceof Jun2dLine) {
			theLine = new Jun3dLine(((Jun2dLine) aLine).from(), ((Jun2dLine) aLine).to());
		} else {
			theLine = (Jun3dLine) aLine;
		}

		JunPlane aPlane = new JunPlane();
		aPlane.setP1_(thePoint);
		aPlane.setA_(theLine.f());
		aPlane.setB_(theLine.g());
		aPlane.setC_(theLine.h());
		aPlane.setD_(-(theLine.f() * thePoint.x() + theLine.g() * thePoint.y() + theLine.h() * thePoint.z()));
		if (Math.pow(aPlane.a(), 2) + Math.pow(aPlane.b(), 2) + Math.pow(aPlane.c(), 2) < Accuracy()) {
			throw SmalltalkException.Error("can not define a plane");
		}
		Jun3dPoint p1 = thePoint;

		Jun3dLine firstLine = theLine;
		Jun3dLine[] threeAxes = new Jun3dLine[] { new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(1, 0, 0)), new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(0, 1, 0)), new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(0, 0, 1)) };
		Jun3dLine firstAxis = null;
		for (int i = 0; i < threeAxes.length; i++) {
			if (firstLine.isParallelWithLine_(threeAxes[i]) == false) {
				firstAxis = threeAxes[i];
				break;
			}
		}
		if (firstAxis == null) {
			throw SmalltalkException.Error("notFoundError");
		}
		Jun3dTransformation aTransformation = Jun3dTransformation.Translate_(firstAxis.to());
		thePoint = null;
		while ((thePoint == null) || thePoint.equal_(p1)) {
			firstLine = firstLine.transform_(aTransformation);
			thePoint = aPlane.intersectingPointWithLine_(firstLine);
		}
		Jun3dPoint p2 = thePoint;

		Jun3dLine secondLine = firstLine;
		threeAxes = new Jun3dLine[] { new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(0, 1, 0)), new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(0, 0, 1)), new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(1, 0, 0)) };
		Jun3dLine secondAxis = null;
		for (int i = 0; i < threeAxes.length; i++) {
			if (secondLine.isParallelWithLine_(threeAxes[i]) == false && threeAxes[i].to() != firstAxis.to()) {
				secondAxis = threeAxes[i];
				break;
			}
		}
		if (secondAxis == null) {
			throw SmalltalkException.Error("notFoundError");
		}
		aTransformation = Jun3dTransformation.Translate_(secondAxis.to());
		thePoint = null;
		while (thePoint == null || thePoint.equal_(p2) || thePoint.equal_(p1)) {
			secondLine = secondLine.transform_(aTransformation);
			thePoint = aPlane.intersectingPointWithLine_(secondLine);
		}
		Jun3dPoint p3 = thePoint;

		if ((p1.to_(p2)).isParallelWithLine_(p1.to_(p3))) {
			Jun3dLine thirdLine = firstLine;
			threeAxes = new Jun3dLine[] { new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(0, 0, 1)), new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(1, 0, 0)), new Jun3dPoint(0, 0, 0).to_(new Jun3dPoint(0, 1, 0)) };
			Jun3dLine thirdAxis = null;
			for (int i = 0; i < threeAxes.length; i++) {
				if (threeAxes[i].to() != firstAxis.to() && threeAxes[i].to() != secondAxis.to()) {
					thirdAxis = threeAxes[i];
					break;
				}
			}
			if (thirdAxis == null) {
				throw SmalltalkException.Error("notFoundError");
			}
			aTransformation = Jun3dTransformation.Translate_(thirdAxis.to());
			thePoint = null;
			while (thePoint == null || thePoint.equal_(p2)) {
				thirdLine = (Jun3dLine) thirdLine.transform_(aTransformation);
				thePoint = aPlane.intersectingPointWithLine_(thirdLine);
			}
			p3 = thePoint;
		}

		try {
			Jun3dPoint normalUnitVector = aPlane.normalUnitVector();
			Jun3dPoint anotherUnitVector = new JunPlane(p1, p2, p3).normalUnitVector();
			if (normalUnitVector.equal_(anotherUnitVector)) {
				aPlane.setP2_(p2);
				aPlane.setP3_(p3);
			} else {
				aPlane.setP2_(p3);
				aPlane.setP3_(p2);
			}
		} catch (Exception e) {
			aPlane.setP2_(p2);
			aPlane.setP3_(p3);
		}

		return aPlane;
	}

	/**
	 * Create a new instance of the JunPlane with a point and a normal vector.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aVector jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category Instance creation
	 */
	public static final JunPlane On_normalVector_(JunPoint aPoint, JunPoint aVector) {
		return JunPlane.On_vertical_(aPoint, new Jun3dLine(aPoint, aPoint.plus_(aVector)));
	}

	/**
	 * Initialize the receiver's parameters with three JunPoints.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.JunPoint
	 * @category initialize-release
	 */
	protected final void _initialize(JunPoint aPoint1, JunPoint aPoint2, JunPoint aPoint3) {
		Jun3dPoint thePoint1 = Jun3dPoint.Coerce_(aPoint1);
		Jun3dPoint thePoint2 = Jun3dPoint.Coerce_(aPoint2);
		Jun3dPoint thePoint3 = Jun3dPoint.Coerce_(aPoint3);
		this.p1 = thePoint1;
		this.p2 = thePoint2;
		this.p3 = thePoint3;

		thePoint2 = thePoint2.minus_(thePoint1);
		thePoint3 = thePoint3.minus_(thePoint1);
		this.a = (thePoint2.y() * thePoint3.z()) - (thePoint2.z() * thePoint3.y());
		this.b = (thePoint2.z() * thePoint3.x()) - (thePoint2.x() * thePoint3.z());
		this.c = (thePoint2.x() * thePoint3.y()) - (thePoint2.y() * thePoint3.x());
		this.d = -((thePoint1.x() * this.a) + (thePoint1.y() * this.b) + thePoint1.z() * this.c);
	}

	/**
	 * Answer a regularized point of this plane at a specified number.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param s double
	 * @param t double
	 * @category accessing
	 */
	public Jun3dPoint atS_andT_(double s, double t) {
		return new Jun3dPoint(this.x0() + this.f1() * s + this.f2() * t, this.y0() + this.g1() * s + this.g2() * t, this.z0() + this.h1() * s + this.h2() * t);
	}

	/**
	 * Answer the first point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint first() {
		return this.p1();
	}

	/**
	 * Answer the last point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint last() {
		return this.p3();
	}

	/**
	 * Answer the middle point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint middle() {
		return this.p2();
	}

	/**
	 * Answer the p1 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p1() {
		return p1;
	}

	/**
	 * Answer the p2 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p2() {
		return p2;
	}

	/**
	 * Answer the p3 point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint p3() {
		return p3;
	}

	/**
	 * Answer the second point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint second() {
		return this.p2();
	}

	/**
	 * Answer the third point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint third() {
		return this.p3();
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		JunPlane aPlane = (JunPlane) anObject;
		return this.isEqualNumber_to_(a, aPlane.a) && this.isEqualNumber_to_(b, aPlane.b) && this.isEqualNumber_to_(c, aPlane.c) && this.isEqualNumber_to_(d, aPlane.d);
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		JunPlane aPlane = (JunPlane) anObject;
		return a == aPlane.a && b == aPlane.b && c == aPlane.c && d == aPlane.d;
	}

	/**
	 * Convert the receiver as an array of Jun3dTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf3dTriangles()
	 * @category converting 
	 */
	public Jun3dTriangle[] asArrayOf3dTriangles() {
		return this.asArrayOfTriangles();
	}

	/**
	 * Convert the receiver as an array of JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		return new JunPlane[] { this.asPlane() };
	}

	/**
	 * Convert to an array of a Jun3dTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category converting
	 */
	public Jun3dTriangle[] asArrayOfTriangles() {
		return new Jun3dTriangle[] { this.asTriangle() };
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[][]
	 * @category converting
	 */
	public Jun3dPoint[][] asArrays() {
		return new Jun3dPoint[][] { new Jun3dPoint[] { this.p1(), this.p2(), this.p3(), this.p1() } };
	}

	/**
	 * Convert to Jun3dCircle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dCircle
	 * @category converting
	 */
	public Jun3dCircle asCircle() {
		return this.asTriangle().asCircle();
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		Jun3dPoint[] points = this.asCircle().trackPointsBy_(36);
		JunOpenGL3dPolygon aPolygon = new JunOpenGL3dPolygon(points);
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunOpenGL3dPolygon reversedPolygon = (JunOpenGL3dPolygon) aPolygon.reversed();
		reversedPolygon.paint_alpha_(StColorValue.Blend(this.defaultColor(), Color.black), this.defaultAlpha());
		compoundObject.add_(reversedPolygon);
		aPolygon.paint_alpha_(this.defaultColor(), this.defaultAlpha());
		compoundObject.add_(aPolygon);
		return compoundObject;
	}

	/**
	 * Convert to a JunOpenGL3dObject with points.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObjectWithPoints()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectWithPoints() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(this.asJunOpenGL3dObject());
		JunOpenGL3dObject p1Object = this.p1().asJunOpenGL3dObject();
		p1Object.paint_(Color.red);
		compoundObject.add_(p1Object);
		JunOpenGL3dObject p2Object = this.p1().asJunOpenGL3dObject();
		p2Object.paint_(Color.green);
		compoundObject.add_(p2Object);
		JunOpenGL3dObject p3Object = this.p1().asJunOpenGL3dObject();
		p3Object.paint_(Color.blue);
		compoundObject.add_(p3Object);
		return this.asJunOpenGL3dObject();
	}

	/**
	 * Convert to a JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category converting
	 */
	public JunPlane asPlane() {
		return this;
	}

	/**
	 * Convert the receiver as an arrays of <code>Point</code>.
	 * 
	 * @return java.awt.Point[][]
	 * @category converting
	 */
	public Point[][] asPointArrays() {
		Jun3dPoint[][] a3dPoints = this.asArrays();
		Point[][] points = new Point[a3dPoints.length][];
		for (int i = 0; i < a3dPoints.length; i++) {
			points[i] = new Point[a3dPoints[i].length];
			for (int j = 0; j < a3dPoints[i].length; j++) {
				points[i][j] = a3dPoints[i][j].asPoint();
			}
		}
		return points;
	}

	/**
	 * Convert to Jun3dTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle
	 * @category converting
	 */
	public Jun3dTriangle asTriangle() {
		return new Jun3dTriangle(this.p1(), this.p2(), this.p3());
	}

	/**
	 * Answer the angle with the specified line.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.curves.JunLine
	 * @throws java.lang.IllegalArgumentException
	 * @category functions
	 */
	public JunAngle angleWithLine_(JunLine aLine) {
		Jun3dLine theLine = null;
		if (aLine instanceof Jun3dLine) {
			theLine = (Jun3dLine) aLine;
		} else {
			Jun2dLine a2dLine = (Jun2dLine) aLine;
			theLine = new Jun3dLine(a2dLine.first(), a2dLine.last());
		}
		double f = theLine.f();
		double g = theLine.g();
		double h = theLine.h();
		double denominator = Math.sqrt(((a * a) + (b * b) + c * c) * ((f * f) + (g * g) + h * h));

		if (denominator < ACCURACY) {
			throw new IllegalArgumentException("unexpected line or plane parameters");
		}

		double numerator = (a * f) + (b * g) + (c * h);
		double gamma = Math.min(Math.max(numerator / denominator, -1), 1);

		return JunAngle.FromRad_((Math.PI / 2.0d) - Math.acos(gamma));
	}

	/**
	 * Answer the angle with the specified plane.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @throws java.lang.IllegalArgumentException
	 * @category functions
	 */
	public JunAngle angleWithPlane_(JunPlane aPlane) {
		double denominator = (Math.sqrt(this.a()) + Math.sqrt(this.b()) + Math.sqrt(this.c())) * (Math.sqrt(aPlane.a()) + Math.sqrt(aPlane.b()) + Math.sqrt(aPlane.c()));
		if (denominator < this.accuracy()) {
			throw new IllegalArgumentException("unexpected plane parameters");
		}
		double numerator = this.a() * aPlane.a() + this.b() * aPlane.b() + this.c() * aPlane.c();
		double gamma = Math.min(Math.max(Math.sqrt(numerator / denominator), -1.0d), 1.0d);
		return JunAngle.FromRad_(Math.acos(gamma));
	}

	/**
	 * Answer the angle with the specified triangle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.surfaces.JunTriangle
	 * @category functions
	 */
	public JunAngle angleWithTriangle_(JunTriangle aTriangle) {
		return this.angleWithPlane_(aTriangle.asPlane());
	}

	/**
	 * Answer the distance from the Point.
	 * 
	 * @return double
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#distanceFromPoint_(jp.co.sra.jun.geometry.basic.Jun3dPoint)
	 * @throws java.lang.IllegalArgumentException
	 * @category functions
	 */
	public double distanceFromPoint_(Jun3dPoint aPoint) {
		double denominator = (a * a) + (b * b) + (c * c);
		if ((denominator - ACCURACY) < 0) {
			throw new IllegalArgumentException("unexpect plane parameters");
		}
		double numerator = (a * aPoint.x()) + (b * aPoint.y()) + (c * aPoint.z()) + d;
		numerator = numerator * numerator;
		return Math.sqrt(numerator / denominator);
	}

	/**
	 * Answer the intersecting line with the plane.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public Jun3dLine intersectingLineWithPlane_(JunPlane aPlane) {
		double f = this.b() * aPlane.c() - (aPlane.b() * this.c());
		double g = this.c() * aPlane.a() - (aPlane.c() * this.a());
		double h = this.a() * aPlane.b() - (aPlane.a() * this.b());
		double denominator = Math.pow(f, 2) + Math.pow(g, 2) + Math.pow(h, 2);
		if (denominator < this.Accuracy()) {
			// parallel
			return null;
		}
		double dc = this.d() * aPlane.c() - (this.c() * aPlane.d());
		double db = this.d() * aPlane.b() - (this.b() * aPlane.d());
		double ad = this.a() * aPlane.d() - (this.d() * aPlane.a());
		double x0 = (g * dc - h * db) / denominator;
		double y0 = (0.0d - (f * dc + h * ad)) / denominator;
		double z0 = (f * db + g * ad) / denominator;
		return new Jun3dLine(new Jun3dPoint(x0, y0, z0), new Jun3dPoint(x0 + f, y0 + g, z0 + h));
	}

	/**
	 * Answer the Point which intersects with the Line.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLine_(Jun3dLine aLine) {
		double f = aLine.f();
		double g = aLine.g();
		double h = aLine.h();
		double denominator = (a * f) + (b * g) + (c * h);
		if (Math.abs(denominator) < aLine.ACCURACY) {
			// The receiver parallels the Line.
			return null;
		}
		double x0 = aLine.x0();
		double y0 = aLine.y0();
		double z0 = aLine.z0();
		double numerator = -((a * x0) + (b * y0) + (c * z0) + d);
		Jun3dPoint thePoint = new Jun3dPoint(x0 + ((f * numerator) / denominator), y0 + ((g * numerator) / denominator), z0 + ((h * numerator) / denominator));
		return thePoint;
	}

	/**
	 * Answer the intersecting point with the line segment.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithLineSegment_(Jun3dLine aLine) {
		Jun3dPoint thePoint = this.intersectingPointWithLine_(aLine);
		if ((thePoint == null) || !aLine.lineSegmentContainsPoint_(thePoint)) {
			return null;
		}
		return thePoint;
	}

	/**
	 * Answer the intersecting point with two planes.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPlane1 jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @param aPlane2 jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public Jun3dPoint intersectingPointWithPlane_wihtPlane_(JunPlane aPlane1, JunPlane aPlane2) {
		double bc = aPlane1.b() * aPlane2.c() - aPlane2.b() * aPlane1.c();
		double ac = aPlane1.a() * aPlane2.c() - aPlane2.a() * aPlane1.c();
		double ab = aPlane1.a() * aPlane2.b() - aPlane2.a() * aPlane1.b();
		double denominator = this.a() * bc - this.b() * ac + this.c() * ab;
		if (Math.abs(denominator) < this.Accuracy()) {
			// parallel two planes
			return null;
		}
		double dc = aPlane1.d() * aPlane2.c() - aPlane2.d() * aPlane1.c();
		double db = aPlane1.d() * aPlane2.b() - aPlane2.d() * aPlane1.b();
		double ad = aPlane1.a() * aPlane2.d() - aPlane2.a() * aPlane1.d();
		return new Jun3dPoint((this.b() * dc - this.d() * bc - this.c() * db) / denominator, (this.d() * ac - this.a() * dc - this.c() * ad) / denominator, (this.b() * ad + this.a() * db - this.d() * ab) / denominator);
	}

	/**
	 * Answer the nearest point at this line from another point. 
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint nearestPointFromPoint_(Jun3dPoint aJun3dPoint) {
		double denominator = Math.pow(this.a(), 2) + Math.pow(this.b(), 2) + Math.pow(this.c(), 2);
		if (Math.abs(denominator) < Accuracy()) {
			this.error_("Can't define a plane");
			return null;
		}
		double numerator = 0.0d - (this.a() * aJun3dPoint.x() + this.b() * aJun3dPoint.y() + this.c() * aJun3dPoint.z() + this.d());
		Jun3dPoint thePoint = new Jun3dPoint(aJun3dPoint.x() + (this.a() * numerator / denominator), aJun3dPoint.y() + (this.b() * numerator / denominator), aJun3dPoint.z() + (this.c() * numerator / denominator));
		return thePoint;
	}

	/**
	 * Answer the normal unit vector of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint normalUnitVector() {
		double dd = Math.sqrt((a * a) + (b * b) + (c * c));

		return new Jun3dPoint(a / dd, b / dd, c / dd);
	}

	/**
	 * Answer the normal vector of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint normalVector() {
		return new Jun3dPoint(a, b, c);
	}

	/**
	 * Answer the projection of the Jun3dLine on the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine projectionOfLine_(Jun3dLine aLine) {
		Jun3dPoint from = (Jun3dPoint) aLine.from();
		Jun3dPoint to = (Jun3dPoint) aLine.to();

		return new Jun3dLine(this.projectionOfPoint_(from), this.projectionOfPoint_(to));
	}

	/**
	 * Answer the projection of the Jun3dPoint on the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category functions
	 */
	public Jun3dPoint projectionOfPoint_(Jun3dPoint aPoint) {
		double denominator = (a * a) + (b * b) + (c * c);
		if (Math.abs(denominator) < ACCURACY) {
			this.normalizeParameters();
			denominator = (a * a) + (b * b) + (c * c);
			if (Math.abs(denominator) < ACCURACY) {
				throw SmalltalkException.Error("can not define a plane");
			}
		}
		double numerator = -((a * aPoint.x()) + (b * aPoint.y()) + (c * aPoint.z()) + d);
		double x = aPoint.x() + ((a * numerator) / denominator);
		double y = aPoint.y() + ((b * numerator) / denominator);
		double z = aPoint.z() + ((c * numerator) / denominator);
		return new Jun3dPoint(x, y, z);
	}

	/**
	 * Answer the reflecting line on the receiver with the specified line.
	 * 
	 * @param aPlane jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category functions
	 */
	public Jun3dLine reflectingLineWithLine_(Jun3dLine aLine) {
		Jun3dPoint intersectingPoint = this.intersectingPointWithLine_(aLine);
		Jun3dPoint reflectingPoint, reflectingVector;
		if (intersectingPoint == null) {
			reflectingPoint = aLine.last();
			reflectingVector = aLine.last().minus_(aLine.first());
		} else {
			if (this.containsPoint_(aLine.first())) {
				Jun3dPoint footPoint = this.nearestPointFromPoint_(aLine.last());
				reflectingPoint = aLine.first();
				reflectingVector = aLine.last().to_(footPoint).atT_(2).minus_(reflectingPoint);
			} else {
				Jun3dPoint nearPoint = this.nearestPointFromPoint_(aLine.first());
				Jun3dPoint farPoint = nearPoint.to_(intersectingPoint).atT_(2);
				reflectingPoint = farPoint.minus_(nearPoint.minus_(aLine.first()));
				if (this.whichSide_(aLine.first()) >= 0) {
					reflectingVector = intersectingPoint.to_(reflectingPoint).normalized().atT_(aLine.length()).minus_(intersectingPoint);
				} else {
					reflectingVector = reflectingPoint.to_(intersectingPoint).normalized().atT_(aLine.length()).minus_(reflectingPoint);
				}
			}
		}
		return new Jun3dLine(reflectingPoint, reflectingPoint.plus_(reflectingVector));
	}

	/**
	 * Answer the reversed triangle of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category functions
	 */
	public JunPlane reversed() {
		return new JunPlane(this.p3(), this.p2(), this.p1());
	}

	/**
	 * Answer the 'f' value of the receiver with the Jun3dPoint.
	 * 
	 * @return double
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category functions
	 */
	public double valueF_(JunPoint aPoint) {
		Jun3dPoint thePoint = Jun3dPoint.Coerce_(aPoint);
		return this.a() * thePoint.x() + this.b() * thePoint.y() + this.c() * thePoint.z() + this.d();
	}

	/**
	 * Answer the value which side at a point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @return int
	 * @category functions
	 */
	public int whichSide_(JunPoint aPoint) {
		double aNumber = this.valueF_(aPoint);
		if (aNumber > 0.0d) {
			return 1;
		} else if (aNumber < 0.0d) {
			return -1;
		}
		return 0;
	}

	/**
	 * Answer the parameter a.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double a() {
		return a;
	}

	/**
	 * Answer the parameter b.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double b() {
		return b;
	}

	/**
	 * Answer the parameter c.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double c() {
		return c;
	}

	/**
	 * Answer the parameter d.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double d() {
		return d;
	}

	/**
	 * Answer the parameter f1.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double f1() {
		return this.p2().x() - this.p1().x();
	}

	/**
	 * Answer the parameter f2.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double f2() {
		return this.p3().x() - this.p1().x();
	}

	/**
	 * Answer the parameter g1.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double g1() {
		return this.p2().y() - this.p1().y();
	}

	/**
	 * Answer the parameter g2.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double g2() {
		return this.p3().y() - this.p1().y();
	}

	/**
	 * Answer the parameter h1.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double h1() {
		return this.p2().z() - this.p1().z();
	}

	/**
	 * Answer the parameter h2.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double h2() {
		return this.p3().z() - this.p1().z();
	}

	/**
	 * Answer the parameter n.
	 * 
	 * @return double
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category parameters
	 */
	public double n() {
		double denominator = Math.pow(this.a(), 2) + Math.pow(this.b(), 2) + Math.pow(this.c(), 2);
		if (denominator < this.Accuracy()) {
			throw SmalltalkException.Error("can not define a plane");
		}
		return 1.0d / Math.sqrt(denominator);
	}

	/**
	 * Answer the parameter n1.
	 * 
	 * @return double
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category parameters
	 */
	public double n1() {
		double denominator = Math.pow(this.f1(), 2) + Math.pow(this.g1(), 2) + Math.pow(this.h1(), 2);
		if (denominator < this.Accuracy()) {
			throw SmalltalkException.Error("can not define a plane");
		}
		return 1.0d / Math.sqrt(denominator);
	}

	/**
	 * Answer the parameter n2.
	 * 
	 * @return double
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category parameters
	 */
	public double n2() {
		double denominator = Math.pow(this.f2(), 2) + Math.pow(this.g2(), 2) + Math.pow(this.h2(), 2);
		if (denominator < this.Accuracy()) {
			throw SmalltalkException.Error("can not define a plane");
		}
		return 1.0d / Math.sqrt(denominator);
	}

	/**
	 * Answer the parameter x0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double x0() {
		return this.p1().x();
	}

	/**
	 * Answer the parameter y0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double y0() {
		return this.p1().y();
	}

	/**
	 * Answer the parameter z0.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double z0() {
		return this.p1().z();
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		this.p1().printOn_(aWriter);
		aWriter.write(" plane: ");
		this.p2().printOn_(aWriter);
		aWriter.write(" and: ");
		this.p3().printOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" on:: ");
		this.p1().storeOn_(aWriter);
		aWriter.write(" on:: ");
		this.p2().storeOn_(aWriter);
		aWriter.write(" on:: ");
		this.p3().storeOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Answer true if the receiver contains the Jun3dPoint, otherwise false.
	 * 
	 * @return boolean
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category testing
	 */
	public boolean containsPoint_(Jun3dPoint aPoint) {
		return Math.abs(this.valueF_(aPoint)) < ACCURACY;
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer true if the receiver is a plane geometry, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isPlane()
	 * @category testing
	 */
	public boolean isPlane() {
		return true;
	}

	/**
	 * Answer the new JunPlane which is rotated by aJunAngle.
	 * 
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category transforming
	 */
	public JunPlane rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun3dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new JunPlane which is scaled by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category transforming
	 */
	public JunPlane scaledBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Scale_(aNumber));
	}

	/**
	 * Answer the new JunPlane which is scaled by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category transforming
	 */
	public JunPlane scaledBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply the JunTransformation to this JunGeometry.
	 * This method is not defined in Smalltalk version.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @category transforming
	 */
	public JunPlane transform_(Jun3dTransformation aTransformation) {
		Object[] arguments = { p1.transform_(aTransformation), p2.transform_(aTransformation), p3.transform_(aTransformation) };
		return (JunPlane) _New(this.getClass(), arguments);
	}

	/**
	 * Answer the copy of the receiver which is translated by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.suraces.JunPlane
	 * @category transforming
	 */
	public JunPlane translatedBy_(double aNumber) {
		return this.transform_(Jun3dTransformation.Translate_(aNumber));
	}

	/**
	 * Answer the copy of the receiver which is translated by the specified amount.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.suraces.JunPlane
	 * @category transforming
	 */
	public JunPlane translatedBy_(Jun3dPoint aPoint) {
		return this.transform_(Jun3dTransformation.Translate_(aPoint));
	}

	/**
	 * Normalize the parameters.
	 * 
	 * @category private
	 */
	protected void normalizeParameters() {
		double max = Math.max(Math.max(Math.abs(a), Math.abs(b)), Math.abs(c));
		a = a / max;
		b = b / max;
		c = c / max;
		d = d / max;
	}

	/**
	 * Set the parameter a.
	 * 
	 * @param aNumber double 
	 * @category private
	 */
	protected void setA_(double aNumber) {
		a = aNumber;
	}

	/**
	 * Set the parameter b.
	 * 
	 * @param aNumber double 
	 * @category private
	 */
	protected void setB_(double aNumber) {
		b = aNumber;
	}

	/**
	 * Set the parameter c.
	 * 
	 * @param aNumber double 
	 * @category private
	 */
	protected void setC_(double aNumber) {
		c = aNumber;
	}

	/**
	 * Set the parameter d.
	 * 
	 * @param aNumber double 
	 * @category private
	 */
	protected void setD_(double aNumber) {
		d = aNumber;
	}

	/**
	 * Set the p1 point.
	 * 
	 * @param aPoint jp.co.sra.jun.geomery.basic.Jun3dPoint 
	 * @category private
	 */
	protected void setP1_(Jun3dPoint aPoint) {
		p1 = Jun3dPoint.Coerce_(aPoint);
	}

	/**
	 * Set the p2 point.
	 * 
	 * @param aPoint jp.co.sra.jun.geomery.basic.Jun3dPoint 
	 * @category private
	 */
	protected void setP2_(Jun3dPoint aPoint) {
		p2 = Jun3dPoint.Coerce_(aPoint);
	}

	/**
	 * Set the p3 point.
	 * 
	 * @param aPoint jp.co.sra.jun.geomery.basic.Jun3dPoint 
	 * @category private
	 */
	protected void setP3_(Jun3dPoint aPoint) {
		p3 = Jun3dPoint.Coerce_(aPoint);
	}
}
