package jp.co.sra.jun.geometry.surfaces;

import java.awt.Point;

import jp.co.sra.jun.collections.sequences.JunDoubleMatrix;
import jp.co.sra.jun.collections.sequences.JunMatrix;
import jp.co.sra.jun.geometry.abstracts.JunSurface;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.support.JunBSplineFunction;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsSurface;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunNurbsSurface class
 * 
 *  @author    nisinaka
 *  @created   1998/12/18 (by nisinaka)
 *  @updated   2004/10/20 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun582 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunNurbsSurface.java,v 8.11 2008/02/20 06:30:58 nisinaka Exp $
 */
public class JunNurbsSurface extends JunSurface {

	/** A JunMatrix of control points. */
	protected JunMatrix controlPoints;

	/** A JunDoubleMatrix of weights. */
	protected JunDoubleMatrix weights;

	/** A knot vector of u-parameter. */
	protected double[] uKnotVector;

	/** A knot vector of v-parameter. */
	protected double[] vKnotVector;

	/**
	 * Create a new instance of JunNurbsSurface and initialize it with the bezier control points.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @return jp.co.sra.jun.geometry.surfaces.JunNurbsSurface
	 * @category Instance creation
	 */
	public static final JunNurbsSurface BezierControlPoints_(JunMatrix aMatrix) {
		int columnSize = aMatrix.columnSize();
		int rowSize = aMatrix.rowSize();
		double[] uKnotVector = new double[rowSize * 2];
		for (int index = 0; index < rowSize; index++) {
			uKnotVector[index] = 0.0;
			uKnotVector[index + rowSize] = 1.0;
		}
		double[] vKnotVector = new double[columnSize * 2];
		for (int index = 0; index < columnSize; index++) {
			vKnotVector[index] = 0.0;
			vKnotVector[index + columnSize] = 1.0;
		}
		return BSplineControlPoints_uKnotVector_vKnotVector_(aMatrix, uKnotVector, vKnotVector);
	}

	/**
	 * Create an instance of JunNurbsSurface and initialize it with the arguments.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @param anArray1 double[]
	 * @param anArray2 double[]
	 * @return jp.co.sra.jun.geometry.surfaces.JunNurbsSurface
	 * @category Instance creation
	 */
	public static final JunNurbsSurface BSplineControlPoints_uKnotVector_vKnotVector_(JunMatrix aMatrix, double[] anArray1, double[] anArray2) {
		int rowSize = aMatrix.rowSize();
		int columnSize = aMatrix.columnSize();
		double[] weights = new double[rowSize * columnSize];
		for (int i = 0; i < weights.length; i++) {
			weights[i] = 1;
		}
		return ControlPoints_weights_uKnotVector_vKnotVector_(aMatrix, new JunDoubleMatrix(rowSize, columnSize, weights), anArray1, anArray2);
	}

	/**
	 * Create a new instance of JunNurbsSurface and initialize it.
	 * 
	 * @param aMatrix1 jp.co.sra.jun.collections.sequences.JunMatrix
	 * @param aMatrix2 jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @param anArray1 double[]
	 * @param anArray2 double[]
	 * @return jp.co.sra.jun.geometry.surfaces.JunNurbsSurface
	 * @category Instance creation
	 */
	public static final JunNurbsSurface ControlPoints_weights_uKnotVector_vKnotVector_(JunMatrix aMatrix1, JunDoubleMatrix aMatrix2, double[] anArray1, double[] anArray2) {
		return new JunNurbsSurface(aMatrix1, aMatrix2, anArray1, anArray2);
	}

	/**
	 * Create a new instance of JunNurbsSurface and initialize it.
	 * 
	 * @param aMatrix1 jp.co.sra.jun.collections.sequences.JunMatrix
	 * @param aMatrix2 jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @param anArray1 double[]
	 * @param anArray2 double[]
	 * @category Instance creation
	 */
	public JunNurbsSurface(JunMatrix aMatrix1, JunDoubleMatrix aMatrix2, double[] anArray1, double[] anArray2) {
		this.controlPoints_(aMatrix1);
		this.weights_(aMatrix2);
		this.uKnotVector_(anArray1);
		this.vKnotVector_(anArray2);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		controlPoints = null;
		weights = null;
		uKnotVector = null;
		vKnotVector = null;
	}

	/**
	 * Answer a regularized point of the receiver on the surface.
	 * 
	 * @param aNumber1 double
	 * @param aNumber2 double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint atU_v_(double aNumber1, double aNumber2) {
		if (aNumber1 < 0 || 1 < aNumber1) {
			return null;
		}
		if (aNumber2 < 0 || 1 < aNumber2) {
			return null;
		}

		JunBSplineFunction uBspline = this.uBspline();
		JunBSplineFunction vBspline = this.vBspline();
		Jun3dPoint numerator = Jun3dPoint.Zero();
		double dominator = 0;
		for (int uIndex = 0; uIndex < this.uSize(); uIndex++) {
			for (int vIndex = 0; vIndex < this.vSize(); vIndex++) {
				double nw = uBspline.i_t_(uIndex, aNumber1) * vBspline.i_t_(vIndex, aNumber2) * this.weightAtU_v_(uIndex, vIndex);
				numerator = numerator.plus_(this.controlPointAtU_v_(uIndex, vIndex).multipliedBy_(nw));
				dominator += nw;
			}
		}
		if (dominator == 0) {
			return this.controlPointAtU_v_(this.uSize() - 1, this.vSize() - 1);
		}

		return numerator.dividedBy_(dominator);
	}

	/**
	 * Answer the control point at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint controlPointAt_(Point aPoint) {
		return (Jun3dPoint) controlPoints._at(aPoint.x, aPoint.y);
	}

	/**
	 * Set the control point at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void controlPointAt_put_(Point aPoint, Jun3dPoint a3dPoint) {
		controlPoints._put(aPoint.x, aPoint.y, a3dPoint);
	}

	/**
	 * Answer the control point at the specified point.
	 * 
	 * @param u int
	 * @param v int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint controlPointAtU_v_(int u, int v) {
		return (Jun3dPoint) controlPoints._at(u, v);
	}

	/**
	 * Answer the JunMatrix of the contol points.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category accessing
	 */
	public JunMatrix controlPoints() {
		return controlPoints;
	}

	/**
	 * Set the JunMatrix as the control points of the receiver.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunMatrix
	 * @category accessing
	 */
	public void controlPoints_(JunMatrix aMatrix) {
		controlPoints = aMatrix;
	}

	/**
	 * Answer the size of the control points.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point controlPointSize() {
		return controlPoints.matrixSize();
	}

	/**
	 * Answer the BSpline of the u-parameter.
	 * 
	 * @return jp.co.sra.jun.geometry.support.JunBSplineFunction
	 * @category accessing
	 */
	public JunBSplineFunction uBspline() {
		return new JunBSplineFunction((double[]) this.uKnotVector().clone(), this.uOrder());
	}

	/**
	 * Answer the knot vector of the u-parameter.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] uKnotVector() {
		return uKnotVector;
	}

	/**
	 * Set the knot vector of the u-parameter.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void uKnotVector_(double[] anArray) {
		uKnotVector = anArray;
	}

	/**
	 * Answer the order number in a row.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int uOrder() {
		return uKnotVector.length - this.uSize();
	}

	/**
	 * Answer the number of the control points in a row.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int uSize() {
		return controlPoints.rowSize();
	}

	/**
	 * Answer the BSpline of the u-parameter.
	 * 
	 * @return jp.co.sra.jun.geometry.support.JunBSplineFunction
	 * @category accessing
	 */
	public JunBSplineFunction vBspline() {
		return new JunBSplineFunction((double[]) this.vKnotVector().clone(), this.uOrder());
	}

	/**
	 * Answer the knot vector of the v-parameter.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] vKnotVector() {
		return vKnotVector;
	}

	/**
	 * Set the knot vector of the v-parameter.
	 * 
	 * @param anArray double[]
	 * @category accessing
	 */
	public void vKnotVector_(double[] anArray) {
		vKnotVector = anArray;
	}

	/**
	 * Answer the order number in a column.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int vOrder() {
		return vKnotVector.length - this.vSize();
	}

	/**
	 * Answer the number of the control points in a column.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int vSize() {
		return controlPoints.columnSize();
	}

	/**
	 * Answer the weight at the specified point.
	 * 
	 * @param u int
	 * @param v int
	 * @return double
	 * @category accessing
	 */
	public double weightAtU_v_(int u, int v) {
		return weights._doubleValueAt(u, v);
	}

	/**
	 * Answer the weights of the receiver.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix weights() {
		return weights;
	}

	/**
	 * Set the weights of the receiver.
	 * 
	 * @param aMatrix jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public void weights_(JunDoubleMatrix aMatrix) {
		weights = aMatrix;
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunNurbsSurface aNurbsSurface = (JunNurbsSurface) anObject;
		if (this.controlPoints().rowSize() != aNurbsSurface.controlPoints().rowSize()) {
			return false;
		}
		if (this.controlPoints().columnSize() != aNurbsSurface.controlPoints().columnSize()) {
			return false;
		}
		for (int u = 0; u < this.controlPoints().rowSize(); u++) {
			for (int v = 0; v < this.controlPoints().columnSize(); v++) {
				if (this.controlPointAtU_v_(u, v).equal_(aNurbsSurface.controlPointAtU_v_(u, v)) == false) {
					return false;
				}
			}
		}

		if (this.weights().rowSize() != aNurbsSurface.weights().rowSize()) {
			return false;
		}
		if (this.weights().columnSize() != aNurbsSurface.weights().columnSize()) {
			return false;
		}
		for (int u = 0; u < this.weights().rowSize(); u++) {
			for (int v = 0; v < this.weights().columnSize(); v++) {
				if (this.isEqualNumber_to_(this.weightAtU_v_(u, v), aNurbsSurface.weightAtU_v_(u, v)) == false) {
					return false;
				}
			}
		}

		int size = this.uKnotVector().length;
		if (size != aNurbsSurface.uKnotVector().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.isEqualNumber_to_(this.uKnotVector()[i], aNurbsSurface.uKnotVector()[i]) == false) {
				return false;
			}
		}

		size = this.vKnotVector().length;
		if (size != aNurbsSurface.vKnotVector().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.isEqualNumber_to_(this.vKnotVector()[i], aNurbsSurface.vKnotVector()[i]) == false) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBox#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		JunNurbsSurface aNurbsSurface = (JunNurbsSurface) anObject;
		if (this.controlPoints().rowSize() != aNurbsSurface.controlPoints().rowSize()) {
			return false;
		}
		if (this.controlPoints().columnSize() != aNurbsSurface.controlPoints().columnSize()) {
			return false;
		}
		for (int u = 0; u < this.controlPoints().rowSize(); u++) {
			for (int v = 0; v < this.controlPoints().columnSize(); v++) {
				if (this.controlPointAtU_v_(u, v).equals(aNurbsSurface.controlPointAtU_v_(u, v)) == false) {
					return false;
				}
			}
		}

		if (this.weights().rowSize() != aNurbsSurface.weights().rowSize()) {
			return false;
		}
		if (this.weights().columnSize() != aNurbsSurface.weights().columnSize()) {
			return false;
		}
		for (int u = 0; u < this.weights().rowSize(); u++) {
			for (int v = 0; v < this.weights().columnSize(); v++) {
				if (this.weightAtU_v_(u, v) != aNurbsSurface.weightAtU_v_(u, v)) {
					return false;
				}
			}
		}

		int size = this.uKnotVector().length;
		if (size != aNurbsSurface.uKnotVector().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.uKnotVector()[i] != aNurbsSurface.uKnotVector()[i]) {
				return false;
			}
		}

		size = this.vKnotVector().length;
		if (size != aNurbsSurface.vKnotVector().length) {
			return false;
		}
		for (int i = 0; i < size; i++) {
			if (this.vKnotVector()[i] != aNurbsSurface.vKnotVector()[i]) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}

	/**
	 * Answer true if the receiver is a plane geometry, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPlane() {
		return true;
	}

	/**
	 * Convert the receiver as a JunOpenGL3dNurbsSurface.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dNurbsSurface
	 * @category converting
	 */
	public JunOpenGL3dNurbsSurface asJunOpenGL3dNurbsSurface() {
		return new JunOpenGL3dNurbsSurface(controlPoints, weights, uKnotVector, vKnotVector);
	}

	/**
	 * Convert the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		return this.asJunOpenGL3dNurbsSurface();
	}

	/**
	 * Render the receiver on the JunOpenGLRenderingContext.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category displaying
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		aRenderingContext.displayNurbsSurface_(this);
	}

	/**
	 * Enumerate the trim curves, but do nothing.
	 * 
	 * @param anObject java.lang.Object
	 * @return java.lang.Object
	 * @category private
	 */
	public Object trimCurvesDo_(Object anObject) {
		return null;
	}

}
