package jp.co.sra.jun.geometry.surfaces;

import java.io.IOException;
import java.io.Writer;

import jp.co.sra.jun.geometry.abstracts.JunSurface;
import jp.co.sra.jun.geometry.basic.JunPoint;

/**
 * JunCircle class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2004/12/17 (by Mitsuhiro Asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun658 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunCircle.java,v 8.13 2008/02/20 06:30:58 nisinaka Exp $
 */
public abstract class JunCircle extends JunSurface {

	protected double radius;

	/**
	 * Answer my center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	protected abstract JunPoint _center();

	/**
	 * Get the receiver's radius.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radius() {
		return radius;
	}

	/**
	 * Answer the this area size
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return Math.PI * Math.pow(this.radius(), 2);
	}

	/**
	 * Get the parameter r.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double r() {
		return this.radius();
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		JunCircle aCircle = (JunCircle) anObject;
		return this.isEqualNumber_to_(this.radius(), aCircle.radius());
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		JunCircle aCircle = (JunCircle) anObject;
		return this.radius() == aCircle.radius();
	}

	/**
	 * Convert the receiver as an array of JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOfPlanes()
	 * @category converting
	 */
	public JunPlane[] asArrayOfPlanes() {
		return new JunPlane[] { this.asPlane() };
	}

	/**
	 * Convert to a JunPlane.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane
	 * @category converting
	 */
	public abstract JunPlane asPlane();

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		this._center().printOn_(aWriter);
		aWriter.write(" radius: ");
		aWriter.write(String.valueOf(this.radius()));
		aWriter.write(')');
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write('(');
		aWriter.write(this._className().toString());
		aWriter.write(" center: ");
		this._center().printOn_(aWriter);
		aWriter.write(" radius: ");
		aWriter.write(String.valueOf(this.radius()));
		aWriter.write(')');
	}

	/**
	 * Answer true if the receiver is a circle, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#isCircle()
	 * @category testing
	 */
	public boolean isCircle() {
		return true;
	}

	/**
	 * Set the receiver's radius.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setR_(double aNumber) {
		this.setRadius_(aNumber);
	}

	/**
	 * Set the receiver's radius.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setRadius_(double aNumber) {
		radius = aNumber;
	}
}
