package jp.co.sra.jun.geometry.forms;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StRectangle;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;

/**
 * JunFormTriangulation2 class
 * 
 *  @author    nisinaka
 *  @created   2005/10/26 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun574 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFormTriangulation2.java,v 8.12 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunFormTriangulation2 extends JunAbstractFormTriangulation {

	protected Jun2dTriangle[] constrainedDelaunayTriangles;

	/**
	 * Create a new instance of JunFormTriangulation2 and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunFormTriangulation2() {
		super();
	}

	/**
	 * Create a new instance of JunFormTriangulation2 and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunFormTriangulation2(Jun2dPoint[] points) {
		super(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.forms.JunForm2dRegion#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		constrainedDelaunayTriangles = null;
	}

	/**
	 * Set my new points.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.geometry.forms.JunForm2dRegion#points_(jp.co.sra.jun.geometry.basic.Jun2dPoint[])
	 * @category accessing
	 */
	public void points_(Jun2dPoint[] points) {
		super.points_(points);
		constrainedDelaunayTriangles = null;
	}

	/**
	 * Answer my current triangles.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @see jp.co.sra.jun.geometry.forms.JunAbstractFormTriangulation#triangles()
	 * @category accessing
	 */
	public Jun2dTriangle[] triangles() {
		return this.trianglesInterim_(null);
	}

	/**
	 * Answer my current triangles.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category accessing
	 */
	public Jun2dTriangle[] trianglesInterim_(StBlockClosure aBlock) {
		if (this.pointsWithoutLast().length < 3) {
			return new Jun2dTriangle[0];
		}

		return this.constrainedDelaunayTrianglesInterim_(aBlock);
	}

	/**
	 * Compute the constrained delaunay triangles.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category computing
	 */
	protected Jun2dTriangle[] constrainedDelaunayTrianglesInterim_(StBlockClosure aBlock) {
		if (constrainedDelaunayTriangles == null) {
			ArrayList aList = new ArrayList();
			this.constrainedDelaunayTriangles_on_interim_(this.pointsWithoutLast(), aList, aBlock);
			constrainedDelaunayTriangles = (Jun2dTriangle[]) aList.toArray(new Jun2dTriangle[aList.size()]);
		}
		return constrainedDelaunayTriangles;
	}

	/**
	 * Compute the constrained delaunay triangles.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param aList java.util.List
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category computing
	 */
	protected void constrainedDelaunayTriangles_on_interim_(Jun2dPoint[] points, List aList, StBlockClosure aBlock) {
		if (points.length < 3) {
			return;
		}

		if (aBlock != null) {
			aBlock.value_value_value_(aList.toArray(new Jun2dTriangle[aList.size()]), null, null);
		}

		Jun2dPoint point1 = points[0];
		Jun2dPoint point2 = points[1];

		if (points.length == 3) {
			Jun2dTriangle triangle = Jun2dTriangle.First_second_third_(point1, points[2], point2);
			if (aBlock != null) {
				aBlock.value_value_value_(aList.toArray(new Jun2dTriangle[aList.size()]), triangle, null);
			}
			aList.add(triangle);
			return;
		}

		Jun2dPoint[] anArray = new Jun2dPoint[points.length - 2];
		System.arraycopy(points, 2, anArray, 0, anArray.length);
		points = anArray;

		JunAngle max = null;
		int target = -1;
		JunAngle[] associations = new JunAngle[points.length];
		for (int i = 0; i < points.length; i++) {
			Jun2dTriangle triangle = Jun2dTriangle.First_second_third_(point1, points[i], point2);
			if (aBlock != null) {
				aBlock.value_value_value_(aList.toArray(new Jun2dTriangle[aList.size()]), triangle, null);
			}
			Jun2dLine line1 = new Jun2dLine(points[i], point1);
			Jun2dLine line2 = new Jun2dLine(points[i], point2);
			JunAngle angle = line1.angleWithLine_(line2);
			if (max == null || max.rad() < angle.rad()) {
				max = angle;
				target = i;
			}
			associations[i] = angle;
		}
		if (target < 0) {
			return;
		}

		Jun2dPoint p = points[target];
		Jun2dTriangle triangle = Jun2dTriangle.First_second_third_(point1, p, point2);
		if (aBlock != null) {
			aBlock.value_value_value_(aList.toArray(new Jun2dTriangle[aList.size()]), triangle, null);
		}
		if (this.containsLineSegment_(new Jun2dLine(point1, p)) == false || this.containsLineSegment_(new Jun2dLine(point2, p)) == false) {
			int nth = this.searchAgain_with_associations_in_on_interim_(point1, point2, associations, points, aList, aBlock);
			triangle = Jun2dTriangle.First_second_third_(point1, points[nth], point2);
			target = nth;
		}
		aList.add(triangle);

		ArrayList list = new ArrayList();
		list.add(points[target]);
		list.add(point2);
		for (int i = 0; i < target; i++) {
			list.add(points[i]);
		}
		this.constrainedDelaunayTriangles_on_interim_((Jun2dPoint[]) list.toArray(new Jun2dPoint[list.size()]), aList, aBlock);

		list = new ArrayList();
		list.add(point1);
		for (int i = target; i < points.length; i++) {
			list.add(points[i]);
		}
		this.constrainedDelaunayTriangles_on_interim_((Jun2dPoint[]) list.toArray(new Jun2dPoint[list.size()]), aList, aBlock);
	}

	/**
	 * Search again for the appropriate point to make a triangle with point1 and point2.
	 * 
	 * @param point1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param point2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param associations jp.co.sra.jun.geometry.basic.JunAngle[]
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param aList java.util.List
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return int
	 * @category computing
	 */
	private int searchAgain_with_associations_in_on_interim_(Jun2dPoint point1, Jun2dPoint point2, JunAngle[] associations, Jun2dPoint[] points, List aList, StBlockClosure aBlock) {
		List associationList = Arrays.asList(associations);

		JunAngle[] angles = new JunAngle[associations.length - 1];
		System.arraycopy(associations, 1, angles, 0, angles.length);
		Arrays.sort(angles);

		for (int i = angles.length - 1; i >= 0; i--) {
			int index = associationList.indexOf(angles[i]);
			Jun2dPoint point = points[index];
			Jun2dTriangle triangle = Jun2dTriangle.First_second_third_(point1, point, point2);
			if (aBlock != null) {
				aBlock.value_value_value_(aList.toArray(new Jun2dTriangle[aList.size()]), null, triangle);
			}
			if (this.containsLineSegment_(new Jun2dLine(point1, point)) && this.containsLineSegment_(new Jun2dLine(point2, point))) {
				return index;
			}
		}

		throw new IllegalStateException("unexpected error");
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		this.displayOn_at_triangles_color_triangle_color_pending_color_(aGraphics, aPoint, this.triangles(), Color.red, null, null, null, null);
	}

	/**
	 * Display the receiver on the graphics with the specified attributes.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param color java.awt.Color
	 * @param triangle jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @param color1 java.awt.Color
	 * @param pending jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @param color2 java.awt.Color
	 * @category displaying
	 */
	public void displayOn_at_triangles_color_triangle_color_pending_color_(Graphics aGraphics, Point aPoint, Jun2dTriangle[] triangles, Color color, Jun2dTriangle triangle, Color color1, Jun2dTriangle pending, Color color2) {
		Graphics2D aGraphics2D = (Graphics2D) aGraphics.create();

		try {
			StRectangle boundingBox = StRectangle.Origin_corner_(this.boundingBox().origin()._toPoint(), this.boundingBox().corner()._toPoint());
			int offsetX = aPoint.x - boundingBox.originX();
			int offsetY = aPoint.y - boundingBox.originY();
			aGraphics2D.translate(offsetX, offsetY);

			aGraphics2D.setColor(Color.white);
			aGraphics2D.fillRect(boundingBox.x(), boundingBox.y(), boundingBox.width(), boundingBox.height());

			StRectangle box = boundingBox.insetBy_(new StRectangle(0, 0, 1, 1));
			aGraphics2D.setColor(Color.gray);
			aGraphics2D.drawRect(box.x(), box.y(), box.width(), box.height());

			for (int i = 0; i < triangles.length; i++) {
				JunForm2dRegion aPolyline = new JunForm2dRegion(triangles[i].asPointArray());
				GeneralPath aGeneralPath = aPolyline.toGeneralPath();
				aGraphics2D.setColor(StColorValue.Blend(color, Color.white));
				aGraphics2D.fill(aGeneralPath);
				aGraphics2D.setColor(color);
				aGraphics2D.draw(aGeneralPath);
			}

			if (triangle != null) {
				JunForm2dRegion aPolyline = new JunForm2dRegion(triangle.asPointArray());
				GeneralPath aGeneralPath = aPolyline.toGeneralPath();
				aGraphics2D.setColor(StColorValue.Blend(color1, Color.white));
				aGraphics2D.fill(aGeneralPath);
				aGraphics2D.setColor(color1);
				aGraphics2D.draw(aGeneralPath);
			}

			if (pending != null) {
				JunForm2dRegion aPolyline = new JunForm2dRegion(pending.asPointArray());
				GeneralPath aGeneralPath = aPolyline.toGeneralPath();
				aGraphics2D.setColor(StColorValue.Blend(color1, Color.white));
				aGraphics2D.fill(aGeneralPath);
				aGraphics2D.setColor(color1);
				aGraphics2D.draw(aGeneralPath);
			}

			aGraphics2D.setColor(Color.black);
			aGraphics2D.draw(this.toGeneralPath());
		} finally {
			if (aGraphics2D != null) {
				aGraphics2D.dispose();
			}
		}
	}

	/**
	 * Display the receiver on the graphics with the specified attributes.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category displaying
	 */
	public void displayOn_triangles_(Graphics aGraphics, Jun2dTriangle[] triangles) {
		this.displayOn_at_triangles_color_triangle_color_pending_color_(aGraphics, new Point(0, 0), triangles, Color.red, null, null, null, null);
	}

	/**
	 * Display the receiver on the graphics with the specified attributes.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param triangles jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @param triangle jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @param pending jp.co.sra.jun.geometry.surfaces.Jun2dTriangle
	 * @category displaying
	 */
	public void displayOn_triangles_triangle_pending_(Graphics aGraphics, Jun2dTriangle[] triangles, Jun2dTriangle triangle, Jun2dTriangle pending) {
		this.displayOn_at_triangles_color_triangle_color_pending_color_(aGraphics, new Point(0, 0), triangles, Color.red, triangle, Color.blue, pending, Color.green);
	}

}
