package jp.co.sra.jun.geometry.forms;

import java.awt.Rectangle;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunForm2dRegion class
 * 
 *  @author    nisinaka
 *  @created   2005/10/26 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunForm2dRegion.java,v 8.14 2008/02/20 06:30:57 nisinaka Exp $
 */
public class JunForm2dRegion extends JunAbstractObject {

	protected Jun2dPoint[] originalPoints;
	protected Jun2dBoundingBox boundingBox;
	protected Jun2dPoint[] processingPoints;
	protected Jun2dLine[] lineSegments;

	/**
	 * Create a new instance of JunForm2dRegion and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunForm2dRegion() {
		super();
	}

	/**
	 * Create a new instance of JunForm2dRegion and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public JunForm2dRegion(Jun2dPoint[] points) {
		this();
		this.points_(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		originalPoints = null;
		boundingBox = null;
		processingPoints = null;
		lineSegments = null;
	}

	/**
	 * Answer my current points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		return originalPoints;
	}

	/**
	 * Set my new points.
	 * 
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public void points_(Jun2dPoint[] points) {
		if (points == null) {
			points = new Jun2dPoint[0];
		}

		ArrayList aList = new ArrayList(points.length);
		Jun2dBoundingBox aBox = null;
		if (points.length > 0) {
			aBox = Jun2dBoundingBox.Origin_extent_(points[0], new Jun2dPoint(1, 1));
			aList.add(points[0]);
		}
		for (int i = 1; i < points.length; i++) {
			aBox = aBox.merge_(Jun2dBoundingBox.Origin_extent_(points[i], new Jun2dPoint(1, 1)));
			aList.add(points[i]);
		}
		if (points.length > 0 && points[0].equal_(points[points.length - 1]) == false) {
			aList.add(points[0]);
		}

		originalPoints = (Jun2dPoint[]) aList.toArray(new Jun2dPoint[aList.size()]);
		boundingBox = aBox;
		processingPoints = null;
		lineSegments = null;
	}

	/**
	 * Answer my current bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = new Jun2dBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Answer the receiver's bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public Rectangle bounds() {
		if (this.boundingBox() == null) {
			return new Rectangle(0, 0, 0, 0);
		}
		return new Rectangle(0, 0, (int) Math.round(this.boundingBox().width()), (int) Math.round(this.boundingBox().height()));
	}

	/**
	 * Answer my points but not contains the last one.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] pointsWithoutLast() {
		if (processingPoints == null) {
			processingPoints = new Jun2dPoint[this.points().length - 1];
			System.arraycopy(this.points(), 0, processingPoints, 0, processingPoints.length);
		}
		return processingPoints;
	}

	/**
	 * Answer my current line segments.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category accessing
	 */
	public Jun2dLine[] segments() {
		if (lineSegments == null) {
			Jun2dPoint[] points = this.points();
			lineSegments = new Jun2dLine[points.length - 1];
			for (int i = 0; i < points.length - 1; i++) {
				lineSegments[i] = new Jun2dLine(points[i], points[i + 1]);
			}
		}
		return lineSegments;
	}

	/**
	 * Answer true if the receiver contains the specified point, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		if (this.boundingBox().containsPoint_(aPoint) == false) {
			return false;
		}

		if (this._indexOfPointCloseEnoughTo(aPoint) >= 0) {
			return true;
		}

		Jun2dLine[] segments = this.segments();
		for (int i = 0; i < segments.length; i++) {
			if (segments[i].lineSegmentContainsPoint_(aPoint)) {
				return true;
			}
		}

		ArrayList intersections = new ArrayList();
		Jun2dLine line1 = new Jun2dLine(aPoint, new Jun2dPoint(aPoint.x() + this.defaultSoFarX(), aPoint.y()));
		for (int i = 0; i < segments.length; i++) {
			Jun2dPoint p = line1.lineSegmentIntersectingPointWithLineSegment_(segments[i]);
			if (p != null) {
				int index = this._indexOfPointCloseEnoughTo(p);
				if (index >= 0) {
					int n = index;
					Jun2dPoint pp1 = p;

					n = (n <= 0) ? this.pointsWithoutLast().length - 1 : n - 1;
					Jun2dPoint p1 = this.pointsWithoutLast()[n];
					while (p1.y() == aPoint.y()) {
						pp1 = p1;

						n = (n <= 0) ? this.pointsWithoutLast().length - 1 : n - 1;
						p1 = this.pointsWithoutLast()[n];
					}

					n = index;
					Jun2dPoint pp2 = p;

					n = (n >= this.pointsWithoutLast().length - 1) ? 0 : n + 1;
					Jun2dPoint p2 = this.pointsWithoutLast()[n];
					while (p2.y() == aPoint.y()) {
						pp2 = p2;

						n = (n >= this.pointsWithoutLast().length - 1) ? 0 : n + 1;
						p2 = this.pointsWithoutLast()[n];
					}

					p = pp1.min_(pp2);
					if (p1.y() >= aPoint.y() && p2.y() >= aPoint.y()) {
						p = null;
					}
					if (p1.y() <= aPoint.y() && p2.y() <= aPoint.y()) {
						p = null;
					}
				}
			}
			if (p != null && intersections.contains(p) == false) {
				intersections.add(p);
			}
		}

		return intersections.size() % 2 == 1;
	}

	/**
	 * Answer true if the receiver contains the specified line segment, otherwise false.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @return boolean
	 * @category testing
	 */
	public boolean containsLineSegment_(Jun2dLine aLine) {
		ArrayList aList = new ArrayList();
		Jun2dLine[] segments = this.segments();
		for (int i = 0; i < segments.length; i++) {
			Jun2dPoint intersection = aLine.lineSegmentIntersectingPointWithLineSegment_(segments[i]);
			if (intersection != null) {
				aList.add(intersection);
			}
		}
		Jun2dPoint[] intersections = (Jun2dPoint[]) aList.toArray(new Jun2dPoint[aList.size()]);
		aList = new ArrayList();
		for (int i = 0; i < intersections.length; i++) {
			if (this._indexOfPointCloseEnoughTo(intersections[i]) < 0) {
				aList.add(intersections[i]);
			}
		}

		if (aList.isEmpty() == false) {
			return false;
		}
		if (this.containsPoint_(aLine.atT_(0.25)) == false) {
			return false;
		}
		if (this.containsPoint_(aLine.atT_(0.5)) == false) {
			return false;
		}
		if (this.containsPoint_(aLine.atT_(0.75)) == false) {
			return false;
		}

		return true;
	}

	/**
	 * Answer my default accuracy.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultAccuracy() {
		return Jun2dLine.Accuracy() * 10.0d;
	}

	/**
	 * Answer my default so far X.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultSoFarX() {
		return 29999;
	}

	/**
	 * Convert to GeneralPath.
	 * 
	 * @return java.awt.geom.GeneralPath
	 * @category converting
	 */
	public GeneralPath toGeneralPath() {
		Jun2dPoint[] points = this.points();
		if (points == null || points.length == 0) {
			return null;
		}

		GeneralPath aGeneralPath = new GeneralPath(GeneralPath.WIND_EVEN_ODD, points.length);
		aGeneralPath.moveTo((float) points[0].x(), (float) points[0].y());
		for (int i = 1; i < points.length; i++) {
			aGeneralPath.lineTo((float) points[i].x(), (float) points[i].y());
		}
		return aGeneralPath;
	}

	/**
	 * Answer the index of the points close enough to the specified point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return int
	 * @category private
	 */
	protected int _indexOfPointCloseEnoughTo(Jun2dPoint aPoint) {
		double accuracy = this.defaultAccuracy();
		Jun2dPoint[] points = this.pointsWithoutLast();
		for (int i = 0; i < points.length; i++) {
			if (aPoint.distance_(points[i]) < accuracy) {
				return i;
			}
		}
		return -1;
	}

}
