package jp.co.sra.jun.geometry.curves;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.TreeMap;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.forms.JunFormTriangulation2;
import jp.co.sra.jun.geometry.surfaces.Jun2dPolygon;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;

/**
 * Jun2dPolyline class
 * 
 *  @author    nisinaka
 *  @created   2006/08/18 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun692 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dPolyline.java,v 8.18 2008/02/20 06:30:57 nisinaka Exp $
 */
public class Jun2dPolyline extends JunPolyline {

	protected Jun2dPoint[] points;
	protected Jun2dBoundingBox boundingBox;
	protected double[] segmentLengths;

	/**
	 * Create a new instance of <code>Jun2dPolyline</code> and initialize it.
	 *
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public Jun2dPolyline(Jun2dPoint aPoint1, Jun2dPoint aPoint2) {
		this(new Jun2dPoint[] { aPoint1, aPoint2 });
	}

	/**
	 * Create a new instance of <code>Jun2dPolyline</code> and initialize it.
	 *
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint3 jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category Instance creation
	 */
	public Jun2dPolyline(Jun2dPoint aPoint1, Jun2dPoint aPoint2, Jun2dPoint aPoint3) {
		this(new Jun2dPoint[] { aPoint1, aPoint2, aPoint3 });
	}

	/**
	 * Create a new instance of <code>Jun2dPolyline</code> and initialize it.
	 *
	 * @param points jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public Jun2dPolyline(Jun2dPoint[] points) {
		this.setPoints_(points);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		points = null;
		boundingBox = null;
		segmentLengths = null;
	}

	/**
	 * Answer my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		if (points == null) {
			points = new Jun2dPoint[0];
		}
		return points;
	}

	/**
	 * Answer my points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint[]
	 * @see jp.co.sra.jun.geometry.curves.JunPolyline#_points()
	 * @category accessing
	 */
	protected JunPoint[] _points() {
		return this.points();
	}

	/**
	 * Answer the point at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category accessing
	 */
	public Jun2dPoint pointAt_(int index) {
		return this.points()[index];
	}

	/**
	 * Answer a regularized point the polyline at the specified number.
	 * 
	 * @param t double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint atT_(double t) {
		if (t <= 0) {
			return this.pointAt_(0);
		} else if (1 <= t) {
			return this.pointAt_(this.numberOfPoints() - 1);
		}

		double targetLength = this.length() * t;
		for (int i = 1; i < this.numberOfPoints(); i++) {
			double minLength = segmentLengths[i - 1];
			double maxLength = segmentLengths[i];
			if (minLength <= targetLength && targetLength <= maxLength) {
				Jun2dLine aLine = this.pointAt_(i - 1).to_(this.pointAt_(i));
				return aLine.atT_((targetLength - minLength) / (maxLength - minLength));
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer a point on the receiver at the specified x-coordinate.
	 * 
	 * @param xValue double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint atX_(double xValue) {
		Jun2dLine[] lines = this.asArrayOfLines();
		for (int i = 0; i < lines.length; i++) {
			if (lines[i].from().x() <= xValue && xValue <= lines[i].to().x()) {
				return lines[i].atX_(xValue);
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer a point on the receiver at the specified y-coordinate.
	 * 
	 * @param yValue double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint atY_(double yValue) {
		Jun2dLine[] lines = this.asArrayOfLines();
		for (int i = 0; i < lines.length; i++) {
			if (lines[i].from().y() <= yValue && yValue <= lines[i].to().y()) {
				return lines[i].atY_(yValue);
			}
		}

		throw SmalltalkException.Error("unexpected error");
	}

	/**
	 * Answer the point at the beginning.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint from() {
		return this.atT_(0.0d);
	}

	/**
	 * Answer the point at the end.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint to() {
		return this.atT_(1.0d);
	}

	/**
	 * Answer the number of the points.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfPoints() {
		return this.size();
	}

	/**
	 * Answer the length of the polyline.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double length() {
		if (segmentLengths == null) {
			double[] lengths = new double[this.numberOfPoints()];
			double length = 0;
			lengths[0] = length;
			for (int i = 1; i < this.numberOfPoints(); i++) {
				length += this.pointAt_(i - 1).distance_(this.pointAt_(i));
				lengths[i] = length;
			}
			segmentLengths = lengths;
		}
		return segmentLengths[segmentLengths.length - 1];
	}

	/**
	 * Answer the receiver's size.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		return this.points().length;
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category bounds accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = this.preferredBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Answer my preferred bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category bounds accessing
	 */
	protected Jun2dBoundingBox preferredBoundingBox() {
		double minX = Double.MAX_VALUE;
		double minY = Double.MAX_VALUE;
		double maxX = Double.MIN_VALUE;
		double maxY = Double.MIN_VALUE;

		Jun2dPoint[] points = this.points();
		for (int i = 0; i < points.length; i++) {
			minX = Math.min(minX, points[i].x());
			minY = Math.min(minY, points[i].y());
			maxX = Math.max(maxX, points[i].x());
			maxY = Math.max(maxY, points[i].y());
		}

		if (minX == Double.MAX_VALUE) {
			minX = 0;
		}
		if (minY == Double.MAX_VALUE) {
			minY = 0;
		}
		if (maxX == Double.MIN_VALUE) {
			maxX = 0;
		}
		if (maxY == Double.MIN_VALUE) {
			maxY = 0;
		}

		return Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(minX, minY), new Jun2dPoint(maxX, maxY));
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dPolyline aPolyline = (Jun2dPolyline) anObject;
		if (this.numberOfPoints() != aPolyline.numberOfPoints()) {
			return false;
		}
		for (int i = 0; i < this.numberOfPoints(); i++) {
			if (this.pointAt_(i).equal_(aPolyline.pointAt_(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dPolyline aPolyline = (Jun2dPolyline) anObject;
		if (this.numberOfPoints() != aPolyline.numberOfPoints()) {
			return false;
		}
		for (int i = 0; i < this.numberOfPoints(); i++) {
			if (this.pointAt_(i).equals(aPolyline.pointAt_(i)) == false) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Convert to an array of Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.geometry.curves.JunPolyline#as2dPoints()
	 * @category converting
	 */
	public Jun2dPoint[] as2dPoints() {
		return this.points();
	}

	/**
	 * Convert to an array of Jun3dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @see jp.co.sra.jun.geometry.curves.JunPolyline#as3dPoints()
	 * @category converting
	 */
	public Jun3dPoint[] as3dPoints() {
		Jun2dPoint[] _2dPoints = this.points();
		Jun3dPoint[] _3dPoints = new Jun3dPoint[_2dPoints.length];
		for (int i = 0; i < _3dPoints.length; i++) {
			_3dPoints[i] = _2dPoints[i].as3dPoint();
		}
		return _3dPoints;
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category converting
	 */
	public Jun2dLine[] asArrayOfLines() {
		return this.asArrayOf2dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf2dLines()
	 * @category converting
	 */
	public Jun2dLine[] asArrayOf2dLines() {
		Jun2dPoint[] points = this.points();
		Jun2dLine[] lines = new Jun2dLine[points.length - 1];
		for (int i = 0; i < points.length - 1; i++) {
			lines[i] = points[i].to_(points[i + 1]);
		}
		return lines;
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category converting
	 */
	public Jun2dPoint[] asArrayOfPoints() {
		return this.points();
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category converting
	 */
	public Jun2dTriangle[] asArrayOfTriangles() {
		return this.asArrayOf2dTriangles();
	}

	/**
	 * Convert to an array of Jun2dTriangle.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category converting
	 */
	public Jun2dTriangle[] asArrayOf2dTriangles() {
		if (this.numberOfPoints() < 3) {
			return new Jun2dTriangle[0];
		}

		Jun2dPoint[] collectionOfPoints = null;
		boolean isLoop = this.from().equals(this.to());
		if (isLoop) {
			collectionOfPoints = this.points();
		} else {
			collectionOfPoints = new Jun2dPoint[this.numberOfPoints() + 1];
			System.arraycopy(this.points(), 0, collectionOfPoints, 0, this.numberOfPoints());
			collectionOfPoints[collectionOfPoints.length - 1] = this.from();
		}
		JunFormTriangulation2 formTriangulation = new JunFormTriangulation2(collectionOfPoints);

		ArrayList triangleCollection = new ArrayList();
		Jun2dTriangle[] triangles = formTriangulation.triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint p1 = triangles[i].third();
			Jun2dPoint p2 = triangles[i].second();
			Jun2dPoint p3 = triangles[i].first();
			if (isLoop) {
				triangleCollection.add(Jun2dTriangle.On_on_on_(p1, p2, p3));
			} else {
				Jun2dPoint fromPoint = this.from();
				Jun2dPoint toPoint = this.to();
				if ((p1.equals(fromPoint) || p2.equals(fromPoint) || p3.equals(fromPoint)) && (p1.equals(toPoint) || p2.equals(toPoint) || p3.equals(toPoint))) {
					// keep it open.
				} else {
					triangleCollection.add(Jun2dTriangle.On_on_on_(p1, p2, p3));
				}
			}
		}

		return (Jun2dTriangle[]) triangleCollection.toArray(new Jun2dTriangle[triangleCollection.size()]);
	}

	/**
	 * Convert the receiver as a <code>Jun2dPolygon</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category converting
	 */
	public Jun2dPolygon asPolygon() {
		return new Jun2dPolygon(this.points());
	}

	/**
	 * Convert the receiver as a <code>Jun2dPolyline</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category converting
	 */
	public Jun2dPolyline asPolyline() {
		return this;
	}

	/**
	 * Convert the receiver as a <code>Jun2dPolygon</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dPolygon
	 * @category converting
	 */
	public Jun2dPolyline reversed() {
		Jun2dPoint[] points = this.points();
		Jun2dPoint[] reversePoints = new Jun2dPoint[points.length];
		for (int i = 0; i < points.length; i++) {
			reversePoints[reversePoints.length - 1 - i] = points[i];
		}
		return new Jun2dPolyline(reversePoints);
	}

	/**
	 * Answer the receiver's subdivide polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category subdividing
	 */
	public Jun2dPolyline subdivide() {
		Jun2dPoint[] points = this.points();
		Collection vertexes = new ArrayList(this.size() * 2 - 1);
		for (int i = 0; i < points.length - 1; i++) {
			vertexes.add(points[i]);
			vertexes.add(points[i].to_(points[i + 1]).center());
		}
		vertexes.add(points[points.length - 1]);
		Jun2dPolyline polyline = new Jun2dPolyline((Jun2dPoint[]) vertexes.toArray(new Jun2dPoint[vertexes.size()]));
		return polyline;
	}

	/**
	 * Answer the receiver's subdivide polyline with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category subdividing
	 */
	public Jun2dPolyline subdivideLevel_(int levelNumber) {
		Jun2dPolyline polyline = (Jun2dPolyline) this.copy();
		for (int i = 0; i < levelNumber; i++) {
			polyline = polyline.subdivide();
		}
		return polyline;
	}

	/**
	 * Subdivide myself as a spline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category subdividing
	 */
	public Jun2dPolyline subdivideAsSpline() {
		return new Jun2dPolyline(this.computeSplinePoints_(this.points()));
	}

	/**
	 * Subdivide myself as a spline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category subdividing
	 */
	public Jun2dPolyline subdivideAsSplineLevel_(int levelNumber) {
		Jun2dPolyline polyline = this;
		for (int i = 0; i < levelNumber; i++) {
			polyline = polyline.subdivideAsSpline();
		}
		return polyline;
	}

	/**
	 * Answer true if the receiver is a 2d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Sew with the triangles between the polylines.
	 * 
	 * @param aPolyline jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category sewing
	 */
	public Jun2dTriangle[] sew_(Jun2dPolyline aPolyline) {
		return this.sew_interim_(aPolyline, null);
	}

	/**
	 * Sew with the triangles between the polylines.
	 * 
	 * @param aPolyline jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category sewing
	 */
	public Jun2dTriangle[] sew_interim_(Jun2dPolyline aPolyline, StBlockClosure aBlock) {
		ArrayList aList = new ArrayList();

		Jun2dPoint[] points = aPolyline.points();
		for (int i = 1; i < points.length; i++) {
			Jun2dPoint startPoint = points[i - 1];
			Jun2dPoint endPoint = points[i];

			int index = this.findIndexOfPoints_from_to_interim_(this.points(), startPoint, endPoint, aBlock);
			if (index >= 0) {
				Jun2dPoint aPoint = this.pointAt_(index);
				aList.add(new Jun2dTriangle(startPoint, endPoint, aPoint));
			}
		}

		points = this.points();
		for (int i = 1; i < points.length; i++) {
			Jun2dPoint startPoint = points[i - 1];
			Jun2dPoint endPoint = points[i];

			int index = this.findIndexOfPoints_from_to_interim_(aPolyline.points(), startPoint, endPoint, aBlock);
			if (index >= 0) {
				Jun2dPoint aPoint = aPolyline.pointAt_(index);
				aList.add(new Jun2dTriangle(endPoint, startPoint, aPoint));
			}
		}

		return (Jun2dTriangle[]) aList.toArray(new Jun2dTriangle[aList.size()]);
	}

	/**
	 * Find an index of points.
	 * 
	 * @param candidatePoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @param startPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param endPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return int
	 * @category sewing
	 */
	protected int findIndexOfPoints_from_to_interim_(Jun2dPoint[] candidatePoints, Jun2dPoint startPoint, Jun2dPoint endPoint, StBlockClosure aBlock) {
		if (candidatePoints == null || candidatePoints.length == 0) {
			return -1;
		}

		double distance = startPoint.distance_(endPoint);

		TreeMap aTreeMap = new TreeMap(new Comparator() {
			public int compare(Object o1, Object o2) {
				double delta = ((JunAngle) o1).rad() - ((JunAngle) o2).rad();
				return (delta < 0) ? 1 : (delta > 0) ? -1 : 0;
			}
		});

		for (int i = 0; i < candidatePoints.length; i++) {
			Jun2dPoint aPoint = candidatePoints[i];
			double distance1 = aPoint.distance_(startPoint);
			double distance2 = aPoint.distance_(endPoint);
			JunAngle key = startPoint.to_(aPoint).angleWithLine_(endPoint.to_(aPoint));
			double[] value = new double[] { distance1, distance2, i };
			aTreeMap.put(key, value);

			if (aBlock != null) {
				aBlock.valueWithArguments_(new Object[] { new Jun2dPoint[] { startPoint, endPoint, aPoint }, $("pending") });
			}
		}

		Iterator i = aTreeMap.keySet().iterator();
		while (i.hasNext()) {
			JunAngle anAngle = (JunAngle) i.next();
			double[] value = (double[]) aTreeMap.get(anAngle);
			double distance1 = value[0];
			double distance2 = value[1];
			int index = (int) value[2];
			if (anAngle.rad() > Accuracy() && Math.abs(distance + distance1 - distance2) > Accuracy() && Math.abs(distance + distance2 - distance1) > Accuracy()) {
				if (aBlock != null) {
					aBlock.valueWithArguments_(new Object[] { new Jun2dPoint[] { startPoint, endPoint, candidatePoints[index] }, $("decided") });
				}
				return index;
			}
		}

		return -1;
	}

	/**
	 * Set my points.
	 * 
	 * @param newPoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category private
	 */
	protected void setPoints_(Jun2dPoint[] newPoints) {
		points = newPoints;
		boundingBox = null;
		segmentLengths = null;
	}

	/**
	 * Compute the spline points from the specified control points.
	 * 
	 * @param controlPoints jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category private
	 */
	protected Jun2dPoint[] computeSplinePoints_(Jun2dPoint[] controlPoints) {
		int size = controlPoints.length;
		if (size < 3) {
			Jun2dPoint[] points = new Jun2dPoint[size];
			System.arraycopy(controlPoints, 0, points, 0, size);
			return points;
		}

		int extras = 0;
		Jun2dPoint[] values = controlPoints;
		if (this.size() > 3 && this.from().equals(this.to())) {
			extras = 2;
			values = new Jun2dPoint[2 * extras + size];
			for (int i = 0; i < extras; i++) {
				values[i] = this.points()[size - extras + i - 1];
				values[size + extras + i] = this.points()[i + 1];
			}
			System.arraycopy(controlPoints, 0, values, extras, size);
		}

		Jun2dPoint[][] derivatives = new Jun2dPoint[3][];
		for (int i = 0; i < derivatives.length; i++) {
			derivatives[i] = new Jun2dPoint[values.length];
		}

		// Beginning of the derivs123 block.
		int n = values.length;
		if (n > 2) {
			double[] v = new double[n];
			v[0] = 4.0;
			Jun2dPoint[] h = new Jun2dPoint[n];
			h[0] = values[0].minus_(values[1].multipliedBy_(2.0)).plus_(values[2]).multipliedBy_(6.0);
			for (int i = 1; i <= n - 3; i++) {
				v[i] = 4.0 - 1.0 / v[i - 1];
				h[i] = values[i].minus_(values[i + 1].multipliedBy_(2.0)).plus_(values[i + 2]).multipliedBy_(6.0).minus_(h[i - 1].dividedBy_(v[i - 1]));
			}
			derivatives[1][n - 2] = h[n - 3].dividedBy_(v[n - 3]);
			for (int i = n - 3; i >= 1; i--) {
				derivatives[1][i] = h[i - 1].minus_(derivatives[1][i + 1]).dividedBy_(v[i - 1]);
			}
		}
		derivatives[1][0] = Jun2dPoint.Zero();
		derivatives[1][n - 1] = Jun2dPoint.Zero();
		for (int i = 0; i <= n - 2; i++) {
			derivatives[0][i] = values[i + 1].minus_(values[i]).minus_(derivatives[1][i].multipliedBy_(2.0).plus_(derivatives[1][i + 1]).dividedBy_(6.0));
			derivatives[2][i] = derivatives[1][i + 1].minus_(derivatives[1][i]);
		}
		// End of the derivs123 block.

		if (extras > 0) {
			for (int i = 0; i < 3; i++) {
				Jun2dPoint[] points = new Jun2dPoint[size];
				System.arraycopy(derivatives[i], extras, points, 0, size);
				derivatives[i] = points;
			}
		}

		LinkedList spline = new LinkedList();
		spline.add(this.from());
		for (int k = 0; k < this.size() - 1; k++) {
			Jun2dPoint d = controlPoints[k];
			Jun2dPoint c = derivatives[0][k];
			Jun2dPoint b = derivatives[1][k].dividedBy_(2.0);
			Jun2dPoint a = derivatives[2][k].dividedBy_(6.0);
			Jun2dPoint p = derivatives[1][k].abs().plus_(derivatives[1][k + 1].abs());
			int steps = (int) Math.max(4, Math.floor((p.x() + p.y()) / 100));
			for (int j = 1; j <= steps; j++) {
				double t = (double) j / steps;
				p = a.multipliedBy_(t).plus_(b).multipliedBy_(t).plus_(c).multipliedBy_(t).plus_(d);
				if (p.equal_(spline.getLast()) == false) {
					spline.add(p);
				}
			}
			p = controlPoints[k + 1];
			if (p.equal_(spline.getLast()) == false) {
				spline.add(p);
			}
		}
		return (Jun2dPoint[]) spline.toArray(new Jun2dPoint[spline.size()]);
	}

}
