package jp.co.sra.jun.geometry.curves;

import java.awt.Color;
import java.awt.Point;
import java.io.IOException;
import java.io.Writer;

import jp.co.sra.smalltalk.SmalltalkException;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.basic.JunPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;
import jp.co.sra.jun.geometry.transformations.JunTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;

/**
 * Jun2dLine class
 * 
 *  @author    nisinaka
 *  @created   1998/10/09 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/20 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dLine.java,v 8.22 2008/02/20 06:30:57 nisinaka Exp $
 */
public class Jun2dLine extends JunLine {

	protected double x0;
	protected double y0;
	protected double f;
	protected double g;

	/**
	 * Create a new instance of <code>Jun2dLine</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected Jun2dLine() {
		super();
	}

	/**
	 * Create a new instance of <code>Jun2dLine</code> with two points.
	 * 
	 * @param fromPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param toPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category Instance creation
	 */
	public Jun2dLine(JunPoint fromPoint, JunPoint toPoint) {
		super();
		this.from_to_(fromPoint, toPoint);
	}

	/**
	 * Create a new instance of the <code>Jun2dLine</code> with two points.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category Instance creation
	 */
	public static final Jun2dLine Between_and_(JunPoint aPoint1, JunPoint aPoint2) {
		Jun2dLine aLine = new Jun2dLine();
		aLine.Between_and_(aPoint1, aPoint2);
		return aLine;
	}

	/**
	 * Answer a new <code>Jun2dLine</code> with unit values.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category Constants access
	 */
	public static final Jun2dLine Unity() {
		Jun2dLine aLine = new Jun2dLine(Jun2dPoint.Zero(), Jun2dPoint.Unity());
		return aLine.normalizedLine();
	}

	/**
	 * Answer a new <code>Jun2dLine</code> with zero values.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category Constants access
	 */
	public static final Jun2dLine Zero() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer my start point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.curves.JunLine#_from()
	 * @category accessing
	 */
	protected JunPoint _from() {
		return this.from();
	}

	/**
	 * Answer my start point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunPoint
	 * @see jp.co.sra.jun.geometry.curves.JunLine#_to()
	 * @category accessing
	 */
	protected JunPoint _to() {
		return this.to();
	}

	/**
	 * Answer the angle between this line and aLine.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category accessing
	 */
	public JunAngle angleWithLine_(Jun2dLine aLine) {
		double denominator = (Math.pow(this.f(), 2) + Math.pow(this.g(), 2)) * (Math.pow(aLine.f(), 2) + Math.pow(aLine.g(), 2));
		denominator = Math.sqrt(denominator);
		if (denominator < Jun2dPoint.Accuracy()) {
			SmalltalkException.Error("unexpected line parameters");
		}
		double numerator = this.f() * aLine.f() + this.g() * aLine.g();
		double gamma = Math.min(Math.max(numerator / denominator, -1.0d), 1.0d);
		return JunAngle.FromRad_(Math.acos(gamma));
	}

	/**
	 * Answer the angle between this line and aLine.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category accessing
	 */
	public JunAngle angleWithLine_(Jun3dLine aLine) {
		Jun2dLine theLine = new Jun2dLine(new Jun2dPoint(aLine.first().x(), aLine.first().y()), new Jun2dPoint(aLine.last().x(), aLine.last().y()));
		return this.angleWithLine_(theLine);
	}

	/**
	 * Answer a regularized point of this curve at a specified number.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param t double
	 * @category accessing
	 */
	public Jun2dPoint atT_(double t) {
		return new Jun2dPoint(x0 + (f * t), y0 + (g * t));
	}

	/**
	 * Answer a point on the receiver at the specified x-coordinate.
	 * 
	 * @param x double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint atX_(double x) {
		Jun2dLine aLine = new Jun2dPoint(x, 0).to_(new Jun2dPoint(x, 1));
		return aLine.intersectingPointWithLine_(this);
	}

	/**
	 * Answer a point on the receiver at the specified y-coordinate.
	 * 
	 * @param y double
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint atY_(double y) {
		Jun2dLine aLine = new Jun2dPoint(0, y).to_(new Jun2dPoint(1, y));
		return aLine.intersectingPointWithLine_(this);
	}

	/**
	 * Set the receiver with two points.
	 * 
	 * @param aPoint1 jp.co.sra.jun.geometry.basic.JunPoint
	 * @param aPoint2 jp.co.sra.jun.geometry.basic.JunPoint
	 * @throws java.lang.IllegalArgumentException
	 * @category accessing
	 */
	public void between_and_(JunPoint aPoint1, JunPoint aPoint2) {
		Jun2dPoint point1 = Jun2dPoint.Coerce_(aPoint1);
		Jun2dPoint point2 = Jun2dPoint.Coerce_(aPoint2);
		if (point1.equals(point2)) {
			throw new IllegalArgumentException("'can not define a line");
		}
		double dx = point2.x() - point1.x();
		double dy = point2.y() - point1.y();
		double denominator = Math.sqrt(dx) + Math.sqrt(dy);
		if (denominator < this.Accuracy()) {
			throw new IllegalArgumentException("'can not define a line");
		}
		denominator = Math.sqrt(denominator);
		this.setX0_((point1.x() + point2.x()) / 2.0d);
		this.setY0_((point1.y() + point2.y()) / 2.0d);
		this.setF_((0 - dy) / denominator);
		this.setG_(dx / denominator);
	}

	/**
	 * Answer the bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall boundingBall() {
		return this.asBoundingBall();
	}

	/**
	 * Answer the bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return this.asBoundingBox();
	}

	/**
	 * Answer the center point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint center() {
		return this.atT_(0.5d);
	}

	/**
	 * Answer the start point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint first() {
		return this.from();
	}

	/**
	 * Answer the start point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint from() {
		return this.atT_(0.0d);
	}

	/**
	 * Initialize this line with two points.
	 * 
	 * @param fromPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @param toPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category accessing
	 */
	public void from_to_(JunPoint fromPoint, JunPoint toPoint) {
		Jun2dPoint point1 = Jun2dPoint.Coerce_(fromPoint);
		Jun2dPoint point2 = Jun2dPoint.Coerce_(toPoint);
		x0 = point1.x();
		y0 = point1.y();
		f = point2.x() - point1.x();
		g = point2.y() - point1.y();
	}

	/**
	 * Answer the end point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint last() {
		return this.to();
	}

	/**
	 * Answer the length of this curve.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.curves.JunLine#length()
	 * @category accessing
	 */
	public double length() {
		return this.first().distance_(this.last());
	}

	/**
	 * Answer the normal unit vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint normalUnitVector() {
		return this.normalizedLine().normalVector();
	}

	/**
	 * Answer the normal vector of this.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint normalVector() {
		return this.to().minus_(this.from());
	}

	/**
	 * Answer the 't' value of the Jun2dPoint.
	 * 
	 * @return double
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public double tAtPoint_(Jun2dPoint aPoint) {
		double denominator = (f * f) + (g * g);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}
		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;
		return (f * x + g * y) / denominator;
	}

	/**
	 * Answer the 't' value of the x value.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category accessing
	 */
	public double tAtX_(double aNumber) {
		if (Math.abs(this.f()) < this.Accuracy()) {
			return Double.NaN;
		}
		return (aNumber - this.x0()) / this.f();
	}

	/**
	 * Answer the 't' value of the y value.
	 * 
	 * @param aNumber double
	 * @return double
	 * @category accessing
	 */
	public double tAtY_(double aNumber) {
		if (Math.abs(this.g()) < this.Accuracy()) {
			return Double.NaN;
		}
		return (aNumber - this.y0()) / this.g();
	}

	/**
	 * Answer the end point of this curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint to() {
		return this.atT_(1.0d);
	}

	/**
	 * Answer true if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dLine aLine = (Jun2dLine) anObject;
		return this.isEqualNumber_to_(x0, aLine.x0) && this.isEqualNumber_to_(y0, aLine.y0) && this.isEqualNumber_to_(f, aLine.f) && this.isEqualNumber_to_(g, aLine.g);
	}

	/**
	 * Answer true if the receiver is equal to the object.
	 * 
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if (this.getClass() != anObject.getClass()) {
			return false;
		}

		Jun2dLine aLine = (Jun2dLine) anObject;
		return x0 == aLine.x0 && y0 == aLine.y0 && f == aLine.f && g == aLine.g;
	}

	/**
	 * Convert the receiver as an arrays of <code>Jun2dPoint</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category converting
	 */
	public Jun2dPoint[][] asArrays() {
		return new Jun2dPoint[][] { new Jun2dPoint[] { this.from(), this.to() } };
	}

	/**
	 * Convert the receiver as an array of the <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category converting
	 */
	public Jun2dLine[] asArrayOfLines() {
		return this.asArrayOf2dLines();
	}

	/**
	 * Convert the receiver as an array of <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asArrayOf2dLines()
	 * @category converting
	 */
	public Jun2dLine[] asArrayOf2dLines() {
		return new Jun2dLine[] { this };
	}

	/**
	 * Convert the receiver as a Jun2dBoundingBall.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category converting
	 */
	public Jun2dBoundingBall asBoundingBall() {
		Jun2dPoint centerPoint = this.center();
		double aRadius = centerPoint.distance_(this.to());
		return new Jun2dBoundingBall(centerPoint, aRadius);
	}

	/**
	 * Convert the receiver as a Jun2dBoundingBox.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category converting
	 */
	public Jun2dBoundingBox asBoundingBox() {
		return Jun2dBoundingBox.Vertex_vertex_(this.from(), this.to());
	}

	/**
	 * Convert the receiver as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		Jun3dPoint[] vertexes = { new Jun3dPoint(this.from().x(), this.from().y(), 0), new Jun3dPoint(this.to().x(), this.to().y(), 0) };
		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(vertexes);
		aPolyline.lineWidth_(1);
		aPolyline.paint_(this.defaultColor());
		return aPolyline;
	}

	/**
	 * Convert the receiver as a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObjectWithPoints()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectWithPoints() {
		Jun3dPoint[] vertexes = { new Jun3dPoint(this.from().x(), this.from().y(), 0), new Jun3dPoint(this.to().x(), this.to().y(), 0) };
		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(vertexes);
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(aPolyline);
		JunOpenGL3dObject fromPoint = this.from().asJunOpenGL3dObject();
		fromPoint.paint_(Color.red);
		compoundObject.add_(fromPoint);
		JunOpenGL3dObject toPoint = this.to().asJunOpenGL3dObject();
		toPoint.paint_(Color.blue);
		compoundObject.add_(toPoint);
		return compoundObject;
	}

	/**
	 * Convert the receiver as a <code>JunNurbsCurve</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.JunNurbsCurve
	 * @see jp.co.sra.jun.geometry.abstracts.JunCurve#asNurbsCurve()
	 * @category converting
	 */
	public JunNurbsCurve asNurbsCurve() {
		return JunNurbsCurve.BezierControlPoints_(new Jun3dPoint[] { this.from().as3dPoint(), this.center().as3dPoint(), this.to().as3dPoint() });
	}

	/**
	 * Convert the receiver as an arrays of <code>Point</code>.
	 * 
	 * @return java.awt.Point[][]
	 * @see jp.co.sra.jun.geometry.curves.JunLine#asPointArrays()
	 * @category converting
	 */
	public Point[][] asPointArrays() {
		Jun2dPoint[][] a2dPoints = this.asArrays();
		Point[][] points = new Point[a2dPoints.length][];
		for (int i = 0; i < a2dPoints.length; i++) {
			points[i] = new Point[a2dPoints[i].length];
			for (int j = 0; j < a2dPoints[i].length; j++) {
				points[i][j] = a2dPoints[i][j].asPoint();
			}
		}
		return points;
	}

	/**
	 * Convert the receiver as a Jun2dPolyline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category converting
	 */
	public Jun2dPolyline asPolyline() {
		return new Jun2dPolyline(new Jun2dPoint[] { this.from(), this.to() });
	}

	/**
	 * Answer the reversed line of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine
	 * @category converting
	 */
	public Jun2dLine reversed() {
		return new Jun2dLine(this.to(), this.from());
	}

	/**
	 * Answer the angle with the specified line.
	 * 
	 * @param aLine jp.co.sra.jun.geometry.curves.JunLine
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.geometry.curves.JunLine#angleWithLine_(jp.co.sra.jun.geometry.curves.JunLine)
	 * @category functions
	 */
	public JunAngle angleWithLine_(JunLine aLine) {
		Jun2dLine theLine = (aLine instanceof Jun2dLine) ? (Jun2dLine) aLine : new Jun2dLine(aLine._from(), aLine._to());
		double denominator = (this.f() * this.f() + this.g() * this.g()) * (theLine.f() * theLine.f() + theLine.g() * theLine.g());
		denominator = Math.sqrt(denominator);
		if (denominator < this.accuracy()) {
			throw new IllegalArgumentException("unexpected line parameters");
		}

		double numerator = this.f() * theLine.f() + this.g() * theLine.g();
		double gamma = Math.min(Math.max(numerator / denominator, -1.0d), 1.0d);
		return JunAngle.FromRad_(Math.acos(gamma));
	}

	/**
	 * Answer the distance between the Jun2dPoint and the receiver.
	 * 
	 * @return double
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#distanceFromPoint_(jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category functions
	 */
	public double distanceFromPoint_(Jun2dPoint aPoint) {
		Jun2dPoint pointOnMe = this.nearestPointFromPoint_(aPoint);
		return aPoint.distance_(pointOnMe);
	}

	/**
	 * Answer the intersecting point of the Jun2dLine and the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dPoint intersectingPointWithLine_(Jun2dLine aLine) {
		double t = this.intersectingTWithLine_(aLine);
		if (Double.isNaN(t)) {
			return null;
		}
		return (Jun2dPoint) this.atT_(t);
	}

	/**
	 * Answer the T parameter of the intersecting point of the Jun2dLine and
	 * the receiver.
	 * 
	 * @return double
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public double intersectingTWithLine_(Jun2dLine aLine) {
		double yourF = aLine.f;
		double yourG = aLine.g;
		double denominator = (yourF * g) - (f * yourG);

		if (Math.abs(denominator) < ACCURACY) {
			// These are parallel lines.
			return Double.NaN;
		}

		return (yourF * (aLine.y0 - y0) - yourG * (aLine.x0 - x0)) / denominator;
	}

	/**
	 * Answer true if the receiver contains the Jun2dPoint, otherwise false.
	 * 
	 * @return boolean
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public boolean lineSegmentContainsPoint_(Jun2dPoint aJun2dPoint) {
		if (!this.containsPoint_(aJun2dPoint)) {
			return false;
		}
		double t = this.tAtPoint_(aJun2dPoint);
		return (-ACCURACY <= t) && (t <= (1 + ACCURACY));
	}

	/**
	 * Answer the distance between the Jun2dPoint and the receiver.
	 * 
	 * @return double
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public double lineSegmentDistanceFromPoint_(Jun2dPoint aPoint) {
		Jun2dPoint pointOnMe = this.lineSegmentNearestPointFromPoint_(aPoint);
		return aPoint.distance_(pointOnMe);
	}

	/**
	 * Answer the intersecting point of the Jun2dLine and the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dPoint lineSegmentIntersectingPointWithLineSegment_(Jun2dLine aLine) {
		Jun2dPoint intersectingPoint = this.intersectingPointWithLine_(aLine);
		if (intersectingPoint == null) {
			return null;
		}
		if (!this.lineSegmentContainsPoint_(intersectingPoint)) {
			return null;
		}
		if (!aLine.lineSegmentContainsPoint_(intersectingPoint)) {
			return null;
		}
		return intersectingPoint;
	}

	/**
	 * Answer the nearest point on the line segment from the Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint lineSegmentNearestPointFromPoint_(Jun2dPoint aPoint) {
		double denominator = (f * f) + (g * g);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}
		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;
		double t = (f * x + g * y) / denominator;
		if (t < 0) {
			t = 0;
		} else if (t > 1) {
			t = 1;
		}
		return (Jun2dPoint) this.atT_(t);
	}

	/**
	 * Answer the nearest point on the receiver from the Jun2dPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category functions
	 */
	public Jun2dPoint nearestPointFromPoint_(Jun2dPoint aPoint) {
		double denominator = (f * f) + (g * g);

		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}

		double x = aPoint.x() - x0;
		double y = aPoint.y() - y0;

		return (Jun2dPoint) this.atT_((f * x + g * y) / denominator);
	}

	/**
	 * Create a normalized line of this.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dLine normalized() {
		return this.normalizedLine();
	}

	/**
	 * Create a normalized line of this.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category functions
	 */
	public Jun2dLine normalizedLine() {
		Jun2dLine aLine = (Jun2dLine) this.copy();
		double n = this.n();
		aLine.f = f * n;
		aLine.g = g * n;

		return aLine;
	}

	/**
	 * Answer the 'f' value of the receiver with the JunPoint.
	 * 
	 * @return double
	 * @param aPoint jp.co.sra.jun.geometry.basic.JunPoint
	 * @category functions
	 */
	public double valueF_(JunPoint aPoint) {
		Jun2dPoint thePoint = Jun2dPoint.Coerce_(aPoint);
		return (this.a() * thePoint.x()) + (this.b() * thePoint.y()) + this.c();
	}

	/**
	 * Answer the value which side at a point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return int
	 * @category functions
	 */
	public int whichSide_(Jun2dPoint aPoint) {
		double aNumber = this.valueF_(aPoint);
		if (aNumber > 0.0d) {
			return 1;
		} else if (aNumber < 0.0d) {
			return -1;
		}
		return 0;
	}

	/**
	 * Answer the a parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double a() {
		return 0.0d - this.g();
	}

	/**
	 * Answer the b parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double b() {
		return this.f();
	}

	/**
	 * Answer the c parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double c() {
		return this.x0() * this.g() - this.y0() * this.f();
	}

	/**
	 * Answer the f parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double f() {
		return f;
	}

	/**
	 * Answer the g parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double g() {
		return g;
	}

	/**
	 * Answer a coefficient for my normalization.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double n() {
		double denominator = (f * f) + (g * g);
		if (denominator < ACCURACY) {
			throw SmalltalkException.Error("can not define a line");
		}
		return 1.0d / Math.sqrt(denominator);
	}

	/**
	 * Answer the x0 parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double x0() {
		return x0;
	}

	/**
	 * Answer the g parameter.
	 * 
	 * @return double
	 * @category parameters
	 */
	public double y0() {
		return y0;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("2dLine (");
		this.from().printOn_(aWriter);
		aWriter.write(" , ");
		this.to().printOn_(aWriter);
		aWriter.write(")");
	}

	/**
	 * Answer the receiver's subdivide polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category subdividing
	 */
	public Jun2dPolyline subdivide() {
		return new Jun2dPolyline(new Jun2dPoint[] { this.from(), this.center(), this.to() });
	}

	/**
	 * Answer the receiver's subdivide polyline with specified level.
	 * 
	 * @param levelNumber int
	 * @return jp.co.sra.jun.geometry.curves.Jun2dPolyline
	 * @category subdividing
	 */
	public Jun2dPolyline subdivideLevel_(int levelNumber) {
		Jun2dPolyline polyline = this.asPolyline();
		for (int i = 0; i < levelNumber; i++) {
			polyline = polyline.subdivide();
		}
		return polyline;
	}

	/**
	 * Answer true if the receiver contains aPoint.
	 * 
	 * @return boolean
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		return this.distanceFromPoint_(aPoint) < ACCURACY;
	}

	/**
	 * Answer true if the receiver is a 2d geometry element.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer true if this line and aLine is parallel, otherwise false.
	 * 
	 * @return boolean
	 * @param aLine jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @category testing
	 */
	public boolean isParallelWithLine_(Jun2dLine aLine) {
		double degrees = this.angleWithLine_(aLine).deg();
		return (Math.abs(0.0d - degrees) < Jun2dPoint.Accuracy() || Math.abs(180.0d - degrees) < Jun2dPoint.Accuracy());
	}

	/**
	 * Answer the new Jun2dLine which is rotated the receiver by anAngle.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @param anAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @category transforming
	 */
	public Jun2dLine rotatedBy_(JunAngle anAngle) {
		return this.transform_(Jun2dTransformation.Rotate_(anAngle));
	}

	/**
	 * Answer the new Jun2dLine which is scaled by aPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dLine scaledBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Scale_(aPoint));
	}

	/**
	 * Apply aJunTransformation to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun2dTransformation
	 * @category transforming
	 */
	public Jun2dLine transform_(Jun2dTransformation aTransformation) {
		return new Jun2dLine(this.from().transform_(aTransformation), this.to().transform_(aTransformation));
	}

	/**
	 * Answer the copy of the receiver which is applied the specified transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#transform_(jp.co.sra.jun.geometry.transformations.JunTransformation)
	 * @category transforming
	 */
	public JunGeometry transform_(JunTransformation aTransformation) {
		return this.transform_((Jun2dTransformation) aTransformation);
	}

	/**
	 * Answer the new Jun2dLine which is translated by anPoint.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category transforming
	 */
	public Jun2dLine translatedBy_(Jun2dPoint aPoint) {
		return this.transform_(Jun2dTransformation.Translate_(aPoint));
	}

	/**
	 * Set the receiver's f parameter.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setF_(double aNumber) {
		f = aNumber;
	}

	/**
	 * Set the receiver's g parameter.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setG_(double aNumber) {
		g = aNumber;
	}

	/**
	 * Set the receiver's x0 parameter.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setX0_(double aNumber) {
		x0 = aNumber;
	}

	/**
	 * Set the receiver's y0 parameter.
	 * 
	 * @param aNumber double
	 * @category private
	 */
	protected void setY0_(double aNumber) {
		y0 = aNumber;
	}
}
