package jp.co.sra.jun.geometry.boxtree;

import java.awt.Color;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.pluralities.Jun3dBoundingBoxes;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;

/**
 * Jun3dBoundingBoxNode class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/05/28 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun666 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun3dBoundingBoxNode.java,v 8.8 2008/02/20 06:30:56 nisinaka Exp $
 */
public class Jun3dBoundingBoxNode extends JunBoundingBoxNode {
	protected Jun3dBoundingBoxNode containerNode;
	protected Jun3dBoundingBox boundingBox;

	/**
	 * Create a new instance of <code>Jun3dBoundingBoxNode</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected Jun3dBoundingBoxNode() {
		super();
	}

	/**
	 * Create a new instance of <code>Jun3dBoundingBoxNode</code> and initialize it.
	 * 
	 * @param objectCollection java.util.Collection
	 * @param boundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category Instance creation
	 */
	public Jun3dBoundingBoxNode(Collection objectCollection, Jun3dBoundingBox boundingBox) {
		super();
		this.containedObjects_(objectCollection);
		this.boundingBox_(boundingBox);
	}

	/**
	 * Create a new instance of <code>Jun3dBoundingBoxNode</code> and initialize it.
	 * 
	 * @param segmentCollection jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category Instance creation
	 */
	public Jun3dBoundingBoxNode(Jun3dLine[] segmentCollection) {
		super();

		JunGeometry.HashEqualitySet equalitySet = new JunGeometry.HashEqualitySet(segmentCollection.length);
		for (int i = 0; i < segmentCollection.length; i++) {
			equalitySet.add(segmentCollection[i]);
		}
		Collection containedObjects = new ArrayList();
		containedObjects.addAll(equalitySet);

		equalitySet = new JunGeometry.HashEqualitySet(segmentCollection.length + 1);
		for (int i = 0; i < segmentCollection.length; i++) {
			equalitySet.add(segmentCollection[i].first());
			equalitySet.add(segmentCollection[i].last());
		}
		Jun3dPoint[] pointCollection = (Jun3dPoint[]) equalitySet.toArray(new Jun3dPoint[equalitySet.size()]);
		Jun3dBoundingBox boundingBox = new Jun3dBoundingBox(pointCollection);

		this.containedObjects_(containedObjects);
		this.boundingBox_(boundingBox);
	}

	/**
	 * Create a new instance of <code>Jun3dBoundingBoxNode</code> and initialize it.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @category Instance creation
	 */
	public Jun3dBoundingBoxNode(Jun3dPoint[] pointCollection) {
		super();

		JunGeometry.HashEqualitySet equalitySet = new JunGeometry.HashEqualitySet(pointCollection.length);
		for (int i = 0; i < pointCollection.length; i++) {
			equalitySet.add(pointCollection[i]);
		}
		Collection containedObjects = new ArrayList(equalitySet);
		Jun3dBoundingBox boundingBox = new Jun3dBoundingBox((Jun3dPoint[]) containedObjects.toArray(new Jun3dPoint[containedObjects.size()]));

		this.containedObjects_(containedObjects);
		this.boundingBox_(boundingBox);
	}

	/**
	 * Create a new instance of <code>Jun3dBoundingBoxNode</code> and initialize it.
	 * 
	 * @param patchCollection jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category Instance creation
	 */
	public Jun3dBoundingBoxNode(Jun3dTriangle[] patchCollection) {
		super();

		JunGeometry.HashEqualitySet equalitySet = new JunGeometry.HashEqualitySet(patchCollection.length);
		for (int i = 0; i < patchCollection.length; i++) {
			equalitySet.add(patchCollection[i]);
		}
		Collection containedObjects = new ArrayList(equalitySet);

		equalitySet = new JunGeometry.HashEqualitySet(patchCollection.length + 1);
		for (int i = 0; i < patchCollection.length; i++) {
			equalitySet.add(patchCollection[i].p1());
			equalitySet.add(patchCollection[i].p2());
			equalitySet.add(patchCollection[i].p3());
		}
		Jun3dPoint[] pointCollection = (Jun3dPoint[]) equalitySet.toArray(new Jun3dPoint[equalitySet.size()]);
		Jun3dBoundingBox boundingBox = new Jun3dBoundingBox(pointCollection);

		this.containedObjects_(containedObjects);
		this.boundingBox_(boundingBox);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		containerNode = null;
		boundingBox = null;
		compareWithBlock = new StBlockClosure() {
			public Object value_value_(Object object, Object box) {
				JunGeometry geometry = (JunGeometry) object;
				Jun3dBoundingBox boundingBox = (Jun3dBoundingBox) box;
				if (geometry.isPoint()) {
					return new Boolean(((Jun3dPoint) geometry).boundingBox().touches_(boundingBox));
				}
				if (geometry.isBoundingBox()) {
					return new Boolean(((Jun3dBoundingBox) geometry).boundingBox().touches_(boundingBox));
				}
				if (geometry.isLine()) {
					return new Boolean(((Jun3dLine) geometry).boundingBox().touches_(boundingBox));
				}
				if (geometry.isTriangle()) {
					return new Boolean(((Jun3dTriangle) geometry).boundingBox().touches_(boundingBox));
				}
				return null;
			}
		};
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (boundingBox == null) {
			boundingBox = new Jun3dBoundingBox();
		}
		return boundingBox;
	}

	/**
	 * Set the receiver's bounding box.
	 * 
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category accessing
	 */
	public void boundingBox_(Jun3dBoundingBox aBoundingBox) {
		boundingBox = aBoundingBox;
		this.flushSubdividedNodes();
	}

	/**
	 * Answer the receiver's container node.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBoxNode
	 * @category accessing
	 */
	public Jun3dBoundingBoxNode containerNode() {
		return containerNode;
	}

	/**
	 * Set the receiver's container node.
	 * 
	 * @param aBoundingBoxNode jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBoxNode
	 * @category accessing
	 */
	public void containerNode_(Jun3dBoundingBoxNode aBoundingBoxNode) {
		containerNode = aBoundingBoxNode;
	}

	/**
	 * Answer the receiver's level number.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.geometry.boxtree.JunBoundingBoxNode#levelNumber()
	 * @category accessing
	 */
	public int levelNumber() {
		if (levelNumber <= 0) {
			Jun3dBoundingBoxNode boxNode = this.containerNode();
			int level = 0;
			while (boxNode != null) {
				boxNode = boxNode.containerNode();
				level = level + 1;
			}
			levelNumber = level;
		}
		return levelNumber;
	}

	/**
	 * Answer the receiver's subdivided nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.boxtree.Jun3dBoundingBoxNode[]
	 * @category accessing
	 */
	public Jun3dBoundingBoxNode[] subdividedNodes() {
		if (subdividedNodes == null) {
			this.subdivide();
		}
		return (Jun3dBoundingBoxNode[]) subdividedNodes.toArray(new Jun3dBoundingBoxNode[subdividedNodes.size()]);
	}

	/**
	 * Answer the subdivided nodes with the specified level value.
	 * 
	 * @param levelValue int
	 * @return jp.co.sra.jun.geometry.boxtree.Jun3dBoundingBoxNode[]
	 * @category accessing
	 */
	public Jun3dBoundingBoxNode[] subdividedNodesLevel_(int levelValue) {
		this.subdivideLevel_(levelValue);
		final Collection aCollection = new ArrayList();
		this.do_(new StBlockClosure() {
			public Object value_(Object each) {
				if (((Jun3dBoundingBoxNode) each).boxSituation() == $("necessary")) {
					aCollection.add(each);
				}
				return null;
			}
		});
		return (Jun3dBoundingBoxNode[]) aCollection.toArray(new Jun3dBoundingBoxNode[aCollection.size()]);
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}
		return this.boundingBox().equal_(((Jun3dBoundingBoxNode) anObject).boundingBox());
	}

	/**
	 * Compute a hash code for this object.
	 * 
	 * @return int
	 * @see java.lang.Object#hashCode()
	 * @category comparing
	 */
	public int hashCode() {
		return this.boundingBox().hashCode();
	}
	
	/**
	 * Convert to a <code>Jun3dBoundingBox</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox
	 * @category converting
	 */
	public Jun3dBoundingBox asBoundingBox() {
		return this.boundingBox();
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		Color colorValue = Color.gray;
		if (boxSituation == $("through")) {
			colorValue = Color.cyan;
		}
		if (boxSituation == $("necessary")) {
			colorValue = Color.magenta;
		}
		if (boxSituation == $("unnecessary")) {
			colorValue = Color.yellow;
		}
		float alphaValue = 0.25f;
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		compoundObject.add_(this.boundingBox().asJunOpenGL3dObjectColor_alpha_(colorValue, alphaValue));
		Jun3dPoint[][] points = this.boundingBox().pps();
		for (int i = 0; i < points.length; i++) {
			JunOpenGL3dPolylineLoop polylineLoop = new JunOpenGL3dPolylineLoop(points[i]);
			polylineLoop.paint_(colorValue);
			polylineLoop.lineWidth_(1);
			compoundObject.add_(polylineLoop);
		}
		return compoundObject;
	}

	/**
	 * Enumerate each objects and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating 
	 */
	public Object do_(StBlockClosure aBlock) {
		switch (aBlock.numArgs()) {
			case 1:
				aBlock.valueWithArguments_(new Object[] { this });
				break;
			case 2:
				aBlock.valueWithArguments_(new Object[] { this, new Integer(this.levelNumber) });
				break;
		}
		if (subdividedNodes != null && subdividedNodes.size() > 0) {
			Jun3dBoundingBoxNode[] geometries = this.subdividedNodes();
			for (int i = 0; i < geometries.length; i++) {
				geometries[i].do_(aBlock);
			}
		}
		return null;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("(");
		aWriter.write(this.shortPrintString());
		aWriter.write(Integer.toString(this.containedObjects().length));
		aWriter.write(", ");
		this.boundingBox().printOn_(aWriter);
		aWriter.write(")");
	}

	/**
	 * Answer the receiver's short print string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.jun.geometry.boxtree.JunBoundingBoxNode#shortPrintString()
	 * @category printing
	 */
	public String shortPrintString() {
		return "3dbbn";
	}

	/**
	 * Answer the receiver's subdivide bounding box node.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBoxNode
	 * @category subdividing
	 */
	public Jun3dBoundingBoxNode subdivide() {
		return this.subdivideLevel_(1);
	}

	/**
	 * Answer the receiver's subdivide bounding box node with the specified level.
	 * 
	 * @param aNumber int
	 * @return jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBoxNode
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category subdividing
	 */
	public Jun3dBoundingBoxNode subdivideLevel_(int aNumber) {
		if (this.levelNumber() == aNumber) {
			return this;
		}
		final Jun3dBoundingBoxNode self = this;
		if (subdividedNodes == null) {
			final List aCollection = new ArrayList(2);
			try {
				((Jun3dBoundingBoxes) this.boundingBox().perform_(this.messageToSubdivide().toString())).do_(new StBlockClosure() {
					public Object value_(Object obj) {
						Jun3dBoundingBox box = (Jun3dBoundingBox) obj;
						JunGeometry[] containedObjects = self.containedObjects();
						Collection objectCollection = new ArrayList();
						for (int i = 0; i < containedObjects.length; i++) {
							if (Boolean.TRUE.equals(self.compare_with_(containedObjects[i], box))) {
								objectCollection.add(containedObjects[i]);
							}
						}
						Jun3dBoundingBoxNode subdividedNode = new Jun3dBoundingBoxNode();
						subdividedNode.containedObjects_(objectCollection);
						subdividedNode.boundingBox_(box);
						subdividedNode.vanishBoolean_(vanishBoolean);
						subdividedNode.messageToSubdivide_(messageToSubdivide);
						subdividedNode.compareWithBlock_(compareWithBlock);
						subdividedNode.containerNode_(self);
						if (objectCollection.isEmpty()) {
							subdividedNode.boxSituation_($("unnecessary"));
						} else {
							subdividedNode.boxSituation_($("necessary"));
						}
						aCollection.add(subdividedNode);
						return null;
					}
				});
			} catch (Exception e) {
				throw new SmalltalkException(e);
			}
			subdividedNodes = aCollection;
		}
		Jun3dBoundingBoxNode[] boxNodes = (Jun3dBoundingBoxNode[]) subdividedNodes.toArray(new Jun3dBoundingBoxNode[subdividedNodes.size()]);
		for (int i = 0; i < boxNodes.length; i++) {
			boxNodes[i] = boxNodes[i].subdivideLevel_(aNumber);
			subdividedNodes.set(i, boxNodes[i]);
		}
		Collection boxSituations = new ArrayList();
		for (int i = 0; i < boxNodes.length; i++) {
			boxSituations.add(boxNodes[i].boxSituation());
		}

		if (boxSituations.contains($("unnecessary")) || boxSituations.contains($("through"))) {
			this.boxSituation_($("through"));
		} else {
			boolean isAllNecessary = true;
			StSymbol[] symbols = (StSymbol[]) boxSituations.toArray(new StSymbol[boxSituations.size()]);
			for (int i = 0; i < symbols.length; i++) {
				if (symbols[i] != $("necessary")) {
					isAllNecessary = false;
					break;
				}
			}
			if (isAllNecessary) {
				this.boxSituation_($("necessary"));
				if (this.vanishBoolean()) {
					this.flushSubdividedNodes();
				} else {
					for (int i = 0; i < boxNodes.length; i++) {
						boxNodes[i].boxSituation_($("unnecessary"));
					}
				}
			} else {
				this.boxSituation_($("through"));
			}
		}
		return this;
	}

	/**
	 * Answer true if the receiver is a 3d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is3d()
	 * @category testing
	 */
	public boolean is3d() {
		return true;
	}
}
