package jp.co.sra.jun.geometry.boxtree;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Frame;
import java.awt.Rectangle;
import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StInterval;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.utilities.JunControlUtility;
import jp.co.sra.jun.goodies.utilities.JunImageUtility;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;

/**
 * Jun2dBoundingBoxTreeTestExamples class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/26 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun666 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dBoundingBoxTreeTestExamples.java,v 8.5 2008/02/20 06:30:56 nisinaka Exp $
 */
public class Jun2dBoundingBoxTreeTestExamples extends Jun2dBoundingBoxNodeTestExamples {
	/**
	 * Example1: Example point collection of 2D line.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example1() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExamplePointCollectionOf2dLine());
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(2, 16));

		return true;
	}

	/**
	 * Example2: Example point collection of 2D sin curve.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example2() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExamplePointCollectionOf2dSinCurve());
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(2, 16));

		return true;
	}

	/**
	 * Example3: Example point collection of 2D half ball.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example3() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExamplePointCollectionOf2dHalfBall());
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(5, 16));

		return true;
	}

	/**
	 * Example4: Example segment collection of 2D line.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example4() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExampleSegmentCollectionOf2dLine());
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(2, 16));

		return true;
	}

	/**
	 * Example5: Example segment collection of 2D sin curve.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example5() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExampleSegmentCollectionOf2dSinCurve());
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(2, 16));

		return true;
	}

	/**
	 * Example6: Example patch collection of 3D half ball.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example6() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExamplePatchCollectionOf2dHalfBall());
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(5, 16));

		return true;
	}

	/**
	 * Example7: Example patch collection of 2D half ball.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example7() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExamplePatchCollectionOf2dHalfBall());
		boundingBoxTree.rootNode().messageToSubdivide_($("subdivide"));
		boundingBoxTree.rootNode().compareWithBlock_(new StBlockClosure() {
			public Object value_value_(Object object, Object box) {
				Jun2dTriangle geometry = (Jun2dTriangle) object;
				Jun2dBoundingBox boundingBox = (Jun2dBoundingBox) box;
				Set set = new HashSet();
				Jun2dTriangle[] triangles = geometry.detailedTrianglesLevel_(1);
				for (int i = 0; i < triangles.length; i++) {
					Jun2dPoint[] points = triangles[i].points();
					for (int j = 0; j < points.length; j++) {
						set.add(points[j]);
					}
				}
				Jun2dPoint[] points = (Jun2dPoint[]) set.toArray(new Jun2dPoint[set.size()]);
				boolean containsPoint = false;
				for (int i = 0; i < points.length; i++) {
					if (boundingBox.containsPoint_(points[i])) {
						containsPoint = true;
						break;
					}
				}
				return new Boolean(containsPoint);
			}
		});
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(2, 5));

		return true;
	}

	/**
	 * Example8: Example patch collection of 3D polyline.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example8() {
		Jun2dBoundingBoxTree boundingBoxTree = new Jun2dBoundingBoxTree(ExamplePatchCollectionOf2dPolyline());
		boundingBoxTree.rootNode().messageToSubdivide_($("subdivide"));
		Jun2dBoundingBoxTreeTestExamples.ExampleDisplay_levelInterval_(boundingBoxTree, new StInterval(1, 6));

		return true;
	}

	/**
	 * Example: Display the specified bounding box node with interval parameter.
	 * 
	 * @param boundingBoxTree jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxTree
	 * @param anInterval jp.co.sra.smalltalk.StInterval
	 * @return jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxTree
	 * @category Examples
	 */
	protected static Jun2dBoundingBoxTree ExampleDisplay_levelInterval_(Jun2dBoundingBoxTree boundingBoxTree, StInterval anInterval) {
		JunOpenGLDisplayModel displayModel = new JunOpenGLDisplayModel();
		displayModel.openIn_(new Rectangle(200, 100, 600 + 6, 400 + 26));
		((Frame) displayModel.getWindow()).setTitle("Bounding Box Tree");
		for (int levelNumber = (int) Math.round(anInterval.start()); levelNumber <= anInterval.last(); levelNumber = levelNumber + (int) anInterval.step()) {
			JunCursors cursor = new JunCursors(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			try {
				cursor._show();
				boundingBoxTree.subdividedNodesLevel_(levelNumber);
			} finally {
				cursor._restore();
			}
			ExampleDisplay_with_(displayModel, boundingBoxTree);
			
			StImage anImage = StImage._OfArea(displayModel.getWindow().getBounds());
			try {
				JunImageUtility.WriteImage_to_(anImage, new File(String.valueOf(levelNumber) + ".bmp"));
			} catch (IOException e) {
				e.printStackTrace();
			}
			
		}
		return boundingBoxTree;
	}

	/**
	 * Example: Display a <code>JunOpenGLDisplayModel</code> with the specified bounding box tree.
	 * 
	 * @param displayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @param boundingBoxTree jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxTree
	 * @category Examples
	 */
	protected static void ExampleDisplay_with_(final JunOpenGLDisplayModel displayModel, final Jun2dBoundingBoxTree boundingBoxTree) {
		final JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunOpenGL3dCompoundObject subObject = new JunOpenGL3dCompoundObject();
		JunGeometry[] objects = boundingBoxTree.containedObjects();
		for (int i = 0; i < objects.length; i++) {
			JunOpenGL3dObject body = null;
			if (objects[i].isPoint()) {
				JunOpenGL3dVertex vertext = (JunOpenGL3dVertex) objects[i].asJunOpenGL3dObjectColor_(Color.gray);
				vertext.size_(3);
				body = vertext;
			}
			if (objects[i].isLine()) {
				JunOpenGL3dCompoundObject compoundBody = new JunOpenGL3dCompoundObject();
				Jun2dLine object = (Jun2dLine) objects[i];
				JunOpenGL3dPolyline line = (JunOpenGL3dPolyline) object.asJunOpenGL3dObjectColor_(Color.black);
				line.lineWidth_(2);
				compoundBody = line.asCompoundObject();
				JunOpenGL3dVertex from = (JunOpenGL3dVertex) object.from().asJunOpenGL3dObjectColor_(Color.black);
				from.size_(3);
				compoundBody.add_(from);
				JunOpenGL3dVertex last = (JunOpenGL3dVertex) object.last().asJunOpenGL3dObjectColor_(Color.black);
				last.size_(3);
				compoundBody.add_(last);
				body = compoundBody;
			}
			if (objects[i].isTriangle()) {
				JunOpenGL3dCompoundObject compoundBody = new JunOpenGL3dCompoundObject();
				Jun2dTriangle object = (Jun2dTriangle) objects[i];
				Jun2dPoint[] a2dPoints = object.asPointArray();
				Jun3dPoint[] a3dPoints = new Jun3dPoint[a2dPoints.length];
				for (int j = 0; j < a2dPoints.length; j++) {
					a3dPoints[j] = a2dPoints[j].as3dPoint();
				}
				JunOpenGL3dPolyline polyline = new JunOpenGL3dPolyline(a3dPoints);
				polyline.paint_(Color.gray);
				polyline.lineWidth_(1);
				compoundBody.add_(polyline);
				JunOpenGL3dVertex p1 = (JunOpenGL3dVertex) object.p3().asJunOpenGL3dObjectColor_(Color.black);
				p1.size_(3);
				compoundBody.add_(p1);
				JunOpenGL3dVertex p2 = (JunOpenGL3dVertex) object.p3().asJunOpenGL3dObjectColor_(Color.black);
				p2.size_(3);
				compoundBody.add_(p2);
				JunOpenGL3dVertex p3 = (JunOpenGL3dVertex) object.p3().asJunOpenGL3dObjectColor_(Color.black);
				p3.size_(3);
				compoundBody.add_(p3);
				body = compoundBody;
			}
			if (body != null) {
				subObject.add_(body);
			}
		}

		compoundObject.add_(subObject);
		compoundObject.add_(boundingBoxTree.asJunOpenGL3dObject());

		Jun3dPoint point = boundingBoxTree.boundingBox().extent().as3dPoint();
		point = new Jun3dPoint(point.x(), point.y(), Math.max(point.x(), point.y()));
		subObject = (JunOpenGL3dCompoundObject) JunOpenGL3dObject.XyUnitArrows().scaledBy_(point);
		compoundObject.add_(subObject);

		JunControlUtility.Do_forMilliseconds_(new StBlockClosure() {
			public Object value() {
				displayModel.displayObject_(compoundObject);
				displayModel.resetView();
				System.out.print("level ");
				System.out.print(Integer.toString(boundingBoxTree.levelNumber()));
				System.out.print(" -> ");
				System.out.print(Integer.toString(boundingBoxTree.subdividedNodes().length));
				System.out.println(boundingBoxTree.subdividedNodes().length > 1 ? " boxes" : " box");
				return null;
			}
		}, 2000);
	}

	/**
	 * Execute all examples.
	 * 
	 * @param args java.lang.String[]
	 * @category Main
	 */
	public static void main(String[] args) {
		new Jun2dBoundingBoxTreeTestExamples();
	}
}
