package jp.co.sra.jun.geometry.boxtree;

import java.awt.Color;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolylineLoop;

/**
 * Jun2dBoundingBoxTree class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/26 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun669 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dBoundingBoxTree.java,v 8.6 2008/02/20 06:30:56 nisinaka Exp $
 */
public class Jun2dBoundingBoxTree extends JunBoundingBoxTree {
	protected Jun2dBoundingBoxNode rootNode;
	protected Jun2dBoundingBoxNode[] subdividedNodes;

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxTree</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected Jun2dBoundingBoxTree() {
		super();
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxTree</code> and initialize it.
	 * 
	 * @param segmentCollection jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category Instance creation
	 */
	public Jun2dBoundingBoxTree(Jun2dLine[] segmentCollection) {
		this();
		this.rootNode_(new Jun2dBoundingBoxNode(segmentCollection));
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxTree</code> and initialize it.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public Jun2dBoundingBoxTree(Jun2dPoint[] pointCollection) {
		this();
		this.rootNode_(new Jun2dBoundingBoxNode(pointCollection));
	}

	/**
	 * Create a new instance of <code>Jun2dBoundingBoxTree</code> and initialize it.
	 * 
	 * @param patchCollection jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category Instance creation
	 */
	public Jun2dBoundingBoxTree(Jun2dTriangle[] patchCollection) {
		this();
		this.rootNode_(new Jun2dBoundingBoxNode(patchCollection));
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		rootNode = null;
		subdividedNodes = null;
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return this.rootNode().boundingBox();
	}

	/**
	 * Answer the receiver's contained points.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry[]
	 * @see jp.co.sra.jun.geometry.boxtree.JunBoundingBoxTree#containedObjects()
	 * @category accessing
	 */
	public JunGeometry[] containedObjects() {
		return this.rootNode().containedObjects();
	}

	/**
	 * Answer the receiver's root node.
	 * 
	 * @return jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode
	 * @category accessing
	 */
	public Jun2dBoundingBoxNode rootNode() {
		return rootNode;
	}

	/**
	 * Set the receiver's root node.
	 * 
	 * param boundingNode jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode
	 * @category accessing
	 */
	public void rootNode_(Jun2dBoundingBoxNode boundingNode) {
		rootNode = boundingNode;
	}

	/**
	 * Answer the receiver's subdivided nodes.
	 * 
	 * @return jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode[]
	 * @category accessing
	 */
	public Jun2dBoundingBoxNode[] subdividedNodes() {
		if (subdividedNodes == null) {
			this.subdividedNodesLevel_(1);
		}
		return subdividedNodes;
	}

	/**
	 * Answer the subdivided nodes with the specified level value.
	 * 
	 * @param levelValue int
	 * @return jp.co.sra.jun.geometry.boxtree.Jun3dBoundingBoxNode[]
	 * @category accessing
	 */
	public Jun2dBoundingBoxNode[] subdividedNodesLevel_(int levelValue) {
		levelNumber = levelValue;
		subdividedNodes = this.rootNode().subdividedNodesLevel_(levelNumber);
		this.rootNode().flushSubdividedNodes();
		return subdividedNodes;
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#equal_(java.lang.Object)
	 * @category comparing
	 */
	public boolean equal_(Object anObject) {
		if (anObject == null || this.getClass() != anObject.getClass()) {
			return false;
		}
		return this.boundingBox().equal_(((Jun2dBoundingBoxTree) anObject).boundingBox());
	}

	/**
	 * Convert to a <code>Jun2dBoundingBox</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox
	 * @category converting
	 */
	public Jun2dBoundingBox asBoundingBox() {
		return this.boundingBox();
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		Jun2dBoundingBoxNode[] nodes = this.subdividedNodes();
		for (int i = 0; i < nodes.length; i++) {
			compoundObject.add_(nodes[i].asJunOpenGL3dObjectColor_alpha_(Color.getHSBColor(i / (float) nodes.length, 1, 1), 0.25f));
			Jun2dPoint[][] pps = nodes[i].boundingBox().pps();
			for (int j = 0; j < pps.length; j++) {
				Jun3dPoint[] vertexes = new Jun3dPoint[pps[j].length];
				for (int k = 0; k < vertexes.length; k++) {
					vertexes[k] = pps[j][k].as3dPoint();
				}
				JunOpenGL3dPolylineLoop polylineLoop = new JunOpenGL3dPolylineLoop(vertexes);
				polylineLoop.paint_(Color.gray);
				polylineLoop.lineWidth_(1);
				compoundObject.add_(polylineLoop);
			}
		}
		return compoundObject;
	}

	/**
	 * Answer true if the receiver is a 2d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}
}
