package jp.co.sra.jun.geometry.boxtree;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Frame;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInterval;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.forms.JunForm2dRegionTestExamples;
import jp.co.sra.jun.geometry.forms.JunFormTriangulation2;
import jp.co.sra.jun.geometry.surfaces.Jun2dTriangle;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.utilities.JunControlUtility;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex;

/**
 * Jun2dBoundingBoxNodeTestExamples class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/06/01 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun666 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dBoundingBoxNodeTestExamples.java,v 8.6 2008/02/20 06:30:56 nisinaka Exp $
 */
public class Jun2dBoundingBoxNodeTestExamples extends JunForm2dRegionTestExamples {
	/**
	 * Example1: Example point collection of 2D line.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example1() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExamplePointCollectionOf2dLine());
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 15));

		return true;
	}

	/**
	 * Example2: Example point collection of 2D sin curve.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example2() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExamplePointCollectionOf2dSinCurve());
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 15));

		return true;
	}

	/**
	 * Example3: Example point collection of 2D half ball.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example3() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExamplePointCollectionOf2dHalfBall());
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 15));

		return true;
	}

	/**
	 * Example4: Example segment collection of 2D line.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example4() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExampleSegmentCollectionOf2dLine());
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 15));

		return true;
	}

	/**
	 * Example5: Example segment collection of 2D sin curve.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example5() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExampleSegmentCollectionOf2dSinCurve());
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(2, 15));

		return true;
	}

	/**
	 * Example6: Example patch collection of 3D half ball.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example6() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExamplePatchCollectionOf2dHalfBall());
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 15));

		return true;
	}

	/**
	 * Example7: Example patch collection of 2D half ball.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example7() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExamplePatchCollectionOf2dHalfBall());
		boundingBoxNode.messageToSubdivide_($("subdivide"));
		boundingBoxNode.compareWithBlock_(new StBlockClosure() {
			public Object value_value_(Object object, Object box) {
				Jun2dTriangle geometry = (Jun2dTriangle) object;
				Jun2dBoundingBox boundingBox = (Jun2dBoundingBox) box;
				Set set = new HashSet();
				Jun2dTriangle[] triangles = geometry.detailedTrianglesLevel_(1);
				for (int i = 0; i < triangles.length; i++) {
					Jun2dPoint[] points = triangles[i].points();
					for (int j = 0; j < points.length; j++) {
						set.add(points[j]);
					}
				}
				Jun2dPoint[] points = (Jun2dPoint[]) set.toArray(new Jun2dPoint[set.size()]);
				boolean containsPoint = false;
				for (int i = 0; i < points.length; i++) {
					if (boundingBox.containsPoint_(points[i])) {
						containsPoint = true;
						break;
					}
				}
				return new Boolean(containsPoint);
			}
		});
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 5));

		return true;
	}

	/**
	 * Example8: Example patch collection of 3D polyline.
	 * 
	 * @return boolean
	 * @category Examples
	 */
	public static boolean Example8() {
		Jun2dBoundingBoxNode boundingBoxNode = new Jun2dBoundingBoxNode(ExamplePatchCollectionOf2dPolyline());
		boundingBoxNode.messageToSubdivide_($("subdivide"));
		Jun2dBoundingBoxNodeTestExamples.ExampleDisplay_levelInterval_(boundingBoxNode, new StInterval(0, 6));

		return true;
	}

	/**
	 * Example: Display the specified bounding box node with interval parameter.
	 * 
	 * @param boundingBoxNode jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode
	 * @param anInterval jp.co.sra.smalltalk.StInterval
	 * @return jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode
	 * @category Examples
	 */
	protected static Jun2dBoundingBoxNode ExampleDisplay_levelInterval_(Jun2dBoundingBoxNode boundingBoxNode, StInterval anInterval) {
		JunOpenGLDisplayModel displayModel = new JunOpenGLDisplayModel();
		displayModel.openIn_(new Rectangle(200, 100, 600 + 6, 400 + 26));
		((Frame) displayModel.getWindow()).setTitle("Bounding Box Node");
		for (int levelNumber = (int) Math.round(anInterval.start()); levelNumber <= anInterval.last(); levelNumber = levelNumber + (int) anInterval.step()) {
			Jun2dBoundingBoxNode[] subdividedNodes;
			JunCursors cursor = new JunCursors(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			try {
				cursor._show();
				subdividedNodes = boundingBoxNode.subdividedNodesLevel_(levelNumber);
			} finally {
				cursor._restore();
			}
			ExampleDisplay_with_with_with_(displayModel, boundingBoxNode, levelNumber, subdividedNodes);
		}
		return boundingBoxNode;
	}

	/**
	 * Example: Display a <code>JunOpenGLDisplayModel</code> with the specified bounding box node.
	 * 
	 * @param displayModel jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @param boundingBoxNode jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode
	 * @param levelNumber int
	 * @param subdividedNodes jp.co.sra.jun.geometry.boxtree.Jun2dBoundingBoxNode[]
	 * @category Examples
	 */
	protected static void ExampleDisplay_with_with_with_(final JunOpenGLDisplayModel displayModel, Jun2dBoundingBoxNode boundingBoxNode, final int levelNumber, final Jun2dBoundingBoxNode[] subdividedNodes) {
		final JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunOpenGL3dCompoundObject subObject = new JunOpenGL3dCompoundObject();
		JunGeometry[] objects = boundingBoxNode.containedObjects();
		for (int i = 0; i < objects.length; i++) {
			JunOpenGL3dObject body = null;
			if (objects[i].isPoint()) {
				JunOpenGL3dVertex vertext = (JunOpenGL3dVertex) objects[i].asJunOpenGL3dObjectColor_(Color.gray);
				vertext.size_(3);
				body = vertext;
			}
			if (objects[i].isLine()) {
				JunOpenGL3dCompoundObject compoundBody = new JunOpenGL3dCompoundObject();
				Jun2dLine object = (Jun2dLine) objects[i];
				JunOpenGL3dPolyline line = (JunOpenGL3dPolyline) object.asJunOpenGL3dObjectColor_(Color.black);
				line.lineWidth_(2);
				compoundBody = line.asCompoundObject();
				JunOpenGL3dVertex from = (JunOpenGL3dVertex) object.from().asJunOpenGL3dObjectColor_(Color.black);
				from.size_(3);
				compoundBody.add_(from);
				JunOpenGL3dVertex last = (JunOpenGL3dVertex) object.last().asJunOpenGL3dObjectColor_(Color.black);
				last.size_(3);
				compoundBody.add_(last);
				body = compoundBody;
			}
			if (objects[i].isTriangle()) {
				JunOpenGL3dCompoundObject compoundBody = new JunOpenGL3dCompoundObject();
				Jun2dTriangle object = (Jun2dTriangle) objects[i];
				Jun2dPoint[] a2dPoints = object.asPointArray();
				Jun3dPoint[] a3dPoints = new Jun3dPoint[a2dPoints.length];
				for (int j = 0; j < a2dPoints.length; j++) {
					a3dPoints[j] = a2dPoints[j].as3dPoint();
				}
				JunOpenGL3dPolyline polyline = new JunOpenGL3dPolyline(a3dPoints);
				polyline.paint_(Color.gray);
				polyline.lineWidth_(1);
				compoundBody.add_(polyline);
				JunOpenGL3dVertex p1 = (JunOpenGL3dVertex) object.p3().asJunOpenGL3dObjectColor_(Color.black);
				p1.size_(3);
				compoundBody.add_(p1);
				JunOpenGL3dVertex p2 = (JunOpenGL3dVertex) object.p3().asJunOpenGL3dObjectColor_(Color.black);
				p2.size_(3);
				compoundBody.add_(p2);
				JunOpenGL3dVertex p3 = (JunOpenGL3dVertex) object.p3().asJunOpenGL3dObjectColor_(Color.black);
				p3.size_(3);
				compoundBody.add_(p3);
				body = compoundBody;
			}
			if (body != null) {
				subObject.add_(body);
			}
		}

		compoundObject.add_(subObject);
		subObject = new JunOpenGL3dCompoundObject();
		for (int n = 0; n < subdividedNodes.length; n++) {
			subObject.add_(subdividedNodes[n].asJunOpenGL3dObject());
		}
		compoundObject.add_(subObject);

		Jun3dPoint point = boundingBoxNode.boundingBox().extent().as3dPoint();
		point = new Jun3dPoint(point.x(), point.y(), Math.max(point.x(), point.y()));
		subObject = (JunOpenGL3dCompoundObject) JunOpenGL3dObject.XyUnitArrows().scaledBy_(point);
		compoundObject.add_(subObject);

		JunControlUtility.Do_forMilliseconds_(new StBlockClosure() {
			public Object value() {
				displayModel.displayObject_(compoundObject);
				displayModel.resetView();
				System.out.print("level ");
				System.out.print(Integer.toString(levelNumber));
				System.out.print(" -> ");
				System.out.print(Integer.toString(subdividedNodes.length));
				System.out.println(subdividedNodes.length > 1 ? " boxes" : " box");
				return null;
			}
		}, 2000);
	}

	/**
	 * Answer the example patch collection of 2D half ball.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category Examples
	 */
	protected static Jun2dTriangle[] ExamplePatchCollectionOf2dHalfBall() {
		final Collection patchCollection = new ArrayList();
		JunOpenGL3dObject.Ball_(4).polygonsDo_(new StBlockClosure() {
			public Object value_(Object obj) {
				JunOpenGL3dPolygon polygon = (JunOpenGL3dPolygon) obj;
				Jun3dPoint[] vertexes = polygon.vertexes();
				Jun3dPoint p1 = vertexes[0];
				Jun3dPoint p2 = vertexes[1];
				Jun3dPoint p3 = vertexes[2];
				Jun3dTriangle triangle = p1.triangle_and_(p2, p3);
				if (triangle.centerOfIncircle().z() >= 0d) {
					Jun2dPoint _p1 = new Jun2dPoint(p1.x(), p1.y()).translatedBy_(new Jun2dPoint(1, 1));
					Jun2dPoint _p2 = new Jun2dPoint(p2.x(), p2.y()).translatedBy_(new Jun2dPoint(1, 1));
					Jun2dPoint _p3 = new Jun2dPoint(p3.x(), p3.y()).translatedBy_(new Jun2dPoint(1, 1));
					Jun2dTriangle _triangle = _p1.triangle_and_(_p2, _p3);
					patchCollection.add(_triangle);
				}
				return null;
			}
		});
		return (Jun2dTriangle[]) patchCollection.toArray(new Jun2dTriangle[patchCollection.size()]);
	}

	/**
	 * Answer the example patch collection of 2D polyline.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun2dTriangle[]
	 * @category Examples
	 */
	protected static Jun2dTriangle[] ExamplePatchCollectionOf2dPolyline() {
		final List patchCollection = new ArrayList();
		Jun2dTriangle[] triangles = new JunFormTriangulation2(JunForm2dRegionTestExamples.ExamplePolyline4()).triangles();
		for (int i = 0; i < triangles.length; i++) {
			Jun2dPoint p1 = triangles[i].p1();
			Jun2dPoint p2 = triangles[i].p2();
			Jun2dPoint p3 = triangles[i].p3();
			Jun2dTriangle triangle = new Jun2dPoint(p1.x(), p1.y()).triangle_and_(new Jun2dPoint(p2.x(), p2.y()), new Jun2dPoint(p3.x(), p3.y()));
			Jun2dTriangle[] detailedTriangles = triangle.detailedTrianglesLevel_(1);
			for (int j = 0; j < detailedTriangles.length; j++) {
				patchCollection.add(detailedTriangles[j]);
			}
		}
		Jun2dBoundingBox boundingBox = null;
		triangles = (Jun2dTriangle[]) patchCollection.toArray(new Jun2dTriangle[patchCollection.size()]);
		for (int i = 0; i < triangles.length; i++) {
			if (boundingBox == null) {
				boundingBox = triangles[i].boundingBox();
			} else {
				boundingBox = boundingBox.merge_(triangles[i].boundingBox());
			}
		}
		for (int i = 0; i < triangles.length; i++) {
			triangles[i] = triangles[i].translatedBy_(boundingBox.origin().negated());
		}
		return triangles;
	}

	/**
	 * Answer the example point collection of 2D half ball.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Examples
	 */
	protected static Jun2dPoint[] ExamplePointCollectionOf2dHalfBall() {
		Collection pointCollection = new ArrayList();
		Jun3dPoint[] ballPoints = JunOpenGL3dObject.Ball_(4).asPointArray();
		for (int i = 0; i < ballPoints.length; i++) {
			if (ballPoints[i].z() >= 0) {
				pointCollection.add(new Jun2dPoint(ballPoints[i].x(), ballPoints[i].y()).translatedBy_(new Jun2dPoint(1, 1)));
			}
		}
		return (Jun2dPoint[]) pointCollection.toArray(new Jun2dPoint[pointCollection.size()]);
	}

	/**
	 * Answer the example point collection of 2D line.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Examples
	 */
	protected static Jun2dPoint[] ExamplePointCollectionOf2dLine() {
		Jun2dLine aLine = new Jun2dPoint(0, 0).to_(new Jun2dPoint(3, 2));
		Collection pointCollection = new ArrayList();
		for (int i = 0; i <= 100; i++) {
			double t = 0.01 * i;
			pointCollection.add(aLine.atT_(t));
		}
		return (Jun2dPoint[]) pointCollection.toArray(new Jun2dPoint[pointCollection.size()]);
	}

	/**
	 * Answer the example point collection of 2D sin curve.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Examples
	 */
	protected static Jun2dPoint[] ExamplePointCollectionOf2dSinCurve() {
		int numberOfPoints = 501;
		Collection pointCollection = new ArrayList(numberOfPoints);
		for (int i = 0; i < numberOfPoints; i++) {
			double x = i / (double) numberOfPoints;
			double y = 0.4 * Math.sin(10 * 2 * Math.PI * x);
			pointCollection.add(new Jun3dPoint(x, y, 0));
		}

		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(pointCollection);
		aPolyline = (JunOpenGL3dPolyline) JunOpenGL3dObject.From_to_with_(new Jun3dPoint(0, 0, 0), new Jun3dPoint(3, 2, 1), aPolyline);
		Jun3dPoint[] a3dPoints = aPolyline.vertexes();
		Jun2dPoint[] a2dPoints = new Jun2dPoint[a3dPoints.length];
		for (int i = 0; i < a2dPoints.length; i++) {
			a2dPoints[i] = a3dPoints[i].as2dPoint();
		}
		return a2dPoints;
	}

	/**
	 * Answer the example segment collection of 2D line.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category Examples
	 */
	protected static Jun2dLine[] ExampleSegmentCollectionOf2dLine() {
		Jun2dLine aLine = new Jun2dPoint(0, 0).to_(new Jun2dPoint(3, 2));
		Collection pointCollection = new ArrayList();
		for (int i = 0; i <= 100; i++) {
			double t = 0.01 * i;
			pointCollection.add(aLine.atT_(t));
		}
		Collection segmentCollection = new ArrayList(pointCollection.size() - 1);
		Jun2dPoint[] points = (Jun2dPoint[]) pointCollection.toArray(new Jun2dPoint[pointCollection.size()]);
		for (int i = 0; i < points.length - 1; i++) {
			segmentCollection.add(points[i].to_(points[i + 1]));
		}
		return (Jun2dLine[]) segmentCollection.toArray(new Jun2dLine[segmentCollection.size()]);
	}

	/**
	 * Answer the example segment collection of 2D sin curve.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category Examples
	 */
	protected static Jun2dLine[] ExampleSegmentCollectionOf2dSinCurve() {
		int numberOfPoints = 501;
		Collection pointCollection = new ArrayList(numberOfPoints);
		for (int i = 0; i < numberOfPoints; i++) {
			double x = i / (double) numberOfPoints;
			double y = 0.4 * Math.sin(10 * 2 * Math.PI * x);
			pointCollection.add(new Jun3dPoint(x, y, 0));
		}

		JunOpenGL3dPolyline aPolyline = new JunOpenGL3dPolyline(pointCollection);
		aPolyline = (JunOpenGL3dPolyline) JunOpenGL3dObject.From_to_with_(new Jun3dPoint(0, 0, 0), new Jun3dPoint(3, 2, 1), aPolyline);
		Jun3dPoint[] vertexes = aPolyline.vertexes();
		Jun2dPoint[] pointArray = new Jun2dPoint[vertexes.length];
		for (int i = 0; i < pointArray.length; i++) {
			pointArray[i] = vertexes[i].as2dPoint();
		}
		Collection segmentCollection = new ArrayList(pointCollection.size() - 1);
		for (int i = 0; i < pointArray.length - 1; i++) {
			segmentCollection.add(pointArray[i].to_(pointArray[i + 1]));
		}
		return (Jun2dLine[]) segmentCollection.toArray(new Jun2dLine[segmentCollection.size()]);
	}

	/**
	 * Execute all examples.
	 * 
	 * @param args java.lang.String[]
	 * @category Main
	 */
	public static void main(String[] args) {
		new Jun2dBoundingBoxNodeTestExamples();
	}
}
