package jp.co.sra.jun.geometry.boundaries;

import java.awt.Point;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.TreeSet;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInterval;
import jp.co.sra.smalltalk.StRectangle;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.surfaces.Jun2dPolygon;
import jp.co.sra.jun.geometry.surfaces.Jun3dPolygon;
import jp.co.sra.jun.system.framework.JunAbstractObject;
import jp.co.sra.jun.topology.elements.JunBody;
import jp.co.sra.jun.topology.elements.JunLoop;
import jp.co.sra.jun.topology.elements.JunVertex;
import jp.co.sra.jun.topology.euleroperators.JunMEKL;
import jp.co.sra.jun.topology.euleroperators.JunMEL;
import jp.co.sra.jun.topology.euleroperators.JunMEV;
import jp.co.sra.jun.topology.euleroperators.JunMEVVL;

/**
 * JunBorderGenerator class
 * 
 *  @author    UNKNOWN
 *  @created   1999/01/08 (by Hirotsugu Kondo)
 *  @updated   2007/07/02 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun796 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunBorderGenerator.java,v 8.15 2008/02/20 06:30:55 nisinaka Exp $
 */
public class JunBorderGenerator extends JunAbstractObject {

	protected static Method[] MarchingCubes;

	static {
		InitializeMarchingCubes();
	}

	/**
	 * Initialize the marching cubes.
	 * 
	 * @category Class initialization
	 */
	protected static void InitializeMarchingCubes() {
		MarchingCubes = new Method[254];
		for (int i = 0; i < MarchingCubes.length; i++) {
			try {
				MarchingCubes[i] = JunBorderGenerator.class.getDeclaredMethod("MarchingCubes" + String.valueOf(i), new Class[] { Jun3dPoint.class, Number.class });
			} catch (NoSuchMethodException e) {
				System.err.println(e.getMessage());
				e.printStackTrace();
			}
		}
	}

	/**
	 * Create a Jun2dPolygonalBoundary with the specified information.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param minNumber double
	 * @param xInterval jp.co.sra.smalltalk.StInterval
	 * @param yInterval jp.co.sra.smalltalk.StInterval
	 * @param interimBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dPolygonalBoundary
	 * @category 2D functions
	 */
	public static Jun2dPolygonalBoundary PolygonalBoundaryWith_min_xInterval_yInterval_interim_(final StBlockClosure evaluationBlock, double minNumber, final StInterval xInterval, final StInterval yInterval, final StBlockClosure interimBlock) {
		final List danglingPolyedges = new ArrayList();
		int maxX = xInterval.size();
		int maxY = yInterval.size();
		final Jun2dPolygonalBoundary polygonalBoundary = new Jun2dPolygonalBoundary();
		final StBlockClosure interpolationBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunBorderVertex vertex = (JunBorderVertex) anObject;
				double x1 = xInterval.at_((int) vertex.vertex1().x());
				double y1 = yInterval.at_((int) vertex.vertex1().y());
				double x2 = xInterval.at_((int) vertex.vertex2().x());
				double y2 = yInterval.at_((int) vertex.vertex2().y());
				double value1 = ((Number) evaluationBlock.value_value_(new Double(x1), new Double(y1))).doubleValue();
				double value2 = ((Number) evaluationBlock.value_value_(new Double(x2), new Double(y2))).doubleValue();
				double t = 0;
				if (value1 != value2) {
					t = (vertex.label() - value1) / (value2 - value1);
				}
				t = Math.max(0, Math.min(t, 1));
				return new Double(t);
			}
		};
		final StBlockClosure completedBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				List aList = (List) anObject;
				JunBorderVertex[] polyedges = (JunBorderVertex[]) aList.toArray(new JunBorderVertex[aList.size()]);
				Jun2dPoint[] points = new Jun2dPoint[polyedges.length];
				for (int i = 0; i < polyedges.length; i++) {
					JunBorderVertex vertex = polyedges[i];
					double t = ((Number) interpolationBlock.value_(vertex)).doubleValue();
					Jun3dPoint v1 = vertex.vertex1();
					Jun3dPoint v2 = vertex.vertex2();
					Jun2dPoint p1 = new Jun2dPoint(xInterval.at_((int) v1.x()), yInterval.at_((int) v1.y()));
					Jun2dPoint p2 = new Jun2dPoint(xInterval.at_((int) v2.x()), yInterval.at_((int) v2.y()));
					points[i] = p2.minus_(p1).multipliedBy_(t).plus_(p1);
				}
				Jun2dPolygon polygon = new Jun2dPolygon(points);
				polygonalBoundary.add_(polygon);
				if (interimBlock != null) {
					interimBlock.value_(polygonalBoundary);
				}
				return null;
			}
		};
		StBlockClosure edgeInsertionBlock = new StBlockClosure() {
			public Object value_value_(Object arg0, Object arg1) {
				JunBorderVertex from = (JunBorderVertex) arg0;
				JunBorderVertex to = (JunBorderVertex) arg1;
				List leadingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (from.equals(polyedge.getLast())) {
						leadingPolyedge = polyedge;
						break;
					}
				}
				List trailingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (to.equals(polyedge.getFirst())) {
						trailingPolyedge = polyedge;
						break;
					}
				}

				if (leadingPolyedge == null) {
					if (trailingPolyedge == null) {
						LinkedList newPolyedge = new LinkedList();
						newPolyedge.add(from);
						newPolyedge.add(to);
						danglingPolyedges.add(newPolyedge);
					} else {
						trailingPolyedge.add(0, from);
					}
				} else {
					if (trailingPolyedge == null) {
						leadingPolyedge.add(to);
					} else {
						if (leadingPolyedge == trailingPolyedge) {
							danglingPolyedges.remove(leadingPolyedge);
							completedBlock.value_(leadingPolyedge);
						} else {
							leadingPolyedge.addAll(trailingPolyedge);
							danglingPolyedges.remove(trailingPolyedge);
						}
					}
				}
				return null;
			}
		};

		for (int xIndex = 1; xIndex < maxX; xIndex++) {
			for (int yIndex = 1; yIndex < maxY; yIndex++) {
				_EdgesWith_at_sign_inUnitSquareAt_do_(new StBlockClosure() {
					public Object value_value_(Object objX, Object objY) {
						int x = ((Number) objX).intValue();
						int y = ((Number) objY).intValue();
						return evaluationBlock.value_value_(new Double(xInterval.at_(x)), new Double(yInterval.at_(y)));
					}
				}, minNumber, 1, new Point(xIndex, yIndex), edgeInsertionBlock);
			}
		}

		List vertexesOnBorderX0 = new ArrayList();
		List vertexesOnBorderX1 = new ArrayList();
		List vertexesOnBorderY0 = new ArrayList();
		List vertexesOnBorderY1 = new ArrayList();
		for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
			LinkedList polyedge = (LinkedList) i.next();
			JunBorderVertex firstVertex = (JunBorderVertex) polyedge.getFirst();
			JunBorderVertex lastVertex = (JunBorderVertex) polyedge.getLast();

			if (firstVertex.onX_(1)) {
				double value = firstVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderX0.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onX_(maxX)) {
				double value = firstVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderX1.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onY_(1)) {
				double value = firstVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderY0.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onY_(maxY)) {
				double value = firstVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderY1.add(new Object[] { new Double(value), firstVertex });
			}
			if (lastVertex.onX_(1)) {
				double value = lastVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderX0.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onX_(maxX)) {
				double value = lastVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderX1.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onY_(1)) {
				double value = lastVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderY0.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onY_(maxY)) {
				double value = lastVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderY1.add(new Object[] { new Double(value), lastVertex });
			}
		}

		double cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.first()), new Double(yInterval.first()))).doubleValue();
		if (cornerValue >= minNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(1, 1, 0), new Jun3dPoint(1, 1, 0), 0);
			vertexesOnBorderX0.add(new Object[] { new Double(1), vertex });
			vertexesOnBorderY0.add(new Object[] { new Double(1), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.last()), new Double(yInterval.first()))).doubleValue();
		if (cornerValue >= minNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(maxX, 1, 0), new Jun3dPoint(maxX, 1, 0), 0);
			vertexesOnBorderX1.add(new Object[] { new Double(1), vertex });
			vertexesOnBorderY0.add(new Object[] { new Double(maxX), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.last()), new Double(yInterval.last()))).doubleValue();
		if (cornerValue >= minNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(maxX, maxY, 0), new Jun3dPoint(maxX, maxY, 0), 0);
			vertexesOnBorderX1.add(new Object[] { new Double(maxY), vertex });
			vertexesOnBorderY1.add(new Object[] { new Double(maxX), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.first()), new Double(yInterval.last()))).doubleValue();
		if (cornerValue >= minNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(1, maxY, 0), new Jun3dPoint(1, maxY, 0), 0);
			vertexesOnBorderX0.add(new Object[] { new Double(maxY), vertex });
			vertexesOnBorderY1.add(new Object[] { new Double(1), vertex });
		}

		ArrayList vertexesOnBorder = new ArrayList();
		Collections.sort(vertexesOnBorderY0, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessX = ((JunBorderVertex) less[1]).vertex1().x();
				double moreX = ((JunBorderVertex) more[1]).vertex1().x();
				return ((lessX < moreX) || (lessX == moreX && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderY0);

		Collections.sort(vertexesOnBorderX1, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessY = ((JunBorderVertex) less[1]).vertex1().y();
				double moreY = ((JunBorderVertex) more[1]).vertex1().y();
				return ((lessY < moreY) || (lessY == moreY && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderX1);

		Collections.sort(vertexesOnBorderY1, new Comparator() {
			public int compare(Object moreObj, Object lessObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessX = ((JunBorderVertex) less[1]).vertex1().x();
				double moreX = ((JunBorderVertex) more[1]).vertex1().x();
				return ((lessX < moreX) || (lessX == moreX && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderY1);

		Collections.sort(vertexesOnBorderX0, new Comparator() {
			public int compare(Object moreObj, Object lessObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessY = ((JunBorderVertex) less[1]).vertex1().y();
				double moreY = ((JunBorderVertex) more[1]).vertex1().y();
				return ((lessY < moreY) || (lessY == moreY && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderX0);

		int size = vertexesOnBorder.size();
		for (int i = 0; i < size; i += 2) {
			JunBorderVertex vertex1 = (JunBorderVertex) ((Object[]) vertexesOnBorder.get(i))[1];
			JunBorderVertex vertex2 = (JunBorderVertex) ((Object[]) vertexesOnBorder.get(i + 1))[1];
			edgeInsertionBlock.value_value_(vertex1, vertex2);
		}

		return polygonalBoundary;
	}

	/**
	 * Create a Jun2dPolygonalBoundary with the specified information.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param maxNumber double
	 * @param xInterval jp.co.sra.smalltalk.StInterval
	 * @param yInterval jp.co.sra.smalltalk.StInterval
	 * @param interimBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dPolygonalBoundary
	 * @category 2D functions
	 */
	public static Jun2dPolygonalBoundary PolygonalBoundaryWith_max_xInterval_yInterval_interim_(final StBlockClosure evaluationBlock, double maxNumber, final StInterval xInterval, final StInterval yInterval, final StBlockClosure interimBlock) {
		final List danglingPolyedges = new ArrayList();
		int maxX = xInterval.size();
		int maxY = yInterval.size();
		final Jun2dPolygonalBoundary polygonalBoundary = new Jun2dPolygonalBoundary();
		final StBlockClosure interpolationBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunBorderVertex vertex = (JunBorderVertex) anObject;
				double x1 = xInterval.at_((int) vertex.vertex1().x());
				double y1 = yInterval.at_((int) vertex.vertex1().y());
				double x2 = xInterval.at_((int) vertex.vertex2().x());
				double y2 = yInterval.at_((int) vertex.vertex2().y());
				double value1 = ((Number) evaluationBlock.value_value_(new Double(x1), new Double(y1))).doubleValue();
				double value2 = ((Number) evaluationBlock.value_value_(new Double(x2), new Double(y2))).doubleValue();
				double t = 0;
				if (value1 != value2) {
					t = (vertex.label() - value1) / (value2 - value1);
				}
				t = Math.max(0, Math.min(t, 1));
				return new Double(t);
			}
		};
		final StBlockClosure completedBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				List aList = (List) anObject;
				JunBorderVertex[] polyedges = (JunBorderVertex[]) aList.toArray(new JunBorderVertex[aList.size()]);
				Jun2dPoint[] points = new Jun2dPoint[polyedges.length];
				for (int i = 0; i < polyedges.length; i++) {
					JunBorderVertex vertex = polyedges[i];
					double t = ((Number) interpolationBlock.value_(vertex)).doubleValue();
					Jun3dPoint v1 = vertex.vertex1();
					Jun3dPoint v2 = vertex.vertex2();
					Jun2dPoint p1 = new Jun2dPoint(xInterval.at_((int) v1.x()), yInterval.at_((int) v1.y()));
					Jun2dPoint p2 = new Jun2dPoint(xInterval.at_((int) v2.x()), yInterval.at_((int) v2.y()));
					points[i] = p2.minus_(p1).multipliedBy_(t).plus_(p1);
				}
				Jun2dPolygon polygon = new Jun2dPolygon(points);
				polygonalBoundary.add_(polygon);
				if (interimBlock != null) {
					interimBlock.value_(polygonalBoundary);
				}
				return null;
			}
		};
		StBlockClosure edgeInsertionBlock = new StBlockClosure() {
			public Object value_value_(Object arg0, Object arg1) {
				JunBorderVertex from = (JunBorderVertex) arg0;
				JunBorderVertex to = (JunBorderVertex) arg1;
				List leadingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (from.equals(polyedge.getLast())) {
						leadingPolyedge = polyedge;
						break;
					}
				}
				List trailingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (to.equals(polyedge.getFirst())) {
						trailingPolyedge = polyedge;
						break;
					}
				}

				if (leadingPolyedge == null) {
					if (trailingPolyedge == null) {
						List newPolyedge = new LinkedList();
						newPolyedge.add(from);
						newPolyedge.add(to);
						danglingPolyedges.add(newPolyedge);
					} else {
						trailingPolyedge.add(0, from);
					}
				} else {
					if (trailingPolyedge == null) {
						leadingPolyedge.add(to);
					} else {
						if (leadingPolyedge == trailingPolyedge) {
							danglingPolyedges.remove(leadingPolyedge);
							completedBlock.value_(leadingPolyedge);
						} else {
							leadingPolyedge.addAll(trailingPolyedge);
							danglingPolyedges.remove(trailingPolyedge);
						}
					}
				}
				return null;
			}
		};

		for (int xIndex = 1; xIndex < maxX; xIndex++) {
			for (int yIndex = 1; yIndex < maxY; yIndex++) {
				_EdgesWith_at_sign_inUnitSquareAt_do_(new StBlockClosure() {
					public Object value_value_(Object objX, Object objY) {
						int x = ((Number) objX).intValue();
						int y = ((Number) objY).intValue();
						return evaluationBlock.value_value_(new Double(xInterval.at_(x)), new Double(yInterval.at_(y)));
					}
				}, maxNumber, -1, new Point(xIndex, yIndex), edgeInsertionBlock);
			}
		}

		List vertexesOnBorderX0 = new ArrayList();
		List vertexesOnBorderX1 = new ArrayList();
		List vertexesOnBorderY0 = new ArrayList();
		List vertexesOnBorderY1 = new ArrayList();
		for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
			LinkedList polyedge = (LinkedList) i.next();
			JunBorderVertex firstVertex = (JunBorderVertex) polyedge.getFirst();
			JunBorderVertex lastVertex = (JunBorderVertex) polyedge.getLast();

			if (firstVertex.onX_(1)) {
				double value = firstVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderX0.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onX_(maxX)) {
				double value = firstVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderX1.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onY_(1)) {
				double value = firstVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderY0.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onY_(maxY)) {
				double value = firstVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderY1.add(new Object[] { new Double(value), firstVertex });
			}
			if (lastVertex.onX_(1)) {
				double value = lastVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderX0.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onX_(maxX)) {
				double value = lastVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderX1.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onY_(1)) {
				double value = lastVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderY0.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onY_(maxY)) {
				double value = lastVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderY1.add(new Object[] { new Double(value), lastVertex });
			}
		}

		double cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.first()), new Double(yInterval.first()))).doubleValue();
		if (cornerValue <= maxNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(1, 1, 0), new Jun3dPoint(1, 1, 0), 0);
			vertexesOnBorderX0.add(new Object[] { new Double(1), vertex });
			vertexesOnBorderY0.add(new Object[] { new Double(1), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.last()), new Double(yInterval.first()))).doubleValue();
		if (cornerValue <= maxNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(maxX, 1, 0), new Jun3dPoint(maxX, 1, 0), 0);
			vertexesOnBorderX1.add(new Object[] { new Double(1), vertex });
			vertexesOnBorderY0.add(new Object[] { new Double(maxX), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.last()), new Double(yInterval.last()))).doubleValue();
		if (cornerValue <= maxNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(maxX, maxY, 0), new Jun3dPoint(maxX, maxY, 0), 0);
			vertexesOnBorderX1.add(new Object[] { new Double(maxY), vertex });
			vertexesOnBorderY1.add(new Object[] { new Double(maxX), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.first()), new Double(yInterval.last()))).doubleValue();
		if (cornerValue <= maxNumber) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(1, maxY, 0), new Jun3dPoint(1, maxY, 0), 0);
			vertexesOnBorderX0.add(new Object[] { new Double(maxY), vertex });
			vertexesOnBorderY1.add(new Object[] { new Double(1), vertex });
		}

		ArrayList vertexesOnBorder = new ArrayList();
		Collections.sort(vertexesOnBorderY0, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessX = ((JunBorderVertex) less[1]).vertex1().x();
				double moreX = ((JunBorderVertex) more[1]).vertex1().x();
				return ((lessX < moreX) || (lessX == moreX && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderY0);

		Collections.sort(vertexesOnBorderX1, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessY = ((JunBorderVertex) less[1]).vertex1().y();
				double moreY = ((JunBorderVertex) more[1]).vertex1().y();
				return ((lessY < moreY) || (lessY == moreY && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderX1);

		Collections.sort(vertexesOnBorderY1, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessX = ((JunBorderVertex) less[1]).vertex1().x();
				double moreX = ((JunBorderVertex) more[1]).vertex1().x();
				return ((lessX < moreX) || (lessX == moreX && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderY1);

		Collections.sort(vertexesOnBorderX0, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessY = ((JunBorderVertex) less[1]).vertex1().y();
				double moreY = ((JunBorderVertex) more[1]).vertex1().y();
				return ((lessY < moreY) || (lessY == moreY && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderX0);

		int size = vertexesOnBorder.size();
		for (int i = 0; i < size; i += 2) {
			JunBorderVertex vertex1 = (JunBorderVertex) ((Object[]) vertexesOnBorder.get(i))[1];
			JunBorderVertex vertex2 = (JunBorderVertex) ((Object[]) vertexesOnBorder.get(i + 1))[1];
			edgeInsertionBlock.value_value_(vertex1, vertex2);
		}

		return polygonalBoundary;
	}

	/**
	 * Create a Jun2dPolygonalBoundary with the specified information.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param minNumber double
	 * @param maxNumber double
	 * @param xInterval jp.co.sra.smalltalk.StInterval
	 * @param yInterval jp.co.sra.smalltalk.StInterval
	 * @param interimBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dPolygonalBoundary
	 * @category 2D functions
	 */
	public static Jun2dPolygonalBoundary PolygonalBoundaryWith_min_max_xInterval_yInterval_interim_(final StBlockClosure evaluationBlock, double minNumber, double maxNumber, final StInterval xInterval, final StInterval yInterval,
			final StBlockClosure interimBlock) {
		final List danglingPolyedges = new ArrayList();
		int maxX = xInterval.size();
		int maxY = yInterval.size();
		final Jun2dPolygonalBoundary polygonalBoundary = new Jun2dPolygonalBoundary();
		final StBlockClosure interpolationBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunBorderVertex vertex = (JunBorderVertex) anObject;
				double x1 = xInterval.at_((int) vertex.vertex1().x());
				double y1 = yInterval.at_((int) vertex.vertex1().y());
				double x2 = xInterval.at_((int) vertex.vertex2().x());
				double y2 = yInterval.at_((int) vertex.vertex2().y());
				double value1 = ((Number) evaluationBlock.value_value_(new Double(x1), new Double(y1))).doubleValue();
				double value2 = ((Number) evaluationBlock.value_value_(new Double(x2), new Double(y2))).doubleValue();
				double t = 0;
				if (value1 != value2) {
					t = (vertex.label() - value1) / (value2 - value1);
				}
				t = Math.max(0, Math.min(t, 1));
				return new Double(t);
			}
		};
		final StBlockClosure completedBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				List aList = (List) anObject;
				JunBorderVertex[] polyedges = (JunBorderVertex[]) aList.toArray(new JunBorderVertex[aList.size()]);
				Jun2dPoint[] points = new Jun2dPoint[polyedges.length];
				for (int i = 0; i < polyedges.length; i++) {
					JunBorderVertex vertex = polyedges[i];
					double t = ((Number) interpolationBlock.value_(vertex)).doubleValue();
					Jun3dPoint v1 = vertex.vertex1();
					Jun3dPoint v2 = vertex.vertex2();
					Jun2dPoint p1 = new Jun2dPoint(xInterval.at_((int) v1.x()), yInterval.at_((int) v1.y()));
					Jun2dPoint p2 = new Jun2dPoint(xInterval.at_((int) v2.x()), yInterval.at_((int) v2.y()));
					points[i] = p2.minus_(p1).multipliedBy_(t).plus_(p1);
				}
				Jun2dPolygon polygon = new Jun2dPolygon(points);
				polygonalBoundary.add_(polygon);
				if (interimBlock != null) {
					interimBlock.value_(polygonalBoundary);
				}
				return null;
			}
		};
		StBlockClosure edgeInsertionBlock = new StBlockClosure() {
			public Object value_value_(Object arg0, Object arg1) {
				JunBorderVertex from = (JunBorderVertex) arg0;
				JunBorderVertex to = (JunBorderVertex) arg1;
				List leadingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (from.equals(polyedge.getLast())) {
						leadingPolyedge = polyedge;
						break;
					}
				}
				List trailingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (to.equals(polyedge.getFirst())) {
						trailingPolyedge = polyedge;
						break;
					}
				}

				if (leadingPolyedge == null) {
					if (trailingPolyedge == null) {
						LinkedList newPolyedge = new LinkedList();
						newPolyedge.add(from);
						newPolyedge.add(to);
						danglingPolyedges.add(newPolyedge);
					} else {
						trailingPolyedge.add(0, from);
					}
				} else {
					if (trailingPolyedge == null) {
						leadingPolyedge.add(to);
					} else {
						if (leadingPolyedge == trailingPolyedge) {
							danglingPolyedges.remove(leadingPolyedge);
							completedBlock.value_(leadingPolyedge);
						} else {
							leadingPolyedge.addAll(trailingPolyedge);
							danglingPolyedges.remove(trailingPolyedge);
						}
					}
				}
				return null;
			}
		};

		for (int xIndex = 1; xIndex < maxX; xIndex++) {
			for (int yIndex = 1; yIndex < maxY; yIndex++) {
				_EdgesWith_at_sign_inUnitSquareAt_do_(new StBlockClosure() {
					public Object value_value_(Object objX, Object objY) {
						int x = ((Number) objX).intValue();
						int y = ((Number) objY).intValue();
						return evaluationBlock.value_value_(new Double(xInterval.at_(x)), new Double(yInterval.at_(y)));
					}
				}, minNumber, 1, new Point(xIndex, yIndex), edgeInsertionBlock);
				_EdgesWith_at_sign_inUnitSquareAt_do_(new StBlockClosure() {
					public Object value_value_(Object objX, Object objY) {
						int x = ((Number) objX).intValue();
						int y = ((Number) objY).intValue();
						return evaluationBlock.value_value_(new Double(xInterval.at_(x)), new Double(yInterval.at_(y)));
					}
				}, maxNumber, -1, new Point(xIndex, yIndex), edgeInsertionBlock);
			}
		}

		List vertexesOnBorderX0 = new ArrayList();
		List vertexesOnBorderX1 = new ArrayList();
		List vertexesOnBorderY0 = new ArrayList();
		List vertexesOnBorderY1 = new ArrayList();
		for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
			LinkedList polyedge = (LinkedList) i.next();
			JunBorderVertex firstVertex = (JunBorderVertex) polyedge.getFirst();
			JunBorderVertex lastVertex = (JunBorderVertex) polyedge.getLast();

			if (firstVertex.onX_(1)) {
				double value = firstVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderX0.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onX_(maxX)) {
				double value = firstVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderX1.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onY_(1)) {
				double value = firstVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderY0.add(new Object[] { new Double(value), firstVertex });
			}
			if (firstVertex.onY_(maxY)) {
				double value = firstVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(firstVertex)).doubleValue();
				vertexesOnBorderY1.add(new Object[] { new Double(value), firstVertex });
			}
			if (lastVertex.onX_(1)) {
				double value = lastVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderX0.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onX_(maxX)) {
				double value = lastVertex.vertex1().y();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderX1.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onY_(1)) {
				double value = lastVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderY0.add(new Object[] { new Double(value), lastVertex });
			}
			if (lastVertex.onY_(maxY)) {
				double value = lastVertex.vertex1().x();
				value += ((Number) interpolationBlock.value_(lastVertex)).doubleValue();
				vertexesOnBorderY1.add(new Object[] { new Double(value), lastVertex });
			}
		}

		double cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.first()), new Double(yInterval.first()))).doubleValue();
		if ((cornerValue >= minNumber) && (cornerValue <= maxNumber)) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(1, 1, 0), new Jun3dPoint(1, 1, 0), 0);
			vertexesOnBorderX0.add(new Object[] { new Double(1), vertex });
			vertexesOnBorderY0.add(new Object[] { new Double(1), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.last()), new Double(yInterval.first()))).doubleValue();
		if ((cornerValue >= minNumber) && (cornerValue <= maxNumber)) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(maxX, 1, 0), new Jun3dPoint(maxX, 1, 0), 0);
			vertexesOnBorderX1.add(new Object[] { new Double(1), vertex });
			vertexesOnBorderY0.add(new Object[] { new Double(maxX), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.last()), new Double(yInterval.last()))).doubleValue();
		if ((cornerValue >= minNumber) && (cornerValue <= maxNumber)) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(maxX, maxY, 0), new Jun3dPoint(maxX, maxY, 0), 0);
			vertexesOnBorderX1.add(new Object[] { new Double(maxY), vertex });
			vertexesOnBorderY1.add(new Object[] { new Double(maxX), vertex });
		}

		cornerValue = ((Number) evaluationBlock.value_value_(new Double(xInterval.first()), new Double(yInterval.last()))).doubleValue();
		if ((cornerValue >= minNumber) && (cornerValue <= maxNumber)) {
			JunBorderVertex vertex = new JunBorderVertex(new Jun3dPoint(1, maxY, 0), new Jun3dPoint(1, maxY, 0), 0);
			vertexesOnBorderX0.add(new Object[] { new Double(maxY), vertex });
			vertexesOnBorderY1.add(new Object[] { new Double(1), vertex });
		}

		ArrayList vertexesOnBorder = new ArrayList();
		Collections.sort(vertexesOnBorderY0, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessX = ((JunBorderVertex) less[1]).vertex1().x();
				double moreX = ((JunBorderVertex) more[1]).vertex1().x();
				return ((lessX < moreX) || (lessX == moreX && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderY0);

		Collections.sort(vertexesOnBorderX1, new Comparator() {
			public int compare(Object lessObj, Object moreObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessY = ((JunBorderVertex) less[1]).vertex1().y();
				double moreY = ((JunBorderVertex) more[1]).vertex1().y();
				return ((lessY < moreY) || (lessY == moreY && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderX1);

		Collections.sort(vertexesOnBorderY1, new Comparator() {
			public int compare(Object moreObj, Object lessObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessX = ((JunBorderVertex) less[1]).vertex1().x();
				double moreX = ((JunBorderVertex) more[1]).vertex1().x();
				return ((lessX < moreX) || (lessX == moreX && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderY1);

		Collections.sort(vertexesOnBorderX0, new Comparator() {
			public int compare(Object moreObj, Object lessObj) {
				Object[] less = (Object[]) lessObj;
				Object[] more = (Object[]) moreObj;
				double lessKey = ((Number) less[0]).doubleValue();
				double moreKey = ((Number) more[0]).doubleValue();
				double lessY = ((JunBorderVertex) less[1]).vertex1().y();
				double moreY = ((JunBorderVertex) more[1]).vertex1().y();
				return ((lessY < moreY) || (lessY == moreY && lessKey <= moreKey)) ? -1 : 1;
			}
		});
		vertexesOnBorder.addAll(vertexesOnBorderX0);

		int size = vertexesOnBorder.size();
		for (int i = 0; i < size; i += 2) {
			JunBorderVertex vertex1 = (JunBorderVertex) ((Object[]) vertexesOnBorder.get(i))[1];
			JunBorderVertex vertex2 = (JunBorderVertex) ((Object[]) vertexesOnBorder.get(i + 1))[1];
			edgeInsertionBlock.value_value_(vertex1, vertex2);
		}

		return polygonalBoundary;
	}

	/**
	 * Contour polylines.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 * @param xInterval jp.co.sra.smalltalk.StInterval
	 * @param yInterval jp.co.sra.smalltalk.StInterval
	 * @param interimBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[][]
	 * @category 2D functions
	 */
	public static Jun2dPoint[][] ContourPolylinesFrom_at_xInterval_yInterval_interim_(final StBlockClosure evaluationBlock, final double aNumber, final StInterval xInterval, final StInterval yInterval, final StBlockClosure interimBlock) {
		final List danglingPolyedges = new ArrayList();
		int maxX = xInterval.size();
		int maxY = yInterval.size();
		final StRectangle border = new StRectangle(1, 1, maxX, maxY);
		final StBlockClosure onBorderBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				LinkedList polyedges = (LinkedList) anObject;
				JunBorderVertex firstVertex = (JunBorderVertex) polyedges.getFirst();
				JunBorderVertex lastVertex = (JunBorderVertex) polyedges.getLast();
				return Boolean.valueOf(firstVertex.onBorderOfRectangle_(border) && lastVertex.onBorderOfRectangle_(border));
			}
		};
		final ArrayList polylines = new ArrayList();
		final StBlockClosure completedBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				List aList = (List) anObject;
				JunBorderVertex[] polyedges = (JunBorderVertex[]) aList.toArray(new JunBorderVertex[aList.size()]);
				Jun2dPoint[] polyline = new Jun2dPoint[polyedges.length];
				for (int i = 0; i < polyedges.length; i++) {
					JunBorderVertex vertex = polyedges[i];
					double x0 = xInterval.at_((int) vertex.vertex1().x());
					double x1 = xInterval.at_((int) vertex.vertex2().x());
					double y0 = yInterval.at_((int) vertex.vertex1().y());
					double y1 = yInterval.at_((int) vertex.vertex2().y());
					double value0 = ((Number) evaluationBlock.value_value_(new Double(x0), new Double(y0))).doubleValue();
					double value1 = ((Number) evaluationBlock.value_value_(new Double(x1), new Double(y1))).doubleValue();
					double interpolation = 0;
					if (value0 != value1) {
						interpolation = (vertex.label() - value0) / (value1 - value0);
					}
					Jun2dPoint aPoint = new Jun2dPoint(x1, y1);
					aPoint = aPoint.minus_(new Jun2dPoint(x0, y0));
					aPoint = aPoint.multipliedBy_(interpolation);
					aPoint = aPoint.plus_(new Jun2dPoint(x0, y0));
					polyline[i] = aPoint;
				}

				polylines.add(polyline);
				if (interimBlock != null) {
					interimBlock.value_(polylines);
				}
				return null;
			}
		};
		StBlockClosure edgeInsertionBlock = new StBlockClosure() {
			public Object value_value_(Object arg0, Object arg1) {
				JunBorderVertex from = (JunBorderVertex) arg0;
				JunBorderVertex to = (JunBorderVertex) arg1;
				List leadingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (from.equals(polyedge.getLast())) {
						leadingPolyedge = polyedge;
						break;
					}
				}
				List trailingPolyedge = null;
				for (Iterator i = danglingPolyedges.iterator(); i.hasNext();) {
					LinkedList polyedge = (LinkedList) i.next();
					if (to.equals(polyedge.getFirst())) {
						trailingPolyedge = polyedge;
						break;
					}
				}

				if (leadingPolyedge == null) {
					if (trailingPolyedge == null) {
						LinkedList newPolyedge = new LinkedList();
						newPolyedge.add(from);
						newPolyedge.add(to);

						if (onBorderBlock.value_(newPolyedge) == Boolean.TRUE) {
							completedBlock.value_(newPolyedge);
						} else {
							danglingPolyedges.add(newPolyedge);
						}
					} else {
						trailingPolyedge.add(0, from);
						if (onBorderBlock.value_(trailingPolyedge) == Boolean.TRUE) {
							danglingPolyedges.remove(trailingPolyedge);
							completedBlock.value_(trailingPolyedge);
						}
					}
				} else {
					if (trailingPolyedge == null) {
						leadingPolyedge.add(to);
						if (onBorderBlock.value_(leadingPolyedge) == Boolean.TRUE) {
							danglingPolyedges.remove(leadingPolyedge);
							completedBlock.value_(leadingPolyedge);
						}
					} else {
						if (leadingPolyedge == trailingPolyedge) {
							leadingPolyedge.add(to);
							danglingPolyedges.remove(leadingPolyedge);
							completedBlock.value_(leadingPolyedge);
						} else {
							leadingPolyedge.addAll(trailingPolyedge);
							danglingPolyedges.remove(trailingPolyedge);

							if (onBorderBlock.value_(leadingPolyedge) == Boolean.TRUE) {
								danglingPolyedges.remove(leadingPolyedge);
								completedBlock.value_(leadingPolyedge);
							}
						}
					}
				}
				return null;
			}
		};

		for (int xIndex = 1; xIndex < xInterval.size(); xIndex++) {
			for (int yIndex = 1; yIndex < yInterval.size(); yIndex++) {
				_EdgesWith_at_sign_inUnitSquareAt_do_(new StBlockClosure() {
					public Object value_value_(Object objX, Object objY) {
						int x = ((Number) objX).intValue();
						int y = ((Number) objY).intValue();
						return evaluationBlock.value_value_(new Double(xInterval.at_(x)), new Double(yInterval.at_(y)));
					}
				}, aNumber, 1, new Point(xIndex, yIndex), edgeInsertionBlock);
			}
		}

		return (Jun2dPoint[][]) polylines.toArray(new Jun2dPoint[polylines.size()][]);
	}

	/**
	 * Contour polygons.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 * @param aNumber2 int
	 * @param xInterval jp.co.sra.smalltalk.StInterval
	 * @param yInterval jp.co.sra.smalltalk.StInterval
	 * @param zInterval jp.co.sra.smalltalk.StInterval
	 * @param interimBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dPolygon[]
	 * @category 3D functions
	 */
	public static Jun3dPolygon[] ContourPolygonsFrom_at_sign_xInterval_yInterval_zInterval_interim_(final StBlockClosure evaluationBlock, double aNumber, int aNumber2, final StInterval xInterval, final StInterval yInterval, final StInterval zInterval,
			StBlockClosure interimBlock) {
		final ArrayList polygons = new ArrayList();
		for (int zIndex = 1; zIndex < zInterval.size(); zIndex++) {
			for (int yIndex = 1; yIndex < yInterval.size(); yIndex++) {
				for (int xIndex = 1; xIndex < xInterval.size(); xIndex++) {
					_TrianglesWith_at_sign_inUnitCubeAt_do_(new StBlockClosure() {
						public Object value_value_value_(Object x, Object y, Object z) {
							x = new Double(xInterval.at_(((Number) x).intValue()));
							y = new Double(yInterval.at_(((Number) y).intValue()));
							z = new Double(zInterval.at_(((Number) z).intValue()));
							return evaluationBlock.value_value_value_(x, y, z);
						}
					}, aNumber, aNumber2, new Jun3dPoint(xIndex, yIndex, zIndex), new StBlockClosure() {
						public Object value_(Object anObject) {
							JunBorderVertex[] triangle = (JunBorderVertex[]) anObject;
							ArrayList points = new ArrayList();
							for (int i = 0; i < triangle.length; i++) {
								Jun3dPoint vertex1 = triangle[i].vertex1();
								double x1 = xInterval.at_((int) Math.round(vertex1.x()));
								double y1 = yInterval.at_((int) Math.round(vertex1.y()));
								double z1 = zInterval.at_((int) Math.round(vertex1.z()));
								Jun3dPoint vertex2 = triangle[i].vertex2();
								double x2 = xInterval.at_((int) Math.round(vertex2.x()));
								double y2 = yInterval.at_((int) Math.round(vertex2.y()));
								double z2 = zInterval.at_((int) Math.round(vertex2.z()));
								double value1 = ((Number) evaluationBlock.value_value_value_(new Double(x1), new Double(y1), new Double(z1))).doubleValue();
								double value2 = ((Number) evaluationBlock.value_value_value_(new Double(x2), new Double(y2), new Double(z2))).doubleValue();
								double t = (value1 == value2) ? 0 : (triangle[i].label() - value1) / (value2 - value1);
								t = Math.max(0, Math.min(t, 1));
								Jun3dPoint p1 = new Jun3dPoint(x1, y1, z1);
								Jun3dPoint p2 = new Jun3dPoint(x2, y2, z2);
								points.add(p2.minus_(p1).multipliedBy_(t).plus_(p1));
							}
							Jun3dPolygon polygon = Jun3dPolygon.Points_((Jun3dPoint[]) points.toArray(new Jun3dPoint[points.size()]));
							if (polygon != null) {
								polygons.add(polygon);
							}
							return polygons;
						}
					});
				}
			}

			if (interimBlock != null) {
				interimBlock.value_(polygons);
			}
		}

		return (Jun3dPolygon[]) polygons.toArray(new Jun3dPolygon[polygons.size()]);
	}

	/**
	 * Contour body.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 * @param aNumber2 int
	 * @param xInterval jp.co.sra.smalltalk.StInterval
	 * @param yInterval jp.co.sra.smalltalk.StInterval
	 * @param zInterval jp.co.sra.smalltalk.StInterval
	 * @param interimBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunBody
	 * @category Solid modeling functions
	 */
	public static synchronized JunBody ContourBodyFrom_at_sign_xInterval_yInterval_zInterval_interim_(final StBlockClosure evaluationBlock, final double aNumber, int aNumber2, final StInterval xInterval, final StInterval yInterval,
			final StInterval zInterval, StBlockClosure interimBlock) {
		final JunBody body = new JunBody();
		final HashMap vertices = new HashMap();
		final ArrayList backloops = new ArrayList();
		final StBlockClosure interpolationBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunBorderVertex vertex = (JunBorderVertex) anObject;
				double x1 = xInterval.at_((int) Math.min(Math.max(Math.round(vertex.vertex1().x()), 1), xInterval.size() - 1));
				double y1 = yInterval.at_((int) Math.min(Math.max(Math.round(vertex.vertex1().y()), 1), yInterval.size() - 1));
				double z1 = zInterval.at_((int) Math.min(Math.max(Math.round(vertex.vertex1().z()), 1), zInterval.size() - 1));
				double x2 = xInterval.at_((int) Math.min(Math.max(Math.round(vertex.vertex2().x()), 1), xInterval.size() - 1));
				double y2 = yInterval.at_((int) Math.min(Math.max(Math.round(vertex.vertex2().y()), 1), yInterval.size() - 1));
				double z2 = zInterval.at_((int) Math.min(Math.max(Math.round(vertex.vertex2().z()), 1), zInterval.size() - 1));
				double value1 = ((Number) (evaluationBlock.value_value_value_(new Double(x1), new Double(y1), new Double(z1)))).doubleValue();
				double value2 = ((Number) (evaluationBlock.value_value_value_(new Double(x2), new Double(y2), new Double(z2)))).doubleValue();
				double t = (value1 == value2) ? 0 : (vertex.label() - value1) / (value2 - value1);
				t = Math.min(Math.max(t, 0), 1);
				Jun3dPoint p1 = new Jun3dPoint(x1, y1, z1);
				Jun3dPoint p2 = new Jun3dPoint(x2, y2, z2);
				return p2.minus_(p1).multipliedBy_(t).plus_(p1);
			}
		};
		final StBlockClosure edgeCreationBlock = new StBlockClosure() {
			public Object value_value_(Object arg1, Object arg2) {
				JunVertex vertex1 = (JunVertex) arg1;
				JunVertex vertex2 = (JunVertex) arg2;
				if (vertex1.hasEdgeToVertex_(vertex2) == false) {
					JunLoop loop = vertex1.commonLoopWithVertex_(vertex2);
					if (loop != null) {
						JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, vertex1, vertex2);
						mel.doOperation();
						backloops.add(mel.newLoop());
					} else {
						TreeSet aTreeSet = new TreeSet();
						JunLoop[] loops = vertex1.loops();
						for (int i = 0; i < loops.length; i++) {
							if (backloops.contains(loops[i])) {
								aTreeSet.add(loops[i]);
							}
						}
						if (aTreeSet.size() != 1) {
							throw SmalltalkException.Error("halt");
						}
						JunLoop loop1 = (JunLoop) aTreeSet.first();

						aTreeSet.clear();
						loops = vertex2.loops();
						for (int i = 0; i < loops.length; i++) {
							if (backloops.contains(loops[i])) {
								aTreeSet.add(loops[i]);
							}
						}
						if (aTreeSet.size() != 1) {
							throw SmalltalkException.Error("halt");
						}
						JunLoop loop2 = (JunLoop) aTreeSet.first();

						JunMEKL mekl = JunMEKL.Body_killLoop_aliveLoop_vertex_vertex_(body, loop1, loop2, vertex1, vertex2);
						mekl.doOperation();
						backloops.remove(loop1);
					}
				}
				return null;
			}
		};
		final StBlockClosure triangleCreationBlock0 = new StBlockClosure() {
			public Object valueWithArguments_(Object[] args) {
				JunBorderVertex point1 = (JunBorderVertex) args[0];
				JunBorderVertex point2 = (JunBorderVertex) args[1];
				JunBorderVertex point3 = (JunBorderVertex) args[2];
				JunMEVVL mevvl = JunMEVVL.Body_point_point_(body, (Jun3dPoint) interpolationBlock.value_(point1), (Jun3dPoint) interpolationBlock.value_(point2));
				mevvl.doOperation();
				JunVertex vertex1 = mevvl.newVertex1();
				vertices.put(point1, vertex1);
				JunVertex vertex2 = mevvl.newVertex2();
				vertices.put(point2, vertex2);
				JunLoop loop = mevvl.newLoop();
				JunMEV mev = JunMEV.Body_vertex_loop_point_(body, vertex2, loop, (Jun3dPoint) interpolationBlock.value_(point3));
				mev.doOperation();
				JunVertex vertex3 = mev.newVertex();
				vertices.put(point3, vertex3);
				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, vertex3, vertex1);
				mel.doOperation();
				backloops.add(vertex1.loopToVertex_(vertex3));
				return null;
			}
		};
		final StBlockClosure triangleCreationBlock1 = new StBlockClosure() {
			public Object value_value_value_(Object arg1, Object arg2, Object arg3) {
				JunVertex vertex1 = (JunVertex) arg1;
				JunBorderVertex point2 = (JunBorderVertex) arg2;
				JunBorderVertex point3 = (JunBorderVertex) arg3;
				JunLoop loop = vertex1.detectLoop_ifNone_(new StBlockClosure() {
					public Object value_(Object anObject) {
						return new Boolean(backloops.contains((JunLoop) anObject));
					}
				}, new StBlockClosure());
				JunMEV mev = JunMEV.Body_vertex_loop_point_(body, vertex1, loop, (Jun3dPoint) interpolationBlock.value_(point2));
				mev.doOperation();
				JunVertex vertex2 = mev.newVertex();
				vertices.put(point2, vertex2);
				mev = JunMEV.Body_vertex_loop_point_(body, vertex2, loop, (Jun3dPoint) interpolationBlock.value_(point3));
				mev.doOperation();
				JunVertex vertex3 = mev.newVertex();
				vertices.put(point3, vertex3);

				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, vertex3, vertex1);
				mel.doOperation();
				backloops.add(mel.newLoop());
				JunLoop foreloop = vertex1.loopToVertex_(vertex2);
				if (foreloop.numberOfVertexes() > 3) {
					throw SmalltalkException.Error("halt");
				}
				if (foreloop.includesVertex_(vertex1) == false) {
					throw SmalltalkException.Error("halt");
				}
				if (foreloop.includesVertex_(vertex2) == false) {
					throw SmalltalkException.Error("halt");
				}
				if (foreloop.includesVertex_(vertex3) == false) {
					throw SmalltalkException.Error("halt");
				}
				backloops.remove(foreloop);
				return null;
			}
		};
		final StBlockClosure triangleCreationBlock2 = new StBlockClosure() {
			public Object value_value_value_(Object arg1, Object arg2, Object arg3) {
				JunVertex vertex1 = (JunVertex) arg1;
				JunVertex vertex2 = (JunVertex) arg2;
				JunBorderVertex point3 = (JunBorderVertex) arg3;
				edgeCreationBlock.value_value_(vertex1, vertex2);
				JunLoop loop = vertex1.loopToVertex_(vertex2);
				JunMEV mev = JunMEV.Body_vertex_loop_point_(body, vertex2, loop, (Jun3dPoint) (interpolationBlock.value_(point3)));
				mev.doOperation();
				JunVertex vertex3 = mev.newVertex();
				vertices.put(point3, vertex3);
				JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, loop, vertex3, vertex1);
				mel.doOperation();
				backloops.add(mel.newLoop());
				JunLoop foreloop = vertex1.loopToVertex_(vertex2);
				if (foreloop.numberOfVertexes() > 3) {
					throw SmalltalkException.Error("halt");
				}
				backloops.remove(foreloop);
				return null;
			}
		};
		final StBlockClosure triangleCreationBlock3 = new StBlockClosure() {
			public Object value_value_value_(Object arg1, Object arg2, Object arg3) {
				JunVertex vertex1 = (JunVertex) arg1;
				JunVertex vertex2 = (JunVertex) arg2;
				JunVertex vertex3 = (JunVertex) arg3;
				edgeCreationBlock.value_value_(vertex1, vertex2);
				JunLoop foreloop = vertex1.loopToVertex_(vertex2);
				edgeCreationBlock.value_value_(vertex2, vertex3);
				foreloop = vertex1.loopToVertex_(vertex2);
				if (foreloop != vertex2.loopToVertex_(vertex3)) {
					throw SmalltalkException.Error("halt");
				}
				if (vertex3.hasEdgeToVertex_(vertex1) == false) {
					JunMEL mel = JunMEL.Body_loop_vertex_vertex_(body, foreloop, vertex3, vertex1);
					mel.doOperation();
					backloops.add(mel.newLoop());
				}
				foreloop = vertex1.loopToVertex_(vertex2);
				if (foreloop.numberOfVertexes() > 3) {
					throw SmalltalkException.Error("halt");
				}
				if (foreloop.includesVertex_(vertex1) == false) {
					throw SmalltalkException.Error("halt");
				}
				if (foreloop.includesVertex_(vertex2) == false) {
					throw SmalltalkException.Error("halt");
				}
				if (foreloop.includesVertex_(vertex3) == false) {
					throw SmalltalkException.Error("halt");
				}
				backloops.remove(foreloop);
				return null;
			}
		};
		final StBlockClosure triangleCreationBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				JunBorderVertex[] triangle = (JunBorderVertex[]) anObject;
				JunVertex vertex1 = (JunVertex) vertices.get(triangle[0]);
				JunVertex vertex2 = (JunVertex) vertices.get(triangle[1]);
				JunVertex vertex3 = (JunVertex) vertices.get(triangle[2]);
				if (vertex1 != null) {
					if (vertex2 != null) {
						if (vertex3 != null) {
							triangleCreationBlock3.value_value_value_(vertex1, vertex2, vertex3);
						} else {
							triangleCreationBlock2.value_value_value_(vertex1, vertex2, triangle[2]);
						}
					} else {
						if (vertex3 != null) {
							triangleCreationBlock2.value_value_value_(vertex3, vertex1, triangle[1]);
						} else {
							triangleCreationBlock1.value_value_value_(vertex1, triangle[1], triangle[2]);
						}
					}
				} else {
					if (vertex2 != null) {
						if (vertex3 != null) {
							triangleCreationBlock2.value_value_value_(vertex2, vertex3, triangle[0]);
						} else {
							triangleCreationBlock1.value_value_value_(vertex2, triangle[2], triangle[0]);
						}
					} else {
						if (vertex3 != null) {
							triangleCreationBlock1.value_value_value_(vertex3, triangle[0], triangle[1]);
						} else {
							triangleCreationBlock0.valueWithArguments_(triangle);
						}
					}
				}
				return null;
			}
		};

		for (int zIndex = 0; zIndex < zInterval.size(); zIndex++) {
			for (int yIndex = 0; yIndex < yInterval.size(); yIndex++) {
				for (int xIndex = 0; xIndex < xInterval.size(); xIndex++) {
					_TrianglesWith_at_sign_inUnitCubeAt_do_(new StBlockClosure() {
						public Object value_value_value_(Object argX, Object argY, Object argZ) {
							int x = ((Number) argX).intValue();
							int y = ((Number) argY).intValue();
							int z = ((Number) argZ).intValue();
							if (1 <= x && x < xInterval.size() && 1 <= y && y < yInterval.size() && 1 <= z && z <= zInterval.size()) {
								return evaluationBlock.value_value_value_(new Double(xInterval.at_(x)), new Double(yInterval.at_(y)), new Double(zInterval.at_(z)));
							} else {
								return new Double(aNumber);
							}
						}
					}, aNumber, aNumber2, new Jun3dPoint(xIndex, yIndex, zIndex), triangleCreationBlock);
				}
			}
			interimBlock.value_(body);
		}
		if (backloops.isEmpty() == false) {
			throw SmalltalkException.Error("halt");
		}

		return body;
	}

	/**
	 * Evaluate a block with the edges.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 * @param aNumber2 int
	 * @param aPoint java.awt.Point
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category Private
	 */
	protected static void _EdgesWith_at_sign_inUnitSquareAt_do_(StBlockClosure evaluationBlock, double aNumber, int aNumber2, Point aPoint, StBlockClosure aBlock) {
		int x0 = aPoint.x;
		int x1 = x0 + 1;
		int y0 = aPoint.y;
		int y1 = y0 + 1;
		int sign = (aNumber2 > 0) ? 1 : (aNumber2 < 0) ? (-1) : 0;
		int index = 0;
		double value = ((Number) evaluationBlock.value_value_(new Integer(x0), new Integer(y1))).doubleValue();
		if (((value - aNumber) * sign) >= 0) {
			index += 1;
		}
		value = ((Number) evaluationBlock.value_value_(new Integer(x1), new Integer(y1))).doubleValue();
		if (((value - aNumber) * sign) >= 0) {
			index += 2;
		}
		value = ((Number) evaluationBlock.value_value_(new Integer(x1), new Integer(y0))).doubleValue();
		if (((value - aNumber) * sign) >= 0) {
			index += 4;
		}
		value = ((Number) evaluationBlock.value_value_(new Integer(x0), new Integer(y0))).doubleValue();
		if (((value - aNumber) * sign) >= 0) {
			index += 8;
		}

		switch (index) {
			case 1:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				break;
			case 2:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				break;
			case 3:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				break;
			case 4:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber));
				break;
			case 5:
				if (true /* sign < 0 */) {
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber));
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				} else {
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber));
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				}
				break;
			case 6:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber));
				break;
			case 7:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber));
				break;
			case 8:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber));
				break;
			case 9:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				break;
			case 10:
				if (true /* sign < 0 */) {
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber));
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber));
				} else {
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber));
					aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				}
				break;
			case 11:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x1, y0, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				break;
			case 12:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber));
				break;
			case 13:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x1, y0, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber));
				break;
			case 14:
				aBlock.value_value_(new JunBorderVertex(new Jun3dPoint(x0, y1, 0), new Jun3dPoint(x1, y1, 0), aNumber), new JunBorderVertex(new Jun3dPoint(x0, y0, 0), new Jun3dPoint(x0, y1, 0), aNumber));
				break;
		}
	}

	/**
	 * Evaluate the block with the triangles.
	 * 
	 * @param evaluationBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param aNumber double
	 * @param aNumber2 int
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aBlock jp.co.sra.smalltalk.StBlockClocure
	 * @category Private
	 */
	protected static void _TrianglesWith_at_sign_inUnitCubeAt_do_(StBlockClosure evaluationBlock, double aNumber, int aNumber2, Jun3dPoint aPoint, StBlockClosure aBlock) {
		int x0 = (int) Math.round(aPoint.x());
		int x1 = x0 + 1;
		int y0 = (int) Math.round(aPoint.y());
		int y1 = y0 + 1;
		int z0 = (int) Math.round(aPoint.z());
		int z1 = z0 + 1;
		int sign = (aNumber2 < 0) ? -1 : (aNumber2 > 0) ? 1 : 0;
		int index = 0;

		if ((((Number) evaluationBlock.value_value_value_(new Integer(x0), new Integer(y0), new Integer(z0))).doubleValue() - aNumber) * sign > 0) {
			index += 1;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x1), new Integer(y0), new Integer(z0))).doubleValue() - aNumber) * sign > 0) {
			index += 2;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x1), new Integer(y1), new Integer(z0))).doubleValue() - aNumber) * sign > 0) {
			index += 4;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x0), new Integer(y1), new Integer(z0))).doubleValue() - aNumber) * sign > 0) {
			index += 8;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x0), new Integer(y0), new Integer(z1))).doubleValue() - aNumber) * sign > 0) {
			index += 16;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x1), new Integer(y0), new Integer(z1))).doubleValue() - aNumber) * sign > 0) {
			index += 32;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x1), new Integer(y1), new Integer(z1))).doubleValue() - aNumber) * sign > 0) {
			index += 64;
		}
		if ((((Number) evaluationBlock.value_value_value_(new Integer(x0), new Integer(y1), new Integer(z1))).doubleValue() - aNumber) * sign > 0) {
			index += 128;
		}

		if (index < 1 || 254 < index) {
			return;
		}

		JunBorderVertex[][] triangles = null;
		try {
			triangles = (JunBorderVertex[][]) MarchingCubes[index - 1].invoke(null, new Object[] { aPoint, new Double(aNumber) });
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InvocationTargetException e) {
			e.printStackTrace();
		}

		for (int i = 0; i < triangles.length; i++) {
			aBlock.value_(triangles[i]);
		}
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes0(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes1(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes2(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes3(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes4(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes5(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes6(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes7(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes8(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes9(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes10(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes11(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes12(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes13(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes14(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes15(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes16(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes17(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes18(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes19(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes20(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes21(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes22(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes23(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes24(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes25(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes26(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes27(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes28(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes29(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes30(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes31(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes32(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes33(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes34(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes35(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes36(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes37(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes38(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes39(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes40(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes41(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes42(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes43(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes44(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes45(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes46(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes47(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes48(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes49(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes50(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes51(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes52(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes53(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes54(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes55(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes56(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes57(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes58(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes59(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes60(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes61(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes62(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes63(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes64(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes65(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes66(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes67(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes68(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes69(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes70(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes71(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes72(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes73(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes74(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes75(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes76(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes77(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes78(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes79(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes80(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes81(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes82(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes83(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes84(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes85(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes86(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes87(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes88(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes89(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes90(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes91(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes92(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes93(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes94(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes95(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes96(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes97(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes98(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes99(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes100(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes101(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes102(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes103(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes104(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes105(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes106(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes107(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes108(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes109(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes110(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes111(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes112(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes113(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes114(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes115(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes116(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes117(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes118(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes119(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes120(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes121(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes122(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes123(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes124(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes125(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes126(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes127(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes128(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes129(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes130(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes131(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes132(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes133(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes134(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes135(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes136(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes137(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes138(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes139(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes140(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes141(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes142(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes143(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes144(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes145(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes146(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes147(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes148(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes149(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes150(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes151(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes152(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes153(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes154(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes155(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes156(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes157(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes158(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes159(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes160(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes161(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes162(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes163(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes164(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes165(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes166(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes167(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes168(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes169(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes170(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes171(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes172(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes173(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes174(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes175(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes176(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes177(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes178(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes179(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes180(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes181(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes182(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes183(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes184(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes185(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes186(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes187(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes188(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes189(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes190(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes191(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes192(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes193(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[5][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[4][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[4][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[4][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes194(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes195(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes196(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes197(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes198(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes199(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes200(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes201(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes202(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes203(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes204(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes205(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes206(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes207(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes208(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes209(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes210(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes211(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes212(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes213(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes214(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes215(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes216(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes217(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes218(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes219(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes220(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes221(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes222(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes223(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes224(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes225(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes226(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes227(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes228(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes229(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes230(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes231(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[4][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[3][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[3][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes232(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes233(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes234(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes235(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes236(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes237(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes238(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 1)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes239(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes240(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes241(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes242(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes243(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes244(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes245(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes246(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes247(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[3][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[2][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[2][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes248(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes249(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes250(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 1, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes251(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[2][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[1][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		triangles[1][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes252(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 1, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(1, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 1)), label);
		return triangles;
	}

	/**
	 * A definition of a function of marching cube method.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aNumber java.lang.Number
	 * @return jp.co.sra.jun.geometry.boundaries.JunBorderVertex[][]
	 * @category Private - marching cubes
	 */
	protected static JunBorderVertex[][] MarchingCubes253(Jun3dPoint aPoint, Number aNumber) {
		double label = aNumber.doubleValue();
		JunBorderVertex[][] triangles = new JunBorderVertex[1][3];
		triangles[0][0] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 1, 0)), label);
		triangles[0][1] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(1, 0, 0)), label);
		triangles[0][2] = JunBorderVertex.Vertex_vertex_label_(aPoint.plus_(new Jun3dPoint(0, 0, 0)), aPoint.plus_(new Jun3dPoint(0, 0, 1)), label);
		return triangles;
	}

}
