package jp.co.sra.jun.geometry.boundaries;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.abstracts.JunGeometry;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls;
import jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes;
import jp.co.sra.jun.geometry.transformations.JunTransformation;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;

/**
 * Jun2dBoundingBall class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/01/24 (by NISHIHARA Satoshi)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @updated   2006/10/10 (by nisinaka)
 *  @updated   2007/05/29 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun666 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dBoundingBall.java,v 8.16 2008/02/20 06:30:55 nisinaka Exp $
 */
public class Jun2dBoundingBall extends JunBoundingBall {

	/**
	 * Create a new Jun2dBoundingBall and initialize it.
	 * 
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param radiusValue double
	 * @category Instance creation
	 */
	public Jun2dBoundingBall(Jun2dPoint centerPoint, double radiusValue) {
		this.center_(centerPoint);
		this.radius_(radiusValue);
	}

	/**
	 * Create a new instance of Jun2dBoundingBall from the point collection.
	 * 
	 * @param pointCollection jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @category Instance creation
	 */
	public Jun2dBoundingBall(Jun2dPoint[] pointCollection) {
		if (pointCollection == null || pointCollection.length == 0) {
			return;
		}

		double x = 0;
		double y = 0;
		for (int i = 0; i < pointCollection.length; i++) {
			x += pointCollection[i].x();
			y += pointCollection[i].y();
		}
		Jun2dPoint centerPoint = new Jun2dPoint(x / pointCollection.length, y / pointCollection.length);

		double maxDistance = 0;
		for (int i = 0; i < pointCollection.length; i++) {
			double distance = centerPoint.distance_(pointCollection[i]);
			if (distance > maxDistance) {
				maxDistance = distance;
			}
		}

		this.center_(centerPoint);
		this.radius_(maxDistance);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		center = Jun2dPoint.Zero();
	}

	/**
	 * Answer my center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint center() {
		return (Jun2dPoint) this._center();
	}

	/**
	 * Set my center point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void center_(Jun2dPoint aPoint) {
		this._center(aPoint);
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#area()
	 * @category accessing
	 */
	public double area() {
		return Math.PI * (Math.pow(this.radius(), 2));
	}

	/**
	 * Answer my depth.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBall#depth()
	 * @category accessing
	 */
	public double depth() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer the bounding ball.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category accessing
	 */
	public Jun2dBoundingBall boundingBall() {
		return this;
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.JunBoundingBox
	 * @category accessing
	 */
	public Jun2dBoundingBox boundingBox() {
		return Jun2dBoundingBox.Origin_extent_(this.center(), Jun2dPoint.Zero()).expandedBy_(this.radius());
	}

	/**
	 * Answer my boudning box.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.JunBoundingBox
	 * @see jp.co.sra.jun.geometry.boundaries.JunBoundingBall#_boundingBox()
	 * @category accessing
	 */
	protected JunBoundingBox _boundingBox() {
		return this.boundingBox();
	}

	/**
	 * Answer the receiver's detailed bounding balls.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls
	 * @category accessing 
	 */
	public Jun2dBoundingBalls detailedBoundingBalls() {
		return this.tetraBoundingBalls();
	}

	/**
	 * Answer the receiver's detailed bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category accessing 
	 */
	public Jun2dBoundingBoxes detailedBoundingBoxes() {
		return this.tetraBoundingBoxes();
	}

	/**
	 * Answer the receiver's tetra bounding balls.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBalls
	 * @category accessing 
	 */
	public Jun2dBoundingBalls tetraBoundingBalls() {
		Jun2dBoundingBalls boundingBalls = new Jun2dBoundingBalls();
		boundingBalls.addAll_(this.boundingBox().tetraBoundingBalls());
		return boundingBalls;
	}

	/**
	 * Answer the receiver's tetra bounding boxes.
	 * 
	 * @return jp.co.sra.jun.geometry.pluralities.Jun2dBoundingBoxes
	 * @category accessing 
	 */
	public Jun2dBoundingBoxes tetraBoundingBoxes() {
		Jun2dBoundingBoxes boundingBoxes = new Jun2dBoundingBoxes();
		boundingBoxes.addAll_(this.boundingBox().tetraBoundingBoxes());
		return boundingBoxes;
	}

	/**
	 * Add a point to the receiver.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category adding
	 */
	public void add_(Jun2dPoint aPoint) {
		if (this.center().distance_(aPoint) > this.radius()) {
			Jun2dBoundingBall ball = this.merge_(new Jun2dBoundingBall(aPoint, 0));
			this.center_(ball.center());
			this.radius_(ball.radius());
		}
	}

	/**
	 * Convert to a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		JunOpenGL3dCompoundObject compoundObject = new JunOpenGL3dCompoundObject();
		JunOpenGL3dObject aBall = JunOpenGL3dObject.PieFrom_to_by_radius_(0, 360, 10, this.radius()).translatedBy_(new Jun3dPoint(this.center().x(), this.center().y(), 0));
		compoundObject.add_(aBall.reversed());
		compoundObject.add_(aBall);
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunOpenGL3dObject) each).paint_alpha_(Jun2dBoundingBall.this.defaultColor(), Jun2dBoundingBall.this.defaultAlpha());
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Expand the receiver by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category functions
	 */
	public Jun2dBoundingBall expandedBy_(double aNumber) {
		return new Jun2dBoundingBall(this.center(), this.radius() + aNumber);
	}

	/**
	 * Inset the receiver by the specified amount.
	 * 
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category functions
	 */
	public Jun2dBoundingBall insetBy_(double aNumber) {
		return new Jun2dBoundingBall(this.center(), this.radius() - aNumber);
	}

	/**
	 * Merge the receiver with the specified bounding ball.
	 * 
	 * @param aBoundingBall jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category functions
	 */
	public Jun2dBoundingBall merge_(Jun2dBoundingBall aBoundingBall) {
		double distance = this.center().distance_(aBoundingBall.center());
		if (distance + aBoundingBall.radius() < this.radius()) {
			return (Jun2dBoundingBall) this.copy();
		}
		if (distance + this.radius() < aBoundingBall.radius()) {
			return (Jun2dBoundingBall) aBoundingBall.copy();
		}

		Jun2dLine aLine = new Jun2dLine(this.center(), aBoundingBall.center());
		Jun2dPoint fromPoint = aLine.atT_(0 - this.radius() / distance);
		Jun2dPoint toPoint = aLine.atT_(1 + aBoundingBall.radius() / distance);
		return new Jun2dBoundingBall(fromPoint.center_(toPoint), fromPoint.distance_(toPoint) / 2);
	}

	/**
	 * Answer true if the receiver contains the specified bounding ball, otherwise false.
	 * 
	 * @param aBoundingBall jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @return boolean
	 * @category testing
	 */
	public boolean contains_(Jun2dBoundingBall aBoundingBall) {
		double delta = this.center().distance_(aBoundingBall.center());
		return delta <= this.radius() && delta + aBoundingBall.radius() <= this.radius();
	}

	/**
	 * Answer true if the receiver contains the point, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Jun2dPoint aPoint) {
		return this.center().distance_(aPoint) <= this.radius() || this.touchesPoint_(aPoint);
	}

	/**
	 * Answer true if the receiver intersects with the bounding ball, otherwise false.
	 * 
	 * @param aBoundingBall jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @return boolean
	 * @category testing
	 */
	public boolean intersects_(Jun2dBoundingBall aBoundingBall) {
		return this.center().distance_(aBoundingBall.center()) <= this.radius() + aBoundingBall.radius();
	}

	/**
	 * Answer true if the receiver touches the point, otherwise false.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return boolean
	 * @category testing
	 */
	public boolean touchesPoint_(Jun2dPoint aPoint) {
		double distance = this.center().distance_(aPoint);
		return this.isEqualNumber_to_(distance, this.radius());
	}

	/**
	 * Answer true if the receiver is a 2d geometry element, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#is2d()
	 * @category testing
	 */
	public boolean is2d() {
		return true;
	}

	/**
	 * Answer <code>true</code> if the receiver touches with the specified bounding ball, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @param aBoundingBox jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category testing
	 */
	public boolean touches_(Jun2dBoundingBall aBoundingBall) {
		if (this.isEqualPoint_to_(this.center(), aBoundingBall.center()) && this.isEqualNumber_to_(this.radius(), aBoundingBall.radius())) {
			return true;
		}
		double distance = this.center().distance_(aBoundingBall.center());
		if (distance <= this.radius()) {
			return this.isEqualNumber_to_(this.radius(), distance + aBoundingBall.radius());
		}
		if (distance <= aBoundingBall.radius()) {
			return this.isEqualNumber_to_(aBoundingBall.radius(), distance + this.radius());
		}
		return this.isEqualNumber_to_(distance, this.radius() + aBoundingBall.radius());
	}

	/**
	 * Answer the copy of the receiver which is applied the specified transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @see jp.co.sra.jun.geometry.abstracts.JunGeometry#transform_(jp.co.sra.jun.geometry.transformations.JunTransformation)
	 * @category transforming
	 */
	public JunGeometry transform_(JunTransformation aTransformation) {
		Jun2dLine aLine = new Jun2dLine(this.center(), this.center().plus_(new Jun2dPoint(this.radius(), 0)));
		aLine = (Jun2dLine) aLine.transform_(aTransformation);
		return new Jun2dBoundingBall(aLine.from(), aLine.from().distance_(aLine.to()));
	}

	/**
	 * Answer a rounded copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category truncation and round off
	 */
	public Jun2dBoundingBall rounded() {
		return new Jun2dBoundingBall(this.center().rounded(), Math.round(this.radius()));
	}

	/**
	 * Answer a truncated copy of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBall
	 * @category truncation and round off
	 */
	public Jun2dBoundingBall truncated() {
		double newRadius = (this.radius() >= 0) ? Math.ceil(this.radius()) : Math.floor(this.radius());
		return new Jun2dBoundingBall(this.center().truncated(), newRadius);
	}

}
