package jp.co.sra.jun.geometry.abstracts;

import java.awt.Color;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.coordinate.JunCoordinateSystem;
import jp.co.sra.jun.geometry.coordinate.JunLocalGeometry;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.Jun3dTriangle;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.geometry.transformations.JunTransformation;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolyline;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunGeometry class
 * 
 *  @author    nisinaka
 *  @created   1998/09/29 (by nisinaka)
 *  @updated   2000/01/06 (by nisinaka)
 *  @updated   2004/10/18 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGeometry.java,v 8.29 2008/02/20 06:30:54 nisinaka Exp $
 */
public abstract class JunGeometry extends JunAbstractObject {

	public static final double ACCURACY = 1.0e-12d;
	public static final double DoublePi = Math.PI * 2;
	public static final double HalfPi = Math.PI / 2;
	public static final double QuarterPi = Math.PI / 4;

	/**
	 * Answer the local geometry class.
	 * 
	 * @return java.lang.Class
	 * @category Class accessing
	 */
	private static Class LocalGeometryClass() {
		return JunLocalGeometry.class;
	}

	/**
	 * Answer the accuracy of geometries.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static final double Accuracy() {
		return ACCURACY;
	}

	/**
	 * Answer the double pi value.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static final double DoublePi() {
		return DoublePi;
	}

	/**
	 * Answer the half pi value.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static final double HalfPi() {
		return HalfPi;
	}

	/**
	 * Answer the pi value.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static final double Pi() {
		return Math.PI;
	}

	/**
	 * Answer the quarter pi value.
	 * 
	 * @return double
	 * @category Constants access
	 */
	public static final double QuarterPi() {
		return QuarterPi;
	}

	/**
	 * Answer the hemisphere point of the ground point with the center point on the X-Y plane.
	 * 
	 * @param groundPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Utilities
	 */
	public static Jun3dPoint HemispherePoint_center_(Jun3dPoint groundPoint, Jun3dPoint centerPoint) {
		return HemispherePoint_center_on_(groundPoint, centerPoint, new JunPlane(new Jun3dPoint(0, 0, 0), new Jun3dPoint(1, 0, 0), new Jun3dPoint(0, 1, 0)));
	}

	/**
	 * Answer the hemisphere point of the ground point with the center point on the ground plane.
	 * 
	 * @param groundPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param centerPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param groundPlane jp.co.sra.jun.geometry.surface.JunPlane
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Utilities
	 */
	public static Jun3dPoint HemispherePoint_center_on_(Jun3dPoint groundPoint, Jun3dPoint centerPoint, JunPlane groundPlane) {
		return centerPoint.to_(groundPlane.nearestPointFromPoint_(groundPoint)).normalized().atT_(groundPlane.distanceFromPoint_(centerPoint));
	}

	/**
	 * Answer the mobius point with the specified mobius radius, theta, strip radius and radius.
	 * 
	 * @param torusRadius double
	 * @param normalizedTheta double
	 * @param tubeRadius double
	 * @param normalizedPhi double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Utilities
	 */
	public static Jun3dPoint MobiusRadius_normalizedTheta_stripRadius_normalizedRadius_(double mobiusRadius, double normalizedTheta, double stripRadius, double normalizedRadius) {
		JunAngle theta = JunAngle.FromRad_(JunGeometry.DoublePi() * normalizedTheta);
		JunAngle phi = theta.multipliedBy_(1.0 / 2.0);
		double x = (mobiusRadius + (stripRadius * phi.cos())) * theta.cos();
		double y = (mobiusRadius + (stripRadius * phi.cos())) * theta.sin();
		double z = stripRadius * phi.sin();
		if (Math.abs(x) < Accuracy()) {
			x = 0.0d;
		}
		if (Math.abs(y) < Accuracy()) {
			y = 0.0d;
		}
		if (Math.abs(z) < Accuracy()) {
			z = 0.0d;
		}
		Jun3dPoint point = new Jun3dPoint(x, y, z);
		x = mobiusRadius * theta.cos();
		y = mobiusRadius * theta.sin();
		if (Math.abs(x) < Accuracy()) {
			x = 0.0d;
		}
		if (Math.abs(y) < Accuracy()) {
			y = 0.0d;
		}
		Jun3dPoint center = new Jun3dPoint(x, y, 0.0d);
		return center.to_(point).atT_(normalizedRadius);
	}

	/**
	 * Sew the specified points with another points with reverse flag.
	 * 
	 * @param firstPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param secondPoints jp.co.sra.jun.geometry.basic.Jun3dPoint[]
	 * @param aBoolean boolean
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category Utilities
	 */
	public static Jun3dPoint[][] SewPoints_withPoints_reverseFlag_(Jun3dPoint[] firstPoints, Jun3dPoint[] secondPoints, boolean aBoolean) {
		ArrayList aCollection = new ArrayList((firstPoints.length - 1) * 2);
		for (int i = 0; i < firstPoints.length - 1; i++) {
			int firstIndex = i;
			int secondIndex = i + 1;
			Jun3dPoint firstPoint = firstPoints[firstIndex];
			Jun3dPoint secondPoint = secondPoints[firstIndex];
			Jun3dPoint thirdPoint = firstPoints[secondIndex];
			Jun3dPoint forthPoint = secondPoints[secondIndex];
			Jun3dPoint[] firstTriPoints = new Jun3dPoint[] { firstPoint, secondPoint, thirdPoint };
			Jun3dPoint[] secondTriPoints = new Jun3dPoint[] { secondPoint, forthPoint, thirdPoint };
			if (aBoolean == false) {
				firstTriPoints = new Jun3dPoint[] { thirdPoint, secondPoint, firstPoint };
				secondTriPoints = new Jun3dPoint[] { thirdPoint, forthPoint, secondPoint };
			}
			aCollection.add(firstTriPoints);
			aCollection.add(secondTriPoints);
		}
		return (Jun3dPoint[][]) aCollection.toArray(new Jun3dPoint[aCollection.size()][]);
	}

	/**
	 * Answer the torus point with the specified torus radius, theta, tube radius and phi.
	 * 
	 * @param torusRadius double
	 * @param normalizedTheta double
	 * @param tubeRadius double
	 * @param normalizedPhi double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Utilities
	 */
	public static Jun3dPoint TorusRadius_normalizedTheta_tubeRadius_normalizedPhi_(double torusRadius, double normalizedTheta, double tubeRadius, double normalizedPhi) {
		JunAngle theta = JunAngle.FromRad_(JunGeometry.DoublePi() * normalizedTheta);
		JunAngle phi = JunAngle.FromRad_(JunGeometry.DoublePi() * normalizedPhi);
		double x = (torusRadius + (tubeRadius * phi.cos())) * theta.cos();
		double y = (torusRadius + (tubeRadius * phi.cos())) * theta.sin();
		double z = tubeRadius * phi.sin();
		if (Math.abs(x) < Accuracy()) {
			x = 0.0d;
		}
		if (Math.abs(y) < Accuracy()) {
			y = 0.0d;
		}
		if (Math.abs(z) < Accuracy()) {
			z = 0.0d;
		}
		Jun3dPoint point = new Jun3dPoint(x, y, z);
		return point;
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
	}

	/**
	 * Answer my area.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double area() {
		return 0.0d;
	}

	/**
	 * Answer my volume.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double volume() {
		return 0.0d;
	}

	/**
	 * Answer the accuracy value of the receiver.
	 * 
	 * @return double
	 * @category constants access
	 */
	public double accuracy() {
		return Accuracy();
	}

	/**
	 * Answer <code>true</code> if the receiver is equal to the object while concerning an accuracy.
	 * 
	 * @param anObject java.lang.Object
	 * @return boolean
	 * @category comparing
	 */
	public abstract boolean equal_(Object anObject);

	/**
	 * Answer <code>true</code> if the receiver is equal to the object.
	 * 
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @category comparing
	 */
	public abstract boolean equals(Object anObject);

	/**
	 * Convert the receiver as an array of <code>Jun2dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun2dLine[]
	 * @category converting
	 */
	public Jun2dLine[] asArrayOf2dLines() {
		return new Jun2dLine[0];
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dLine</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.curves.Jun3dLine[]
	 * @category converting
	 */
	public Jun3dLine[] asArrayOf3dLines() {
		return new Jun3dLine[0];
	}

	/**
	 * Convert the receiver as an array of <code>Jun3dTriangle</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.Jun3dTriangle[]
	 * @category converting 
	 */
	public Jun3dTriangle[] asArrayOf3dTriangles() {
		return new Jun3dTriangle[0];
	}

	/**
	 * Convert the receiver as an array of <code>JunPlane</code>.
	 * 
	 * @return jp.co.sra.jun.geometry.surfaces.JunPlane[]
	 * @category converting 
	 */
	public JunPlane[] asArrayOfPlanes() {
		return new JunPlane[0];
	}

	/**
	 * Convert to a global geometry.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @category converting
	 */
	public JunGeometry asGlobalGeometry() {
		return this;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		throw SmalltalkException.SubclassResponsibility();
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code> with color.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param colorValue java.awt.Color
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectColor_(final Color colorValue) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dObject();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				((JunOpenGL3dObject) anObject).paint_(colorValue);
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code> with color and alpha.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param colorValue java.awt.Color
	 * @param alphaValue float
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectColor_alpha_(final Color colorValue, final float alphaValue) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dObject();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				each.paint_(colorValue);
				each.alpha_(alphaValue);
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code> with color and halftone.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @param colorValue java.awt.Color
	 * @param halftoneValue double
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectColor_halftone_(final Color colorValue, final double halftoneValue) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dObject();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				each.paint_(colorValue);
				each.halftone_(halftoneValue);
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert to a <code>JunOpenGL3dObject</code> with points.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObjectWithPoints() {
		return this.asJunOpenGL3dObject();
	}

	/**
	 * Convert the receiver as a compound object of <code>JunOpenGL3dPolyline</code>s.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dPolylines() {
		JunOpenGL3dCompoundObject aBody = new JunOpenGL3dCompoundObject();
		Jun3dLine[] lines = this.asArrayOf3dLines();
		for (int i = 0; i < lines.length; i++) {
			aBody.add_(new JunOpenGL3dPolyline(new Jun3dPoint[] { lines[i].first(), lines[i].last() }));
		}
		return aBody;
	}

	/**
	 * Convert the receiver as a compound object of <code>JunOpenGL3dPolyline</code>s.
	 * 
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dPolylinesColor_(final Color aColor) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dPolylines();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				((JunOpenGL3dObject) anObject).paint_(aColor);
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert the receiver as a compound object of <code>JunOpenGL3dPolyline</code>s.
	 * 
	 * @param aColor java.awt.Color
	 * @param halftone double
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dPolylinesColor_halftone_(final Color aColor, final double halftone) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dPolylines();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				each.paint_(aColor);
				each.halftone_(halftone);
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert the receiver as a compound object of <code>JunOpenGL3dPolyline</code>s.
	 * 
	 * @param aColor java.awt.Color
	 * @param halftone double
	 * @param lineWidth float
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dPolylinesColor_halftone_lineWidth_(final Color aColor, final double halftone, final float lineWidth) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dPolylines();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				each.paint_(aColor);
				each.halftone_(halftone);
				if (each.isPolyline()) {
					((JunOpenGL3dPolyline) each).lineWidth_(lineWidth);
				}
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert the receiver as a compound object of <code>JunOpenGL3dPolyline</code>s.
	 * 
	 * @param aColor java.awt.Color
	 * @param lineWidth float
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dPolylinesColor_lineWidth_(final Color aColor, final float lineWidth) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dPolylines();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				each.paint_(aColor);
				if (each.isPolyline()) {
					((JunOpenGL3dPolyline) each).lineWidth_(lineWidth);
				}
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert the receiver as a compound object of <code>JunOpenGL3dPolyline</code>s.
	 * 
	 * @param lineWidth float
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dPolylinesLineWidth_(final float lineWidth) {
		JunOpenGL3dObject compoundObject = this.asJunOpenGL3dPolylines();
		compoundObject.objectsDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunOpenGL3dObject each = (JunOpenGL3dObject) anObject;
				if (each.isPolyline()) {
					((JunOpenGL3dPolyline) each).lineWidth_(lineWidth);
				}
				return null;
			}
		});
		return compoundObject;
	}

	/**
	 * Convert this as a local geometry in aJunCoordinateSystem.
	 * 
	 * @param aCoordinateSystem jp.co.sra.jun.geometry.coordinate.JunCoordinateSystem
	 * @return jp.co.sra.jun.geometry.coordinate.JunLocalGeometry
	 * @category converting
	 */
	public JunLocalGeometry asLocalGeometryIn_(JunCoordinateSystem aCoordinateSystem) {
		JunGeometry localGeometry = aCoordinateSystem.interprete_(this);
		return (JunLocalGeometry) _New(LocalGeometryClass(), new Object[] { localGeometry, aCoordinateSystem });
	}

	/**
	 * Answer the distance from specified Point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @return double
	 * @category functions (java only)
	 */
	public double distanceFromPoint_(Jun2dPoint aPoint) {
		throw SmalltalkException.SubclassResponsibility();
	}

	/**
	 * Answer the distance from specified Point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return double
	 * @category functions (java only)
	 */
	public double distanceFromPoint_(Jun3dPoint aPoint) {
		throw SmalltalkException.SubclassResponsibility();
	}

	/**
	 * Answer a new JunGeometry interpreted by aJunCoordinateSystem.
	 * 
	 * @param aJunCoordinateSystem jp.co.sra.jun.geometry.coordinate.JunCoordinateSystem
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @category converting
	 */
	public JunGeometry interpretedBy_(JunCoordinateSystem aJunCoordinateSystem) {
		return this.transform_(aJunCoordinateSystem._globalInvTransformation());
	}

	/**
	 * Answer the default color.
	 * 
	 * @return java.awt.Color
	 * @category defaults
	 */
	public Color defaultColor() {
		return Color.gray;
	}

	/**
	 * Answer the default alpha value.
	 * 
	 * @return float
	 * @category defaults
	 */
	public float defaultAlpha() {
		return 0.25f;
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		this.storeOn_(aWriter);
	}

	/**
	 * Answer <code>true</code> if the receiver is a 2d geometry element, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean is2d() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a 3d geometry element, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean is3d() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is an angle geometry element, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isAngle() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a bounding ball, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isBoundingBall() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a bounding balls, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isBoundingBalls() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a bounding box, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isBoundingBox() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a bounding boxes, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isBoundingBoxes() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a bounding object, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isBoundingObject() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a bounding objects, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isBoundingObjects() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a circle, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCircle() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a circular cone, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCircularCone() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a circular cylinder, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCircularCylinder() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a compound surface, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCompoundSurface() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a curve, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isCurve() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the accuracy equal a distance from a firstNumber to a
	 * secondNumber, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @param firstNumber double
	 * @param secondNumber double
	 * @category testing
	 */
	public boolean isEqualNumber_to_(double firstNumber, double secondNumber) {
		return Math.abs(firstNumber - secondNumber) < this.accuracy();
	}

	/**
	 * Answer <code>true</code> if the accuracy equal a distance from a firstPoint to a
	 * secondPoint, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @param firstPoint jp.co.sra.jun.geometory.basic.Jun3dPoint
	 * @param secondPoint jp.co.sra.jun.geometory.basic.Jun3dPoint
	 * @category testing
	 */
	public boolean isEqualPoint_to_(Jun3dPoint firstPoint, Jun3dPoint secondPoint) {
		return firstPoint.distance_(secondPoint) < this.accuracy();
	}

	/**
	 * Answer <code>true</code> if the accuracy equal a distance from a firstPoint to a
	 * secondPoint, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @param firstPoint jp.co.sra.jun.geometory.basic.Jun2dPoint
	 * @param secondPoint jp.co.sra.jun.geometory.basic.Jun2dPoint
	 * @category testing
	 */
	public boolean isEqualPoint_to_(Jun2dPoint firstPoint, Jun2dPoint secondPoint) {
		return firstPoint.distance_(secondPoint) < this.accuracy();
	}

	/**
	 * Answer <code>true</code> if the receiver is a geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isGeometry() {
		return true;
	}

	/**
	 * Answer <code>true</code> if the receiver is a global geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isGlocalGeometry() {
		return true;
	}

	/**
	 * Answer <code>true</code> if the receiver is a line geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isLine() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a lines geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isLines() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a plane geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPlane() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a point geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPoint() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a polygon geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPolygon() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a polyline geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPolyline() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a polylines geometry, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isPolylines() {
		return false;
	}

	/**
	 * Answer <code>true</code> if this is a kind of Solid geometry object, otherwise
	 * false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSolid() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a sphere, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSphere() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a surface, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isSurface() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a tetrahedron, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTetrahedron() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a topology, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTopology() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a triangle, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTriangle() {
		return false;
	}

	/**
	 * Answer <code>true</code> if the receiver is a triangles, otherwise <code>false</code>.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isTriangles() {
		return false;
	}

	/**
	 * Answer the copy of the receiver which is applied the specified transformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.JunTransformation
	 * @return jp.co.sra.jun.geometry.abstracts.JunGeometry
	 * @category transforming
	 */
	public JunGeometry transform_(JunTransformation aTransformation) {
		throw SmalltalkException.SubclassResponsibility();
	}

	/**
	 * Display this object with a <code>JunOpenGLDisplayModel</code>.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category viewing
	 */
	public JunOpenGLDisplayModel show() {
		return this.asJunOpenGL3dObject().show();
	}

	/**
	 * Display this object with a <code>JunOpenGLDisplayModel</code> with arrows.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category viewing
	 */
	public JunOpenGLDisplayModel showWithArrows() {
		JunOpenGL3dObject arrowObject = this.is2d() ? JunOpenGL3dObject.XyArrows() : JunOpenGL3dObject.XyzArrows();
		JunOpenGL3dCompoundObject compoundObject = this.asJunOpenGL3dObject().asCompoundObject();
		compoundObject.add_(arrowObject);
		return compoundObject.show();
	}

	/**
	 * Display this object with a <code>JunOpenGLDisplayModel</code> with axes.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category viewing
	 */
	public JunOpenGLDisplayModel showWithAxes() {
		return this.asJunOpenGL3dObject().showWithAxes();
	}

	/**
	 * Display this object with a <code>JunOpenGLDisplayModel</code> with unit arrows.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category viewing
	 */
	public JunOpenGLDisplayModel showWithUnitArrows() {
		JunOpenGL3dObject arrowObject = this.is2d() ? JunOpenGL3dObject.XyUnitArrows() : JunOpenGL3dObject.XyzUnitArrows();
		JunOpenGL3dCompoundObject compoundObject = this.asJunOpenGL3dObject().asCompoundObject();
		compoundObject.add_(arrowObject);
		return compoundObject.show();
	}

	/**
	 * Display this object with a <code>JunOpenGLDisplayModel</code> with unit arrows.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category viewing
	 */
	public JunOpenGLDisplayModel showWithUnitAxes() {
		return this.asJunOpenGL3dObject().showWithUnitAxes();
	}

	/**
	 * HashEqualityDictionary class
	 */
	public static class HashEqualityDictionary extends HashMap {
		/**
		 * Create a new instance of <code>HashEqualityDictionary</code> and initialize it.
		 * 
		 * @category Instance creation
		 */
		public HashEqualityDictionary() {
			super();
		}

		/**
		 * Create a new instance of <code>HashEqualityDictionary</code> and initialize it.
		 * 
		 * @param initialCapacity int
		 * @category Instance creation
		 */
		public HashEqualityDictionary(int initialCapacity) {
			super(initialCapacity);
		}

		/**
		 * Create a new instance of <code>HashEqualityDictionary</code> and initialize it.
		 * 
		 * @param m java.util.Map
		 * @category Instance creation
		 */
		public HashEqualityDictionary(Map m) {
			super(m);
		}

		/**
		 * Returns the value to which the specified key is mapped in this identity hash map.
		 *
		 * @param key jp.co.sra.jun.geometry.abstracts.JunGeometry
		 * @see java.util.HashMap#get(java.lang.Object)
		 * @return java.lang.Object
		 * @category accessing
		 */
		public Object get(JunGeometry key) {
			Object anObject = super.get(key);
			if (anObject != null) {
				return anObject;
			}

			Iterator i = this.keySet().iterator();
			if (key == null) {
				while (i.hasNext()) {
					JunGeometry anotherObject = (JunGeometry) i.next();
					if (anotherObject == null)
						return super.get(anotherObject);
				}
			} else {
				while (i.hasNext()) {
					JunGeometry anotherObject = (JunGeometry) i.next();
					if (key.equal_(anotherObject))
						return super.get(anotherObject);
				}
			}
			return null;
		}

		/**
		 * Associates the specified value with the specified key in this map.
		 * If the map previously contained a mapping for this key, the old
		 * value is replaced.
		 *
		 * @param key jp.co.sra.jun.geometry.abstracts.JunGeometry
		 * @param value java.lang.Object
		 * @return java.lang.Object
		 * @see java.util.HashMap#put(java.lang.Object, java.lang.Object)
		 * @category accessing
		 */
		public Object put(JunGeometry key, Object value) {
			if (this.containsKey(key)) {
				return this.get(key);
			}
			return super.put(key, value);
		}

		/**
		 * Answer <code>true</code> if this map contains a mapping for the specified key.
		 * 
		 * @param key jp.co.sra.jun.geometry.abstracts.JunGeometry
		 * @return boolean
		 * @see java.util.HashMap#containsKey(java.lang.Object)
		 * @category testing
		 */
		public boolean containsKey(JunGeometry key) {
			Iterator i = this.keySet().iterator();
			if (key == null) {
				while (i.hasNext()) {
					JunGeometry anotherObject = (JunGeometry) i.next();
					if (anotherObject == null)
						return true;
				}
			} else {
				while (i.hasNext()) {
					JunGeometry anotherObject = (JunGeometry) i.next();
					if (key.equal_(anotherObject))
						return true;
				}
			}
			return false;
		}
	}

	/**
	 * HashEqualitySet class
	 */
	public static class HashEqualitySet extends HashSet {
		/**
		 * Create a new instance of <code>HashEqualitySet</code> and initialize it.
		 * 
		 * @category Instance creation
		 */
		public HashEqualitySet() {
			super();
		}

		/**
		 * Create a new instance of <code>HashEqualitySet</code> and initialize it.
		 * 
		 * @param initialCapacity int
		 * @category Instance creation
		 */
		public HashEqualitySet(int initialCapacity) {
			super(initialCapacity);
		}

		/**
		 * Adds the specified geometry to this set if it is not already present.
		 * 
		 * @param anObject jp.co.sra.jun.geometry.abstracts.JunGeometry
		 * @return boolean
		 * @see java.util.HashSet#add(java.lang.Object)
		 * @category adding
		 */
		public boolean add(JunGeometry anObject) {
			if (this.contains(anObject)) {
				return false;
			}
			return super.add(anObject);
		}

		/**
		 * Adds all of the elements in the specified collection to this collection.
		 * 
		 * @param geometries jp.co.sra.jun.geometry.abstracts.JunGeometry
		 * @return boolean
		 * @see java.util.AbstractCollection#addAll(java.util.Collection)
		 * @category adding
		 */
		public boolean addAll(JunGeometry[] geometries) {
			boolean modified = false;
			for (int i = 0; i < geometries.length; i++) {
				if (this.add(geometries[i])) {
					modified = true;
				}
			}
			return modified;
		}

		/**
		 * Answer <tt>true</tt> if this set contains the specified geometry.
		 * 
		 * @param anObject jp.co.sra.jun.geometry.abstracts.JunGeometry
		 * @return boolean
		 * @see java.util.HashSet#contains(java.lang.Object)
		 * @category testing
		 */
		public boolean contains(JunGeometry anObject) {
			Iterator i = this.iterator();
			if (anObject == null) {
				while (i.hasNext()) {
					JunGeometry anotherObject = (JunGeometry) i.next();
					if (anotherObject == null)
						return true;
				}
			} else {
				while (i.hasNext()) {
					JunGeometry anotherObject = (JunGeometry) i.next();
					if (anObject.equal_(anotherObject))
						return true;
				}
			}
			return false;
		}
	}
}
