package jp.co.sra.jun.dxf.support;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StReadStream;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunDXFScanner class
 * 
 *  @author    nisinaka
 *  @created   2002/07/11 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun447 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDXFScanner.java,v 8.12 2008/02/20 06:30:54 nisinaka Exp $
 */
public class JunDXFScanner extends JunAbstractObject {
	protected StReadStream source;
	protected int mark;
	protected String dataKey;
	protected String dataString;
	protected Number dataNumber;
	protected char lineSeparator;
	protected int lines;
	protected StBlockClosure failBlock;
	protected int groupCode;

	/**
	 * Shorten a string by ellipsis if too long.
	 * 
	 * @param aString java.lang.String
	 * @param count int
	 * @return java.lang.String
	 * @category Utilities
	 */
	public static String ContractTo(String aString, int count) {
		int length = aString.length();
		if (length <= count) {
			return aString;
		}
		int half = (count / 2) - 1;
		return aString.substring(0, half) + "..." + aString.substring(length - count + half + 3);
	}

	/**
	 * Determine the line separator character.
	 * 
	 * @category accessing
	 */
	protected void findSeparator() {
		char separator = 0;
		while (separator == 0 && source.atEnd() == false) {
			char ch = source.next();
			if (ch == '\n') {
				separator = ch;
			} else if (ch == '\r') {
				separator = ch;
				if (source.peek() == '\n') {
					source.next();
				}
			}
		}
		if (separator != 0) {
			this.lineSeparator_(separator);
		}
		source.position_(0);
	}

	/**
	 * Initialize the JunDXFScanner.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		this.initScanner();
		lineSeparator = '\r';
	}

	/**
	 * Initialize the scanner information.
	 * 
	 * @category initialize-release
	 */
	protected void initScanner() {
		lines = 0;
		failBlock = new StBlockClosure() {
			public Object value_(Object anObject) {
				String errorMessage = (String) anObject;
				String label = errorMessage + " near " + ContractTo(dataString, 10);
				String string = source.upToEnd();

				if (string.length() == 0) {
					string = "--> end of file";
				} else {
					string = "--> " + ContractTo(string, 30);
				}

				throw SmalltalkException.Error(label + "\n" + string);
			}
		};

	}

	/**
	 * Answer the current line separator character.
	 * 
	 * @return char
	 * @category accessing
	 */
	protected char lineSeparator() {
		return lineSeparator;
	}

	/**
	 * Set the new line separator character.
	 * 
	 * @param newLineSeparator char
	 * @category accessing
	 */
	protected void lineSeparator_(char newLineSeparator) {
		lineSeparator = newLineSeparator;
	}

	/**
	 * Get next pair of the data.<br>
	 * 0- 9 : String<br>
	 * 10-59 : Double precision 3D point<br>
	 * 60-79 : 16-bit integer<br>
	 * 90-99 : 32-bit integer<br>
	 * 
	 * @category scanning
	 */
	protected void nextPair() {
		while (source.atEnd() == false) {
			char ch = source.peek();
			if (Character.isWhitespace(ch)) {
				source.next();
			} else {
				break;
			}
		}

		mark = source.position();
		if (source.atEnd()) {
			dataKey = "eof";
			dataString = "EOF";

			return;
		}
		String lineString = source.upTo_(this.lineSeparator());
		lines++;

		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < lineString.length(); i++) {
			char ch = lineString.charAt(i);
			if (Character.isDigit(ch)) {
				buf.append(ch);
			}
		}

		dataKey = buf.toString();
		groupCode = Integer.parseInt(dataKey);

		lineString = source.upTo_(this.lineSeparator());
		lines++;
		buf = new StringBuffer();
		for (int i = 0; i < lineString.length(); i++) {
			char ch = lineString.charAt(i);
			if (Character.isWhitespace(ch) == false) {
				buf.append(ch);
			}
		}

		dataString = buf.toString();

		if ((10 <= groupCode) && (groupCode <= 59)) {
			dataNumber = new Double(dataString);
		} else if ((60 <= groupCode) && (groupCode < 100)) {
			dataNumber = new Long(dataString);
		}
	}

	/**
	 * Initialize the source.
	 * 
	 * @param readStream jp.co.sra.smalltalk.StReadStream
	 * @category initialize-release
	 */
	protected void on_(StReadStream readStream) {
		source = readStream;
		mark = source.position();
	}

	/**
	 * Initialize the source and determine the EOL character.
	 * 
	 * @param readStream jp.co.sra.smalltalk.StReadStream
	 * @category initialize-release
	 */
	protected void onSmart_(StReadStream readStream) {
		this.on_(readStream);
		this.findSeparator();
	}

	/**
	 * Reset the previous pair.
	 * 
	 * @category scanning
	 */
	protected void unNextPair() {
		source.position_(mark);
		lines -= 2;
	}
}
