package jp.co.sra.jun.delaunay.twoD;

import java.awt.*;
import java.io.*;
import jp.co.sra.smalltalk.*;

/**
 * Jun2dDelaunayEdge class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2002/01/14 (by Ryouichi Matsuda)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: Jun2dDelaunayEdge.java,v 8.12 2008/02/20 06:30:54 nisinaka Exp $
 */
public class Jun2dDelaunayEdge extends Jun2dDelaunayElement {
	protected Jun2dDelaunayHalfEdge halfEdge12;
	protected Jun2dDelaunayHalfEdge halfEdge21;
	protected boolean constrained;
	protected boolean dirty;

	/**
	 * Create a new instance of <code>Jun2dDelaunayEdge</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public Jun2dDelaunayEdge() {
		super();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean arrange() {
		Jun2dDelaunayVertex v1;
		Jun2dDelaunayVertex v2;

		if (constrained || !dirty) {
			return false;
		}

		this.setClean();

		if ((this.loop12() == null) || (this.loop21() == null)) {
			return false;
		}

		v1 = this.halfEdge12().next().vertex();
		v2 = this.halfEdge21().next().vertex();

		if ((this.vertex1().areaWith_with_(v2, v1) >= 0) && (this.vertex2().areaWith_with_(v1, v2) >= 0)) {
			boolean isArrange;

			if ((this.loop12().area() < DECIMAL_12) || (this.loop21().area() < DECIMAL_12)) {
				isArrange = true;
			} else {
				double maxCos;
				double arrangedMaxCos;

				maxCos = this.vertex2().cosOfAngleWith_with_(this.vertex1(), v2);
				maxCos = Math.max(this.vertex1().cosOfAngleWith_with_(v2, this.vertex2()), maxCos);
				maxCos = Math.max(this.vertex1().cosOfAngleWith_with_(this.vertex2(), v1), maxCos);
				maxCos = Math.max(this.vertex2().cosOfAngleWith_with_(v1, this.vertex1()), maxCos);
				arrangedMaxCos = v1.cosOfAngleWith_with_(v2, this.vertex2());
				arrangedMaxCos = Math.max(v2.cosOfAngleWith_with_(this.vertex2(), v1), arrangedMaxCos);
				arrangedMaxCos = Math.max(v1.cosOfAngleWith_with_(this.vertex1(), v2), arrangedMaxCos);
				arrangedMaxCos = Math.max(v2.cosOfAngleWith_with_(v1, this.vertex1()), arrangedMaxCos);
				isArrange = (arrangedMaxCos < maxCos);
			}

			if (isArrange) {
				this.basicArrange();

				return true;
			}
		}

		return false;
	}

	/**
	 * DOCUMENT ME!
	 */
	public void basicArrange() {
		Jun2dDelaunayHalfEdge hEdge1;
		Jun2dDelaunayHalfEdge hEdge2;
		Jun2dDelaunayHalfEdge hEdge3;
		Jun2dDelaunayHalfEdge hEdge4;
		Jun2dDelaunayLoop loop1;
		Jun2dDelaunayLoop loop2;

		if (constrained) {
			throw SmalltalkException.Halt("constrained");
		}

		hEdge1 = this.halfEdge12().next();
		hEdge2 = hEdge1.next();
		hEdge3 = this.halfEdge21().next();
		hEdge4 = hEdge3.next();
		this.halfEdge12().vertex_(hEdge2.pair().vertex());
		this.halfEdge12().next_(hEdge2);
		hEdge2.next_(hEdge3);
		hEdge3.next_(this.halfEdge12());
		this.halfEdge21().vertex_(hEdge4.pair().vertex());
		this.halfEdge21().next_(hEdge4);
		hEdge4.next_(hEdge1);
		hEdge1.next_(this.halfEdge21());
		hEdge4.vertex().halfEdge_(hEdge4);
		hEdge2.vertex().halfEdge_(hEdge2);
		loop1 = this.loop12();
		loop2 = this.loop21();
		loop1.halfEdge_(this.halfEdge12());
		loop2.halfEdge_(this.halfEdge21());

		final Jun2dDelaunayLoop loop1_ = loop1;
		loop1.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				hEdge.loop_(loop1_);

				return null;
			}
		});

		final Jun2dDelaunayLoop loop2_ = loop2;
		loop2.halfEdgesDo_(new StBlockClosure() {
			public Object value_(Object hEdge_) {
				Jun2dDelaunayHalfEdge hEdge = (Jun2dDelaunayHalfEdge) hEdge_;

				hEdge.loop_(loop2_);

				return null;
			}
		});
		this.halfEdge12().prev().edge().setDirty();
		this.halfEdge12().next().edge().setDirty();
		this.halfEdge21().next().edge().setDirty();
		this.halfEdge21().prev().edge().setDirty();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return boolean
	 */
	public boolean contains_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		return this.containsX_y_(aJun2dDelaunayVertex.x(), aJun2dDelaunayVertex.y());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return boolean
	 */
	public boolean containsX_y_(double xNumber, double yNumber) {
		return (this.squaredDistanceFromX_y_(xNumber, yNumber) < DECIMAL_12);
	}

	/**
	 * DOCUMENT ME!
	 */
	public void disableArrange() {
		constrained = true;
	}

	/**
	 * Display the receiver on aGraphics.
	 * 
	 * @param aGraphicsContext java.awt.Graphics
	 */
	public void displayOn_(Graphics aGraphicsContext) {
		Color oldColor;

		if (constrained) {
			oldColor = aGraphicsContext.getColor();
			aGraphicsContext.setColor(Color.red);
			halfEdge12.displayOn_(aGraphicsContext);
			aGraphicsContext.setColor(oldColor);
		} else {
			halfEdge12.displayOn_(aGraphicsContext);
		}
	}

	/**
	 * DOCUMENT ME!
	 */
	public void enableArrange() {
		constrained = false;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public Jun2dDelaunayHalfEdge halfEdge12() {
		return halfEdge12;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayHalfEdge
	 */
	public Jun2dDelaunayHalfEdge halfEdge21() {
		return halfEdge21;
	}

	/**
	 * Initialize the receiver when created.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		halfEdge12 = new Jun2dDelaunayHalfEdge();
		halfEdge21 = new Jun2dDelaunayHalfEdge();
		halfEdge12.setEdge_(this);
		halfEdge21.setEdge_(this);
		halfEdge12.setPair_(halfEdge21);
		halfEdge21.setPair_(halfEdge12);
		constrained = false;
		dirty = true;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayEdge jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayEdge
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	public Jun2dDelaunayVertex intersectionWith_(Jun2dDelaunayEdge aJun2dDelaunayEdge) {
		return this.intersectionWithLineSegmentFrom_to_(aJun2dDelaunayEdge.vertex1(), aJun2dDelaunayEdge.vertex2());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param fromPoint jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param toPoint jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return boolean
	 */
	public boolean intersectsWithLineSegmentFrom_to_(Jun2dDelaunayVertex fromPoint, Jun2dDelaunayVertex toPoint) {
		return halfEdge12.intersectsWithLineSegmentFrom_to_(fromPoint, toPoint);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean isArrangeable() {
		Jun2dDelaunayVertex v1;
		Jun2dDelaunayVertex v2;

		if (this.isConstrained()) {
			return false;
		}

		v1 = this.halfEdge12().next().vertex;
		v2 = this.halfEdge21().next().vertex;

		return (this.vertex1().areaWith_with_(v2, v1) >= 0) && (this.vertex2().areaWith_with_(v1, v2) >= 0);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean isConstrained() {
		return constrained;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 */
	public double length() {
		return Math.sqrt(this.lengthSquared());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return double
	 */
	public double lengthSquared() {
		return this.vertex1().dotProduct_(this.vertex2());
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 */
	public Jun2dDelaunayLoop loop12() {
		return halfEdge12.loop();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayLoop jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 */
	public void loop12_(Jun2dDelaunayLoop aJun2dDelaunayLoop) {
		halfEdge12.loop_(aJun2dDelaunayLoop);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 */
	public Jun2dDelaunayLoop loop21() {
		return halfEdge21.loop();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayLoop jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayLoop
	 */
	public void loop21_(Jun2dDelaunayLoop aJun2dDelaunayLoop) {
		halfEdge21.loop_(aJun2dDelaunayLoop);
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException DOCUMENT ME!
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Edge(");
		aWriter.write(Double.toString(this.vertex1().x()));
		aWriter.write("@");
		aWriter.write(Double.toString(this.vertex1().y()));
		aWriter.write(" ");
		aWriter.write(Double.toString(this.vertex2().x()));
		aWriter.write("@");
		aWriter.write(Double.toString(this.vertex2().y()));
		aWriter.write(")");
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return boolean
	 */
	public boolean recursiveArrange() {
		if (this.arrange()) {
			this.halfEdge12().prev().edge().recursiveArrange();
			this.halfEdge12().next().edge().recursiveArrange();
			this.halfEdge21().next().edge().recursiveArrange();
			this.halfEdge21().prev().edge().recursiveArrange();

			return true;
		}

		return false;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	public Jun2dDelaunayVertex vertex1() {
		return halfEdge21.vertex();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	public void vertex1_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		halfEdge21.vertex_(aJun2dDelaunayVertex);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	public Jun2dDelaunayVertex vertex2() {
		return halfEdge12.vertex();
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	public void vertex2_(Jun2dDelaunayVertex aJun2dDelaunayVertex) {
		halfEdge12.vertex_(aJun2dDelaunayVertex);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return double
	 */
	protected double distanceFromX_y_(double xNumber, double yNumber) {
		return Math.sqrt(this.squaredDistanceFromX_y_(xNumber, yNumber));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param aJun2dDelaunayVertex1 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * @param aJun2dDelaunayVertex2 jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 * 
	 * @return jp.co.sra.jun.delaunay.twoD.Jun2dDelaunayVertex
	 */
	protected Jun2dDelaunayVertex intersectionWithLineSegmentFrom_to_(Jun2dDelaunayVertex aJun2dDelaunayVertex1, Jun2dDelaunayVertex aJun2dDelaunayVertex2) {
		Jun2dDelaunayVertex myV1;
		Jun2dDelaunayVertex myV2;
		Jun2dDelaunayVertex yourV1;
		Jun2dDelaunayVertex yourV2;
		double myF;
		double myG;
		double yourF;
		double yourG;
		double denominator;
		double myT;
		double yourT;

		myV1 = this.vertex1();
		myV2 = this.vertex2();
		yourV1 = aJun2dDelaunayVertex1;
		yourV2 = aJun2dDelaunayVertex2;

		if (Math.max(myV1.x(), myV2.x()) < Math.min(yourV1.x(), yourV2.x())) {
			return null;
		}

		if (Math.max(yourV1.x(), yourV2.x()) < Math.min(myV1.x(), myV2.x())) {
			return null;
		}

		if (Math.max(myV1.y(), myV2.y()) < Math.min(yourV1.y(), yourV2.y())) {
			return null;
		}

		if (Math.max(yourV1.y(), yourV2.y()) < Math.min(myV1.y(), myV2.y())) {
			return null;
		}

		myF = myV2.x() - myV1.x();
		myG = myV2.y() - myV1.y();
		yourF = yourV2.x() - yourV1.x();
		yourG = yourV2.y() - yourV1.y();
		denominator = (yourF * myG) - (myF * yourG);

		if (Math.abs(denominator) < DECIMAL_12) {
			return null;
		}

		myT = ((yourF * (yourV1.y() - myV1.y())) - (yourG * (yourV1.x() - myV1.x()))) / denominator;

		if (!(-DECIMAL_12 <= myT && myT <= (1.0d + DECIMAL_12))) {
			return null;
		}

		yourT = ((myG * (myV1.x() - yourV1.x())) - (myF * (myV1.y() - yourV1.y()))) / denominator;

		if (!(-DECIMAL_12 <= yourT && yourT <= (1.0d + DECIMAL_12))) {
			return null;
		}

		return new Jun2dDelaunayVertex((myF * myT) + myV1.x(), (myG * myT) + myV1.y(), (myV1.z() != Double.NaN) ? ((myV2.z() != Double.NaN) ? (((myV2.z() - myV1.z()) * myT) + myV1.z()) : myV1.z()) : ((myV2.z() != Double.NaN) ? myV2.z() : Double.NaN));
	}

	/**
	 * DOCUMENT ME!
	 */
	protected void setClean() {
		dirty = false;
	}

	/**
	 * DOCUMENT ME!
	 */
	protected void setDirty() {
		dirty = true;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param xNumber double
	 * @param yNumber double
	 * 
	 * @return double
	 */
	protected double squaredDistanceFromX_y_(double xNumber, double yNumber) {
		Jun2dDelaunayVertex v1;
		Jun2dDelaunayVertex v2;
		double f;
		double g;
		double dominator;
		double x0;
		double y0;
		double t;
		double x;
		double y;

		v1 = this.vertex1();
		v2 = this.vertex2();
		f = v2.x() - v1.x();
		g = v2.y() - v1.y();
		dominator = Math.pow(f, 2) + Math.pow(g, 2);

		if (dominator < DECIMAL_12) {
			return (Math.pow(v1.x() - xNumber, 2) + Math.pow(v1.y() - yNumber, 2)) / DECIMAL_12;
		}

		x0 = xNumber - v1.x();
		y0 = yNumber - v1.y();
		t = ((f * x0) + (g * y0)) / dominator;
		t = Math.max((Math.min(t, 1.0d)), 0.0d);
		x = (f * t) + v1.x();
		y = (g * t) + v1.y();

		return Math.pow(xNumber - x, 2) + Math.pow(yNumber - y, 2);
	}
}
