package jp.co.sra.gl4jun;

import java.awt.Canvas;
import java.awt.Container;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;

import jp.co.sra.smalltalk.SmalltalkException;

/**
 * GLjCanvas class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1999/08/19 (by MATSUDA Ryouichi)
 *  @updated   2006/07/18 (by nisinaka)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: GLjCanvas.java,v 8.10 2008/02/20 06:30:52 nisinaka Exp $
 */
public class GLjCanvas extends Canvas {

	protected int renderingMode;

	protected transient GLjRenderingContext renderingContext;
	protected transient boolean resizeRequest;

	/**
	 * Create a new instance of GLjCanvas and initialize it.
	 *
	 * @category Instance creation
	 */
	public GLjCanvas() {
		this.initialize();
		this.addComponentListener(new ComponentAdapter() {
			public void componentResized(ComponentEvent e) {
				resizeRequest = true;
			}
		});
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		renderingMode = GLjRenderingMode.DOUBLE_BUFFER;
		renderingContext = null;
		resizeRequest = false;
	}

	/**
	 * Delete the rendering context before removed.
	 * 
	 * @see java.awt.Component#removeNotify()
	 * @category initialize-release
	 */
	public void removeNotify() {
		if (renderingContext != null) {
			Container c = this.getParent();
			while (c != null && (c instanceof java.applet.Applet) == false) {
				c = c.getParent();
			}

			renderingContext.gljDelete(c != null);
			renderingContext = null;
		}

		super.removeNotify();
	}

	/**
	 * Set the rendering mode.
	 * 
	 * @param aMode int
	 * @category accessing
	 */
	public void setRenderingMode(int aMode) {
		switch (aMode) {
			case GLjRenderingMode.NOTHING:
			case GLjRenderingMode.IMAGE:
			case GLjRenderingMode.SINGLE_BUFFER:
			case GLjRenderingMode.DIRECT:
			case GLjRenderingMode.DOUBLE_BUFFER:
				renderingMode = aMode;
				break;
			default:
				throw new SmalltalkException("Invalid rendering mode : " + aMode);
		}
	}

	/**
	 * Answer my default rendering context.
	 * 
	 * @return jp.co.sra.gl4jun.GLjRenderingContext
	 * @category defaults
	 */
	protected GLjRenderingContext defaultRenderingContext() {
		return new GLjRenderingContext(this);
	}

	/**
	 * Paint the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category painting
	 */
	public void paint(Graphics aGraphics) {
		this.render(aGraphics);
		this.superimposeOn(aGraphics);
	}

	/**
	 * Render the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category rendering
	 */
	protected void render(Graphics aGraphics) {
		if (this.isShowing() == false) {
			return;
		}

		synchronized (GLjInterface.Current()) {
			if (renderingContext == null) {
				renderingContext = this.defaultRenderingContext();
				renderingContext.setMode(renderingMode);
				resizeRequest = false;
				renderingContext.gljInitialize();
			}
		}

		if (renderingContext.getMode() != GLjRenderingMode.NOTHING) {
			synchronized (renderingContext.OpenGLInterface()) {
				if (resizeRequest) {
					renderingContext.gljResize(this.getSize());
					resizeRequest = false;
				}

				renderingContext.gljCurrent();
				this.render(renderingContext);

				if (renderingContext.getMode() == GLjRenderingMode.IMAGE) {
					Image anImage = renderingContext.gljImageRGB();
					aGraphics.drawImage(anImage, 0, 0, null);
				}

				renderingContext.gljFlush();
			}
		}
	}

	/**
	 * Render the receiver on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.gl4jun.GLjRenderingContext
	 * @category rendering
	 */
	protected void render(GLjRenderingContext aRenderingContext) {
		this.render(aRenderingContext.OpenGLInterface());
	}

	/**
	 * Render the receiver on the OpenGL interface.
	 * 
	 * @param openGLInterface jp.co.sra.gl4jun.GLjInterface
	 * @category rendering
	 */
	protected void render(GLjInterface openGLInterface) {
	}

	/**
	 * Superimpose on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category superimpose
	 */
	protected void superimposeOn(Graphics aGraphics) {
	}

	/**
	 * Update for the change notice.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category updating
	 */
	public void update(Graphics aGraphics) {
		this.paint(aGraphics);
	}

}
