/**
 * StPL-x11.c:
 *
 * 	@author:    Hoshi Takanori
 * 	@version:   2.0
 * 	@created:   2001/11/30 (by Hoshi Takanori)
 * 	@updated:   N/A
 *
 * 	$Id: StPL-x11.c,v 2.1 2006/04/28 05:22:27 hoshi Exp $
 */

#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include "jp_co_sra_smalltalk_SystemInterface.h"
#include "version.h"

#define IMAGE_FROM_USER 1
#define IMAGE_OF_AREA 2
#define IMAGE_OF_WHOLE_SCREEN 3
#define RECTANGLE_FROM_USER 4
#define COLOR_SPUIT 5

#define RGB_PIXEL(red, grren, blue) \
		(0xff000000 | ((red) << 16) | ((green) << 8) | (blue))

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    getVersion
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_jp_co_sra_smalltalk_SystemInterface_getVersion
  (JNIEnv *env, jclass cls)
{
	return (*env)->NewStringUTF(env, STPL_JNI_VERSION);
}

// ==================================================================

static unsigned char get_component
  (unsigned long pixel, unsigned long mask)
{
	int bits;

	pixel &= mask;

	if (pixel == 0) {
		return 0;
	} else if (pixel == mask) {
		return 0xff;
	}

	while ((mask & 1) == 0) {
		pixel >>= 1;
		mask >>= 1;
	}

	if (mask < 0x80) {
		for (bits = 0; mask != 0; ++bits, mask >>= 1) {}
		switch (bits) {
		case 2:		return (pixel << 6) | (pixel << 4) | (pixel << 2) | pixel;
		case 3:		return (pixel << 5) | (pixel << 2) | (pixel >> 1);
		case 4:		return (pixel << 4) | pixel;
		case 5:		return (pixel << 3) | (pixel >> 2);
		case 6:		return (pixel << 2) | (pixel >> 4);
		case 7:		return (pixel << 1) | (pixel >> 6);
		}
	}

	while (mask > 0xff) {
		pixel >>= 1;
		mask >>= 1;
	}

	return pixel;
}

static jint *capture_screen
  (Display *display, Window root_window, XRectangle *rect)
{
	XImage *image;
	jint *pixels;
	int x, y;
	unsigned long pixel;
	unsigned char red, green, blue;

	image = XGetImage(display, root_window,
			rect->x, rect->y, rect->width, rect->height, AllPlanes, ZPixmap);
	if (image == NULL) {
		return NULL;
	}

	pixels = (jint *) malloc(sizeof(jint) * rect->width * rect->height);
	if (pixels == NULL) {
		XDestroyImage(image);
		return NULL;
	}

	for (y = 0; y < rect->height; ++y) {
		for (x = 0; x < rect->width; ++x) {
			pixel = XGetPixel(image, x, y);
			red   = get_component(pixel, image->red_mask);
			green = get_component(pixel, image->green_mask);
			blue  = get_component(pixel, image->blue_mask);
			pixels[y * rect->width + x] = RGB_PIXEL(red, grren, blue);
		}
	}

	XDestroyImage(image);

	return pixels;
}

// ==================================================================

static int select_rectangle
  (int operation, Display *display, Window root_window, XRectangle *rect)
{
	XGCValues gc_values;
	GC gc;
	Cursor cursor;
	unsigned int event_mask;
	int status, presses, start_x = 0, start_y = 0;
	XEvent event;
	Bool done;

	gc_values.function = GXinvert;
	gc_values.foreground = XBlackPixel(display, XDefaultScreen(display));
	gc_values.background = XWhitePixel(display, XDefaultScreen(display));
	gc_values.plane_mask = gc_values.foreground ^ gc_values.background;
	gc_values.subwindow_mode = IncludeInferiors;
	gc = XCreateGC(display, root_window,
			GCFunction | GCForeground | GCBackground | GCSubwindowMode,
			&gc_values);

	cursor = XCreateFontCursor(display, XC_crosshair);

	event_mask = ButtonPressMask | ButtonReleaseMask | ButtonMotionMask;
	status = XGrabPointer(display, root_window, False, event_mask,
			GrabModeSync, GrabModeAsync, root_window, cursor, CurrentTime);
	if (status != GrabSuccess) {
		XCloseDisplay(display);
		return -1;
	}

	presses = 0;
	rect->width = 0;
	rect->height = 0;
	done = False;

	while (done == False) {
		if (rect->width > 0 && rect->height > 0) {
			XDrawRectangle(display, root_window, gc,
					rect->x, rect->y, rect->width - 1, rect->height - 1);
		}

		XAllowEvents(display, SyncPointer, CurrentTime);
		XWindowEvent(display, root_window, event_mask, &event);

		if (rect->width > 0 && rect->height > 0) {
			XDrawRectangle(display, root_window, gc,
					rect->x, rect->y, rect->width -1, rect->height - 1);
		}

		switch (event.type) {
		case ButtonPress:
			if (presses == 0) {
				rect->x = start_x = event.xbutton.x_root;
				rect->y = start_y = event.xbutton.y_root;
				rect->width = 1;
				rect->height = 1;
			}
			++presses;
			break;

		case ButtonRelease:
			--presses;
			if (presses == 0) {
				done = True;
			}
			break;

		case MotionNotify:
			while (XCheckMaskEvent(display, ButtonMotionMask, &event)) {}

			if (operation == COLOR_SPUIT) {
				rect->x = start_x = event.xbutton.x_root;
				rect->y = start_y = event.xbutton.y_root;
				rect->width = 1;
				rect->height = 1;
			} else {
				if (event.xbutton.x_root >= start_x) {
					rect->x = start_x;
					rect->width = event.xbutton.x_root - start_x + 1;
				} else {
					rect->x = event.xbutton.x_root;
					rect->width = start_x - event.xbutton.x_root + 1;
				}

				if (event.xbutton.y_root >= start_y) {
					rect->y = start_y;
					rect->height = event.xbutton.y_root - start_y + 1;
				} else {
					rect->y = event.xbutton.y_root;
					rect->height = start_y - event.xbutton.y_root + 1;
				}
			}
			break;

		default:
			break;
		}
	}

	XUngrabPointer(display, CurrentTime);
	XFreeCursor(display, cursor);
	XFreeGC(display, gc);

	return event.xbutton.button == 1 ? 0 : -1;
}

// ==================================================================

static jintArray do_operation
  (JNIEnv *env, int operation, XRectangle *rect, jintArray dimension)
{
	Display *display;
	Window root_window;
	jsize size = 0;
	jint *ptr = NULL, dimension_buf[2];
	jintArray array = NULL;

	display = XOpenDisplay(NULL);		/* Is this OK??? */
	if (display == NULL) {
		return NULL;
	}

	root_window = XRootWindow(display, XDefaultScreen(display));

	if (operation == IMAGE_FROM_USER
	 || operation == RECTANGLE_FROM_USER
	 || operation == COLOR_SPUIT) {
		if (select_rectangle(operation, display, root_window, rect) != 0) {
			XCloseDisplay(display);
			return NULL;
		}
	} else if (operation == IMAGE_OF_WHOLE_SCREEN) {
		rect->x = 0;
		rect->y = 0;
		rect->width = XDisplayWidth(display, XDefaultScreen(display));
		rect->height = XDisplayHeight(display, XDefaultScreen(display));
	}

	if (operation == IMAGE_FROM_USER
	 || operation == IMAGE_OF_AREA
	 || operation == IMAGE_OF_WHOLE_SCREEN
	 || operation == COLOR_SPUIT) {
		size = rect->width * rect->height;
		ptr = capture_screen(display, root_window, rect);
	} else if (operation == RECTANGLE_FROM_USER) {
		size = 4;
		ptr = (jint *) malloc(sizeof(jint) * size);
		if (ptr != NULL) {
			ptr[0] = rect->x;
			ptr[1] = rect->y;
			ptr[2] = rect->width;
			ptr[3] = rect->height;
		}
	}

	XCloseDisplay(display);

	if (ptr != NULL) {
		array = (*env)->NewIntArray(env, size);
		if (array != NULL) {
			(*env)->SetIntArrayRegion(env, array, 0, size, ptr);
		}
		free(ptr);

		if (dimension != NULL) {
			dimension_buf[0] = rect->width;
			dimension_buf[1] = rect->height;
			(*env)->SetIntArrayRegion(env, dimension, 0, 2, dimension_buf);
		}
	}

	return array;
}

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageFromUser
 * Signature: ([I)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageFromUser
  (JNIEnv *env, jobject obj, jintArray dimension)
{
	XRectangle rect;

	return do_operation(env, IMAGE_FROM_USER, &rect, dimension);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageOfArea
 * Signature: (IIII)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageOfArea
  (JNIEnv *env, jobject obj, jint x, jint y, jint width, jint height)
{
	XRectangle rect;

	rect.x = x;
	rect.y = y;
	rect.width = width;
	rect.height = height;
	return do_operation(env, IMAGE_OF_AREA, &rect, NULL);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageOfWholeScreen
 * Signature: ([I)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageOfWholeScreen
  (JNIEnv *env, jobject obj, jintArray dimension)
{
	XRectangle rect;

	return do_operation(env, IMAGE_OF_WHOLE_SCREEN, &rect, dimension);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilRectangleFromUser
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilRectangleFromUser
  (JNIEnv *env, jobject obj)
{
	XRectangle rect;

	return do_operation(env, RECTANGLE_FROM_USER, &rect, NULL);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilColorSpuit
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilColorSpuit
  (JNIEnv *env, jobject obj)
{
	XRectangle rect;
	jintArray array;
	jint pixel = 0;

	array = do_operation(env, COLOR_SPUIT, &rect, NULL);
	if (array != NULL) {
		(*env)->GetIntArrayRegion(env, array, 0, 1, &pixel);
		// I believe that array will be GC'ed...
	}
	return pixel;
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilMousePoint
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilMousePoint
  (JNIEnv *env, jobject obj)
{
	Display *display;
	Window root_window, root, child;
	int root_x, root_y, win_x, win_y;
	unsigned int mask;
	jintArray array;
	jint buf[2];

	display = XOpenDisplay(NULL);		/* Is this OK??? */
	if (display == NULL) {
		return NULL;
	}

	root_window = XRootWindow(display, XDefaultScreen(display));
	XQueryPointer(display, root_window,
			&root, &child, &root_x, &root_y, &win_x, &win_y, &mask);

	XCloseDisplay(display);

	array = (*env)->NewIntArray(env, 2);
	if (array != NULL) {
		buf[0] = root_x;
		buf[1] = root_y;
		(*env)->SetIntArrayRegion(env, array, 0, 2, buf);
	}

	return array;
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilSetMousePoint
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilSetMousePoint
  (JNIEnv *env, jobject obj, jint x, jint y)
{
	Display *display;
	Window root_window;

	display = XOpenDisplay(NULL);		/* Is this OK??? */
	if (display == NULL) {
		return;
	}

	root_window = XRootWindow(display, XDefaultScreen(display));
	XWarpPointer(display, None, root_window, 0, 0, 0, 0, x, y);

	XCloseDisplay(display);
}

// ==================================================================

#define BROWSE_PREFIX "netscape '"
#define BROWSE_PREFIX_LENGTH 10

#define BROWSE_SUFFIX "' &"
#define BROWSE_SUFFIX_LENGTH 3

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilBrowse
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilBrowse
  (JNIEnv *env, jobject obj, jstring url)
{
	const char *jstr = (*env)->GetStringUTFChars(env, url, NULL);
	jsize jlen = (*env)->GetStringUTFLength(env, url);
	int len = BROWSE_PREFIX_LENGTH + jlen + BROWSE_SUFFIX_LENGTH + 1;
	char *buf = (char *) malloc(len);

	if (buf != NULL) {
		strcpy(buf, BROWSE_PREFIX);
		memcpy(buf + BROWSE_PREFIX_LENGTH, jstr, jlen);
		strcpy(buf + BROWSE_PREFIX_LENGTH + jlen, BROWSE_SUFFIX);
		system(buf);
		free(buf);
	}
	(*env)->ReleaseStringUTFChars(env, url, jstr);
}
