/**
 * StPL-mac.c:
 *
 * 	@author:    Hoshi Takanori
 * 	@version:   2.0
 * 	@created:   2002/02/28 (by Hoshi Takanori)
 * 	@updated:   N/A
 *
 * 	$Id: StPL-mac.c,v 2.1 2006/04/28 06:01:05 hoshi Exp $
 */

#include <Carbon/Carbon.h>
#include <stdlib.h>

#include "jp_co_sra_smalltalk_SystemInterface.h"
#include "version.h"

#define IMAGE_FROM_USER 1
#define IMAGE_OF_AREA 2
#define IMAGE_OF_WHOLE_SCREEN 3
#define RECTANGLE_FROM_USER 4
#define COLOR_SPUIT 5

#define RGB_PIXEL(red, grren, blue) \
		(0xff000000 | ((red) << 16) | ((green) << 8) | (blue))

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    getVersion
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_jp_co_sra_smalltalk_SystemInterface_getVersion
  (JNIEnv *env, jclass cls)
{
	return (*env)->NewStringUTF(env, STPL_JNI_VERSION);
}

// ==================================================================

static jint *capture_screen
  (const Rect *rect)
{
	Rect bounds;
	GWorldPtr gworld;
	PixMapHandle pixmap;
	BitMap screen_bits;
	int width, height, row_bytes, x, y;
	unsigned char *base_addr, *row, red, green, blue;
	jint *pixels, *p;

	width = rect->right - rect->left;
	height = rect->bottom - rect->top;

	SetRect(&bounds, 0, 0, width, height);
	if (NewGWorld(&gworld, 32, &bounds, NULL, NULL, 0) != noErr) {
		return NULL;
	}

	pixmap = GetGWorldPixMap(gworld);
	LockPixels(pixmap);
	CopyBits(GetQDGlobalsScreenBits(&screen_bits), (BitMap *) *pixmap,
			rect, &bounds, srcCopy, NULL);

	pixels = (jint *) malloc(sizeof(jint) * width * height);
	if (pixels == NULL) {
		UnlockPixels(pixmap);
		DisposeGWorld(gworld);
		return NULL;
	}

	base_addr = GetPixBaseAddr(pixmap);
	row_bytes = (**pixmap).rowBytes & 0x3fff;
	for (p = pixels, y = 0; y < height; ++y) {
		row = base_addr + y * row_bytes;
		for (x = 0; x < width; ++x) {
			++row;
			red   = *row++;
			green = *row++;
			blue  = *row++;
			*p++ = RGB_PIXEL(red, grren, blue);
		}
	}

	UnlockPixels(pixmap);
	DisposeGWorld(gworld);

	return pixels;
}

// ==================================================================

static void InvertSelection(const Rect *rect)
{
	Rect r;

	if (rect->right - rect->left <= 2
	 || rect->bottom - rect->top <= 2) {
		InvertRect(rect);
	} else {
		r = *rect;
		r.bottom = r.top + 1;
		InvertRect(&r);

		r = *rect;
		r.top = r.bottom - 1;
		InvertRect(&r);

		r = *rect;
		++r.top;
		--r.bottom;
		r.right = r.left + 1;
		InvertRect(&r);

		r = *rect;
		++r.top;
		--r.bottom;
		r.left = r.right - 1;
		InvertRect(&r);
	}
}

static int select_rectangle
  (int operation, Rect *rect)
{
	BitMap screen_bits;
	WindowRef window;
	GrafPtr saved_port;
	CursHandle cursor;
	Cursor arrow;
	Point start, mouse, prev;
	Boolean pressed = false, done = false;

	GetQDGlobalsScreenBits(&screen_bits);
	if (CreateNewWindow(kOverlayWindowClass,
			kWindowNoUpdatesAttribute | kWindowNoActivatesAttribute |
			kWindowOpaqueForEventsAttribute | kWindowNoShadowAttribute,
			&screen_bits.bounds, &window) != noErr) {
		return -1;
	}

	ShowWindow(window);
	GetPort(&saved_port);
	SetPort(GetWindowPort(window));
	BeginAppModalStateForWindow(window);

	CopyBits(&screen_bits,
			GetPortBitMapForCopyBits(GetWindowPort(window)),
			&screen_bits.bounds, &screen_bits.bounds, srcCopy, NULL);

	cursor = GetCursor(crossCursor);
	SetCursor(*cursor);

	SetPt(&prev, 0, 0);
	while (done == false) {			// evil loop...
		if (pressed == false) {
			if (Button()) {
				GetMouse(&start);
				SetRect(rect, start.h, start.v, start.h + 1, start.v + 1);
				InvertSelection(rect);
				prev = start;
				pressed = true;
			}
		} else {
			if (Button()) {
				GetMouse(&mouse);
				if (mouse.h != prev.h || mouse.v != prev.v) {
					InvertSelection(rect);
					if (operation == COLOR_SPUIT) {
						SetRect(rect,
								mouse.h, mouse.v, mouse.h + 1, mouse.v + 1);
					} else {
						if (mouse.h >= start.h) {
							rect->left = start.h;
							rect->right = mouse.h + 1;
						} else {
							rect->left = mouse.h;
							rect->right = start.h + 1;
						}
						if (mouse.v >= start.v) {
							rect->top = start.v;
							rect->bottom = mouse.v + 1;
						} else {
							rect->top = mouse.v;
							rect->bottom = start.v + 1;
						}
					}
					InvertSelection(rect);
					prev = mouse;
				}
			} else {
				InvertSelection(rect);
				done = true;
			}
		}
	}

	SetCursor(GetQDGlobalsArrow(&arrow));

	EndAppModalStateForWindow(window);
	SetPort(saved_port);
	DisposeWindow(window);

	return (pressed && done) ? 0 : -1;
}

// ==================================================================

static jintArray do_operation
  (JNIEnv *env, int operation, Rect *rect, jintArray dimension)
{
	jsize size = 0;
	jint *ptr = NULL, dimension_buf[2];
	jintArray array = NULL;

	if (operation == IMAGE_FROM_USER
	 || operation == RECTANGLE_FROM_USER
	 || operation == COLOR_SPUIT) {
		if (select_rectangle(operation, rect) != 0) {
			return NULL;
		}
	} else if (operation == IMAGE_OF_WHOLE_SCREEN) {
		GetRegionBounds(GetGrayRgn(), rect);
	}

	if (operation == IMAGE_FROM_USER
	 || operation == IMAGE_OF_AREA
	 || operation == IMAGE_OF_WHOLE_SCREEN
	 || operation == COLOR_SPUIT) {
		size = (rect->right - rect->left) * (rect->bottom - rect->top);
		ptr = capture_screen(rect);
	} else if (operation == RECTANGLE_FROM_USER) {
		size = 4;
		ptr = (jint *) malloc(sizeof(jint) * size);
		if (ptr != NULL) {
			ptr[0] = rect->left;
			ptr[1] = rect->top;
			ptr[2] = rect->right - rect->left;
			ptr[3] = rect->bottom - rect->top;
		}
	}

	if (ptr != NULL) {
		array = (*env)->NewIntArray(env, size);
		if (array != NULL) {
			(*env)->SetIntArrayRegion(env, array, 0, size, ptr);
		}
		free(ptr);

		if (dimension != NULL) {
			dimension_buf[0] = rect->right - rect->left;
			dimension_buf[1] = rect->bottom - rect->top;
			(*env)->SetIntArrayRegion(env, dimension, 0, 2, dimension_buf);
		}
	}

	return array;
}

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageFromUser
 * Signature: ([I)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageFromUser
  (JNIEnv *env, jobject obj, jintArray dimension)
{
	Rect rect;

	return do_operation(env, IMAGE_FROM_USER, &rect, dimension);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageOfArea
 * Signature: (IIII)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageOfArea
  (JNIEnv *env, jobject obj, jint x, jint y, jint width, jint height)
{
	Rect rect;

	SetRect(&rect, x, y, x + width, y + height);
	return do_operation(env, IMAGE_OF_AREA, &rect, NULL);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilImageOfWholeScreen
 * Signature: ([I)[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilImageOfWholeScreen
  (JNIEnv *env, jobject obj, jintArray dimension)
{
	Rect rect;

	return do_operation(env, IMAGE_OF_WHOLE_SCREEN, &rect, dimension);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilRectangleFromUser
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilRectangleFromUser
  (JNIEnv *env, jobject obj)
{
	Rect rect;

	return do_operation(env, RECTANGLE_FROM_USER, &rect, NULL);
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilColorSpuit
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilColorSpuit
  (JNIEnv *env, jobject obj)
{
	Rect rect;
	jintArray array;
	jint pixel = 0;

	array = do_operation(env, COLOR_SPUIT, &rect, NULL);
	if (array != NULL) {
		(*env)->GetIntArrayRegion(env, array, 0, 1, &pixel);
		// I believe that array will be GC'ed...
	}
	return pixel;
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilMousePoint
 * Signature: ()[I
 */
JNIEXPORT jintArray JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilMousePoint
  (JNIEnv *env, jobject obj)
{
	Point pt;
	jintArray array;
	jint buf[2];

	GetMouse(&pt);

	array = (*env)->NewIntArray(env, 2);
	if (array != NULL) {
		buf[0] = pt.h;
		buf[1] = pt.v;
		(*env)->SetIntArrayRegion(env, array, 0, 2, buf);
	}

	return array;
}

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilSetMousePoint
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilSetMousePoint
  (JNIEnv *env, jobject obj, jint x, jint y)
{
	CGDisplayMoveCursorToPoint(kCGDirectMainDisplay, CGPointMake(x, y));
}

// ==================================================================

/*
 * Class:     jp_co_sra_smalltalk_SystemInterface
 * Method:    utilBrowse
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_jp_co_sra_smalltalk_SystemInterface_utilBrowse
  (JNIEnv *env, jobject obj, jstring url)
{
	const char *jstr = (*env)->GetStringUTFChars(env, url, NULL);
	jsize jlen = (*env)->GetStringUTFLength(env, url);
	CFURLRef cfurl = CFURLCreateWithBytes(
			kCFAllocatorDefault, jstr, jlen, kCFStringEncodingUTF8, NULL);

	if (cfurl != NULL) {
		LSOpenCFURLRef(cfurl, NULL);
		CFRelease(cfurl);
	}
	(*env)->ReleaseStringUTFChars(env, url, jstr);
}
