/**
 * JunQTMacOSXUtils.m:
 *
 * 	@author:	Hoshi Takanori
 * 	@version:	2.0
 * 	@updated:	2004/01/09 (by Hoshi Takanori)
 *
 * 	$Id: JunQTMacOSXUtils.m,v 2.0 2005/02/17 00:38:39 hoshi Exp $
 */

#import <Cocoa/Cocoa.h>
#import <Movies.h>

#include <jawt.h>
#include <jawt_md.h>

#include "JunQTMacOSXUtils.h"


/*
 * static variables
 */

static NSMutableArray *portAssociations = nil;


/*
 * class interfaces
 */

@interface PortAssociation : NSObject
{
	NSView *parent;
	NSQuickDrawView *view;
	CGrafPtr port;
	Movie movie;
}
+ (PortAssociation *) findPortAssociationForMovie: (Movie) aMovie;
- (id) initWithParent: (NSView *) newParent;
- (void) dealloc;
- (NSRect) getFrame;
- (void) createView;
- (void) releaseView;
- (CGrafPtr) getPort;
- (GDHandle) gdHandle;
- (void) setMovie: (Movie) newMovie gdHandle: (GDHandle) newGDH;
- (void) unsetMovie;
- (void) setMovieBox: (const Rect *) newBox;
- (Boolean) isPortInitialized;
@end

@interface PortAssociationAwt : PortAssociation
{
	GDHandle gdh;
	Rect box;
}
- (NSRect) getFrame;
- (CGrafPtr) getPort;
- (GDHandle) gdHandle;
- (void) setMovie: (Movie) newMovie gdHandle: (GDHandle) newGDH;
- (void) unsetMovie;
- (void) setMovieBox: (const Rect *) newBox;
- (Boolean) isPortInitialized;
@end

@interface PortAssociationSwing : PortAssociationAwt
{
	JNIEnv *env;
	jobject component;
}
- (id) initWithParent: (NSView *) newParent
		env: (JNIEnv *) newEnv component: (jobject) newComponent;
- (void) dealloc;
- (NSRect) getFrame;
- (void) createView;
- (void) unsetMovie;
- (void) setMovieBox: (const Rect *) newBox;
@end


/*
 * port association for both AWT and Swing component on Jaguar
 */

@implementation PortAssociation

+ (PortAssociation *) findPortAssociationForMovie: (Movie) aMovie
{
	PortAssociation *assoc;
	int i;

	if (portAssociations == nil || aMovie == nil) {
		return nil;
	}

	for (i = 0; i < [portAssociations count]; ++i) {
		assoc = [portAssociations objectAtIndex: i];
		if (assoc->movie == aMovie) {
			return assoc;
		}
	}

	return nil;
}

- (id) initWithParent: (NSView *) newParent
{
	self = [super init];

	if (self != nil) {
		parent = newParent;
	}

	return self;
}

- (void) dealloc
{
	[self releaseView];

	[super dealloc];
}

- (NSRect) getFrame
{
	return NSMakeRect(0, 0, 0, 0);
}

- (void) createView
{
	view = [[NSQuickDrawView alloc] initWithFrame: [self getFrame]];
	[view setAutoresizingMask: NSViewWidthSizable | NSViewHeightSizable];
	[parent addSubview: view];
}

- (void) releaseView
{
	if (view != nil) {
		[view removeFromSuperview];
		[view release];
		view = nil;
		port = nil;
	}
}

- (CGrafPtr) getPort
{
	[view lockFocus];
	port = [view qdPort];
	[view unlockFocus];

	return port;
}

- (GDHandle) gdHandle
{
	CGrafPtr dummy;
	GDHandle gdh;

	GetMovieGWorld(movie, &dummy, &gdh);
	return gdh;
}

- (void) setMovie: (Movie) newMovie gdHandle: (GDHandle) newGDH
{
	movie = newMovie;
	SetMovieGWorld(movie, port, newGDH);
}

- (void) unsetMovie
{
	movie = nil;
}

- (void) setMovieBox: (const Rect *) newBox
{
	SetMovieBox(movie, newBox);
}

- (Boolean) isPortInitialized
{
	return true;
}

@end


/*
 * port association for AWT component on Panther
 */

@implementation PortAssociationAwt

- (NSRect) getFrame
{
	NSRect frame;

	frame = [parent frame];
	return NSMakeRect(0, 0, frame.size.width, frame.size.height);
}

- (CGrafPtr) getPort
{
	if (view == nil) {
		[self createView];
	}

	if (port == nil && view != nil && [view lockFocusIfCanDraw]) {
		port = [view qdPort];
		[view unlockFocus];
	}

	return port;
}

- (GDHandle) gdHandle
{
	return gdh;
}

- (void) setMovie: (Movie) newMovie gdHandle: (GDHandle) newGDH
{
	movie = newMovie;
	gdh = newGDH;
}

- (void) unsetMovie
{
	movie = nil;
	port = nil;
}

- (void) setMovieBox: (const Rect *) newBox
{
	if (port == nil) {
		box = *newBox;
	} else {
		SetMovieBox(movie, newBox);
	}
}

- (Boolean) isPortInitialized
{
	if (port != nil) {
		return true;
	}

	if ([self getPort] == nil) {
		return false;
	}

	SetMovieGWorld(movie, port, gdh);
	[self setMovieBox: &box];

	return true;
}

@end


/*
 * port association for Swing component on Panther
 */

static Boolean check_frame_size(JNIEnv *env, jobject component, NSSize size)
{
	jclass cls;
	jmethodID mid;
	jobject cp;
	jfieldID fid;
	int width, height;

	// java: cp = component.getContentPane();
	cls = (*env)->FindClass(env, "javax/swing/JFrame");
	if (cls == 0) { return false; }
	mid = (*env)->GetMethodID(env, cls,
			"getContentPane", "()Ljava/awt/Container;");
	if (mid == 0) { return false; }
	cp = (*env)->CallObjectMethod(env, component, mid);
	if (cp == 0) { return false; }

	// java: width = component.width, height = component.height;
	cls = (*env)->FindClass(env, "java/awt/Component");
	if (cls == 0) { return false; }
	fid = (*env)->GetFieldID(env, cls, "width", "I");
	if (fid == 0) { return false; }
	width = (*env)->GetIntField(env, cp, fid);
	fid = (*env)->GetFieldID(env, cls, "height", "I");
	if (fid == 0) { return false; }
	height = (*env)->GetIntField(env, cp, fid);

	return width == (int) size.width && height == (int) size.height;
}

@implementation PortAssociationSwing

- (id) initWithParent: (NSView *) newParent
		env: (JNIEnv *) newEnv component: (jobject) newComponent
{
	self = [super initWithParent: newParent];

	if (self != nil) {
		env = newEnv;
		component = (*env)->NewGlobalRef(env, newComponent);
	}

	return self;
}

- (void) dealloc
{
	(*env)->DeleteGlobalRef(env, component);

	[super dealloc];
}

- (NSRect) getFrame
{
	return NSMakeRect(
			box.left, box.top, box.right - box.left, box.bottom - box.top);
}

- (void) createView
{
	NSRect frame;

	frame = [parent frame];
	if (check_frame_size(env, component, frame.size)) {
		[super createView];
	}
}

- (void) unsetMovie
{
	[super unsetMovie];

	[self releaseView];
}

- (void) setMovieBox: (const Rect *) newBox
{
	Rect r;

	if (port == nil) {
		box = *newBox;
	} else {
		r = *newBox;
		OffsetRect(&r, - box.left, - box.top);
		SetMovieBox(movie, &r);
	}
}

@end


/*
 * functions
 */

static Boolean is_jaguar()
{
	OSErr err;
	long result;

	err = Gestalt(gestaltSystemVersion, &result);
	return err == noErr && result < 0x1030;
}

static Boolean is_jframe(JNIEnv *env, jobject component)
{
	jclass cls;

	cls = (*env)->FindClass(env, "javax/swing/JFrame");
	return cls != 0 && (*env)->IsInstanceOf(env, component, cls);
}

PortAssociation *JunMacOSX_CreatePortAssociation1(JNIEnv *env,
		jobject component, JAWT_MacOSXDrawingSurfaceInfo *platformInfo)
{
	PortAssociation *assoc;

	if (portAssociations == nil) {
		portAssociations = [[NSMutableArray alloc] init];
	}

	if (is_jaguar()) {
		assoc = [[PortAssociation alloc]
				initWithParent: platformInfo->cocoaViewRef];
		[assoc createView];
	} else if (! is_jframe(env, component)) {
		assoc = [[PortAssociationAwt alloc]
				initWithParent: platformInfo->cocoaViewRef];
		[assoc createView];
	} else {
		assoc = [[PortAssociationSwing alloc]
				initWithParent: platformInfo->cocoaViewRef
				env: env component: component];
	}

	[portAssociations addObject: assoc];
	[assoc release];

	return assoc;
}

PortAssociation *JunMacOSX_CreatePortAssociation2(PortAssociation *assoc)
{
	if (is_jaguar()) {
		[assoc getPort];
	}

	return assoc;
}

void JunMacOSX_DestroyPortAssociation(PortAssociation *assoc)
{
	[portAssociations removeObject: assoc];
}

void JunMacOSX_GetMoviePort(Movie movie, PortAssociation **assoc, GDHandle *gdh)
{
	*assoc = [PortAssociation findPortAssociationForMovie: movie];
	*gdh = [*assoc gdHandle];
}

void JunMacOSX_SetMoviePort(Movie movie, PortAssociation *assoc, GDHandle gdh)
{
	if (assoc != nil) {
		[assoc setMovie: movie gdHandle: gdh];
	} else {
		assoc = [PortAssociation findPortAssociationForMovie: movie];
		[assoc unsetMovie];
	}
}

void JunMacOSX_SetMovieBox(Movie movie, const Rect *box)
{
	PortAssociation *assoc;

	assoc = [PortAssociation findPortAssociationForMovie: movie];
	[assoc setMovieBox: box];
}

Boolean JunMacOSX_IsMoviePortInitialized(Movie movie)
{
	PortAssociation *assoc;

	assoc = [PortAssociation findPortAssociationForMovie: movie];
	return (assoc == nil) ? false : [assoc isPortInitialized];
}
