/*
 * This file is part of jDiffChaser.
 *
 *  jDiffChaser is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  jDiffChaser is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with jDiffChaser; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdiffchaser.testing;

import java.awt.Frame;
import org.jdiffchaser.scenarihandling.Player;
import org.jdiffchaser.scenarihandling.PlayerException;
import org.jdiffchaser.scenarihandling.PlayerListener;
import org.jdiffchaser.scenarihandling.RemoteControlFrame;
import org.jdiffchaser.utils.LogUtilities;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileInputStream;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;



public class DefaultPlayer {
    
    private static final Logger LOGGER = Logger.getLogger(DefaultPlayer.class.getName());    
    
    private static final int TIME_TO_WAIT = 2000;
        
    static{
        String os = System.getProperty("os.name").toLowerCase();
        System.out.println("os.name is " + os);
        if(os.indexOf("win")>=0){
            System.out.println("Loading Windows native library");
            System.loadLibrary("RCtrlFrameOnTop");     
            System.out.println("Native lib loaded");
        }
    }    
        
    public static class DefaultScenarioPlayer extends Player{
        public DefaultScenarioPlayer(String fullClassName, JFrame clientWindow, String screenshotsDirectory) 
            throws PlayerException{
            super(fullClassName, clientWindow, screenshotsDirectory);
        }

       public DefaultScenarioPlayer(String fullClassName, JFrame clientWindow, String hostName, 
                          int jmxPort, String screenshotsDirectory) throws PlayerException{
            super(fullClassName, clientWindow, hostName, jmxPort, screenshotsDirectory);
        }

        /**
         * args array containing the position parameters
         * extensive use of reflection to be able to test any version of your app, no matter
         * the package the version is in.
         */
        public boolean launch(String[] args){          
            
            LOGGER.fine("Launching application");
            LOGGER.finer("Classpath is" + System.getProperty("java.class.path"));
            LOGGER.finer("Args, before truncate, are" + Arrays.asList(args));
            try{
                String[] mainArgs = new String[0];
                if(mainArgs.length>0){
                    mainArgs = new String[args.length - 2];
                    System.arraycopy(args, 2, mainArgs, 0, args.length - 2);
                }
                
                Method mainMethod = mainClass.getMethod("main", new Class[]{String[].class});
                LOGGER.finer("main method parameters types are " + Arrays.asList(mainMethod.getParameterTypes()));
                mainMethod.invoke(null, new Object[]{mainArgs});
                Thread.sleep(1000); //needed to make the getFrame method successful in ALL cases... weird thing.
                Method getFrameMethod = mainClass.getMethod("getFrame", null);
                LOGGER.finer("getFrameMethod is " + getFrameMethod);
                clientWindow = (JFrame) getFrameMethod.invoke(null, null);
                LOGGER.fine("Application launched");
            }catch(Throwable t){ 
                t.printStackTrace();
                return false;
            }
            return true;
        }

        public boolean exit(final long timeBeforeExit, final boolean withRestart){
            new Thread(){
                public void run(){
                    Player.setWithRestart(withRestart);
                    try{
                        Thread.sleep(timeBeforeExit);
                    }catch(InterruptedException ie){
                        //
                    }
                    System.exit(0);
                }
            }.start();
            return true;
        }
        
        /**
         * In a dedicated player, override this method to have a correct version number.
         * If using this player, create a static getVersion method in the main class you're testing
         */
        public String getVersion(){
            Method getVersionMethod = null;
            try{
                getVersionMethod = mainClass.getMethod("getVersion", null);
            }catch(Throwable t){}
            if(getVersionMethod!=null){
                try {
                    return (String) getVersionMethod.invoke(null, null);
                }catch(Throwable t){
                    return "<Using DefaultPlayer: unknown version>";
                }
            }else{
                return "<Using DefaultPlayer: unknown version>";
            }
        }        
    }
    
    private static class PlayerStateListener implements PlayerListener{
        
        private String screenshotsDirectory;
        
        public PlayerStateListener(String scrDir){
            this.screenshotsDirectory = scrDir;
        }
        
        public void playerStarted(){
            LOGGER.info("Player Started");
        }

        public void playerAborted(){
            LOGGER.info("Player Aborted!!!");        
        }

        public void playerTerminated(){
            LOGGER.info("Player successfully terminated");  
            JFrame dummyF = new JFrame("dummyF"); //dummy frame to make the OptionPane "OnTop"
            dummyF.setVisible(true);
            try{
                dummyF.setAlwaysOnTop(true);
            }catch(NoSuchMethodError error){
                RemoteControlFrame.setWindowAlwaysOnTop("dummyF", true);
            }
            JOptionPane.showMessageDialog(dummyF, "End of scenario reached :)"
                                                + "\nYou can now exit the application and\ncheck the screenshot \n"
                                                + "into " + screenshotsDirectory + " directory."
                                          , "Player stopped", 
                                          JOptionPane.INFORMATION_MESSAGE);
            System.exit(0);
        }
        
    }
    
    private static PlayerListener stateListener;
    
    /**
     * args[0] must be the fully qualified class name to test (e.g.: ch.toto.positions.PositionContainer)
     *
     */
    public static void main(String[] args){
        
        if(args.length<2){
            System.out.println("Usage: DefaultPlayer [MainClassName] [screenshotDirectory] [ [App Arg1] [App Arg2] [...]]");
            System.exit(0);
        }          
        
       JFrame testFrame = new JFrame("ClientFrame");
       try{
           LogUtilities.configureSimpleConsoleLogs(Level.FINER.toString());

           JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
           chooser.showOpenDialog(null);

           File file = chooser.getSelectedFile();
           byte[] fileBytes = new byte[(int)file.length()];
           FileInputStream in = new FileInputStream(file);
           in.read(fileBytes);
           in.close();
           
           File screenshotsDir = new File(args[1]);
           if(!screenshotsDir.exists()){
               screenshotsDir.mkdirs();
           }
           
           stateListener = new PlayerStateListener(args[1]);
           
           final DefaultScenarioPlayer pl = new DefaultScenarioPlayer(args[0], 
                                                                      testFrame, 
                                                                      args[1]);
           
           
           pl.launch(args);
           
           testFrame.addWindowListener(new WindowAdapter(){
               public void windowClosed(WindowEvent we){
                   try{
                       pl.closePlayer();
                   }catch(Exception e){
                       e.printStackTrace();
                   }
               }
           });
           
           boolean ready = waitForPlayerToBeReady(pl);
           pl.addStateListener(stateListener);
           if(ready){
               pl.startScenario(fileBytes, true);
           }else{
               LOGGER.warning("Unable to play because we're unable to reach a ready application to test");
               System.exit(1);
           }
       }catch(Exception e){
           e.printStackTrace();
       }
    }
   
    private static boolean waitForPlayerToBeReady(Player localPlayer){
        long totalWaitingTime = 0;
        while(!localPlayer.isReadyToBeTested()){
            totalWaitingTime += TIME_TO_WAIT;
            if(totalWaitingTime > 30000){
                return false;
            }
            try{
                System.out.println("Waiting for app to be ready to test");
                Thread.sleep(TIME_TO_WAIT);
            }catch(Exception e){
                e.printStackTrace();
            }

        }
        return true;
    }    
    

    
}
