/*
 * $Id:FrameRuntimeElement.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.twister.frame;

import net.sf.jame.core.config.ListConfigElementEvents;
import net.sf.jame.core.config.ListRuntimeElement;
import net.sf.jame.core.config.RuntimeElement;
import net.sf.jame.core.config.ValueChangeEvent;
import net.sf.jame.core.config.ValueChangeListener;
import net.sf.jame.twister.frame.filter.FrameFilterConfigElement;
import net.sf.jame.twister.frame.filter.FrameFilterRuntimeElement;
import net.sf.jame.twister.frame.layer.GroupLayerConfigElement;
import net.sf.jame.twister.frame.layer.GroupLayerRuntimeElement;

/**
 * @author Andrea Medeghini
 */
public class FrameRuntimeElement extends RuntimeElement {
	private final ListRuntimeElement<GroupLayerRuntimeElement> layerListElement = new ListRuntimeElement<GroupLayerRuntimeElement>(1);
	private final ListRuntimeElement<FrameFilterRuntimeElement> filterListElement = new ListRuntimeElement<FrameFilterRuntimeElement>(0);
	private FilterListElementListener filtersListener;
	private LayerListElementListener layersListener;
	private FrameConfigElement frameElement;

	/**
	 * Constructs a new frame.
	 * 
	 * @param frameElement
	 */
	public FrameRuntimeElement(final FrameConfigElement frameElement) {
		if (frameElement == null) {
			throw new IllegalArgumentException("frameElement is null");
		}
		this.frameElement = frameElement;
		createFilters(frameElement);
		filtersListener = new FilterListElementListener();
		frameElement.getFilterListElement().addChangeListener(filtersListener);
		createLayers(frameElement);
		layersListener = new LayerListElementListener();
		frameElement.getLayerListElement().addChangeListener(layersListener);
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#dispose()
	 */
	@Override
	public void dispose() {
		if ((frameElement != null) && (filtersListener != null)) {
			frameElement.getFilterListElement().removeChangeListener(filtersListener);
		}
		filtersListener = null;
		if ((frameElement != null) && (layersListener != null)) {
			frameElement.getLayerListElement().removeChangeListener(layersListener);
		}
		layersListener = null;
		filterListElement.dispose();
		layerListElement.dispose();
		frameElement = null;
		super.dispose();
	}

	private void createFilters(final FrameConfigElement frameElement) {
		for (int i = 0; i < frameElement.getFilterConfigElementCount(); i++) {
			final FrameFilterConfigElement filterElement = frameElement.getFilterConfigElement(i);
			final FrameFilterRuntimeElement filter = new FrameFilterRuntimeElement(filterElement);
			appendFilter(filter);
		}
	}

	private void createLayers(final FrameConfigElement frameElement) {
		for (int i = 0; i < frameElement.getLayerConfigElementCount(); i++) {
			final GroupLayerConfigElement layerElement = frameElement.getLayerConfigElement(i);
			final GroupLayerRuntimeElement layer = new GroupLayerRuntimeElement(layerElement);
			appendLayer(layer);
		}
	}

	/**
	 * Returns a layer.
	 * 
	 * @param index the layer index.
	 * @return the layer.
	 */
	public GroupLayerRuntimeElement getLayer(final int index) {
		return layerListElement.getElement(index);
	}

	/**
	 * Returns the layer index.
	 * 
	 * @param layer the layer.
	 * @return the index.
	 */
	public int indexOfLayer(final GroupLayerRuntimeElement layer) {
		return layerListElement.indexOfElement(layer);
	}

	/**
	 * Returns the number of layers.
	 * 
	 * @return the number of layers.
	 */
	public int getLayerCount() {
		return layerListElement.getElementCount();
	}

	private void appendLayer(final GroupLayerRuntimeElement layer) {
		layerListElement.appendElement(layer);
	}

	private void insertLayerAfter(final int index, final GroupLayerRuntimeElement layer) {
		layerListElement.insertElementAfter(index, layer);
	}

	private void insertLayerBefore(final int index, final GroupLayerRuntimeElement layer) {
		layerListElement.insertElementBefore(index, layer);
	}

	private void removeLayer(final int index) {
		layerListElement.getElement(index).dispose();
		layerListElement.removeElement(index);
	}

	/**
	 * Returns a filter.
	 * 
	 * @param index the filter index.
	 * @return the filter.
	 */
	public FrameFilterRuntimeElement getFilter(final int index) {
		return filterListElement.getElement(index);
	}

	/**
	 * Returns the filter index.
	 * 
	 * @param filter the filter.
	 * @return the index.
	 */
	public int indexOfFilter(final FrameFilterRuntimeElement filter) {
		return filterListElement.indexOfElement(filter);
	}

	/**
	 * Returns the number of filters.
	 * 
	 * @return the number of filters.
	 */
	public int getFilterCount() {
		return filterListElement.getElementCount();
	}

	private void appendFilter(final FrameFilterRuntimeElement filter) {
		filterListElement.appendElement(filter);
	}

	private void insertFilterAfter(final int index, final FrameFilterRuntimeElement filter) {
		filterListElement.insertElementAfter(index, filter);
	}

	private void insertFilterBefore(final int index, final FrameFilterRuntimeElement filter) {
		filterListElement.insertElementBefore(index, filter);
	}

	private void removeFilter(final int index) {
		filterListElement.getElement(index).dispose();
		filterListElement.removeElement(index);
	}

	/**
	 * @see net.sf.jame.core.config.RuntimeElement#isChanged()
	 */
	@Override
	public boolean isChanged() {
		final boolean layersChanged = layerListElement.isChanged();
		final boolean filtersChanged = filterListElement.isChanged();
		return super.isChanged() || layersChanged || filtersChanged;
	}

	private class LayerListElementListener implements ValueChangeListener, ListConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case ELEMENT_ADDED: {
					appendLayer(new GroupLayerRuntimeElement((GroupLayerConfigElement) e.getParams()[0]));
					fireChanged();
					break;
				}
				case ELEMENT_INSERTED_AFTER: {
					insertLayerAfter(((Integer) e.getParams()[1]).intValue(), new GroupLayerRuntimeElement((GroupLayerConfigElement) e.getParams()[0]));
					fireChanged();
					break;
				}
				case ELEMENT_INSERTED_BEFORE: {
					insertLayerBefore(((Integer) e.getParams()[1]).intValue(), new GroupLayerRuntimeElement((GroupLayerConfigElement) e.getParams()[0]));
					fireChanged();
					break;
				}
				case ELEMENT_REMOVED: {
					removeLayer(((Integer) e.getParams()[1]).intValue());
					fireChanged();
					break;
				}
				default: {
					break;
				}
			}
		}
	}

	private class FilterListElementListener implements ValueChangeListener, ListConfigElementEvents {
		/**
		 * @see net.sf.jame.core.config.ValueChangeListener#valueChanged(net.sf.jame.core.config.ValueChangeEvent)
		 */
		public void valueChanged(final ValueChangeEvent e) {
			switch (e.getEventType()) {
				case ELEMENT_ADDED: {
					appendFilter(new FrameFilterRuntimeElement((FrameFilterConfigElement) e.getParams()[0]));
					fireChanged();
					break;
				}
				case ELEMENT_INSERTED_AFTER: {
					insertFilterAfter(((Integer) e.getParams()[1]).intValue(), new FrameFilterRuntimeElement((FrameFilterConfigElement) e.getParams()[0]));
					fireChanged();
					break;
				}
				case ELEMENT_INSERTED_BEFORE: {
					insertFilterBefore(((Integer) e.getParams()[1]).intValue(), new FrameFilterRuntimeElement((FrameFilterConfigElement) e.getParams()[0]));
					fireChanged();
					break;
				}
				case ELEMENT_REMOVED: {
					removeFilter(((Integer) e.getParams()[1]).intValue());
					fireChanged();
					break;
				}
				default: {
					break;
				}
			}
		}
	}
}
