/*
 * $Id:MovieClipTableModel.java 486 2008-01-27 09:20:26Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.swing;

import java.util.ArrayList;
import java.util.List;

import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableModel;

import net.sf.jame.service.AsyncService;
import net.sf.jame.service.AsyncService.ServiceCallback;
import net.sf.jame.service.AsyncService.ServiceVoidCallback;
import net.sf.jame.service.clip.MovieClipDataRow;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class MovieClipTableModel extends ServiceAdapter implements TableModel {
	private static final Logger logger = Logger.getLogger(MovieClipTableModel.class);
	private static final long serialVersionUID = 1L;
	public static final int CLIPID = 0;
	public static final int NAME = 1;
	public static final int DESCRIPTION = 2;
	public static final int DURATION = 3;
	public static final int STATUS = 4;
	private final List<TableModelListener> listeners;
	private final List<MovieClipDataRow> model;
	private final AsyncService service;

	/**
	 * @param service
	 */
	public MovieClipTableModel(final AsyncService service) {
		listeners = new ArrayList<TableModelListener>();
		model = new ArrayList<MovieClipDataRow>();
		this.service = service;
		service.addServiceListener(new ServiceVoidCallback() {
			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
			 */
			public void executed() {
			}

			/**
			 * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
			 */
			public void failed(final Throwable throwable) {
			}
		}, this);
		this.reload();
	}

	/**
	 * @see javax.swing.table.TableModel#addTableModelListener(javax.swing.event.TableModelListener)
	 */
	public void addTableModelListener(final TableModelListener listener) {
		listeners.add(listener);
	}

	/**
	 * @see javax.swing.table.TableModel#removeTableModelListener(javax.swing.event.TableModelListener)
	 */
	public void removeTableModelListener(final TableModelListener listener) {
		listeners.remove(listener);
	}

	/**
	 * @param e
	 */
	protected void fireTableChanged(final TableModelEvent e) {
		for (final TableModelListener listener : listeners) {
			listener.tableChanged(e);
		}
	}

	/**
	 * @see javax.swing.table.TableModel#getColumnClass(int)
	 */
	public Class<?> getColumnClass(final int columnIndex) {
		switch (columnIndex) {
			case CLIPID: {
				return String.class;
			}
			case NAME: {
				return String.class;
			}
			case DESCRIPTION: {
				return String.class;
			}
			case DURATION: {
				return String.class;
			}
			case STATUS: {
				return String.class;
			}
			default: {
				break;
			}
		}
		return null;
	}

	/**
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	public int getColumnCount() {
		return 5;
	}

	/**
	 * @see javax.swing.table.TableModel#getColumnName(int)
	 */
	public String getColumnName(final int columnIndex) {
		switch (columnIndex) {
			case CLIPID: {
				return ServiceSwingResources.getInstance().getString("column.id");
			}
			case NAME: {
				return ServiceSwingResources.getInstance().getString("column.name");
			}
			case DESCRIPTION: {
				return ServiceSwingResources.getInstance().getString("column.description");
			}
			case DURATION: {
				return ServiceSwingResources.getInstance().getString("column.duration");
			}
			case STATUS: {
				return ServiceSwingResources.getInstance().getString("column.status");
			}
			default: {
				break;
			}
		}
		return null;
	}

	/**
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount() {
		return model.size();
	}

	/**
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	public Object getValueAt(final int rowIndex, final int columnIndex) {
		final MovieClipDataRow dataRow = model.get(rowIndex);
		switch (columnIndex) {
			case CLIPID: {
				return String.valueOf(dataRow.getClipId());
			}
			case NAME: {
				return dataRow.getClipName();
			}
			case DESCRIPTION: {
				return dataRow.getDescription();
			}
			case DURATION: {
				final long h = dataRow.getDuration() / 3600000;
				final long t1 = dataRow.getDuration() % 3600000;
				final long m = t1 / 60000;
				final long t2 = t1 % 60000;
				final long s = t2 / 1000;
				final long ms = t2 % 1000;
				return String.format("%d:%02d:%02d:%03d", h, m, s, ms);
			}
			case STATUS: {
				return String.valueOf(dataRow.getStatus() == 0 ? ServiceSwingResources.getInstance().getString("label.unlocked") : ServiceSwingResources.getInstance().getString("label.locked") + " (" + dataRow.getStatus() + ")");
			}
			default: {
				break;
			}
		}
		return null;
	}

	/**
	 * @see javax.swing.table.TableModel#isCellEditable(int, int)
	 */
	public boolean isCellEditable(final int rowIndex, final int columnIndex) {
		// if (model.get(rowIndex).getStatus() != 0) {
		// return false;
		// }
		// switch (columnIndex) {
		// case NAME: {
		// return true;
		// }
		// case DESCRIPTION: {
		// return true;
		// }
		// default: {
		// break;
		// }
		// }
		return false;
	}

	/**
	 * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
	 */
	public void setValueAt(final Object aValue, final int rowIndex, final int columnIndex) {
		// final MovieClipDataRow dataRow = model.get(rowIndex);
		// if (aValue != null) {
		// switch (columnIndex) {
		// case NAME: {
		// dataRow.setClipName((String) aValue);
		// saveRow(rowIndex);
		// fireTableChanged(new TableModelEvent(this, rowIndex, rowIndex, MovieClipTableModel.NAME, TableModelEvent.UPDATE));
		// break;
		// }
		// case DESCRIPTION: {
		// dataRow.setDescription((String) aValue);
		// saveRow(rowIndex);
		// fireTableChanged(new TableModelEvent(this, rowIndex, rowIndex, MovieClipTableModel.DESCRIPTION, TableModelEvent.UPDATE));
		// break;
		// }
		// default: {
		// break;
		// }
		// }
		// }
	}

	/**
	 * 
	 */
	public void clear() {
		final int lastRow = model.size() - 1;
		model.clear();
		if (lastRow >= 0) {
			fireTableChanged(new TableModelEvent(this, 0, lastRow, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
		}
	}

	/**
	 * 
	 */
	public void reload() {
		service.loadClips(new ServiceCallback<List<MovieClipDataRow>>() {
			/**
			 * @param value
			 */
			public void executed(final List<MovieClipDataRow> clips) {
				try {
					if (MovieClipTableModel.logger.isInfoEnabled()) {
						MovieClipTableModel.logger.info("Reload executed");
					}
					SwingUtilities.invokeAndWait(new Runnable() {
						/**
						 * @see java.lang.Runnable#run()
						 */
						public void run() {
							MovieClipTableModel.this.reload(clips);
						}
					});
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}

			/**
			 * @see net.sf.jame.service.AsyncService.ServiceCallback#failed(java.lang.Throwable)
			 */
			public void failed(final Throwable throwable) {
				try {
					MovieClipTableModel.logger.error("Reload failed", throwable);
					SwingUtilities.invokeAndWait(new Runnable() {
						/**
						 * @see java.lang.Runnable#run()
						 */
						public void run() {
							MovieClipTableModel.this.reload(new ArrayList<MovieClipDataRow>());
						}
					});
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
		});
	}

	private void reload(final List<MovieClipDataRow> clips) {
		int lastRow = model.size() - 1;
		model.clear();
		if (lastRow >= 0) {
			fireTableChanged(new TableModelEvent(this, 0, lastRow, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
		}
		for (final MovieClipDataRow clip : clips) {
			model.add(clip);
		}
		lastRow = model.size() - 1;
		if (lastRow >= 0) {
			fireTableChanged(new TableModelEvent(this, 0, lastRow, TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT));
		}
	}

	// private void saveRow(final int rowIndex) {
	// final MovieClipDataRow dataRow = model.get(rowIndex);
	// service.saveClip(new ServiceVoidCallback() {
	// /**
	// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#executed()
	// */
	// public void executed() {
	// if (MovieClipTableModel.logger.isInfoEnabled()) {
	// MovieClipTableModel.logger.info("Save executed");
	// }
	// }
	//
	// /**
	// * @see net.sf.jame.service.AsyncService.ServiceVoidCallback#failed(java.lang.Throwable)
	// */
	// public void failed(final Throwable throwable) {
	// MovieClipTableModel.logger.error("Save failed", throwable);
	// }
	// }, dataRow);
	// }
	/**
	 * @param rowIndex
	 * @return
	 */
	public MovieClipDataRow getClip(final int rowIndex) {
		return model.get(rowIndex);
	}

	/**
	 * @see net.sf.jame.service.swing.ServiceAdapter#clipCreated(net.sf.jame.service.clip.MovieClipDataRow)
	 */
	@Override
	public void clipCreated(final MovieClipDataRow clip) {
		SwingUtilities.invokeLater(new TableModelAdapter(this) {
			/**
			 * @see java.lang.Runnable#run()
			 */
			@Override
			public void run() {
				for (int rowIndex = 0; rowIndex < model.size(); rowIndex++) {
					if (model.get(rowIndex).getClipId() == clip.getClipId()) {
						model.remove(rowIndex);
						MovieClipTableModel.this.fireTableChanged(new TableModelEvent(tableModel, rowIndex, rowIndex, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
						break;
					}
				}
				final int rowIndex = model.size();
				model.add(clip);
				MovieClipTableModel.this.fireTableChanged(new TableModelEvent(tableModel, rowIndex, rowIndex, TableModelEvent.ALL_COLUMNS, TableModelEvent.INSERT));
			}
		});
	}

	/**
	 * @see net.sf.jame.service.swing.ServiceAdapter#clipDeleted(net.sf.jame.service.clip.MovieClipDataRow)
	 */
	@Override
	public void clipDeleted(final MovieClipDataRow clip) {
		SwingUtilities.invokeLater(new TableModelAdapter(this) {
			/**
			 * @see java.lang.Runnable#run()
			 */
			@Override
			public void run() {
				for (int rowIndex = 0; rowIndex < model.size(); rowIndex++) {
					if (model.get(rowIndex).getClipId() == clip.getClipId()) {
						model.remove(rowIndex);
						MovieClipTableModel.this.fireTableChanged(new TableModelEvent(tableModel, rowIndex, rowIndex, TableModelEvent.ALL_COLUMNS, TableModelEvent.DELETE));
						break;
					}
				}
			}
		});
	}

	/**
	 * @see net.sf.jame.service.swing.ServiceAdapter#clipUpdated(net.sf.jame.service.clip.MovieClipDataRow)
	 */
	@Override
	public void clipUpdated(final MovieClipDataRow clip) {
		SwingUtilities.invokeLater(new TableModelAdapter(this) {
			/**
			 * @see java.lang.Runnable#run()
			 */
			@Override
			public void run() {
				for (int rowIndex = 0; rowIndex < model.size(); rowIndex++) {
					if (model.get(rowIndex).getClipId() == clip.getClipId()) {
						model.set(rowIndex, clip);
						MovieClipTableModel.this.fireTableChanged(new TableModelEvent(tableModel, rowIndex, rowIndex, TableModelEvent.ALL_COLUMNS, TableModelEvent.UPDATE));
						break;
					}
				}
			}
		});
	}

	private class TableModelAdapter implements Runnable {
		public TableModel tableModel;

		public TableModelAdapter(final TableModel tableModel) {
			this.tableModel = tableModel;
		}

		public void run() {
		}
	}
}
