/*
 * $Id:PostProcessorJob.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.spool.impl;

import java.awt.RenderingHints;
import java.awt.image.DataBufferInt;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.util.HashMap;
import java.util.Map;

import net.sf.jame.core.xml.XML;
import net.sf.jame.service.Service;
import net.sf.jame.service.job.RenderJobDataRow;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.SpoolJobInterface;
import net.sf.jame.twister.ImageTile;
import net.sf.jame.twister.IntegerVector2D;
import net.sf.jame.twister.TwisterClip;
import net.sf.jame.twister.TwisterClipController;
import net.sf.jame.twister.TwisterClipXMLImporter;
import net.sf.jame.twister.TwisterConfig;
import net.sf.jame.twister.TwisterRuntime;
import net.sf.jame.twister.renderer.OverlayTwisterRenderer;
import net.sf.jame.twister.renderer.Surface;
import net.sf.jame.twister.renderer.TwisterRenderer;
import net.sf.jame.twister.renderer.TwisterRenderingHints;

import org.w3c.dom.Document;

/**
 * @author Andrea Medeghini
 */
public class PostProcessorJob implements SpoolJobInterface {
	private final JobListener listener;
	private final String jobId;
	private long lastUpdate;
	private boolean started;
	private boolean aborted;
	private boolean terminated;
	private Thread thread;
	private RenderJobDataRow jobDataRow;
	private final Service service;

	/**
	 * @param service
	 * @param jobId
	 * @param listener
	 */
	public PostProcessorJob(final Service service, final String jobId, final JobListener listener) {
		lastUpdate = System.currentTimeMillis();
		this.listener = listener;
		this.service = service;
		this.jobId = jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getJobId()
	 */
	public String getJobId() {
		return jobId;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getFrameNumber()
	 */
	public int getFrameNumber() {
		return jobDataRow.getFrameNumber();
	}

	/**
	 * @param frameNumber
	 */
	public void setFrameNumber(final int frameNumber) {
		jobDataRow.setFrameNumber(frameNumber);
		lastUpdate = System.currentTimeMillis();
		listener.stateChanged(this);
	}

	/**
	 * @return the jobDataRow
	 */
	public RenderJobDataRow getJobDataRow() {
		return jobDataRow;
	}

	/**
	 * @param jobDataRow the jobDataRow to set
	 */
	public void setJobDataRow(final RenderJobDataRow jobDataRow) {
		this.jobDataRow = jobDataRow;
	}

	/**
	 * @see net.sf.jame.service.spool.SpoolJobInterface#getClip()
	 */
	public TwisterClip getClip() throws IOException {
		try {
			final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
			final InputStream is = service.getClipInputStream(jobDataRow.getClipId());
			final Document doc = XML.loadDocument(is, "twister-clip.xml");
			return importer.importFromElement(doc.getDocumentElement());
		}
		catch (final Exception e) {
			throw new IOException(e.getMessage());
		}
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder builder = new StringBuilder();
		builder.append("id = ");
		builder.append(jobId);
		builder.append(", frameNumber = ");
		builder.append(getFrameNumber());
		return builder.toString();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#getLastUpdate()
	 */
	public long getLastUpdate() {
		return lastUpdate;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#reset()
	 */
	public void reset() {
		started = false;
		aborted = false;
		terminated = false;
		lastUpdate = System.currentTimeMillis();
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#start()
	 */
	public void start() {
		started = true;
		aborted = false;
		terminated = false;
		if (thread == null) {
			thread = new Thread(new RenderTask(), "PostProcessorJob Task");
			thread.setPriority(Thread.MIN_PRIORITY);
			thread.setDaemon(true);
			thread.start();
		}
		listener.started(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#stop()
	 */
	public void stop() {
		started = false;
		if (thread != null) {
			// thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
		listener.stopped(this);
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#abort()
	 */
	public void abort() {
		aborted = true;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#dispose()
	 */
	public void dispose() {
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isStarted()
	 */
	public boolean isStarted() {
		return started;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isAborted()
	 */
	public boolean isAborted() {
		return aborted;
	}

	/**
	 * @see net.sf.jame.service.spool.JobInterface#isTerminated()
	 */
	public boolean isTerminated() {
		return terminated;
	}

	private void terminate() {
		listener.terminated(this);
	}

	private class RenderTask implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			RandomAccessFile raf = null;
			InputStream is = null;
			try {
				is = service.getClipInputStream(jobDataRow.getClipId());
				final TwisterClipXMLImporter importer = new TwisterClipXMLImporter();
				final Document doc = XML.loadDocument(is, "twister-clip.xml");
				final TwisterClip clip = importer.importFromElement(doc.getDocumentElement());
				if (clip.getSequenceCount() > 0) {
					final int frameCount = (jobDataRow.getStopTime() - jobDataRow.getStartTime()) * jobDataRow.getFrameRate();
					int frameTimeInMillis = 0;
					final int iw = jobDataRow.getImageWidth();
					final int ih = jobDataRow.getImageHeight();
					final byte[] row = new byte[iw * 4];
					final Surface surface = new Surface(iw, ih);
					surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_ALPHA_INTERPOLATION, RenderingHints.VALUE_ALPHA_INTERPOLATION_SPEED);
					surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
					surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
					surface.getGraphics2D().setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
					final TwisterClipController controller = new TwisterClipController(clip);
					controller.init();
					final TwisterConfig config = controller.getConfig();
					final TwisterRuntime overlayRuntime = new TwisterRuntime(config);
					final TwisterRenderer overlayRenderer = new OverlayTwisterRenderer(overlayRuntime);
					final Map<Object, Object> overlayHints = new HashMap<Object, Object>();
					overlayHints.put(TwisterRenderingHints.KEY_MEMORY, TwisterRenderingHints.MEMORY_LOW);
					overlayHints.put(TwisterRenderingHints.KEY_TYPE, TwisterRenderingHints.TYPE_OVERLAY);
					overlayRenderer.setRenderingHints(overlayHints);
					final IntegerVector2D size = new IntegerVector2D(iw, ih);
					overlayRenderer.setTile(new ImageTile(size, size, new IntegerVector2D(0, 0), new IntegerVector2D(0, 0)));
					final int[] data = ((DataBufferInt) (surface.getImage().getRaster().getDataBuffer())).getData();
					raf = service.getProfileRandomAccessFile(jobDataRow.getProfileId());
					int startFrameNumber = 0;
					if (jobDataRow.getFrameNumber() > 0) {
						startFrameNumber = jobDataRow.getFrameNumber() + 1;
						frameTimeInMillis = jobDataRow.getStartTime() * 1000 + (jobDataRow.getFrameNumber() * 1000) / jobDataRow.getFrameRate();
						controller.redoAction(frameTimeInMillis, false);
						overlayRenderer.prepareImage(false);
						overlayRenderer.render();
					}
					for (int frameNumber = 0; frameNumber < startFrameNumber; frameNumber++) {
						frameTimeInMillis = jobDataRow.getStartTime() * 1000 + (frameNumber * 1000) / jobDataRow.getFrameRate();
						controller.redoAction(frameTimeInMillis, false);
						if (overlayRuntime.getEffectElement().getEffectRuntime() != null) {
							overlayRuntime.getEffectElement().getEffectRuntime().setSize(size);
							overlayRuntime.getEffectElement().getEffectRuntime().prepareEffect();
						}
					}
					for (int frameNumber = startFrameNumber; frameNumber < frameCount; frameNumber++) {
						frameTimeInMillis = jobDataRow.getStartTime() * 1000 + (frameNumber * 1000) / jobDataRow.getFrameRate();
						controller.redoAction(frameTimeInMillis, false);
						overlayRenderer.prepareImage(false);
						overlayRenderer.render();
						long pos = frameNumber * iw * ih * 4;
						for (int j = 0, k = 0; k < ih; k++) {
							raf.seek(pos);
							raf.readFully(row);
							for (int i = 0; i < row.length; i += 4) {
								data[j] = (((row[i + 3]) & 0xFF) << 24) | (((row[i + 0]) & 0xFF) << 16) | (((row[i + 1]) & 0xFF) << 8) | ((row[i + 2]) & 0xFF);
								j += 1;
							}
							pos += row.length;
							Thread.yield();
						}
						if (overlayRuntime.getEffectElement().getEffectRuntime() != null) {
							overlayRuntime.getEffectElement().getEffectRuntime().setSize(size);
							overlayRuntime.getEffectElement().getEffectRuntime().prepareEffect();
							overlayRuntime.getEffectElement().getEffectRuntime().renderImage(surface);
						}
						overlayRenderer.drawImage(surface.getGraphics2D());
						pos = frameNumber * iw * ih * 4;
						for (int j = 0, k = 0; k < ih; k++) {
							for (int i = 0; i < row.length; i += 4) {
								row[i + 0] = (byte) ((data[j] & 0x00FF0000) >> 16);
								row[i + 1] = (byte) ((data[j] & 0x0000FF00) >> 8);
								row[i + 2] = (byte) ((data[j] & 0x000000FF) >> 0);
								row[i + 3] = (byte) ((data[j] & 0xFF000000) >> 24);
								j += 1;
							}
							raf.seek(pos);
							raf.write(row);
							pos += row.length;
							Thread.yield();
						}
						setFrameNumber(frameNumber);
						if (aborted) {
							break;
						}
					}
					overlayRenderer.dispose();
					overlayRuntime.dispose();
					surface.dispose();
				}
			}
			catch (final Exception e) {
				aborted = true;
				e.printStackTrace();
			}
			finally {
				if (raf != null) {
					try {
						raf.close();
					}
					catch (final IOException e) {
					}
				}
				if (is != null) {
					try {
						is.close();
					}
					catch (final IOException e) {
					}
				}
			}
			terminated = true;
			terminate();
		}
	}
}
