/*
 * $Id:DefaultJobService.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.spool;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class DefaultJobService<T extends JobInterface> implements JobService<T> {
	private static final Logger logger = Logger.getLogger(DefaultJobService.class);
	private List<JobServiceListener> listeners = new LinkedList<JobServiceListener>();
	private final HashMap<String, ScheduledJob> scheduledJobs = new HashMap<String, ScheduledJob>();
	private final HashMap<String, T> terminatedJobs = new HashMap<String, T>();
	private final HashMap<String, T> spooledJobs = new HashMap<String, T>();
	private final JobFactory<T> jobFactory;
	private final Object monitor = new Object();
	private Thread thread;
	private boolean running;
	private boolean dirty;
	private int jobCount;
	private final int maxJobCount;

	/**
	 * @param jobFactory
	 */
	public DefaultJobService(final JobFactory<T> jobFactory) {
		this.jobFactory = jobFactory;
		maxJobCount = Integer.getInteger("jame.maxJobCount", 5);
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#start()
	 */
	public void start() {
		if (thread == null) {
			thread = new Thread(new ServiceHandler(), "Local JobService Thread");
			thread.setDaemon(true);
			running = true;
			thread.start();
		}
		fireStateChanged("Local service started");
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#stop()
	 */
	public void stop() {
		if (thread != null) {
			running = false;
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
		fireStateChanged("Local service stopped");
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#getJobCount()
	 */
	public int getJobCount() {
		synchronized (spooledJobs) {
			return spooledJobs.size();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#deleteJob(java.lang.String)
	 */
	public void deleteJob(final String jobId) {
		synchronized (scheduledJobs) {
			synchronized (spooledJobs) {
				final JobInterface job = spooledJobs.remove(jobId);
				if (job != null) {
					if (job.isStarted()) {
						job.abort();
						job.stop();
					}
					job.dispose();
				}
				if (scheduledJobs.remove(jobId) != null) {
					if (jobCount > 0) {
						jobCount -= 1;
					}
				}
			}
			synchronized (terminatedJobs) {
				terminatedJobs.remove(jobId);
			}
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#abortJob(java.lang.String)
	 */
	public void abortJob(final String jobId) {
		synchronized (spooledJobs) {
			final JobInterface job = spooledJobs.get(jobId);
			if (job != null) {
				job.abort();
			}
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#createJob(net.sf.jame.service.spool.JobListener)
	 */
	public String createJob(final JobListener listener) {
		synchronized (spooledJobs) {
			final T job = jobFactory.createJob(JobIDFactory.newJobId(), new ServiceJobListener(listener));
			spooledJobs.put(job.getJobId(), job);
			return job.getJobId();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#getJob(java.lang.String)
	 */
	public T getJob(final String jobId) {
		synchronized (spooledJobs) {
			synchronized (terminatedJobs) {
				T job = spooledJobs.get(jobId);
				if (job == null) {
					job = terminatedJobs.get(jobId);
				}
				return job;
			}
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#runJob(java.lang.String)
	 */
	public void runJob(final String jobId) {
		synchronized (scheduledJobs) {
			synchronized (spooledJobs) {
				final T job = spooledJobs.get(jobId);
				if (job != null) {
					scheduledJobs.put(jobId, new ScheduledJob(job));
				}
			}
		}
		synchronized (monitor) {
			dirty = true;
			monitor.notify();
		}
	}

	private class ServiceHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				final long pollingTime = 10 * 1000L;
				while (running) {
					synchronized (scheduledJobs) {
						synchronized (terminatedJobs) {
							final Iterator<ScheduledJob> jobIterator = scheduledJobs.values().iterator();
							while (jobIterator.hasNext()) {
								final ScheduledJob scheduledJob = jobIterator.next();
								if (scheduledJob.isTerminated()) {
									jobIterator.remove();
									scheduledJob.stop();
									scheduledJob.dispose();
									terminatedJobs.put(scheduledJob.getJob().getJobId(), scheduledJob.getJob());
									if (jobCount > 0) {
										jobCount -= 1;
									}
									logger.info("Job terminated " + scheduledJob.getJob() + " (jobs = " + jobCount + ")");
								}
								else if (scheduledJob.isStarted() && (System.currentTimeMillis() - scheduledJob.getJob().getLastUpdate()) > 60 * 1000L) {
									jobIterator.remove();
									scheduledJob.abort();
									scheduledJob.stop();
									scheduledJob.dispose();
									terminatedJobs.put(scheduledJob.getJob().getJobId(), scheduledJob.getJob());
									if (jobCount > 0) {
										jobCount -= 1;
									}
									logger.info("Job terminated " + scheduledJob.getJob() + " (jobs = " + jobCount + ")");
								}
								else if (!scheduledJob.isAborted() && !scheduledJob.isStarted() && jobCount < maxJobCount) {
									jobCount += 1;
									scheduledJob.start();
									logger.info("Job started " + scheduledJob.getJob() + " (jobs = " + jobCount + ")");
								}
							}
						}
					}
					synchronized (spooledJobs) {
						synchronized (terminatedJobs) {
							final Iterator<T> jobIterator = terminatedJobs.values().iterator();
							while (jobIterator.hasNext()) {
								final T job = jobIterator.next();
								if ((System.currentTimeMillis() - job.getLastUpdate()) > 120 * 1000L) {
									logger.info("Remove terminated job " + job);
									spooledJobs.remove(job.getJobId());
									jobIterator.remove();
									job.dispose();
								}
							}
						}
					}
					synchronized (monitor) {
						if (!dirty) {
							monitor.wait(pollingTime);
						}
						else {
							Thread.yield();
						}
						dirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
		}
	}

	private class ServiceJobListener implements JobListener {
		private final JobListener listener;

		/**
		 * @param listener
		 */
		protected ServiceJobListener(final JobListener listener) {
			this.listener = listener;
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stateChanged(net.sf.jame.service.spool.JobInterface)
		 */
		public void stateChanged(final JobInterface job) {
			listener.stateChanged(job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#started(net.sf.jame.service.spool.JobInterface)
		 */
		public void started(final JobInterface job) {
			listener.started(job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stopped(net.sf.jame.service.spool.JobInterface)
		 */
		public void stopped(final JobInterface job) {
			listener.stopped(job);
		}

		/**
		 * @see net.sf.jame.service.spool.JobListener#stopped(net.sf.jame.service.spool.JobInterface)
		 */
		public void terminated(final JobInterface job) {
			listener.terminated(job);
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}
	}

	private class ScheduledJob {
		private final T job;

		/**
		 * @param job
		 */
		public ScheduledJob(final T job) {
			this.job = job;
		}

		/**
		 * @return
		 */
		public boolean isTerminated() {
			return job.isTerminated();
		}

		/**
		 * @return
		 */
		public boolean isAborted() {
			return job.isAborted();
		}

		/**
		 * @return
		 */
		public boolean isStarted() {
			return job.isStarted();
		}

		/**
		 * 
		 */
		public void stop() {
			job.stop();
		}

		/**
		 * 
		 */
		public void start() {
			job.start();
		}

		/**
		 * 
		 */
		public void abort() {
			job.abort();
		}

		/**
		 * @return the job
		 */
		public T getJob() {
			return job;
		}

		/**
		 * 
		 */
		public void dispose() {
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#addServiceListener(net.sf.jame.service.spool.JobServiceListener)
	 */
	public void addServiceListener(JobServiceListener listener) {
		listeners.add(listener);
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#removeServiceListener(net.sf.jame.service.spool.JobServiceListener)
	 */
	public void removeServiceListener(JobServiceListener listener) {
		listeners.remove(listener);
	}
	
	/**
	 * @param message
	 */
	protected void fireStateChanged(String message) {
		for (JobServiceListener listener : listeners) {
			listener.stateChanged(message);
		}
	}
}
