/*
 * $Id:RenderJobService.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.service.job;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.List;

import net.sf.jame.service.DataTableException;
import net.sf.jame.service.Session;
import net.sf.jame.service.SessionException;

/**
 * @author Andrea Medeghini
 */
public class RenderJobService {
	private final RenderJobFileCache fileCache;
	private final RenderJobDataTable dataTable;

	/**
	 * @param workdir
	 * @throws IOException
	 */
	public RenderJobService(final File workdir) throws IOException {
		fileCache = new RenderJobFileCache(new File(workdir, "job"));
		dataTable = new RenderJobDataTable();
	}

	/**
	 * @param session
	 * @param job
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void create(final Session session, final RenderJobDataRow job) throws SessionException, DataTableException {
		dataTable.create(session, job);
	}

	/**
	 * @param session
	 * @param job
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void delete(final Session session, final RenderJobDataRow job) throws SessionException, DataTableException {
		dataTable.delete(session, job);
		try {
			fileCache.delete(job.getJobId());
		}
		catch (final IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param session
	 * @param job
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void save(final Session session, final RenderJobDataRow job) throws SessionException, DataTableException {
		dataTable.save(session, job);
	}

	/**
	 * @param session
	 * @param id
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public boolean validate(final Session session, final int id) throws SessionException, DataTableException {
		final RenderJobDataRow job = dataTable.load(session, id);
		return fileCache.exists(job.getJobId());
	}

	/**
	 * @param session
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public List<RenderJobDataRow> loadAll(final Session session) throws SessionException, DataTableException {
		return dataTable.loadAll(session);
	}

	/**
	 * @param session
	 * @param profileId
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public List<RenderJobDataRow> loadAll(final Session session, final int profileId) throws SessionException, DataTableException {
		return dataTable.loadAll(session, profileId);
	}

	/**
	 * @param session
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void deleteAll(final Session session) throws SessionException, DataTableException {
		final List<RenderJobDataRow> jobs = this.loadAll(session);
		for (final RenderJobDataRow job : jobs) {
			delete(session, job);
		}
	}

	/**
	 * @param session
	 * @throws SessionException
	 */
	public void cleanup(final Session session) throws SessionException {
		final File[] files = fileCache.list();
		for (final File file : files) {
			try {
				final int id = Integer.parseInt(file.getName().substring(0, file.getName().indexOf(fileCache.getSuffix())));
				dataTable.load(session, id);
			}
			catch (final NumberFormatException e) {
				if (file.exists() && !file.delete()) {
					throw new SessionException("Can't delete the file: " + file.getName());
				}
			}
			catch (final DataTableException e) {
				if (file.exists() && !file.delete()) {
					throw new SessionException("Can't delete the file: " + file.getName());
				}
			}
		}
	}

	/**
	 * @param jobId
	 * @return
	 * @throws FileNotFoundException
	 */
	public InputStream getInputStream(final int jobId) throws FileNotFoundException {
		return fileCache.getInputStream(jobId);
	}

	/**
	 * @param jobId
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getOutputStream(final int jobId) throws FileNotFoundException {
		return fileCache.getOutputStream(jobId);
	}

	/**
	 * @param jobId
	 * @param append
	 * @return
	 * @throws FileNotFoundException
	 */
	public OutputStream getOutputStream(final int jobId, final boolean append) throws FileNotFoundException {
		return fileCache.getOutputStream(jobId, append);
	}

	/**
	 * @param jobId
	 * @return
	 * @throws FileNotFoundException
	 */
	public RandomAccessFile getRandomAccessFile(final int jobId) throws FileNotFoundException {
		return fileCache.getRandomAccessFile(jobId);
	}

	/**
	 * @param session
	 * @param jobId
	 * @return
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public RenderJobDataRow getJob(final Session session, final int jobId) throws SessionException, DataTableException {
		return dataTable.load(session, jobId);
	}

	/**
	 * @param session
	 * @param profileId
	 */
	public void deleteAll(final Session session, final int profileId) throws SessionException, DataTableException {
		dataTable.deleteAll(session, profileId);
	}

	/**
	 * @param session
	 * @param job
	 * @throws SessionException
	 * @throws DataTableException
	 */
	public void saveStatus(final Session session, final RenderJobDataRow job) throws SessionException, DataTableException {
		dataTable.saveStatus(session, job);
	}

	/**
	 * @param session
	 * @param job
	 * @throws IOException
	 */
	public void clean(final Session session, final RenderJobDataRow job) throws SessionException, DataTableException, IOException {
		fileCache.delete(job.getJobId());
	}

	/**
	 * @param session
	 * @throws DataTableException
	 * @throws SessionException
	 */
	public void init(final Session session) throws DataTableException, SessionException {
		if (dataTable.updateTable(session)) {
			fileCache.deleteAll();
		}
		dataTable.init(session);
	}
}
