/*
 * $Id:AbstractSpoolJobService.java 491 2008-01-28 21:59:31Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.networking.spool;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.RandomAccessFile;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import net.sf.jame.networking.DiscoveryService;
import net.sf.jame.networking.EventMessage;
import net.sf.jame.networking.RequestIDFactory;
import net.sf.jame.networking.RequestMessage;
import net.sf.jame.networking.ResponseMessage;
import net.sf.jame.networking.ServiceEndpoint;
import net.sf.jame.networking.ServiceException;
import net.sf.jame.networking.ServiceListener;
import net.sf.jame.networking.ServiceMessage;
import net.sf.jame.networking.ServiceSession;
import net.sf.jame.networking.SessionHandler;
import net.sf.jame.service.spool.DistributedSpoolJobInterface;
import net.sf.jame.service.spool.JobEvent;
import net.sf.jame.service.spool.JobFactory;
import net.sf.jame.service.spool.JobIDFactory;
import net.sf.jame.service.spool.JobInterface;
import net.sf.jame.service.spool.JobListener;
import net.sf.jame.service.spool.JobService;
import net.sf.jame.service.spool.JobServiceListener;
import net.sf.jame.service.spool.JobStatus;
import net.sf.jame.service.spool.SpoolJobInterface;
import net.sf.jame.service.spool.impl.DistributedJobEncoder;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public abstract class AbstractSpoolJobService implements JobService<SpoolJobInterface> {
	private static final Logger logger = Logger.getLogger(AbstractSpoolJobService.class);
	private List<JobServiceListener> listeners = new LinkedList<JobServiceListener>();
	private final List<ExecutorTask> tasks = new LinkedList<ExecutorTask>();
	private final HashMap<String, ScheduledJob> scheduledJobs = new HashMap<String, ScheduledJob>();
	private final HashMap<String, DistributedSpoolJobInterface> terminatedJobs = new HashMap<String, DistributedSpoolJobInterface>();
	private final HashMap<String, DistributedSpoolJobInterface> spooledJobs = new HashMap<String, DistributedSpoolJobInterface>();
	private final DiscoveryService discoveryService;
	private final JobFactory<? extends DistributedSpoolJobInterface> jobFactory;
	private final Object dispatchMonitor = new Object();
	private final Object serviceMonitor = new Object();
	private final Object monitor = new Object();
	private Thread thread;
	private boolean running;
	private boolean dispatchDirty;
	private boolean serviceDirty;
	private boolean dirty;
	private String serviceName;

	/**
	 * @param serviceName
	 * @param discoveryService
	 * @param jobFactory
	 */
	public AbstractSpoolJobService(final String serviceName, final DiscoveryService discoveryService, final JobFactory<? extends DistributedSpoolJobInterface> jobFactory) {
		this.discoveryService = discoveryService;
		this.jobFactory = jobFactory;
		this.serviceName = serviceName;
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#start()
	 */
	public void start() {
		if (thread == null) {
			thread = new Thread(new ServiceHandler(), serviceName + " JobService Thread");
			thread.setDaemon(true);
			running = true;
			thread.start();
		}
		fireStateChanged(serviceName + " service started");
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#stop()
	 */
	public void stop() {
		if (thread != null) {
			running = false;
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
		fireStateChanged(serviceName + " service stopped");
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#getJobCount()
	 */
	public int getJobCount() {
		synchronized (spooledJobs) {
			return spooledJobs.size();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#deleteJob(java.lang.String)
	 */
	public void deleteJob(final String jobId) {
		synchronized (scheduledJobs) {
			synchronized (spooledJobs) {
				final DistributedSpoolJobInterface job = spooledJobs.remove(jobId);
				if (job != null) {
					final ScheduledJob scheduledJob = scheduledJobs.remove(jobId);
					if (scheduledJob != null) {
						if (scheduledJob.isStarted()) {
							job.abort();
							scheduledJob.stop();
						}
					}
					else {
						if (job.isStarted()) {
							job.abort();
							job.stop();
						}
					}
					job.dispose();
				}
			}
			synchronized (terminatedJobs) {
				terminatedJobs.remove(jobId);
			}
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#abortJob(java.lang.String)
	 */
	public void abortJob(final String jobId) {
		synchronized (scheduledJobs) {
			synchronized (spooledJobs) {
				final DistributedSpoolJobInterface job = spooledJobs.get(jobId);
				if (job != null) {
					final ScheduledJob scheduledJob = scheduledJobs.get(jobId);
					if (scheduledJob != null) {
						scheduledJob.abort();
					}
					else {
						job.abort();
					}
				}
			}
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#createJob(java.lang.String)
	 */
	public String createJob(final JobListener listener) {
		synchronized (spooledJobs) {
			final DistributedSpoolJobInterface job = jobFactory.createJob(JobIDFactory.newJobId(), listener);
			spooledJobs.put(job.getJobId(), job);
			return job.getJobId();
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#getJob(java.lang.String)
	 */
	public SpoolJobInterface getJob(final String jobId) {
		synchronized (spooledJobs) {
			synchronized (terminatedJobs) {
				DistributedSpoolJobInterface job = spooledJobs.get(jobId);
				if (job == null) {
					job = terminatedJobs.get(jobId);
				}
				return job;
			}
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#runJob(java.lang.String)
	 */
	public void runJob(final String jobId) {
		synchronized (scheduledJobs) {
			synchronized (spooledJobs) {
				final DistributedSpoolJobInterface job = spooledJobs.get(jobId);
				if (job != null) {
					scheduledJobs.put(jobId, new ScheduledJob(new JobSessionHandler(job)));
				}
				synchronized (dispatchMonitor) {
					dispatchDirty = true;
					dispatchMonitor.notify();
				}
			}
		}
	}

	private class ServiceHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				final long pollingTime = 30 * 1000L;
				final Thread dispatcherThread = new Thread(new DispatcherHandler(), serviceName + " JobService Dispatcher Thread");
				dispatcherThread.setDaemon(true);
				dispatcherThread.start();
				final Thread executorThread = new Thread(new ExecutorHandler(), serviceName + " JobService Executor Thread");
				executorThread.setDaemon(true);
				executorThread.start();
				discoveryService.start();
				try {
					while (running) {
						synchronized (scheduledJobs) {
							synchronized (terminatedJobs) {
								final Iterator<ScheduledJob> jobIterator = scheduledJobs.values().iterator();
								while (jobIterator.hasNext()) {
									final ScheduledJob job = jobIterator.next();
									if (job.isTerminated()) {
										terminatedJobs.put(job.getJob().getJobId(), job.getJob());
										logger.info("Job terminated " + job.getJob());
										jobIterator.remove();
										job.stop();
										job.dispose();
									}
									else if (job.isStarted() && (System.currentTimeMillis() - job.getJob().getLastUpdate()) > 20 * 1000L) {
										job.abort();
										job.stop();
										job.reset();
										job.setSession(null);
										synchronized (dispatchMonitor) {
											dispatchDirty = true;
											dispatchMonitor.notify();
										}
									}
								}
							}
						}
						synchronized (spooledJobs) {
							synchronized (terminatedJobs) {
								final Iterator<DistributedSpoolJobInterface> jobIterator = terminatedJobs.values().iterator();
								while (jobIterator.hasNext()) {
									final JobInterface job = jobIterator.next();
									if ((System.currentTimeMillis() - job.getLastUpdate()) > 120 * 1000L) {
										logger.info("Remove terminated job " + job);
										spooledJobs.remove(job.getJobId());
										jobIterator.remove();
										job.dispose();
									}
								}
							}
						}
						synchronized (serviceMonitor) {
							if (!serviceDirty) {
								serviceMonitor.wait(pollingTime);
							}
							serviceDirty = false;
						}
					}
				}
				catch (final InterruptedException e) {
				}
				discoveryService.stop();
				dispatcherThread.interrupt();
				executorThread.interrupt();
				dispatcherThread.join();
				executorThread.join();
			}
			catch (final InterruptedException e) {
			}
		}
	}

	private class DispatcherHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long pollingTime = 10 * 1000L;
			try {
				final List<ScheduledJob> jobs = new LinkedList<ScheduledJob>();
				while (running) {
					try {
						synchronized (scheduledJobs) {
							final Iterator<ScheduledJob> jobIterator = scheduledJobs.values().iterator();
							while (jobIterator.hasNext()) {
								final ScheduledJob job = jobIterator.next();
								if (!job.getJob().isAborted()) {
									if (job.getSession() == null) {
										jobs.add(job);
									}
									else {
										sendKeepAlive(job);
									}
								}
							}
						}
						dispatchJobs(jobs);
						jobs.clear();
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					synchronized (dispatchMonitor) {
						if (!dispatchDirty) {
							dispatchMonitor.wait(pollingTime);
						}
						dispatchDirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
		}

		private void dispatchJobs(final List<ScheduledJob> jobs) {
			final List<ServiceEndpoint> services = discoveryService.getEndpoints();
			fireStateChanged(services.size() + (services.size() == 1 ? " peer available" : " peers available"));
			if (services.size() > 0) {
				Collections.shuffle(services);
				int i = 0;
				while (i < jobs.size()) {
					for (final ServiceEndpoint service : services) {
						final ScheduledJob job = jobs.get(i);
						try {
							final ServiceSession session = service.createSession(job);
							if (session != null) {
								job.setSession(session);
								job.start();
							}
						}
						catch (final Exception e) {
							e.printStackTrace();
						}
						i += 1;
						if (i >= jobs.size()) {
							break;
						}
					}
				}
			}
			else {
				logger.warn("There are no services to process the jobs");
			}
		}

		private void sendKeepAlive(final ScheduledJob job) {
			try {
				job.getSession().sendKeepAliveMessage();
			}
			catch (ServiceException e) {
				e.printStackTrace();
			}
		}
	}

	private class ExecutorHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long pollingTime = 60 * 1000L;
			final List<ExecutorTask> tasksToRun = new LinkedList<ExecutorTask>();
			try {
				while (running) {
					try {
						synchronized (tasks) {
							tasksToRun.addAll(tasks);
							tasks.clear();
						}
						for (final ExecutorTask task : tasksToRun) {
							task.run();
						}
						tasksToRun.clear();
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					synchronized (monitor) {
						if (!dirty) {
							monitor.wait(pollingTime);
						}
						dirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
		}
	}

	private class ExecutorTask implements Runnable {
		private final ServiceSession session;
		private final ServiceMessage message;

		/**
		 * @param session
		 * @param message
		 */
		public ExecutorTask(final ServiceSession session, final ServiceMessage message) {
			this.session = session;
			this.message = message;
		}

		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			try {
				logger.debug("Ready to send message " + message + " on session " + session.getSessionId());
				if (!session.isExpired()) {
					session.sendMessage(message);
				}
				else {
					logger.error("Failed to send the message " + message + " on session " + session.getSessionId());
					logger.error("Invalidate session " + session.getSessionId());
					session.invalidate();
				}
			}
			catch (final Exception e) {
				logger.error("Failed to send the message " + message + " on session " + session.getSessionId());
				logger.error("Invalidate session " + session.getSessionId());
				session.invalidate();
				e.printStackTrace();
			}
		}
	}

	private class ScheduledJob implements ServiceListener {
		private final JobSessionHandler jobSession;

		/**
		 * @param jobSession
		 */
		public ScheduledJob(final JobSessionHandler jobSession) {
			this.jobSession = jobSession;
		}

		/**
		 * @return
		 */
		public boolean isStarted() {
			return jobSession.isStarted();
		}

		/**
		 * @return
		 */
		public boolean isAborted() {
			return jobSession.isAborted();
		}

		/**
		 * @return
		 */
		public boolean isTerminated() {
			return jobSession.isTerminated();
		}

		/**
		 * 
		 */
		public void reset() {
			jobSession.reset();
		}

		/**
		 * 
		 */
		public void start() {
			jobSession.start();
		}

		/**
		 * 
		 */
		public void abort() {
			jobSession.abort();
		}

		/**
		 * 
		 */
		public void stop() {
			jobSession.stop();
		}

		/**
		 * 
		 */
		public void sendHelloRequest() {
			jobSession.sendHelloRequest();
		}

		/**
		 * 
		 */
		public void sendAbortRequest() {
			jobSession.sendAbortRequest();
		}

		/**
		 * 
		 */
		public void dispose() {
			jobSession.dispose();
		}

		/**
		 * @return
		 */
		public boolean isExpired() {
			return jobSession.isExpired();
		}

		/**
		 * @param session
		 */
		public void setSession(final ServiceSession session) {
			jobSession.setSession(session);
		}

		/**
		 * @return
		 */
		public ServiceSession getSession() {
			return jobSession.getSession();
		}

		/**
		 * @return the job
		 */
		public DistributedSpoolJobInterface getJob() {
			return jobSession.getJob();
		}

		/**
		 * @see net.sf.jame.networking.ServiceListener#onMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void onMessage(final ServiceMessage message) throws ServiceException {
			jobSession.onMessage(message);
		}
	}

	private class JobSessionHandler implements SessionHandler {
		private ServiceSession session;
		private final DistributedSpoolJobInterface job;

		/**
		 * @param job
		 */
		public JobSessionHandler(final DistributedSpoolJobInterface job) {
			this.job = job;
		}

		/**
		 * @return
		 */
		public boolean isStarted() {
			return job.isStarted();
		}

		/**
		 * @return
		 */
		public boolean isAborted() {
			return job.isAborted();
		}

		/**
		 * @return
		 */
		public boolean isTerminated() {
			return job.isTerminated();
		}

		/**
		 * 
		 */
		public void reset() {
			job.reset();
		}

		/**
		 * 
		 */
		public void start() {
			job.start();
			sendHelloRequest();
		}

		/**
		 * 
		 */
		public void abort() {
			sendAbortRequest();
			job.abort();
		}

		/**
		 * 
		 */
		public void stop() {
			// sendDeleteRequest();
			job.stop();
		}

		/**
		 * 
		 */
		public void sendHelloRequest() {
			if (isExpired()) {
				return;
			}
			synchronized (tasks) {
				try {
					final RequestMessage request = createHelloRequest();
					tasks.add(new ExecutorTask(session, request));
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * 
		 */
		public void sendAbortRequest() {
			if (isExpired()) {
				return;
			}
			synchronized (tasks) {
				try {
					final RequestMessage request = createAbortRequest(job);
					tasks.add(new ExecutorTask(session, request));
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * 
		 */
		public void sendDeleteRequest() {
			if (isExpired()) {
				return;
			}
			synchronized (tasks) {
				try {
					final RequestMessage request = createDeleteRequest(job);
					tasks.add(new ExecutorTask(session, request));
				}
				catch (final Exception e) {
					e.printStackTrace();
				}
			}
			synchronized (monitor) {
				dirty = true;
				monitor.notify();
			}
		}

		/**
		 * @return
		 */
		public DistributedSpoolJobInterface getJob() {
			return job;
		}

		/**
		 * @return
		 */
		public boolean isExpired() {
			return session == null || session.isExpired();
		}

		/**
		 * 
		 */
		public void dispose() {
			if (session != null) {
				session.dispose();
				session = null;
			}
		}

		/**
		 * @return the session
		 */
		public ServiceSession getSession() {
			return session;
		}

		/**
		 * @param session the session to set
		 */
		public synchronized void setSession(final ServiceSession session) {
			this.session = session;
		}

		/**
		 * @see net.sf.jame.p2p.ServiceListener#onMessage(net.sf.jame.p2p.ServiceMessage)
		 */
		public synchronized void onMessage(final ServiceMessage message) {
			if (isExpired()) {
				return;
			}
			logger.debug("Received message " + message + " on session " + session.getSessionId());
			try {
				switch (message.getMessageType()) {
					case ServiceMessage.MESSAGE_TYPE_RESPONSE: {
						final ResponseMessage response = (ResponseMessage) message;
						// if (response.getReturnCode() != 0) {
						// job.abort();
						// job.stop();
						// job.reset();
						// session = null;
						// }
						if (response.getReturnCode() == 0) {
							switch (response.getResponseType()) {
								case RequestMessage.TYPE_HELLO: {
									if (job.isAborted()) {
										break;
									}
									final String jobId = (String) response.getUserData();
									job.setRemoteJobId(jobId);
									synchronized (tasks) {
										final RequestMessage request = createPutRequest(job);
										tasks.add(new ExecutorTask(session, request));
									}
									synchronized (monitor) {
										dirty = true;
										monitor.notify();
									}
									break;
								}
								case RequestMessage.TYPE_PUT: {
									break;
								}
								case RequestMessage.TYPE_GET: {
									if (job.isAborted()) {
										break;
									}
									final int frameNumber = (Integer) ((Object[]) response.getUserData())[1];
									if (!session.isLocalSession()) {
										// final String jobId = (String) ((Object[]) response.getUserData())[0];
										final byte[] data = (byte[]) ((Object[]) response.getUserData())[2];
										if (data != null) {
											RandomAccessFile raf = null;
											final int tw = job.getJobDataRow().getTileWidth();
											final int th = job.getJobDataRow().getTileHeight();
											final int bw = job.getJobDataRow().getBorderWidth();
											final int bh = job.getJobDataRow().getBorderHeight();
											final int sw = tw + 2 * bw;
											final int sh = th + 2 * bh;
											final long pos = frameNumber * sw * sh * 4;
											try {
												raf = job.getRAF();
												raf.seek(pos);
												raf.write(data);
											}
											catch (final IOException e) {
												e.printStackTrace();
											}
											finally {
												if (raf != null) {
													try {
														raf.close();
													}
													catch (final IOException e) {
													}
												}
											}
										}
									}
									job.setFrameNumber(frameNumber);
									break;
								}
								case RequestMessage.TYPE_ABORT: {
									synchronized (tasks) {
										final RequestMessage request = createDeleteRequest(job);
										tasks.add(new ExecutorTask(session, request));
									}
									synchronized (monitor) {
										dirty = true;
										monitor.notify();
									}
									break;
								}
								case RequestMessage.TYPE_DELETE: {
									job.setTerminated(true);
									synchronized (serviceMonitor) {
										serviceDirty = true;
										serviceMonitor.notify();
									}
									break;
								}
								default: {
									break;
								}
							}
						}
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_REQUEST: {
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_FAILURE: {
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_EVENT: {
						final EventMessage event = (EventMessage) message;
						final JobEvent jobEvent = (JobEvent) event.getUserData();
						switch (jobEvent.getEventType()) {
							case JobEvent.EVENT_TYPE_STATUS: {
								if (job.isAborted()) {
									break;
								}
								synchronized (tasks) {
									final JobStatus status = (JobStatus) jobEvent.getEventData();
									final RequestMessage request = createGetRequest(job, status.getFrameNumber());
									tasks.add(new ExecutorTask(session, request));
								}
								synchronized (monitor) {
									dirty = true;
									monitor.notify();
								}
								break;
							}
							case JobEvent.EVENT_TYPE_START: {
								if (job.isAborted()) {
									break;
								}
								break;
							}
							case JobEvent.EVENT_TYPE_STOP: {
								if (job.isAborted()) {
									break;
								}
								synchronized (tasks) {
									final RequestMessage request = createDeleteRequest(job);
									tasks.add(new ExecutorTask(session, request));
								}
								synchronized (monitor) {
									dirty = true;
									monitor.notify();
								}
								break;
							}
							default: {
								break;
							}
						}
						break;
					}
					case ServiceMessage.MESSAGE_TYPE_KEEPALIVE: {
						break;
					}
					default: {
						break;
					}
				}
			}
			catch (final Exception e) {
				e.printStackTrace();
			}
		}

		private RequestMessage createPutRequest(final DistributedSpoolJobInterface job) throws Exception {
			final RequestMessage message = new RequestMessage();
			message.setRequestId(RequestIDFactory.newRequestId());
			message.setRequestType(RequestMessage.TYPE_PUT);
			if (!session.isLocalSession()) {
				byte[] jobData = null;
				if (job.getFrameNumber() > 0) {
					RandomAccessFile raf = job.getRAF();
					try {
						raf = job.getRAF();
						final int tw = job.getJobDataRow().getTileWidth();
						final int th = job.getJobDataRow().getTileHeight();
						final int bw = job.getJobDataRow().getBorderWidth();
						final int bh = job.getJobDataRow().getBorderHeight();
						final int sw = tw + 2 * bw;
						final int sh = th + 2 * bh;
						final byte[] data = new byte[sw * sh * 4];
						final long pos = job.getFrameNumber() * sw * sh * 4;
						raf.seek(pos);
						raf.readFully(data);
						jobData = data;
					}
					finally {
						if (raf != null) {
							raf.close();
						}
					}
				}
				final DistributedJobEncoder encoder = new DistributedJobEncoder(job.getClip(), job.getJobDataRow().getJob(), jobData);
				message.setUserData(new Object[] { job.getRemoteJobId(), job.getFrameNumber(), encoder.getBytes() });
			}
			else {
				final ByteArrayOutputStream baos = new ByteArrayOutputStream();
				final ObjectOutputStream oos = new ObjectOutputStream(baos);
				oos.writeInt(job.getJobDataRow().getJobId());
				oos.close();
				baos.close();
				message.setUserData(new Object[] { job.getRemoteJobId(), job.getFrameNumber(), baos.toByteArray() });
			}
			return message;
		}

		private RequestMessage createHelloRequest() {
			final RequestMessage message = new RequestMessage();
			message.setRequestId(RequestIDFactory.newRequestId());
			message.setRequestType(RequestMessage.TYPE_HELLO);
			return message;
		}

		private RequestMessage createGetRequest(final DistributedSpoolJobInterface job, final int frameNumber) throws Exception {
			final RequestMessage message = new RequestMessage();
			message.setRequestId(RequestIDFactory.newRequestId());
			message.setRequestType(RequestMessage.TYPE_GET);
			message.setUserData(new Object[] { job.getRemoteJobId(), frameNumber });
			return message;
		}

		private RequestMessage createAbortRequest(final DistributedSpoolJobInterface job) throws Exception {
			final RequestMessage message = new RequestMessage();
			message.setRequestId(RequestIDFactory.newRequestId());
			message.setRequestType(RequestMessage.TYPE_ABORT);
			message.setUserData(job.getRemoteJobId());
			return message;
		}

		private RequestMessage createDeleteRequest(final DistributedSpoolJobInterface job) throws Exception {
			final RequestMessage message = new RequestMessage();
			message.setRequestId(RequestIDFactory.newRequestId());
			message.setRequestType(RequestMessage.TYPE_DELETE);
			message.setUserData(job.getRemoteJobId());
			return message;
		}
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#addServiceListener(net.sf.jame.service.spool.JobServiceListener)
	 */
	public void addServiceListener(JobServiceListener listener) {
		listeners.add(listener);
	}

	/**
	 * @see net.sf.jame.service.spool.JobService#removeServiceListener(net.sf.jame.service.spool.JobServiceListener)
	 */
	public void removeServiceListener(JobServiceListener listener) {
		listeners.remove(listener);
	}

	/**
	 * @param message
	 */
	protected void fireStateChanged(String message) {
		for (JobServiceListener listener : listeners) {
			listener.stateChanged(message);
		}
	}
}
