/*
 * $Id:LocalService.java 476 2008-01-13 18:41:22Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.networking;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * @author Andrea Medeghini
 */
public class LocalService {
	private static final Logger logger = Logger.getLogger(LocalService.class);
	private final HashMap<String, ServiceSession> sessions = new HashMap<String, ServiceSession>();
	private final ServiceProcessor processor;
	private final String serviceName;
	private Thread thread;
	private boolean running;
	private boolean dirty;
	private final Object monitor = new Object();

	/**
	 * @param serviceName
	 * @param processor
	 */
	public LocalService(final String serviceName, final ServiceProcessor processor) {
		this.serviceName = "LocalService " + serviceName;
		this.processor = processor;
	}

	/**
	 * @return the name
	 */
	public String getServiceName() {
		return serviceName;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		final StringBuilder builder = new StringBuilder();
		builder.append("serviceName = ");
		builder.append(serviceName);
		return builder.toString();
	}

	/**
	 * 
	 */
	public void start() {
		processor.start();
		if (thread == null) {
			thread = new Thread(new CleanupHandler(), "LocalService Cleanup Thread");
			thread.setDaemon(true);
			running = true;
			thread.start();
		}
	}

	/**
	 * 
	 */
	public void stop() {
		processor.stop();
		if (thread != null) {
			running = false;
			thread.interrupt();
			try {
				thread.join();
			}
			catch (final InterruptedException e) {
			}
			thread = null;
		}
	}

	/**
	 * @see net.sf.jame.networking.ServiceEndpoint#createSession(net.sf.jame.networking.ServiceListener)
	 */
	public ServiceSession createSession(final ServiceListener listener) throws ServiceException {
		final SessionHandler sessionHandler = processor.createSessionHandler();
		final ServiceConsumer clientConsumer = new LocalServiceConsumer(listener);
		final ServiceConsumer serverConsumer = new LocalServiceConsumer(sessionHandler);
		final ServiceSession clientSession = new LocalServiceSession(SessionIDFactory.newSessionId(), clientConsumer, new LocalServiceProducer(serverConsumer));
		logger.info("Local client session created " + clientSession.getSessionId());
		synchronized (sessions) {
			sessions.put(clientSession.getSessionId(), clientSession);
		}
		final ServiceSession serverSession = new LocalServiceSession(SessionIDFactory.newSessionId(), serverConsumer, new LocalServiceProducer(clientConsumer));
		logger.info("Local server session created " + serverSession.getSessionId());
		sessionHandler.setSession(serverSession);
		synchronized (sessions) {
			sessions.put(serverSession.getSessionId(), serverSession);
		}
		return clientSession;
	}

	private class LocalServiceSession extends ServiceSession {
		/**
		 * @param sessionId
		 * @param consumer
		 * @param producer
		 */
		public LocalServiceSession(final String sessionId, final ServiceConsumer consumer, final ServiceProducer producer) {
			super(sessionId, consumer, producer);
		}

		/**
		 * @see net.sf.jame.networking.ServiceSession#dispose()
		 */
		@Override
		public void dispose() {
			logger.debug("Session disposed " + getSessionId());
			super.dispose();
		}

		/**
		 * @see net.sf.jame.networking.ServiceSession#isLocalSession()
		 */
		@Override
		public boolean isLocalSession() {
			return true;
		}
	}

	private class LocalServiceProducer implements ServiceProducer {
		private ServiceListener listener;

		/**
		 * @param listener
		 */
		public LocalServiceProducer(final ServiceListener listener) {
			this.listener = listener;
		}

		/**
		 * @see net.sf.jame.networking.ServiceProducer#dispose()
		 */
		public void dispose() {
			listener = null;
		}

		/**
		 * @see net.sf.jame.networking.ServiceProducer#sendMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void sendMessage(final ServiceMessage message) throws ServiceException {
			listener.onMessage(message);
		}

		/**
		 * @see net.sf.jame.networking.ServiceProducer#sendKeepAliveMessage()
		 */
		public void sendKeepAliveMessage() throws ServiceException {
			listener.onMessage(new KeepAliveMessage());
		}
	}

	private class LocalServiceConsumer implements ServiceConsumer {
		private ServiceListener listener;
		private long lastMessageTime = System.currentTimeMillis();

		/**
		 * @param listener
		 */
		public LocalServiceConsumer(final ServiceListener listener) {
			this.listener = listener;
		}

		/**
		 * @see net.sf.jame.networking.ServiceConsumer#consumeMessages()
		 */
		public void consumeMessages() throws ServiceException {
		}

		/**
		 * @see net.sf.jame.networking.ServiceConsumer#dispose()
		 */
		public void dispose() {
			listener = null;
		}

		/**
		 * @see net.sf.jame.networking.ServiceConsumer#isTimeout()
		 */
		public boolean isTimeout() {
			return (System.currentTimeMillis() - lastMessageTime) > 120 * 1000L;
		}

		/**
		 * @see net.sf.jame.networking.ServiceListener#onMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void onMessage(final ServiceMessage message) throws ServiceException {
			if (listener != null) {
				lastMessageTime = System.currentTimeMillis();
				listener.onMessage(message);
			}
		}

		/**
		 * @see net.sf.jame.networking.ServiceConsumer#consumeMessage(net.sf.jame.networking.ServiceMessage)
		 */
		public void consumeMessage(ServiceMessage message) throws ServiceException {
			if (listener != null) {
				lastMessageTime = System.currentTimeMillis();
				listener.onMessage(message);
			}
		}
	}

	private class CleanupHandler implements Runnable {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			final long pollingTime = 30 * 1000L;
			final List<ServiceSession> activeSessions = new LinkedList<ServiceSession>();
			try {
				while (running) {
					try {
						synchronized (sessions) {
							final Iterator<ServiceSession> sessionIterator = sessions.values().iterator();
							while (sessionIterator.hasNext()) {
								final ServiceSession session = sessionIterator.next();
								if (session.isExpired()) {
									logger.info("Dispose expired session " + session.getSessionId());
									sessionIterator.remove();
								}
								else {
									activeSessions.add(session);
								}
							}
						}
						for (final ServiceSession session : activeSessions) {
							try {
								session.consumeMessages();
							}
							catch (final Exception e) {
								e.printStackTrace();
								session.invalidate();
							}
						}
						activeSessions.clear();
					}
					catch (final Exception e) {
						e.printStackTrace();
					}
					synchronized (monitor) {
						if (!dirty) {
							monitor.wait(pollingTime);
						}
						dirty = false;
					}
				}
			}
			catch (final InterruptedException e) {
			}
			synchronized (sessions) {
				for (final ServiceSession session : sessions.values()) {
					logger.info("Dispose session " + session.getSessionId());
					session.dispose();
				}
				sessions.clear();
			}
		}
	}
}
