/*
 * $Id:Pipeline.java 456 2008-01-05 21:56:57Z andreamedeghini $
 *
 * JAME is a Java real-time multi-thread fractal graphics platform
 * Copyright (C) 2001, 2008 Andrea Medeghini
 * andreamedeghini@users.sf.net
 * http://jame.sourceforge.net
 * http://sourceforge.net/projects/jame
 * http://jame.dev.java.net
 * http://jugbrescia.dev.java.net
 *
 * This file is part of JAME.
 *
 * JAME is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JAME is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with JAME.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package net.sf.jame.media;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.util.LinkedList;

import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.LineUnavailableException;

public final class Pipeline {
	private static final boolean debug = false;
	private Graphics2D graphics;
	private final Color color;
	private Effect effect;
	private final Context context;
	private final LinkedList events;
	private float mouse_x = 0;
	private float mouse_y = 0;
	private boolean mouse_pressed = false;

	public Pipeline(final Context context, final Color color) {
		this.color = color;
		this.context = context;
		events = new LinkedList();
		try {
			AudioSystem.getMixer(null).open();
			if (context.debug()) {
				context.println("mixer opened");
			}
		}
		catch (final LineUnavailableException e) {
			e.printStackTrace();
		}
	}

	public void kill() {
		AudioSystem.getMixer(null).close();
		if (context.debug()) {
			context.println("mixer closed");
		}
	}

	public void enqueueEvent(final EngineEvent event) {
		synchronized (events) {
			events.addLast(event);
		}
	}

	private void processEvents() {
		synchronized (events) {
			while (events.size() > 0) {
				final Object object = events.removeFirst();
				if (object instanceof EngineMouseEvent) {
					final EngineMouseEvent event = (EngineMouseEvent) object;
					// if (debug)
					// {
					// System.out.println(event);
					// }
					switch (event.event) {
						case EngineMouseEvent.PRESSED: {
							mouse_pressed = true;
							break;
						}
						case EngineMouseEvent.RELEASED: {
							mouse_pressed = false;
							break;
						}
						case EngineMouseEvent.MOVED: {
							mouse_x = event.mouse.x;
							mouse_y = event.mouse.y;
							break;
						}
						default:
							break;
					}
				}
			}
		}
	}

	public void rendering(final Graphics2D graphics, final int w, final int h, final Movie movie) {
		this.graphics = graphics;
		graphics.setPaint(color);
		graphics.fillRect(0, 0, w, h);
		effect = null;
		this.rendering(movie);
		processEvents();
	}

	private void rendering(final Movie movie) {
		// if (context.debug() && debug)
		// {
		// if (movie.getParent() != null)
		// {
		// System.out.println(movie.getParent().toString() + "->" + movie.toString());
		// }
		// else
		// {
		// System.out.println("root->" + movie.toString());
		// }
		// }
		final Timeline timeline = movie.getTimeline();
		final AffineTransform old_transform = graphics.getTransform();
		final AffineTransform transform = movie.getTransform();
		Layer layer;
		Sequence sequence;
		Effect old_effect;
		AbstractObject object;
		Shape old_clip;
		Shape clip;
		if (transform != null) {
			graphics.transform(transform);
		}
		for (int i = timeline.getLayers() - 1; i >= 0; i--) {
			layer = timeline.getLayer(i);
			old_clip = graphics.getClip();
			clip = layer.getClip();
			if (clip != null) {
				graphics.clip(clip);
			}
			if (layer.getSequences() > 0) {
				sequence = layer.getSequence();
				if (sequence != null) {
					old_effect = effect;
					if (sequence instanceof GraphicsSequence) {
						if (effect != null) {
							effect = effect.add(((GraphicsSequence) sequence).getEffect());
						}
						else {
							effect = ((GraphicsSequence) sequence).getEffect();
						}
					}
					// if (context.debug() && debug && (effect != null))
					// {
					// System.out.println(effect.toString());
					// }
					object = sequence.getObject();
					if (object instanceof AbstractShape) {
						this.rendering((AbstractShape) object);
					}
					else if (object instanceof AbstractImage) {
						this.rendering((AbstractImage) object);
					}
					else if (object instanceof Movie) {
						this.rendering((Movie) object);
					}
					else if (object instanceof AbstractButton) {
						this.rendering((AbstractButton) object);
					}
					effect = old_effect;
				}
			}
			graphics.setClip(old_clip);
		}
		graphics.setTransform(old_transform);
	}

	private void rendering(final AbstractShape shape) {
		// if (context.debug() && debug)
		// {
		// if (shape.getParent() != null)
		// {
		// System.out.println(shape.getParent().toString() + "->" + shape.toString());
		// }
		// else
		// {
		// System.out.println("null->" + shape.toString());
		// }
		// }
		final Shape path = shape.getShape();
		if (path != null) {
			if (effect != null) {
				shape.applyEffect(effect);
			}
			final AffineTransform old_transform = graphics.getTransform();
			final Stroke old_stroke = graphics.getStroke();
			final Paint old_paint = graphics.getPaint();
			final Color old_color = graphics.getColor();
			final AffineTransform transform = shape.getTransform();
			final Stroke stroke = shape.getStroke();
			final Paint paint1 = shape.getPaint1();
			final Paint paint2 = shape.getPaint2();
			if (transform != null) {
				graphics.transform(transform);
			}
			if (stroke != null) {
				graphics.setStroke(stroke);
			}
			if (paint2 != null) {
				graphics.setPaint(paint2);
				graphics.fill(path);
			}
			if (paint1 != null) {
				graphics.setPaint(paint1);
				graphics.draw(path);
			}
			graphics.setTransform(old_transform);
			graphics.setStroke(old_stroke);
			graphics.setPaint(old_paint);
			graphics.setPaint(old_color);
		}
	}

	private void rendering(final AbstractImage image) {
		// if (context.debug() && debug)
		// {
		// if (image.getParent() != null)
		// {
		// System.out.println(image.getParent().toString() + "->" + image.toString());
		// }
		// else
		// {
		// System.out.println("null->" + image.toString());
		// }
		// }
		final BufferedImage img = image.getImage();
		if (img != null) {
			if (image instanceof AbstractRenderedImage) {
				final Renderer renderer = ((AbstractRenderedImage) image).getRenderer();
				if (renderer != null) {
					renderer.render();
				}
			}
			if (effect != null) {
				image.applyEffect(effect);
			}
			final AffineTransform old_transform = graphics.getTransform();
			final AffineTransform transform = image.getTransform();
			if (transform != null) {
				graphics.transform(transform);
			}
			graphics.drawImage(img, null, null);
			graphics.setTransform(old_transform);
		}
	}

	private void rendering(final AbstractButton button) {
		// if (context.debug() && debug)
		// {
		// if (button.getParent() != null)
		// {
		// System.out.println(button.getParent().toString() + "->" + button.toString());
		// }
		// else
		// {
		// System.out.println("null->" + button.toString());
		// }
		// }
		try {
			final AbstractShape shape = button.getButtonShape();
			if (shape != null) {
				final AffineTransform old_transform = graphics.getTransform();
				final AffineTransform transform = button.getTransform();
				if (transform != null) {
					graphics.transform(transform);
				}
				final AffineTransform inverse = graphics.getTransform().createInverse();
				final Shape hit = button.getHit();
				if (hit != null) {
					synchronized (events) {
						final ButtonHandler handler = button.getHandler();
						final Point2D.Float mouse = new Point2D.Float(mouse_x, mouse_y);
						inverse.transform(mouse, mouse);
						if (hit.contains(mouse.x, mouse.y)) {
							if (!mouse_pressed) {
								button.actived = true;
							}
							if (!button.entered) {
								if (handler != null) {
									handler.process(new EngineButtonEvent(EngineButtonEvent.ENTERED, inverse, mouse));
								}
								button.entered = true;
							}
							if ((mouse_pressed) && (button.actived) && (!button.pressed)) {
								if (handler != null) {
									handler.process(new EngineButtonEvent(EngineButtonEvent.PRESSED, inverse, mouse));
								}
								button.pressed = true;
							}
							else if ((!mouse_pressed) && (button.actived) && (button.pressed)) {
								if (handler != null) {
									handler.process(new EngineButtonEvent(EngineButtonEvent.RELEASED, inverse, mouse));
								}
								button.pressed = false;
							}
						}
						else {
							if (!mouse_pressed) {
								button.actived = false;
							}
							button.pressed = false;
							if (button.entered) {
								if (handler != null) {
									handler.process(new EngineButtonEvent(EngineButtonEvent.EXITED, inverse, mouse));
								}
								button.entered = false;
							}
						}
					}
				}
				this.rendering(shape);
				graphics.setTransform(old_transform);
			}
		}
		catch (final NoninvertibleTransformException e) {
		}
	}
}
